/* $Id: pk_access.c,v 1.2 1997/12/23 16:47:32 gordo Exp $ */
/*
 * Simple pf_key interface for use as a library.
 *
 * Copyright (C) 1997, Gordon Oliver
 *
 * No warranty whatsoever is offered for this software.
 *
 * Permission to use, modify, or do whatever you wish with
 * this software is hereby granted, so long as you do not
 * remove this copyright notice, and so long as you do not
 * hold me responsible for any damage that it might cause.
 * You must also add a notice that you changed this software.
 *
 * This is _not_ the GNU GPL. You can sell this if you so choose.
 * You do _not_ have to include source for this work or
 * derived works.
 *
 * I would appreciate credit in which you use this software or
 * the linux PF_KEY interface.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <assert.h>

#include <pk_access.h>

struct pk_message *
pkmsg_new()
{
    struct pk_message *new;
    static int seq = 1024;

    new = malloc(sizeof *new);
    if (!new)
	return NULL;
    memset(new, 0, sizeof *new);
    new->iov_size = 10;
    new->iov = malloc(10 * sizeof new->iov[0]);
    if (!new->iov)
    {
	free(new);
	return NULL;
    }
    new->iov_len = 1;
    new->iov[0].iov_base = &new->hdr;
    new->iov[0].iov_len = sizeof new->hdr;
    new->hdr.sadb_msg_version = PF_KEY_V2;
    new->hdr.sadb_msg_seq = seq++;
    new->hdr.sadb_msg_pid = getpid();
    new->flags = 0;
    return new;
}

static inline int
pkmsg_putiov(struct pk_message *pm, void *base, int size)
{
    if (pm->iov_len == pm->iov_size)
    {
	struct iovec *niov;
	int nsize;

	nsize = pm->iov_size + (pm->iov_size>>1);
	niov = realloc(pm->iov, nsize * sizeof niov[0]);
	if (!niov)
	    return -1;	/* XXX: maybe die */
	pm->iov_size = nsize;
	pm->iov = niov;
    }
    pm->iov[pm->iov_len].iov_base = base;
    pm->iov[pm->iov_len++].iov_len = size;
    return 0;
}

struct sadb_ext *
pkmsg_add(struct pk_message *pm, int type, int size)
{
    struct sadb_ext **ptr;

    if (pm->flags & 1)
	return NULL;
    if (size % 8)
	return NULL;
    if (type <= SADB_EXT_MAX)
	ptr = &pm->n_ex[type];
    else
    {
	int i;

	for (i = 0; pm->x_ex[i] && i < 50; i++)
	    if (pm->x_ex[i]->sadb_ext_type == type)
		return NULL;
	ptr = &pm->x_ex[i];
    }
    if (*ptr)
	return NULL;
    *ptr = malloc(size);
    if (!*ptr)
	return NULL;
    if (pkmsg_putiov(pm, *ptr, size))
    {
	free(*ptr);
	*ptr = NULL;
	return NULL;
    }
    memset((*ptr), 0, size);
    (*ptr)->sadb_ext_type = type;
    (*ptr)->sadb_ext_len = size/8;
    return *ptr;
}

int
pkmsg_send(int s, struct pk_message *pm)
{
    struct msghdr msg;
    int i, tot_len;
    int count;

    tot_len = 0;
    for (i = 0; i < pm->iov_len; i++)
	tot_len += pm->iov[i].iov_len;
    pm->hdr.sadb_msg_len = tot_len/8;
    memset(&msg, 0, sizeof msg);
    msg.msg_iov = pm->iov;
    msg.msg_iovlen = pm->iov_len;
    count = sendmsg(s, &msg, 0);
    if (count != tot_len)
	return -1;	/* XXX not really very good*/
    return count;
}

struct pk_message *
pkmsg_receive(int s)
{
    char *buf;
    struct iovec iov;
    struct pk_message *new;
    struct msghdr msg;
    struct sadb_ext *se;
    int tot_len, len;
    int count;

    new = malloc(sizeof *new);
    if (!new)
	return NULL;
    memset(new, 0, sizeof new[0]);
    memset(&msg, 0, sizeof msg);
    memset(&iov, 0, sizeof iov);
    iov.iov_base = &new->hdr;
    iov.iov_len = sizeof new->hdr;
    msg.msg_iov = &iov;
    msg.msg_iovlen = 1;
    count = recvmsg(s, &msg, 0);
    if (count != sizeof new->hdr)
    {
	free(new);
	return NULL;
    }
    new->iov = malloc(2 * sizeof new->iov[0]);
    new->iov_size = 2;
    pkmsg_putiov(new, &new->hdr, sizeof new->hdr);
    tot_len = new->hdr.sadb_msg_len * 8 - sizeof new->hdr;
    buf = malloc(tot_len);
    new->alloced = buf;
    if (!buf)
    {
	/* XXX die nazi scum */abort();
    }

    if (tot_len == 0)
	return new;
    memset(&msg, 0, sizeof msg);
    memset(&iov, 0, sizeof iov);
    iov.iov_base = buf;
    iov.iov_len = tot_len;
    msg.msg_iov = &iov;
    msg.msg_iovlen = 1;
    count = recvmsg(s, &msg, 0);
    if (count != tot_len)
	/* XXX die nazi scum */(void)NULL;
    count = 0;
    len = 0;
    while (len < tot_len)
    {
	se = (struct sadb_ext *)(buf + len);
	if (se->sadb_ext_type <= SADB_EXT_MAX)
	{
 	    assert(!new->n_ex[se->sadb_ext_type]);
	    new->n_ex[se->sadb_ext_type] = se;
	}
	else
	{
	    int i;

	    for (i = 0; new->x_ex[i] && i < 50; i++)
		assert(new->x_ex[i]->sadb_ext_type != se->sadb_ext_type);
	    new->x_ex[i] = se;
	}
	pkmsg_putiov(new, se, se->sadb_ext_len * 8);
	len += se->sadb_ext_len * 8;
	count += 1;
    }
    new->flags = 1;
    return new;
}

void
pkmsg_free(struct pk_message *pm)
{
    int i;

    if (pm->flags & 1)
    {
	if (pm->alloced)
	    free(pm->alloced);
    }
    else
    {
	for (i = 0; i <= SADB_EXT_MAX; i++)
	    if (pm->n_ex[i])
		free(pm->n_ex[i]);
	for (i = 0; i < 50 && pm->x_ex[i]; i++)
	    free(pm->x_ex[i]);
    }
    if (pm->iov)
	free(pm->iov);
    free(pm);
}

char *req_names[] =
{
	"Reserved",	/* SADB_RESERVED */
	"GetSPI",	/* SADB_GETSPI */
	"Update",	/* SADB_UPDATE */
	"Add",		/* SADB_ADD */
	"Delete",	/* SADB_DELETE */
	"Get",		/* SADB_GET */
	"Acquire",	/* SADB_ACQUIRE */
	"Register",	/* SADB_REGISTER */
	"Expire",	/* SADB_EXPIRE */
	"Flush",	/* SADB_FLUSH */
	"Dump",		/* SADB_DUMP */
};

char *ext_names[] =
{
	"Reserved",		/* SADB_EXT_RESERVED */
	"Sa",			/* SADB_EXT_SA */
	"CurrentLife",		/* SADB_EXT_LIFETIME_CURRENT */
	"HardLife",		/* SADB_EXT_LIFETIME_HARD */
	"SoftLife",		/* SADB_EXT_LIFETIME_SOFT */
	"SrcAddr",		/* SADB_EXT_ADDRESS_SRC */
	"DestAddr",		/* SADB_EXT_ADDRESS_DST */
	"ProxyAddr",		/* SADB_EXT_ADDRESS_PROXY */
	"AuthKey",		/* SADB_EXT_KEY_AUTH */
	"EncryptKey",		/* SADB_EXT_KEY_ENCRYPT */
	"SrcIdent",		/* SADB_EXT_IDENTITY_SRC */
	"DestIdent",		/* SADB_EXT_IDENTITY_DST */
	"Sensitivity",		/* SADB_EXT_SENSITIVITY */
	"Proposal",		/* SADB_EXT_PROPOSAL */
	"SupportAuth",		/* SADB_EXT_SUPPORTED_AUTH */
	"SupportEnc",		/* SADB_EXT_SUPPORTED_ENCRYPT */
	"SPIRange",		/* SADB_EXT_SPIRANGE */
};


char *sa_type_names[] =
{
	"Unspecified",		/* SADB_SATYPE_UNSPEC*/
	"AH",			/* SADB_SATYPE_AH*/
	"ESP",			/* SADB_SATYPE_ESP*/
	"RSVP",			/* SADB_SATYPE_RSVP*/
	"OSPFV2",		/* SADB_SATYPE_OSPFV2*/
	"RIPV2",		/* SADB_SATYPE_RIPV2*/
	"Mobile-IP",		/* SADB_SATYPE_MIP*/
};

char *state_names[] =
{
	"Larval", "Mature", "Dying", "Dead"
};

char *auth_names[] =
{
	"None",
	"MD5-HMAC",
	"SHA1-HMAC",
	"MD5-KPDK",
	"DES_MAC"
};

char *encrypt_names[] =
{
	"None",
	"DES-CBC",
	"3DES-CBC",
	"RC5",
	"IDEA",
	"CAST",
	"BLOWFISH",
	"3IDEA",
	"ARCFOUR",
	"DES-IV64",
	"DES-IV32",
};

char *family_names[] =
{
	"Unspecified",
	"Unix",
	"Inet",
	"AX25",
	"Ipx",
	"AppleTalk",
	"NetRom",
	"Bridge",
	"AAL5",
	"X25",
};

#define do_prt(s, v, n)						\
			    if ((v) < sizeof (n)/sizeof (n)[0])	\
				printf("%s: %s\n", s, (n)[v]);	\
			    else				\
				printf("%s: Unknown(%d)\n", s, v)
void
pkmsg_print(struct pk_message *pm)
{
    int i;

    do_prt("Type", pm->hdr.sadb_msg_type, req_names);
    if (pm->hdr.sadb_msg_errno)
	printf("Error: %d\n", pm->hdr.sadb_msg_errno);
    do_prt("Domain", pm->hdr.sadb_msg_satype, sa_type_names);
    printf("Sequence: %d\n", pm->hdr.sadb_msg_seq);
    printf("PID: %d\n", pm->hdr.sadb_msg_pid);
    for (i = 0; i <= SADB_EXT_MAX; i++)
	if (pm->n_ex[i])
	{
	    do_prt("Extension:", pm->n_ex[i]->sadb_ext_type, ext_names);
	    switch (pm->n_ex[i]->sadb_ext_type)
	    {
	    case SADB_EXT_RESERVED:
		break;
	    case SADB_EXT_SA:
		{
		    struct sadb_sa *sa = (struct sadb_sa *) pm->n_ex[i];

		    printf("\tSPI: %lx\n", ntohl(sa->sadb_sa_spi));
		    printf("\tReplay: %d\n", sa->sadb_sa_replay);
		    do_prt("\tState:", sa->sadb_sa_state, state_names);
		    do_prt("\tAuth:", sa->sadb_sa_auth, auth_names);
		    do_prt("\tEncrypt:", sa->sadb_sa_encrypt, encrypt_names);
		    printf("\tFlags: %x\n", sa->sadb_sa_flags);
		}
		break;
	    case SADB_EXT_LIFETIME_CURRENT:
	    case SADB_EXT_LIFETIME_HARD:
	    case SADB_EXT_LIFETIME_SOFT:
		{
		    struct sadb_lifetime *lf;

		    lf = (struct sadb_lifetime *) pm->n_ex[i];
		    printf("\tAlloc: %d\n",(int)lf->sadb_lifetime_allocations);
		    printf("\tBytes: %u\n",(__u32)lf->sadb_lifetime_bytes);
		    printf("\tAddTime: %u\n",
			   (__u32)lf->sadb_lifetime_addtime);
		    printf("\tUseTime: %u\n",
			   (__u32)lf->sadb_lifetime_usetime);
		}
		break;
	    case SADB_EXT_ADDRESS_SRC:
	    case SADB_EXT_ADDRESS_DST:
	    case SADB_EXT_ADDRESS_PROXY:
		{
		    struct sockaddr *saddr;

		    saddr = (struct sockaddr *)((char *)pm->n_ex[i]
						+ sizeof(struct sadb_address));
		    do_prt("\tFamily:", saddr->sa_family, family_names);
		    switch (saddr->sa_family)
		    {
		    case AF_INET:
			{
			    struct sockaddr_in *sin=(struct sockaddr_in *)saddr;

			    printf("\tAddr: %s\n", inet_ntoa(sin->sin_addr));
			    printf("\tPort: %hd\n", ntohs(sin->sin_port));
			}
		    default:
		    }
		}
		break;
	    case SADB_EXT_KEY_AUTH:
	    case SADB_EXT_KEY_ENCRYPT:
		break;
	    case SADB_EXT_IDENTITY_SRC:
	    case SADB_EXT_IDENTITY_DST:
		break;
	    case SADB_EXT_SENSITIVITY:
		break;
	    case SADB_EXT_PROPOSAL:
		{
		    struct sadb_prop *prop;
		    struct sadb_comb *comb;
		    int count;

		    prop = (struct sadb_prop *)pm->n_ex[i];
		    count = ((prop->sadb_prop_len * 8 - sizeof prop[0])
			     / sizeof comb[0]);
		    comb = (struct sadb_comb *)&prop[1];
		    printf("\tReplay: %d", (int)prop->sadb_prop_replay);
		    for (i = 0; i < count; i++)
		    {
			printf("\t#%d:\n", i);
		        do_prt("\t\tAuth:", comb[i].sadb_comb_auth,auth_names);
			printf("\t\t\tMin Key: %d\n",
				comb[i].sadb_comb_auth_minbits);
			printf("\t\t\tMax Key: %d\n",
				comb[i].sadb_comb_auth_maxbits);
		        do_prt("\t\tEncrypt:", comb[i].sadb_comb_encrypt,
						encrypt_names);
			printf("\t\t\tMin Key: %d\n",
				comb[i].sadb_comb_auth_minbits);
			printf("\t\t\tMax Key: %d\n",
				comb[i].sadb_comb_auth_maxbits);
			printf("\t\tFlags: %08x\n", comb[i].sadb_comb_flags);
			printf("\t\tSoft: %u Packets, %u Bytes, %u Time, %u Used\n", 
				(unsigned int) comb[i].sadb_comb_soft_allocations,
				(unsigned int) comb[i].sadb_comb_soft_bytes,
				(unsigned int) comb[i].sadb_comb_soft_addtime,
				(unsigned int) comb[i].sadb_comb_soft_usetime);
			printf("\t\tHard: %u Packets, %u Bytes, %u Time, %u Used\n", 
				(unsigned int) comb[i].sadb_comb_hard_allocations,
				(unsigned int) comb[i].sadb_comb_hard_bytes,
				(unsigned int) comb[i].sadb_comb_hard_addtime,
				(unsigned int) comb[i].sadb_comb_hard_usetime);
		    }
		}
		break;
	    case SADB_EXT_SUPPORTED_AUTH:
		{
		    struct sadb_supported *sup;
		    struct sadb_alg *algs;
		    int count, j;

		    sup = (struct sadb_supported *)pm->n_ex[i];
		    algs = (struct sadb_alg *)&sup[1];
		    count = ((sup->sadb_supported_len * 8 - sizeof sup[0])
			     / sizeof algs[0]);
		    for (j = 0; j < count; j++)
		    {
			printf("\t#%d:\n", j);
		        do_prt("\t\tAuth:", algs[j].sadb_alg_id,auth_names);
			printf("\t\tIV Len: %d\n", algs[j].sadb_alg_ivlen);
			printf("\t\tMin Key: %d\n", algs[j].sadb_alg_minbits);
			printf("\t\tMax Key: %d\n", algs[j].sadb_alg_maxbits);
		    }
		}
		break;
	    case SADB_EXT_SUPPORTED_ENCRYPT:
		{
		    struct sadb_supported *sup;
		    struct sadb_alg *algs;
		    int count, j;

		    sup = (struct sadb_supported *)pm->n_ex[i];
		    algs = (struct sadb_alg *)&sup[1];
		    count = ((sup->sadb_supported_len * 8 - sizeof sup[0])
			     / sizeof algs[0]);
		    for (j = 0; j < count; j++)
		    {
			printf("\t#%d:\n", j);
		        do_prt("\t\tEncrypt:",
				algs[j].sadb_alg_id, encrypt_names);
			printf("\t\tIV Len: %d\n", algs[j].sadb_alg_ivlen);
			printf("\t\tMin Key: %d\n", algs[j].sadb_alg_minbits);
			printf("\t\tMax Key: %d\n", algs[j].sadb_alg_maxbits);
		    }
		}
		break;
	    case SADB_EXT_SPIRANGE:
		{
		    struct sadb_spirange *sr;

		    sr = (struct sadb_spirange *) pm->n_ex[i];
		    printf("\tMinimum: %lx\n", ntohl(sr->sadb_spirange_min));
		    printf("\tMaximum: %lx\n", ntohl(sr->sadb_spirange_max));
		}
		break;
	    }
	}
    for (i = 0; pm->x_ex[i] && i <= 50; i++)
    {
	do_prt("Extension:", pm->x_ex[i]->sadb_ext_type, ext_names);
    }
}

/* EOF $Id: pk_access.c,v 1.2 1997/12/23 16:47:32 gordo Exp $ */
