/* $Id: ipsec_access.c,v 1.1 1997/12/02 19:25:30 gordo Exp $ */
/*
 * Simple ipsec interface for use as a library.
 *
 * Copyright (C) 1997, Gordon Oliver
 *
 * No warranty whatsoever is offered for this software.
 *
 * Permission to use, modify, or do whatever you wish with
 * this software is hereby granted, so long as you do not
 * remove this copyright notice, and so long as you do not
 * hold me responsible for any damage that it might cause.
 * You must also add a notice that you changed this software.
 *
 * This is _not_ the GNU GPL. You can sell this if you so choose.
 * You do _not_ have to include source for this work or
 * derived works.
 *
 * I would appreciate credit in which you use this software or
 * the linux PF_KEY interface.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <netinet/in.h>

#include "pk_access.h"

int
ipsec_dump(int s, int type)
{
    struct pk_message *pm;
    int count;

    pm = pkmsg_new();
    pm->hdr.sadb_msg_type = SADB_DUMP;
    pm->hdr.sadb_msg_satype = type;
    count = pkmsg_send(s, pm);
    pkmsg_free(pm);	/* or return it */
    if (count < 0)
	return -1;
    return 0;
}

int
ipsec_register(int s, int type)
{
    struct pk_message *pm;
    int count;

    pm = pkmsg_new();
    pm->hdr.sadb_msg_type = SADB_REGISTER;
    pm->hdr.sadb_msg_satype = type;
    count = pkmsg_send(s, pm);
    pkmsg_free(pm);	/* or return it */
    if (count < 0)
	return -1;
    return 0;
}

int
ipsec_flush(int s, int type)
{
    struct pk_message *pm;
    int count;

    pm = pkmsg_new();
    pm->hdr.sadb_msg_type = SADB_FLUSH;
    pm->hdr.sadb_msg_satype = type;
    count = pkmsg_send(s, pm);
    pkmsg_free(pm);	/* or return it */
    if (count < 0)
	return -1;
    return 0;
}

static int
do_addressed(int s, struct pk_message *pm, struct sockaddr *sa, struct sockaddr *da, struct sockaddr *pa)
{
    struct sadb_address *saddr;
    struct sadb_ext *se;
    int count;

    if (sa)
    {
	se = pkmsg_add(pm, SADB_EXT_ADDRESS_SRC,
		       sizeof(struct sadb_address) + sizeof sa[0]);
	saddr = (struct sadb_address *)se;
	saddr->sadb_address_proto = AF_INET;
	memcpy(&saddr[1], sa, sizeof *sa);
    }

    if (da)
    {
	se = pkmsg_add(pm, SADB_EXT_ADDRESS_DST,
		       sizeof(struct sadb_address) + sizeof da[0]);
	saddr = (struct sadb_address *)se;
	saddr->sadb_address_proto = AF_INET;
	memcpy(&saddr[1], da, sizeof *da);
    }

    if (pa)
    {
	se = pkmsg_add(pm, SADB_EXT_ADDRESS_PROXY,
		       sizeof(struct sadb_address) + sizeof pa[0]);
	saddr = (struct sadb_address *)se;
	saddr->sadb_address_proto = AF_INET;
	memcpy(&saddr[1], pa, sizeof *pa);
    }

    count = pkmsg_send(s, pm);
    pkmsg_free(pm);
    if (count < 0)
	return -1;
    return 0;
}

int
ipsec_acquire(int s, int type, struct sockaddr *sa, struct sockaddr *da, struct sockaddr *pa)
{
    struct pk_message *pm;
    struct sadb_ext *se;

    pm = pkmsg_new();
    pm->hdr.sadb_msg_type = SADB_ACQUIRE;
    pm->hdr.sadb_msg_satype = type;

    se = pkmsg_add(pm, SADB_EXT_PROPOSAL,
		   sizeof(struct sadb_prop) + 2*sizeof(struct sadb_comb));
    /* need to fill in the possibilities */

    return do_addressed(s, pm, sa, da, pa);
}

int
ipsec_delete(int s, int type, __u32 spi, struct sockaddr *sa, struct sockaddr *da, struct sockaddr *pa)
{
    struct pk_message *pm;
    struct sadb_ext *se;
    struct sadb_sa *sasa;

    pm = pkmsg_new();
    pm->hdr.sadb_msg_type = SADB_DELETE;
    pm->hdr.sadb_msg_satype = type;

    se = pkmsg_add(pm, SADB_EXT_SA, sizeof(struct sadb_sa));
    sasa = (struct sadb_sa *)se;
    sasa->sadb_sa_spi = htonl(spi);

    return do_addressed(s, pm, sa, da, pa);
}

int
ipsec_get(int s, int type, __u32 spi, struct sockaddr *sa, struct sockaddr *da, struct sockaddr *pa)
{
    struct pk_message *pm;
    struct sadb_ext *se;
    struct sadb_sa *sasa;

    pm = pkmsg_new();
    pm->hdr.sadb_msg_type = SADB_GET;
    pm->hdr.sadb_msg_satype = type;

    se = pkmsg_add(pm, SADB_EXT_SA, sizeof(struct sadb_sa));
    sasa = (struct sadb_sa *)se;
    sasa->sadb_sa_spi = htonl(spi);

    return do_addressed(s, pm, sa, da, pa);
}


int
ipsec_getspi(int s, int type, struct sockaddr *sa, struct sockaddr *da, struct sockaddr *pa)
{
    struct pk_message *pm;
    struct sadb_ext *se;
    struct sadb_spirange *spir;

    pm = pkmsg_new();
    pm->hdr.sadb_msg_type = SADB_GETSPI;
    pm->hdr.sadb_msg_satype = type;

    se = pkmsg_add(pm, SADB_EXT_SPIRANGE, sizeof(struct sadb_spirange));
    spir = (struct sadb_spirange *)se;
    spir->sadb_spirange_min = 0;
    spir->sadb_spirange_max = 0xffffffff;

    return do_addressed(s, pm, sa, da, pa);
}

int
ipsec_make_key(struct pk_message *pm, int ktype, char *kmat, int ksize)
{
    int mesg_size;
    struct sadb_ext *se;
    struct sadb_key *sk;

    mesg_size = (ksize + 63) >> 6;	/* multiple of 64 */
    if (ksize & 7)
	fprintf(stderr, "key sizes not a multiple of 8 are broken");
    se = pkmsg_add(pm, ktype, sizeof(struct sadb_key) + mesg_size * 8);
    sk = (struct sadb_key *)se;
    if (!se)
	return -1;
    memcpy(&sk[1], kmat, ksize>>8);
    sk->sadb_key_bits = ksize;
    return 0;
}

int
ipsec_update_ah(int s, __u32 spi, int algo,
	        struct sockaddr *sa, struct sockaddr *da, struct sockaddr *pa,
	        char *kmat, int ksize)
{
    struct pk_message *pm;
    struct sadb_ext *se;
    struct sadb_sa *sasa;

    pm = pkmsg_new();
    pm->hdr.sadb_msg_type = SADB_UPDATE;
    pm->hdr.sadb_msg_satype = SADB_SATYPE_AH;

    se = pkmsg_add(pm, SADB_EXT_SA, sizeof(struct sadb_sa));
    if (!se)
	goto err_out;
    sasa = (struct sadb_sa *)se;
    sasa->sadb_sa_spi = htonl(spi);
    sasa->sadb_sa_auth = algo;
    sasa->sadb_sa_state = SADB_SASTATE_MATURE;

    if (ipsec_make_key(pm, SADB_EXT_KEY_AUTH, kmat, ksize))
	goto err_out;

    return do_addressed(s, pm, sa, da, pa);

err_out:
    pkmsg_free(pm);
    return -1;
}

int
ipsec_update_esp(int s, __u32 spi, int algo,
	         struct sockaddr *sa, struct sockaddr *da, struct sockaddr *pa,
	         char *esp_kmat, int esp_ksize,
	         char *ah_kmat, int ah_ksize)
{
    struct pk_message *pm;
    struct sadb_ext *se;
    struct sadb_sa *sasa;

    pm = pkmsg_new();
    pm->hdr.sadb_msg_type = SADB_UPDATE;
    pm->hdr.sadb_msg_satype = SADB_SATYPE_AH;

    se = pkmsg_add(pm, SADB_EXT_SA, sizeof(struct sadb_sa));
    if (!se)
	return -1;
    sasa = (struct sadb_sa *)se;
    sasa->sadb_sa_spi = htonl(spi);
    sasa->sadb_sa_auth = algo;
    sasa->sadb_sa_state = SADB_SASTATE_MATURE;

    if (ipsec_make_key(pm, SADB_EXT_KEY_ENCRYPT, esp_kmat, esp_ksize))
	goto err_out;
    if (ipsec_make_key(pm, SADB_EXT_KEY_AUTH, ah_kmat, ah_ksize))
	goto err_out;

    return do_addressed(s, pm, sa, da, pa);

err_out:
    pkmsg_free(pm);
    return -1;
}
/* EOF $Id: ipsec_access.c,v 1.1 1997/12/02 19:25:30 gordo Exp $ */
