/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)es_timers.c	1.6 96/04/02 Sun Microsystems"

/*
 * System includes
 */
#include <skip_os.h>

#include <xview/notify.h>

#include "es_timers.h"


/* Number of timers active currently
 */
int			timers_active=0;

/* List of active timers
 */
static timerentry_t	*my_timers = NULL;

extern int		errno;

static int		timer_debug = 0;
#define			TIMER_DEBUG2(s, t, u)	if (timer_debug) { \
							printf(s, t, u); \
						}
#define			TIMER_DEBUG3(s, t,u,v)	if (timer_debug) { \
							printf(s, t, u, v); \
						}

/* tick()
 *
 * tell the timer module that one time unit has elapsed.  This has the
 * side effect that an expired timers will be serviced.
 *
 * Returns: none
 */

Notify_value
tick(Xv_opaque win)
{
	struct itimerval	timer;
	timerentry_t		*curr, *next;

	timer.it_value.tv_sec = TIMER_RES;
	timer.it_value.tv_usec = 0;
	timer.it_interval.tv_sec = timer.it_interval.tv_usec = 0;

	notify_set_itimer_func(win, tick, ITIMER_REAL, &timer, NULL);

	if ((curr = my_timers) == NULL) {
		return (NOTIFY_DONE);
	}

	/* tick one unit off the head of the queue, has it expired?
	 */
	curr->t_timeout--;
	for (; curr && curr->t_timeout == 0; curr = next) {

		/* get hold the next timer entry while we still have the
		 * current one around (it will be going away)
		 */
		next=curr->t_next;

		/* timer has expired: unlink from the list
		 */
		my_timers = next;

		timers_active--;

		TIMER_DEBUG2("timer %s (%p) has popped\n", curr->t_id, curr);

		/* and call the required function
		 */
		(void) (*curr->t_func)(curr->t_param);

		free(curr);
	}
	return (NOTIFY_DONE);
}


/* start_timer()
 *
 * call function func(param) after t ticks have elapsed
 *
 * Returns: timer identifier on success, 0 otherwise
 */

timerentry_t * 
start_timer(char *id, unsigned long t, void (func)(), int param)
{
	timerentry_t	*new, **prevp, *curr;

	new = (timerentry_t *) calloc(1, sizeof *new);

	TIMER_DEBUG3("starting timer %s (%p) param=%d\n", id, new, param);

	if (new == NULL) {
		return NULL;
	}

	/* refuse a timer of value 0
	 */
	if (t < 1) {
		return NULL;
	}

	new->t_func    = func;
	new->t_param   = param;
	new->t_timeout = t;
	strncpy(new->t_id, id, MAX_TIMER_ID);

	/* locate this timer in the list of active timers
	 */
	for (prevp = &my_timers; (curr = *prevp); prevp = &curr->t_next) {
		if (curr->t_timeout <= new->t_timeout) {
			new->t_timeout -= curr->t_timeout;
		} else {
			break;
		}
	}

	/* and insert in the list
	 */
	if (*prevp) {

		/* reduce timeout of next in queue appropriately
		 */
		(*prevp)->t_timeout -= new->t_timeout;
	}

	new->t_next = *prevp;
	*prevp = new;

	timers_active++;

	return new;
}


/* cancel_timer()
 *
 * cancel call back of function func(param) after t ticks have elapsed
 *
 * Returns: 0 on success, non-zero on error
 */

int
cancel_timer(timerentry_t *old)
{
	timerentry_t	**prevp, *curr;

	if (old)
		TIMER_DEBUG2("cancelling timer %s (%p)\n", old->t_id, old);

	/* find the timer in the current timer list
	 */
	for (prevp = &my_timers; (curr = *prevp); prevp = &curr->t_next) {
		if (curr == old) {
			break;
		}
	}

	if (curr == NULL) {
		/* bad timer id, was not found in search
		 */
		errno=ENXIO;
		return -1;
	}

	timers_active--;

	/* unlink the timer from the list
	 */
	*prevp = curr->t_next;

	/* need to add this timer's delta onto the next member
	 */
	if (*prevp) {

		(*prevp)->t_timeout += curr->t_timeout;
	}

	/* and free the old timer structure
	 */
	free(old);

	return 0;
}
	

/* change_timer()
 *
 * change an existing timer parameters
 *
 * Returns: 0 on success, non-zero on error
 */

int
change_timer(timerentry_t *old, void (newfunc)(), int newparam)
{
	timerentry_t	*curr;

	if (old)
		TIMER_DEBUG2("changing timer %s (%p)\n", old->t_id, old);

	/* find the timer in the current timer list
	 */
	for (curr = my_timers; curr; curr = curr->t_next) {
		if (curr == old) {
			break;
		}
	}

	if (curr == NULL) {
		/* bad timer id, was not found in search
		 */
		errno=ENXIO;
		return -1;
	}

	TIMER_DEBUG2("old param=%d, new param=%d\n", curr->t_param, newparam);
	curr->t_func = newfunc;
	curr->t_param = newparam;
	return 0;
}
	

/* time_has_passed()
 *
 * Returns: non-zero (true) once every TIMER_RES seconds
 */

int
time_has_passed()
{
	static time_t	then=0;
	time_t		now;

	if (time(&now) < 0) {
		perror("time");
		exit(1);
	}

	return now - then > TIMER_RES? then=now, 1 : 0;
}


/* print_timers()
 *
 * print details of timers current active
 *
 * Returns: 0 on success, non-zero on error
 */

void
print_timers()
{
	timerentry_t	*curr = my_timers;
	register	abs=0;

	while (curr) {
		abs += curr->t_timeout;
		printf("timer %s: (id=%p) will pop in %d ticks (delta %d) "
			"call %p(%x)\n",
			curr->t_id, curr, abs, curr->t_timeout, curr->t_func, 
			curr->t_param);
		curr = curr->t_next;
	}
}

