/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/
 
#pragma ident "@(#)skipstat.c	1.23 96/08/14 Sun Microsystems"

/*
 * System includes
 */
#include <skip_os.h>

/*
 * SKIP includes
 */
#include <skip_proto.h>
#include <skip_types.h>
#include <skip_ioctl.h>
#include <skip_acl.h>
#include <skip_lib.h>

char		*progname;
extern int	errno;

extern char	*optarg;
extern int	optind;

ioctl_if_stat_t		if_stats;
ioctl_key_stat_t	key_stats;
ioctl_hdr_stat_t	hdr_stats;

/*
 * Format error message if invalid options
 */
static void
usage()
{

	fprintf(stderr, "usage:\t %s "
		"[-amCIkh] [-c <version>] [-i <interface>]\n"
		"\t-C:	display cryptographic and MAC algorithms supported\n"
		"\t-c:	display cryptographic algorithm statistics\n"
		"\t-m:	display MAC algorithms statistics\n"
		"\t-I:	display SKIP interface statistics\n"
		"\t-k:	display SKIP key statistics\n"
		"\t-h:	display SKIP header statistics\n"
		"\t-K:	display SKIP local master key IDs\n"
		"\t-a:	all %s options\n",
		 progname, progname);
			
	fprintf(stderr, "\n");
	return;
}

/*
 * get_if_stats()
 *
 * Get and display per SKIP interface statistics
 */
static int
get_if_stats(char *if_name)
{

	if (skip_get_stats_if(if_name, &if_stats)) {
		fprintf(stderr, "%s\n", skip_errmsg);
		return(-1);
	}

	printf("\nSKIP interface (%s) statistics:\n", if_name);

	printf("skip_if_ipkts:\t\t%lu\n", if_stats.skip_if_ipkts);
	printf("skip_if_opkts:\t\t%lu\n", if_stats.skip_if_opkts);
	printf("skip_if_encrypts:\t%lu\n", if_stats.skip_if_encrypts);
	printf("skip_if_decrypts:\t%lu\n", if_stats.skip_if_decrypts);
	printf("skip_if_drops:\t\t%lu\n", if_stats.skip_if_drops);
	printf("skip_if_notv4:\t\t%lu\n", if_stats.skip_if_notv4);
	printf("skip_if_bypasses:\t%lu\n", if_stats.skip_if_bypasses);
	printf("skip_if_raw_in:\t\t%lu\n", if_stats.skip_if_raw_in);
	printf("skip_if_raw_out:\t%lu\n", if_stats.skip_if_raw_out);

	return(0);
}

/*
 * get_key_stats()
 *
 * Get and display SKIP key statistics
 */
static int
get_key_stats(char *if_name)
{

	int key_idle, key_bytes;

	/*
	 * Get the key management parameters
	 */
	if (skip_var_get(if_name, "skip_key_max_bytes", &key_bytes) < 0) {
		fprintf(stderr, "%s\n", skip_errmsg);
		return(-1);
	}

	if (skip_var_get(if_name, "skip_key_max_idle", &key_idle) < 0) {
		fprintf(stderr, "%s\n", skip_errmsg);
		return(-1);
	}

	/*
	 * Now, get the keys statistics
	 */
	if (skip_get_stats_key(if_name, &key_stats)) {
		fprintf(stderr, "%s\n", skip_errmsg);
		return(-1);
	}

	printf("\nSKIP key management parameters:\n");
	printf("skip_key_max_idle:\t%d\n", key_idle);
	printf("skip_key_max_bytes:\t%d\n", key_bytes);

	printf("\nSKIP key statistics:\n");

	printf("skip_encrypt_keys_active:\t%lu\n",
					key_stats.skip_encrypt_keys_active);
	printf("skip_decrypt_keys_active:\t%lu\n",
					key_stats.skip_decrypt_keys_active);
	printf("skip_key_lookups:\t\t%lu\n",
					key_stats.skip_key_lookups);
	printf("skip_keymgr_requests:\t\t%lu\n",
					key_stats.skip_keymgr_requests);
	printf("skip_key_reclaims:\t\t%lu\n",
					key_stats.skip_key_reclaims);
	printf("skip_hash_collisions:\t\t%lu\n",
					key_stats.skip_hash_collisions);

	return(0);
}

/*
 * get_hdr_stats()
 *
 * Get and display SKIP header statistics
 */
static int
get_hdr_stats(char *if_name)
{

	if (skip_get_stats_hdr(if_name, &hdr_stats)) {
		fprintf(stderr, "%s\n", skip_errmsg);
		return(-1);
	}

	printf("\nSKIP header statistics:\n");

	printf("skip_hdr_bad_versions:\t\t%lu\n",
			hdr_stats.skip_hdr_bad_versions);
	printf("skip_hdr_short_ekps:\t\t%lu\n",
			hdr_stats.skip_hdr_short_ekps);
	printf("skip_hdr_short_mids:\t\t%lu\n",
			hdr_stats.skip_hdr_short_mids);
	printf("skip_hdr_bad_kp_algs:\t\t%lu\n",
			hdr_stats.skip_hdr_bad_kp_algs);

	printf("V1 skip_hdr_encodes:\t\t%lu\n",
			hdr_stats.skip_hdr_encodes);
	printf("V1 skip_hdr_decodes:\t\t%lu\n",
			hdr_stats.skip_hdr_decodes);
	printf("V1 skip_hdr_runts:\t\t%lu\n",
			hdr_stats.skip_hdr_runts);
	printf("V1 skip_hdr_short_nodeids:\t%lu\n",
			hdr_stats.skip_hdr_short_nodeids);

	printf("IPSP skip_ipsp_decodes:\t\t%lu\n",
			hdr_stats.skip_ipsp_decodes);
	printf("IPSP skip_ipsp_encodes:\t\t%lu\n",
			hdr_stats.skip_ipsp_encodes);
	printf("IPSP skip_hdr_bad_nsid:\t\t%lu\n",
			hdr_stats.skip_hdr_bad_nsid);
	printf("IPSP skip_hdr_bad_mac_algs:\t%lu\n",
			hdr_stats.skip_hdr_bad_mac_algs);
	printf("IPSP skip_hdr_bad_mac_size:\t%lu\n",
			hdr_stats.skip_hdr_bad_mac_size);
	printf("IPSP skip_hdr_bad_mac_val:\t%lu\n",
			hdr_stats.skip_hdr_bad_mac_val);
	printf("IPSP skip_hdr_bad_next:\t\t%lu\n",
			hdr_stats.skip_hdr_bad_next);
	printf("IPSP skip_hdr_bad_esp_spi:\t%lu\n",
			hdr_stats.skip_hdr_bad_esp_spi);
	printf("IPSP skip_hdr_bad_ah_spi:\t%lu\n",
			hdr_stats.skip_hdr_bad_ah_spi);
	printf("IPSP skip_hdr_bad_iv:\t\t%lu\n",
			hdr_stats.skip_hdr_bad_iv);
	printf("IPSP skip_hdr_short_r_mkeyid:\t%lu\n",
			hdr_stats.skip_hdr_short_r_mkeyid);
	printf("IPSP skip_hdr_short_s_mkeyid:\t%lu\n",
			hdr_stats.skip_hdr_short_s_mkeyid);
	printf("IPSP skip_hdr_bad_r_mkeyid:\t%lu\n",
			hdr_stats.skip_hdr_bad_r_mkeyid);
	printf("\n");

	return(0);
}

/*
 * get_crypt_list()
 *
 * Get and display crypto and MAC modules list
 */
static int
get_crypt_list(char *if_name)
{
	int			idx;

	printf("\nCryptographic algorithms (SKIP version 1):\n");

	for (idx = 0; idx < SKIP_MAXCRYPTORS; idx++) {
		if (skip_supported_kp_alg(idx, SKIP_V1)) {
			printf("Crypto Module Id:\t%u", idx);
			printf("\tCrypto Name:\t%s\n",
				skip_kp_alg_to_name(idx, SKIP_V1));
		}
        }

	printf("\nCryptographic algorithms (SKIP):\n");

	for (idx = 1; idx < SKIP_MAXCRYPTORS; idx++) {
		if (skip_supported_kp_alg(idx, SKIP_V2)) {
			printf("Crypto Module Id:\t%u", idx);
			printf("\tCrypto Name:\t%s\n",
				skip_kp_alg_to_name(idx, SKIP_V2));
		}
        }

	printf("\nMAC algorithms (SKIP):\n");

	for (idx = 1; idx < SKIP_MAX_MAC; idx++) {
		if (skip_supported_mac_alg(idx, SKIP_V2)) {
			printf("MAC Module Id:\t\t%u", idx);
			printf("\tMAC Name:\t%s\n",
				skip_mac_alg_to_name(idx, SKIP_V2));
		}
        }
	printf("\n");

	return(0);
}

/*
 * get_crypt_stats()
 *
 * Get and display crypto modules statistics
 */
static int
get_crypt_stats(char *if_name, int vers)
{
	ioctl_crypt_stat_t	crypt_stats;
	crypt_mod_stat_t	*p_stats = crypt_stats.crypt_stats;
	int			idx;

	if ((vers == 0) || (vers == SKIP_V1)) {

		crypt_stats.version = SKIP_V1;
	
		if (skip_get_crypt_stats(if_name, &crypt_stats)) {
			fprintf(stderr, "%s\n", skip_errmsg);
			return(-1);
		}
	
		printf("\nCryptographic algorithm statistics "
					"(SKIP version 1):\n");
	
		for (idx = 0; idx < SKIP_MAXCRYPTORS; idx++) {
	
			if (skip_supported_kp_alg(idx, SKIP_V1)) {
				printf("\nCrypto Module Name:\t%s\n",
					skip_kp_alg_to_name(idx, SKIP_V1));
		
				printf("encrypts:\t\t%lu\n", 
						p_stats->encrypts);
				printf("encrypterrs:\t\t%lu\n",
						p_stats->encrypterrs);
				printf("decrypts:\t\t%lu\n",
						p_stats->decrypts);
				printf("decrypterrs:\t\t%lu\n",
						p_stats->decrypterrs);
	
				/*
			  	 * Next cryptor
			 	 */
				p_stats++;
			}
	        }
		printf("\n");
	}

	if ((vers == 0) || (vers == SKIP_V2)) {

		crypt_stats.version = SKIP_V2;
	
		if (skip_get_crypt_stats(if_name, &crypt_stats)) {
			fprintf(stderr, "%s\n", skip_errmsg);
			return(-1);
		}
	
		printf("\nCryptographic algorithm statistics (SKIP):\n");
	
		p_stats = crypt_stats.crypt_stats;
	
		for (idx = 1; idx < SKIP_MAXCRYPTORS; idx++) {
	
			if (skip_supported_kp_alg(idx, SKIP_V2)) {
				printf("\nCrypto Module Name:\t%s\n",
					skip_kp_alg_to_name(idx, SKIP_V2));
		
				printf("encrypts:\t\t%lu\n",
						p_stats->encrypts);
				printf("encrypterrs:\t\t%lu\n",
						p_stats->encrypterrs);
				printf("decrypts:\t\t%lu\n",
						p_stats->decrypts);
				printf("decrypterrs:\t\t%lu\n",
						p_stats->decrypterrs);
	
				/*
			  	 * Next cryptor
			 	 */
				p_stats++;
			}
	        }
		printf("\n");
	}

	return(0);
}

/*
 * get_mac_stats()
 *
 * Get and display MAC modules statistics
 */
static int
get_mac_stats(char *if_name)
{
	ioctl_mac_stat_t	mac_stats;
	mac_mod_stat_t		*p_stats = mac_stats.mac_stats;
	unsigned int		idx;

	if (skip_get_mac_stats(if_name, &mac_stats)) {
		fprintf(stderr, "%s\n", skip_errmsg);
		return(-1);
	}

	printf("\nMAC algorithm statistics (SKIP):\n");

	p_stats = mac_stats.mac_stats;

	for (idx = 1; idx <= mac_stats.nb_algs; idx++) {

		if (!skip_supported_mac_alg(p_stats->module_id, SKIP_V2)) {
			continue;
		}

		printf("\nMAC Module Name:\t%s\n",
			skip_mac_alg_to_name(p_stats->module_id, SKIP_V2));
	
		printf("in_mac:\t\t\t%lu\n", p_stats->in_mac);
		printf("in_mac_errs:\t\t%lu\n", p_stats->in_mac_errs);
		printf("out_mac:\t\t%lu\n", p_stats->out_mac);
		printf("out_mac_errs:\t\t%lu\n", p_stats->out_mac_errs);

		/*
	  	 * Next MAC module statistics
	 	 */
		p_stats++;
        }
	printf("\n");

	return(0);
}

/*
 * Get list of all local Master Key IDs
 */
static void
get_keyid_list(char *ifname)
{
	ioctl_keyid_t		keyid_list;
	skip_keyid_list_t	*p_keyid_lst;
	char			keyid[MAXVARSZ*2];
	int			nsid;
	int			key_idx;

	if (skip_get_list_keyids(ifname, &keyid_list) < 0) {
		fprintf(stderr, "%s.\n", skip_errmsg);
		return;
	}

	p_keyid_lst = keyid_list.keyid_lst;

	printf("\nLocal Master Key ID list:\n");

	for (nsid = 0; nsid < SKIP_MAX_NSID; nsid++) {

		if (p_keyid_lst->count == 0) {
			p_keyid_lst++;
			continue;
		}

		printf("\tnsid: %d, total keys = %d\n",
						nsid, p_keyid_lst->count);

		for(key_idx = 0; key_idx <p_keyid_lst->count; key_idx++) {
			printf("\t\tKey #");
			if (skip_keyid_to_s(&p_keyid_lst->mkeyid[key_idx],
						nsid, keyid) < 0) {
				printf("%d. *** Bad key ID value.\n",
								key_idx + 1);
			} else {
				printf("%d. %s\n", (key_idx + 1), keyid);
			}

		}

		p_keyid_lst++;
	}

}

/*
 * skipstat tool
 *
 * Get SKIP driver statistics
 *
 */
int
SKIPSTAT(int argc, char *argv[])
{
	char		*ifname = skip_default_if();

	int		opt, req, rc;
	int		opt_all, opt_if, opt_key, opt_drv, opt_hdr;		
	int		opt_crypt_list, opt_crypt_stats;
	int		opt_vers, opt_mac, opt_mkeyid;

	progname = argv[0];
	opt_all = opt_if = opt_key = opt_drv = opt_hdr = 0;
	opt_crypt_list = opt_crypt_stats = opt_mac = opt_mkeyid = 0;

	/*
	 * Command line parsing
	 */
	optind = 1;
	while ((opt = getopt(argc,argv, "aCmc:hIki:K")) != -1) {

		switch (opt) {
		
		case 'a':
			opt_all = 1;
			goto precheck;

		case 'k':
			opt_key = 1;
			break;

		case 'h':
			opt_hdr = 1;
			break;

		case 'I':
			opt_if = 1;
			break;

		case 'i':
			ifname = optarg;
			break;

		case 'm':
			opt_mac = 1;
			break;

		case 'C':
			opt_crypt_list = 1;
			break;

		case 'c':
			opt_crypt_stats = 1;
			opt_vers = atoi(optarg);
			break;

		case 'K':
			opt_mkeyid = 1;
			break;

		default:
			usage();
			return(1);

		}
	}

precheck:
	/*
	 * Useful pre-checks
	 */
	if (skip_pre_checks(ifname)) {
		switch (errno) {
		case EACCES:
			fprintf(stderr, "%s: you must be root to run this "
				"program\n", progname);
			break;

		case ENODEV:
			fprintf(stderr, "%s\n", skip_errmsg);
			fprintf(stderr, "%s: Check that SKIP is correctly "
				"installed on this system\n", progname);
			break;
		default:
			fprintf(stderr, "%s\n", skip_errmsg);
		}
		return(1);
	}

	if (opt_crypt_stats && (opt_vers != 1) && (opt_vers != 2)) {
		fprintf(stderr, "%s: SKIP version must be 1 or 2\n", progname);
		return(1);
	}

	/*
	 * Get all algorithms configured in the kernel
	 */
	if (skip_var_init(progname, ifname)) {
		fprintf(stderr, "%s\n", skip_errmsg);
		return(1);
	}

	/*
	 * Check for default action 
	 */
	req = opt_if+opt_key+opt_drv+opt_hdr+opt_crypt_list+opt_crypt_stats;
	req += opt_mac+opt_mkeyid;
	if (!opt_all &&  !req) {
		rc = get_if_stats(ifname);
		return(rc);
	}

	/*
	 * Process each command line option
	 */
	if (opt_all || opt_crypt_list) {
		get_crypt_list(ifname);
	}

	if (opt_all || opt_crypt_stats) {
		get_crypt_stats(ifname, (opt_all ? 0 : opt_vers));
	}

	if (opt_all || opt_mac) {
		get_mac_stats(ifname);
	}

	if (opt_all || opt_if) {
		get_if_stats(ifname);
	}

	if (opt_all || opt_key) {
		get_key_stats(ifname);
	}

	if (opt_all || opt_hdr) {
		get_hdr_stats(ifname);
	}

	if (opt_all || opt_mkeyid) {
		get_keyid_list(ifname);
	}

	return (0);
}
