/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skip_sunos.c	1.9 96/07/08 Sun Microsystems"

/*
 * System includes
 */
#include <skip_os.h>

/*
 * SKIP includes
 */
#include <skip_proto.h>
#include <skip_types.h>
#include <skip_ioctl.h>
#include <skip_acl.h>
#include <skip_lib.h>

#define BUFSIZE		10000


extern char *sys_errlist[];

int skip_if_plumb(char *, int);

/* _skip_error()
 *
 * Set skip_errmsg string
 */
static void
_skip_error(char *errstr)
{
	sprintf(skip_errmsg, "%s: %s: %s", SKIPNAME, errstr, sys_errlist[errno]);
}

/* _skip_iferror()
 *
 * Set skip_errmsg string with interface name information
 */
static void
_skip_iferror(char *ifname, char *errstr)
{
	sprintf(skip_errmsg, "%s: %s: %s: %s",
		SKIPNAME, ifname, errstr, sys_errlist[errno]);
}

/* _skip_invalid_ifname()
 *
 * Make sure an interface name is not too long
 */
static int
_skip_invalid_ifname(char *ifname)
{
	if (strlen(ifname) > (size_t) (IFNAMSIZ - 1)) {
		sprintf(skip_errmsg, SKIP_MSG_IFNAME2LONG);
		return (1);
	}
	return (0);
}

/*
 * open a connection to the SKIP kernel
 */
int
_skip_open(char *name)
{
	int		skip_fd, minor = 1;
	char		device[sizeof SKIPDEV "999"];

	do {
		(void) sprintf(device, SKIPDEV "%d", minor++);
		skip_fd = open(device, O_RDWR);
	} while (skip_fd < 0 && errno == EBUSY);

	if (skip_fd < 0) {
		sprintf(skip_errmsg, "%s: cannot open device %s: %s",
				SKIPNAME, SKIPDEV, sys_errlist[errno]);
		return (-1);
	}
	return (skip_fd);
}

/*
 * close a connection to the SKIP kernel
 */
void
_skip_close(int skip_fd)
{
	(void) close(skip_fd);
}

/* _skip_open_ip()
 *
 * Returns: file descriptor on success, -1 otherwise.
 */
static int
_skip_open_ip()
{
	int			ip_fd;

	ip_fd = socket(AF_INET, SOCK_DGRAM, 0);
	if (ip_fd < 0) {
		_skip_error(SKIP_MSG_IPOPEN);
		return(-1);
	}
	return (ip_fd);
}

/* _skip_gethostbyname()
 *
 * Non-name service based gethostbyname() - eats /etc/hosts
 *
 * Returns: as gethostbyname()
*/
static struct hostent *
_skip_gethostbyname(char *hostname)
{
#define	ETC_HOSTS		"/etc/hosts"
#define	MAXLINESZ		128
	static long		addr;
	static char		line[MAXLINESZ], *s, *addrs[1];
	static struct hostent	h, *hp = &h;
	FILE			*fp;

	fp = fopen(ETC_HOSTS, "r");
	if (fp == NULL) {
		return (NULL);
	}
	
	memset((char *) hp, 0, sizeof (*hp));

	while (!feof(fp)) {
		if (fgets(line, MAXLINESZ, fp) == NULL) {
			break;
		}
		s = strtok(line, " \t\n");
		if (s == NULL) {
			continue;
		}
		if (s[0] == '#') {
			continue;
		}
		if ((addr = inet_addr(s)) == -1) {
			continue;
		}
		for (; s; s = strtok(NULL, " \t\n")) {
			if (strchr(s, '#')) {
				break;
			}
			if (!strcmp(s, hostname)) {
				/*
				 * found the host
				 */
				addrs[0] = (char *) &addr;
				hp->h_name = s;
				hp->h_addrtype = AF_INET;
				hp->h_length = sizeof(addr);
				hp->h_addr_list = addrs;
				break;
			}
		}
		if (hp->h_name) {
			break;
		}
	}
	fclose(fp);
	return (hp->h_name ? hp : NULL);
}

/* skip_hostname()
 *
 * Return the name of the local system
 *
 * Returns: 0 on success, -1 otherwise
 */
int
skip_hostname(char *hostname, int len)
{
	int	rc;

	rc = gethostname(hostname, len);
	if (rc) {
		_skip_error("gethostname");
	}
	return (rc);
}

/* skip_default_if()
 *
 * Return the name of the default network interface
 *
 * Returns: interface name
 */
char *
skip_default_if()
{
#define	SKIP_DEFAULT_IF		"le0"
	static char		devname[MAXPATHLEN];
	char			cbuf[sizeof(struct ifreq) * 16];
	struct ifconf		ifc;
	struct ifreq		*ifr, ifflags;
	struct hostent		*hp;
	struct sockaddr_in	*sa;
	char			hostname[MAXHOSTNAMELEN];
	int			s, n;

	strcpy(devname, SKIP_DEFAULT_IF);

	if (skip_hostname(hostname, MAXHOSTNAMELEN)) {
		return (devname);
	}

	hp = _skip_gethostbyname(hostname);

	if (hp == NULL) {
		return(devname);
	}

	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		return (devname);
	}

	ifc.ifc_len = sizeof (cbuf);
	ifc.ifc_buf = cbuf;
	if (ioctl(s, SIOCGIFCONF, (char *)&ifc) < 0) {
		goto out;
	}

	n = ifc.ifc_len / sizeof (struct ifreq);
	ifr = ifc.ifc_req;

	/*
	 * scan list trying to find the principal interface, based on
	 * primary host address.  Note any up interfaces in passing,
	 * just in case the primary can't be found.
	 * 
	 */
	for (; n > 0; n--, ifr++) {
		strcpy(ifflags.ifr_name, ifr->ifr_name);
		if (ioctl(s, SIOCGIFFLAGS, (char *) &ifflags) < 0) {
			goto out;
		}
		if ((ifflags.ifr_flags & (IFF_LOOPBACK|IFF_UP|IFF_RUNNING)) ==
					(IFF_UP|IFF_RUNNING)) {
			/*
			 * note the interface name just in case
			 */
			strcpy(devname, ifr->ifr_name);

			/*
			 * but try to find an exact match
			 */
			sa = (struct sockaddr_in *) &ifr->ifr_addr;
			if (memcmp(hp->h_addr_list[0], (void *) &sa->sin_addr,
							hp->h_length) == 0) {
					break;
			}
		}
	}
out:
 	(void) close(s);
	return (devname);
}
 

/* skip_pre_checks()
 *
 * check that permissions and software installation is correct
 *
 * Returns: 0 on success, non-zero otherwise
 */
int
skip_pre_checks(char *name)
{
	struct ifreq	flags;
	int		fd;

	if (_skip_invalid_ifname(name)) {
		return (-1);
	}

	fd = _skip_open_ip();
	if (fd < 0) {
		return(-1);
	}

	strcpy(flags.ifr_name, name);
	if (ioctl(fd, SIOCGIFFLAGS, (caddr_t) &flags) < 0) {
		_skip_iferror(name, SKIP_MSG_IFCONFIG);
		close(fd);
		return(-1);
	}
	close(fd);

	fd = _skip_open(name);
	if (fd < 0) {
		return (-1);
	}
	close(fd);

	return (0);
}

/* skip_if_state()
 *
 * save/restore if state
 *
 * Returns: 0 on success, -1 otherwise
 */
int
skip_if_state(char *name, int mode)
{
	static struct ifreq	addr, broadaddr, dstaddr, netmask, flags;
	static int		saved = 0;
	int			ip_fd;

	if (_skip_invalid_ifname(name)) {
		return (-1);
	}

	if ((ip_fd = _skip_open_ip()) < 0) {
		return(-1);
	}

	switch (mode) {
	case Save:
		/*
		 * save flags state
		 */
		strcpy(flags.ifr_name, name);
		if (ioctl(ip_fd, SIOCGIFFLAGS, (caddr_t)&flags) < 0) {
			_skip_iferror(name, "SIOCGIFFLAGS");
			close(ip_fd);
			return(-1);
		}
		/*
		 * save addr state
		 */
		strcpy(addr.ifr_name, name);
		if (ioctl(ip_fd, SIOCGIFADDR, (caddr_t)&addr) < 0) {
			_skip_iferror(name, "SIOCGIFADDR");
			close(ip_fd);
			return(-1);
		}
		if (flags.ifr_flags & IFF_POINTOPOINT) {
			strcpy(dstaddr.ifr_name, name);
			if (ioctl(ip_fd, SIOCGIFDSTADDR, (caddr_t)&dstaddr)
									< 0) {
				_skip_iferror(name, "SIOCGIFDSTADDR");
				close(ip_fd);
				return(-1);
			}
		}
		/*
		 * save broadcast address state
		 */
		if (flags.ifr_flags & IFF_BROADCAST) {
			strcpy(broadaddr.ifr_name, name);
			if (ioctl(ip_fd, SIOCGIFBRDADDR, (caddr_t)&broadaddr)	
									< 0) {
				_skip_iferror(name, "SIOCGIFBRDADDR");
				close(ip_fd);
				return(-1);
			}
		}
		/*
		 * save netmask state
		 */
		strcpy(netmask.ifr_name, name);
		if (ioctl(ip_fd, SIOCGIFNETMASK, (caddr_t)&netmask) < 0) {
			_skip_iferror(name, "SIOCGIFNETMASK");
			close(ip_fd);
			return(-1);
		}
		saved = 1;
		break;

	case Restore:
		if (saved == 0) {
			break;
		}
		/*
		 * restore addr state
		 */
		if (ioctl(ip_fd, SIOCSIFADDR, (caddr_t)&addr) < 0) {
			_skip_iferror(name, "SIOCSIFADDR");
			close(ip_fd);
			return(-1);
		}
		if (flags.ifr_flags & IFF_POINTOPOINT) {
			if (ioctl(ip_fd, SIOCSIFDSTADDR, (caddr_t)&dstaddr)
									< 0) {
				_skip_iferror(name, "SIOCSIFDSTADDR");
				close(ip_fd);
				return(-1);
			}
		}
		/*
		 * restore broadcast address state
		 */
		if (flags.ifr_flags & IFF_BROADCAST) {
			if (ioctl(ip_fd, SIOCSIFBRDADDR, (caddr_t)&broadaddr)
									< 0) {
				_skip_iferror(name, "SIOCSIFBRDADDR");
				close(ip_fd);
				return(-1);
			}
		}
		/*
		 * restore netmask state
		 */
		if (ioctl(ip_fd, SIOCSIFNETMASK, (caddr_t)&netmask) < 0) {
			_skip_iferror(name, "SIOCSIFNETMASK");
			close(ip_fd);
			return(-1);
		}
		/*
		 * restore flags state
		 */
		if (ioctl(ip_fd, SIOCSIFFLAGS, (caddr_t)&flags) < 0) {
			_skip_iferror(name, "SIOCSIFFLAGS");
			close(ip_fd);
			return(-1);
		}
#ifdef notdef
		/*
		 * restore mtu
		 */
		if (mtu.ifr_metric != 0) {
			mtu.ifr_metric -= SKIP_HDR_SZ;
			if (ioctl(ip_fd, SIOCSIFMTU, (caddr_t)&mtu) < 0) {
				_skip_iferror(name, "SIOCSIFMTU");
				close(ip_fd);
				return(-1);
			}
		}
#endif
		break;

	default:
		_skip_iferror(name, "bad command (internal error)");
		close(ip_fd);
		return(-1);
	}
	close(ip_fd);
	return (0);
}

int
skip_if_set_mtu(char * name, int val)
{
	static struct ifreq	mtu;
	int			ip_fd;

	if (_skip_invalid_ifname(name)) {
		return (-1);
	}

	if ((ip_fd = _skip_open_ip()) < 0) {
		return (-1);
	}

	strcpy(mtu.ifr_name, name);
	mtu.ifr_metric = val;
	if (ioctl(ip_fd, SIOCSIFMTU, (caddr_t)&mtu) < 0) {
		_skip_iferror(name, "SIOCSIFFLAGS");
		close(ip_fd);
		return(-1);
	}
	close(ip_fd);
	return (0);
}

int
skip_if_get_mtu(char * name)
{
	static struct ifreq	mtu;
	int			ip_fd;

	if (_skip_invalid_ifname(name)) {
		return (-1);
	}

	if ((ip_fd = _skip_open_ip()) < 0) {
		return (-1);
	}

	strcpy(mtu.ifr_name, name);
	if (ioctl(ip_fd, SIOCGIFMTU, (caddr_t)&mtu) < 0) {
		_skip_iferror(name, "SIOCGIFMTU");
		close(ip_fd);
		return(-1);
	}
	close(ip_fd);
	return (mtu.ifr_metric);
}


int
skip_if_get_broadcast(char *name, struct in_addr *addr)
{
	static struct ifreq	f;
	struct sockaddr_in	*sin;
	int			ip_fd;

	if (_skip_invalid_ifname(name)) {
		return (-1);
	}

	if ((ip_fd = _skip_open_ip()) < 0) {
		return (-1);
	}

	strcpy(f.ifr_name, name);
	if (ioctl(ip_fd, SIOCGIFFLAGS, (caddr_t)&f) < 0) {
		_skip_iferror(name, "SIOCGIFFLAGS");
		close(ip_fd);
		return(-1);
	}
	if (f.ifr_flags & IFF_BROADCAST) {
		if (ioctl(ip_fd, SIOCGIFBRDADDR, (caddr_t) &f)	< 0) {
			_skip_iferror(name, "SIOCGIFBRDADDR");
			close(ip_fd);
			return(-1);
		}
		close(ip_fd);
		sin = (struct sockaddr_in *) &f.ifr_broadaddr;
		*addr = sin->sin_addr;
		return (0);
	}
	_skip_iferror(name, "non-broadcast interface");
	close(ip_fd);
	return (-1);
}

int
skip_if_module_present(char *name)
{
	return (skip_get_mode(name) != -1);
}

/* skip_if_plumb()
 *
 * If secure == 1, insert SKIP between IP and the specified network interface
 * otherwise remove it.
 *
 * Returns: 0 on success, -1 on error
 */
int
skip_if_plumb(char *name, int secure)
{
	skip_es_req_t	req;
	int		rc = 0;
	int		fd;

	if (_skip_invalid_ifname(name)) {
		return (-1);
	}

	fd = _skip_open(name);
	if (fd < 0) {
		sprintf(skip_errmsg, "%s: failed to open %s: %s",
			SKIPNAME, SKIPDEV, sys_errlist[errno]);
		return (-1);
	}
	strcpy(req.if_name, name); 
	if (secure) {
		if (_skip_ioctl(fd, SKIP_ES_ADD_IF, (char *) &req,
						sizeof(req.if_name)) < 0) {
			_skip_error("SKIP_ES_ADD_IF");
			close(fd);
			return (-1);
		}
	} else {
		if (_skip_ioctl(fd, SKIP_ES_DEL_IF, (char *) &req,
						sizeof(req.if_name)) < 0) {
			if (errno != ENXIO) {
				/*
				 * silently allow the removal of
				 * an already removed interface
				 */
				_skip_error("SKIP_ES_DEL_IF");
				close(fd);
				return (-1);
			}
		}
	}
	close(fd);
	return (rc);
}

/* skip_if_unplumb()
 *
 * Remove network interface from under IP (SYSV only)
 *
 * Returns: 0 on success, -1 on error
 */
/*ARGSUSED*/
int 
skip_if_unplumb(char *name)
{
	return (0);
}


/* _skip_ioctl()
 *
 * Returns: -1 on error, bytes returned by ioctl()
 */
int
_skip_ioctl(s, cmd, buf, buflen)
	int s;
	int cmd;
	char *buf;
	int buflen;
{
	skip_io_t	ioc;
	int		rc;

	memset((char *)&ioc, 0, sizeof(ioc));
	ioc.ic_cmd = cmd;
	ioc.ic_timout = 0;
	ioc.ic_len = buflen;
	ioc.ic_dp = buf;
	rc = ioctl(s, SKIPIOACL, &ioc);
	return (rc < 0 ? rc : ioc.ic_len);
}


/* skip_time_stamp()
 *
 * Format a "time stamp"
 */
void
skip_time_stamp(char *buf)
{
	struct tm		*tm;
	long			clock;

	clock = time((time_t *) 0);
	tm = localtime(&clock);

	strftime(buf, ERRSTRSZ, "%c", tm);
}

/* skip_open_device()
 *
 * Open the SKIP kernel device
 */
int
skip_open_device(char *dev)
{
	return(open(dev, O_RDWR));
}

/* skip_getmsg()
 *
 * Retrieve a message from the SKIP kernel
 *
 * Returns: -1 on error, message size on success
 */
int
skip_getmsg(int fd, void *buf, int buflen)
{
	return(read(fd, (char *) buf, buflen));
}

/* skip_putmsg()
 *
 * Send a message to the SKIP kernel
 *
 * Returns: -1 on error, message size on success
 */
int
skip_putmsg(int fd, void *buf, int buflen)
{
	return(write(fd, (char *) buf, buflen));
}

/*
 * Fill in a buffer with "random" value
 */
void
skip_fill_random(char *addr, unsigned int len)
{
	struct timeval	tv;

	if (len > sizeof tv) {
		memcpy(addr, (char *) skip_fill_random, len);
	} else {
		(void) gettimeofday(&tv, NULL);
		memcpy(addr, (char *) &tv, len);
	}
}
