/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skip_ctrl.c	1.47 96/06/30 Sun Microsystems"

/*
 * System includes
 */
#include <skip_os.h>

/*
 * SKIP includes
 */
#include <skip_proto.h>
#include <skip_types.h> 
#include <skip_ioctl.h>
#include <skip_acl.h>
#include <skip_lib.h>

char			skip_errmsg[ERRSTRSZ];
extern int		errno;

/* skip_acl_list()
 *
 * extract list of entries in the access control list
 *
 * Returns the number of entries or -1 on error.
 */
int
skip_acl_list(char *if_name, void (*f)())
{
	skip_es_req_t		*reqp;
	skip_param_t		*params;
	skip_fd_t		skip_fd;
	int			nentries, i, len;

	skip_fd = _skip_open(if_name);

	if (skip_fd == SKIPFD_ERR) {
		return (-1);
	}
	/*
	 * make a buffer big enough to retrieve all the entries
	 */
	len = sizeof (reqp->if_name) + 
		(SKIP_MAX_HOSTS  * sizeof(skip_param_t));
	reqp = (skip_es_req_t *) SYSMEM_ALLOC(len);

	if (reqp == NULL) {
		sprintf(skip_errmsg, SKIP_MSG_NOMEM);
		_skip_close(skip_fd);
		errno = ENOMEM;
		return (-1);
	}
	memset(reqp, 0, len);
	strcpy(reqp->if_name, if_name);

	if ((len = _skip_ioctl(skip_fd, SKIP_ES_ACL_LIST,
					(char *)reqp, len)) < 0) {
		sprintf(skip_errmsg, SKIP_MSG_BAD_LIST, errno);
		_skip_close(skip_fd);
		SYSMEM_FREE(reqp, 0);
		return (-1);
	}

	nentries = (len - sizeof(reqp->if_name))/sizeof (skip_param_t);
	params = (skip_param_t *) &reqp->params;

	for (i = 0; i < nentries; i++) {
		(*f)(params);
		params++;
	}

	SYSMEM_FREE(reqp, 0);
	_skip_close(skip_fd);
	return (nentries);
}

/* skip_es_params_ok()
 *
 * validate parameters to an ACL operation
 *
 * Returns: 0 if OK, errno otherwise
 */
static int
skip_es_params_ok(skip_param_t *params)
{
	char		msg[ERRSTRSZ];

	if ((params->ip_addr.s_addr & params->mask.s_addr) !=
						params->ip_addr.s_addr) {
		/*
		 * address and mask are incompatible
		 */
		strcpy(msg, inet_ntoa(params->mask));
		sprintf(skip_errmsg, SKIP_MSG_BAD_MASK,
				inet_ntoa(params->ip_addr), msg);
		return (EFAULT);
	}

	if (params->comp_alg) {
		sprintf(skip_errmsg, SKIP_MSG_NOCOMP);
		return (EPROTONOSUPPORT);
	}

	/*
	 * Protect against bad configuration from the application.
	 */
	switch (params->version) {
	case SKIP_NONE:
		if (params->kij_alg ||
			params->kp_alg ||
			params->mac_alg ||
			params->comp_alg) {
			/*
			 * cleartext, all algs must be zero
			 */
			sprintf(skip_errmsg, SKIP_MSG_ZERO_ALGS);
			return (EPROTONOSUPPORT);
		}
		break;

	case SKIP_V1:
		if (!params->kij_alg || 
			!params->kp_alg ||
			params->mac_alg || 
			params->comp_alg) {
			/*
			 * V1, kij/kp algs required
			 */
			sprintf(skip_errmsg, SKIP_MSG_V1_ALGS);
			return (EPROTONOSUPPORT);
		}
		break;

	case SKIP_V2:
		if (((params->kp_alg || params->mac_alg) && !params->kij_alg) ||
			params->comp_alg) {
			/*
			 * V2, kp/mac algs require kij_alg
			 */
			sprintf(skip_errmsg, SKIP_MSG_NEED_KIJ);
			return (EPROTONOSUPPORT);
		}
		if ((!params->kp_alg && !params->mac_alg) && 
					(params->ip_mode == SKIP_TRS_ON)) {
			/*
			 * Transport mode requires kp/mac algs
			 */
			sprintf(skip_errmsg, SKIP_MSG_NEED_KIJ);
			return (EPROTONOSUPPORT);
		}
		break;

	case SKIP_RAW:
		if ((!params->kp_alg && !params->mac_alg) || 
			params->comp_alg) {
			/*
			 * raw mode, kp and/or mac algs required
			 */
			sprintf(skip_errmsg, SKIP_MSG_NEED_KPMAC);
			return (EPROTONOSUPPORT);
		}
		break;

	default:
		sprintf(skip_errmsg, SKIP_BAD_VERSION);
		return (EPROTONOSUPPORT);
	}
	return (0);
}

/*
 * add an entry to the access control list
 */
int
skip_acl_add(char *if_name, skip_param_t *params)
{
	skip_es_req_t		*reqp;
	skip_fd_t		skip_fd;
	int			len;
	char			msg[ERRSTRSZ];

	/*
	 * Sanity checking...
	 */
	if (skip_es_params_ok(params)) {
		return (-1);
	}

	skip_fd = _skip_open(if_name);

	if (skip_fd == SKIPFD_ERR) {
		return (-1);
	}

	reqp = (skip_es_req_t *) SYSMEM_ALLOC(sizeof (reqp->if_name) +
						sizeof (skip_param_t));
	if (reqp == NULL) {
		sprintf(skip_errmsg, SKIP_MSG_NOMEM);
		_skip_close(skip_fd);
		return (-1);
	}

	/*
	 * set the if_name name and the acl information
	 */
	strcpy(reqp->if_name, if_name);
	BCOPY(params, &reqp->params, sizeof(*params));

	len = sizeof (reqp->if_name) + sizeof (skip_param_t);
	
	if (_skip_ioctl(skip_fd, SKIP_ES_ACL_ADD, (char *) reqp, len) < 0) {
		switch (errno)  {
		case EEXIST:
			if (params->flags & SKIP_NOMADIC) {
				if (skip_keyid_to_s(&params->r_mkeyid,
							params->r_nsid, msg)) {

					sprintf(skip_errmsg, SKIP_MSG_BAD_KEY,
						params->r_nsid);
				} else {
					sprintf(skip_errmsg, SKIP_MSG_DUPKEY,
					msg,
					skip_nsids[params->r_nsid].name);
				}
			} else {
				if (SKIP_IS_HOSTACL(params)) {
					sprintf(skip_errmsg, SKIP_MSG_DUPADDR,
					inet_ntoa(params->ip_addr));
				} else {
					strcpy(msg, inet_ntoa(params->mask));
					sprintf(skip_errmsg, SKIP_MSG_DUPNET,
					inet_ntoa(params->ip_addr), msg);
				}
			}
			break;
	
		case EFAULT:
			strcpy(msg, inet_ntoa(params->mask));
			sprintf(skip_errmsg, SKIP_MSG_BAD_MASK,
				inet_ntoa(params->ip_addr), msg);
			break;

		case EPROTONOSUPPORT:
			sprintf(skip_errmsg, SKIP_BAD_MODE);
			break;

		case EADDRNOTAVAIL:
			sprintf(skip_errmsg, SKIP_BAD_TUNNEL_ADDR);
			break;

		default:
			sprintf(skip_errmsg, SKIP_MSG_BAD_ADD,
				errno, inet_ntoa(params->ip_addr));
		}
		SYSMEM_FREE(reqp, 0);
		_skip_close(skip_fd);
		return (-1);
	}
	SYSMEM_FREE(reqp, 0);
	_skip_close(skip_fd);
	return (0);
}

/*
 * remove an entry from the access control list
 */
int
skip_acl_del(char *if_name, skip_param_t *params)
{
	skip_es_req_t		*reqp;
	skip_fd_t		skip_fd;
	int			len;
	char			msg[ERRSTRSZ];

	skip_fd = _skip_open(if_name);

	if (skip_fd == SKIPFD_ERR) {
		return (-1);
	}

	reqp = (skip_es_req_t *) SYSMEM_ALLOC(sizeof (reqp->if_name) +
						sizeof (skip_param_t));
	if (reqp == NULL) {
		sprintf(skip_errmsg, SKIP_MSG_NOMEM);
		_skip_close(skip_fd);
		return (-1);
	}

	/*
	 * set the if_name name and the acl information
	 */
	strcpy(reqp->if_name, if_name);
	BCOPY(params, &reqp->params, sizeof(*params));

	len = sizeof (reqp->if_name) + sizeof (skip_param_t);
	if (_skip_ioctl(skip_fd, SKIP_ES_ACL_DEL, (char *) reqp, len) < 0) {
		switch (errno)  {
		case ENOENT:
			if (params->flags & SKIP_NOMADIC) {
				if (skip_keyid_to_s(&params->r_mkeyid,
							params->r_nsid, msg)) {

					sprintf(skip_errmsg, SKIP_MSG_BAD_KEY,
						params->r_nsid);
				} else {
					sprintf(skip_errmsg, SKIP_MSG_NO_KEY,
						msg,
					skip_nsids[params->r_nsid].name);
				}
			} else {
				if (SKIP_IS_HOSTACL(params)) {
					sprintf(skip_errmsg, SKIP_MSG_NO_ADDR,
					inet_ntoa(params->ip_addr));
				} else {
					strcpy(msg, inet_ntoa(params->mask));
					sprintf(skip_errmsg, SKIP_MSG_NO_NET,
					inet_ntoa(params->ip_addr), msg);
				}
			}
			break;
	
		case EFAULT:
			sprintf(msg, inet_ntoa(params->mask));
			sprintf(skip_errmsg, SKIP_MSG_BAD_MASK,
					inet_ntoa(params->ip_addr), msg);
			break;

		default:
			sprintf(skip_errmsg, SKIP_MSG_BAD_DEL,
					errno, inet_ntoa(params->ip_addr));
		}
		SYSMEM_FREE(reqp, 0);
		_skip_close(skip_fd);
		return (-1);
	}
	SYSMEM_FREE(reqp, 0);
	_skip_close(skip_fd);
	return (0);
}

/*
 * get a host's access control details
 */
int
skip_acl_get(char *if_name, skip_param_t *params)
{
	skip_es_req_t		*reqp;
	skip_fd_t		skip_fd;
	int			len;

	skip_fd = _skip_open(if_name);

	if (skip_fd == SKIPFD_ERR) {
		return (-1);
	}

	len = sizeof (reqp->if_name) + sizeof (skip_param_t);
	reqp = (skip_es_req_t *) SYSMEM_ALLOC(len);
						
	if (reqp == NULL) {
		sprintf(skip_errmsg, SKIP_MSG_NOMEM);
		_skip_close(skip_fd);
		return (-1);
	}

	/*
	 * set the if_name name and the acl information buffer
	 */
	strcpy(reqp->if_name, if_name);
	BCOPY(params, &reqp->params, sizeof(*params));

	if (_skip_ioctl(skip_fd, SKIP_ES_ACL_GET, (char *) reqp, len) < 0) {
		sprintf(skip_errmsg, SKIP_MSG_BAD_GET,
					errno, inet_ntoa(params->ip_addr));
		SYSMEM_FREE(reqp, 0);
		_skip_close(skip_fd);
		return (-1);
	}

	*params = * (skip_param_t *) &reqp->params;
	SYSMEM_FREE(reqp, 0);
	_skip_close(skip_fd);
	return (0);
}

/*
 * returns current access control mode
 */
int
skip_get_mode(char *if_name)
{
	skip_es_mode_t		req;
	skip_fd_t		skip_fd;
	int			rc;

	skip_fd = _skip_open(if_name);

	if (skip_fd == SKIPFD_ERR) {
		return (-1);
	}

	/*
	 * set the if_name name and the mode information
	 */
	strcpy(req.if_name, if_name);

	rc = _skip_ioctl(skip_fd, SKIP_ES_GET_MODE,
				(char *) &req, sizeof(req)); 
	if (rc < 0) {
		sprintf(skip_errmsg, SKIP_MSG_BAD_GET_MODE, errno);
		_skip_close(skip_fd);
		return (-1);
	}
	_skip_close(skip_fd);

	return (req.if_mode);
}

/*
 * turn access control on/off
 */
int
skip_set_mode(char *if_name, unsigned char if_mode)
{
	skip_es_mode_t		req;
	skip_fd_t		skip_fd;

	skip_fd = _skip_open(if_name);

	if (skip_fd == SKIPFD_ERR) {
		return (-1);
	}

	/*
	 * set the if_name name and the mode information
	 */
	strcpy(req.if_name, if_name);
	req.if_mode = if_mode;

	if (_skip_ioctl(skip_fd, SKIP_ES_SET_MODE,
					(char *) &req, sizeof(req)) < 0) {

		sprintf(skip_errmsg, SKIP_MSG_BAD_GET_MODE, errno);
		_skip_close(skip_fd);
		return (-1);
	}
	if (if_mode == SkipInteractive) {
		return (skip_fd);
	}
	_skip_close(skip_fd);
	return (0);
}

int
skip_access_ctrl_on(char *if_name)
{
	return (skip_set_mode(if_name, SkipAccessControlOn));
}

int
skip_access_ctrl_off(char *if_name)
{
	return (skip_set_mode(if_name, SkipAccessControlOff));
}

/*
 * skip_get_stats_if()
 *
 * Get if_name statistics
 */
int 
skip_get_stats_if(char *if_name, ioctl_if_stat_t *p_stats)
{

	skip_fd_t		skip_fd;
	skip_ctrl_stats_t 	if_stats;

	skip_fd = _skip_open(if_name);

	if (skip_fd == SKIPFD_ERR) {
		return (-1);
	}

	BZERO((char *)&if_stats, sizeof(skip_ctrl_stats_t));

	/*
	 * set the if_name name 
	 */
	strcpy(if_stats.if_name, if_name);

	if (_skip_ioctl(skip_fd, SKIP_ES_GET_IF_STATS, 
			(char *) &if_stats, sizeof(skip_ctrl_stats_t)) < 0) {
		sprintf(skip_errmsg, SKIP_MSG_IFSTATS, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	*p_stats = if_stats.if_stats;

	_skip_close(skip_fd);
	return (0);
}

/*
 * skip_get_stats_key()
 *
 * Get key statistics
 */
int 
skip_get_stats_key(char *if_name, ioctl_key_stat_t *p_stats)
{

	skip_fd_t		skip_fd;

	skip_fd = _skip_open(if_name);

	if (skip_fd == SKIPFD_ERR) {
		return (-1);
	}

	BZERO((char *)p_stats, sizeof(ioctl_key_stat_t));

	if (_skip_ioctl(skip_fd, SKIP_KEY_GET_KEY_STATS, 
			(char *) p_stats, sizeof(ioctl_key_stat_t)) < 0) {
		sprintf(skip_errmsg, SKIP_MSG_KEY_STATS, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	_skip_close(skip_fd);
	return (0);
}

/*
 * skip_get_stats_hdr()
 *
 * Get header statistics
 */
int 
skip_get_stats_hdr(char *if_name, ioctl_hdr_stat_t *p_stats)
{

	skip_fd_t		skip_fd;

	skip_fd = _skip_open(if_name);

	if (skip_fd == SKIPFD_ERR) {
		return (-1);
	}

	BZERO((char *)p_stats, sizeof(ioctl_hdr_stat_t));

	if (_skip_ioctl(skip_fd, SKIP_KEY_GET_HDR_STATS, 
			(char *) p_stats, sizeof(ioctl_hdr_stat_t)) < 0) {
		sprintf(skip_errmsg, SKIP_MSG_HDR_STATS, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	_skip_close(skip_fd);
	return (0);
}

/*
 * skip_get_kij_list()
 *
 * Get installed crypto modules list
 */
int 
skip_get_kij_list(char *if_name, ioctl_alg_list_t *kij_list)
{

	skip_fd_t	skip_fd;

	skip_fd = _skip_open(if_name);

	if (skip_fd == SKIPFD_ERR) {
		return (-1);
	}

	if (_skip_ioctl(skip_fd, SKIP_KEY_GET_KIJ_ALGS, 
			(char *) kij_list, sizeof(ioctl_alg_list_t)) < 0) {
		sprintf(skip_errmsg, SKIP_MSG_GET_KIJ_ALGS, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	_skip_close(skip_fd);
	return (0);
}
/*
 * skip_get_crypt_list()
 *
 * Get installed crypto modules list
 */
int 
skip_get_crypt_list(char *if_name, ioctl_alg_list_t *p_crypt)
{

	skip_fd_t	skip_fd;

	skip_fd = _skip_open(if_name);

	if (skip_fd == SKIPFD_ERR) {
		return (-1);
	}

	if (_skip_ioctl(skip_fd, SKIP_KEY_GET_CRYPTORS, 
			(char *) p_crypt, sizeof(ioctl_alg_list_t)) < 0) {
		sprintf(skip_errmsg, SKIP_MSG_GET_KP_ALGS, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	_skip_close(skip_fd);
	return (0);
}

/*
 * skip_get_crypt_stat()
 *
 * Get crypto modules statistics
 */
int 
skip_get_crypt_stats(char *if_name, ioctl_crypt_stat_t *p_stats)
{

	skip_fd_t skip_fd;

	skip_fd = _skip_open(if_name);

	if (skip_fd == SKIPFD_ERR) {
		return (-1);
	}

	if (_skip_ioctl(skip_fd, SKIP_KEY_GET_CPT_STATS, 
			(char *) p_stats, sizeof(ioctl_crypt_stat_t)) < 0) {
		sprintf(skip_errmsg, SKIP_MSG_GET_ALGSTATS, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	_skip_close(skip_fd);
	return (0);
}

/*
 * skip_get_mac_stats()
 *
 * Get crypto modules statistics
 */
int 
skip_get_mac_stats(char *if_name, ioctl_mac_stat_t *p_stats)
{

	skip_fd_t	skip_fd;

	skip_fd = _skip_open(if_name);

	if (skip_fd == SKIPFD_ERR) {
		return (-1);
	}

	if (_skip_ioctl(skip_fd, SKIP_KEY_GET_MAC_STATS, 
			(char *) p_stats, sizeof(ioctl_mac_stat_t)) < 0) {
		sprintf(skip_errmsg, SKIP_MSG_GET_MACSTATS, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	_skip_close(skip_fd);
	return (0);
}

/*
 * skip_var_get()
 *
 * Get named variable value
 */
int 
skip_var_get(char *if_name, char  *name, int *value)
{

	skip_fd_t		skip_fd;
	ioctl_skip_sym_t	p_symb = { { 0 } };

	skip_fd = _skip_open(if_name);

	if (skip_fd == SKIPFD_ERR) {
		return (-1);
	}

	/*
	 * Fill in ioctl 
	 */
	p_symb.var_lg = strlen(name);
	strcpy(p_symb.var_name, name);

	if (_skip_ioctl(skip_fd, SKIP_KEY_GET_VAR, 
			(char *) &p_symb, sizeof(ioctl_skip_sym_t)) < 0) {
		sprintf(skip_errmsg, SKIP_MSG_GET_VAR, name, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	*value = p_symb.var_value;

	_skip_close(skip_fd);
	return (0);
}

/*
 * skip_var_set()
 *
 * Set named variable value
 */
int 
skip_var_set(char *if_name, char  *name, int value)
{

	skip_fd_t		skip_fd;
	ioctl_skip_sym_t	p_symb = { { 0 } };

	skip_fd = _skip_open(if_name);

	if (skip_fd == SKIPFD_ERR) {
		return (-1);
	}

	/*
	 * Fill in ioctl 
	 */
	p_symb.var_value = value;
	p_symb.var_lg = strlen(name);
	strcpy(p_symb.var_name, name);

	if (_skip_ioctl(skip_fd, SKIP_KEY_SET_VAR, 
			(char *) &p_symb, sizeof(ioctl_skip_sym_t)) < 0) {
		sprintf(skip_errmsg, SKIP_MSG_SET_VAR, name, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	_skip_close(skip_fd);
	return (0);
}

/*
 * skip_get_status_keys()
 *
 * Get crypt/decrypt keys status
 */
int 
skip_get_status_keys(char *if_name, ioctl_key_status_t *p_status)
{

	skip_fd_t	skip_fd;

	skip_fd = _skip_open(if_name);

	if (skip_fd == SKIPFD_ERR) {
		return (skip_fd);
	}

	if (_skip_ioctl(skip_fd, SKIP_KEY_STATUS, 
			(char *) p_status, sizeof(ioctl_key_status_t)) < 0) {
		sprintf(skip_errmsg, SKIP_MSG_GET_KEY_STATUS, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	_skip_close(skip_fd);
	return (0);
}

/*
 * skip_get_list_keyids()
 *
 * Get list of local Master Key IDs
 */
int 
skip_get_list_keyids(char *if_name, ioctl_keyid_t *p_keyidlst)
{

	skip_fd_t	skip_fd;

	skip_fd = _skip_open(if_name);

	if (skip_fd == SKIPFD_ERR) {
		return (skip_fd);
	}

	if (_skip_ioctl(skip_fd, SKIP_KEY_LIST, 
			(char *) p_keyidlst, sizeof(ioctl_keyid_t)) < 0) {
		sprintf(skip_errmsg, SKIP_MSG_GET_LOCAL_KEYS, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	_skip_close(skip_fd);
	return (0);
}
