/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/
#pragma ident "@(#)skipdb.C	1.21 96/10/14"
#include <skip_os.h>
#include "AuthPolicy.h"
#include "X509skip.h"
#include "HashCert.h"
#include "SkipCert.h"
#include "LL.h"
#include "db.h"
#include "skip_certbase.h"

AuthPolicy auth;
extern char *certs_base;

void
usage(char *s)
{
	fprintf(stderr,"%s [add|extract|init|list|rm]\n",s);
	exit(1);
}

void
add_usage()
{
	fprintf(stderr,"skipdb add -n NSID -t CERTYPE cert\n");
	exit(1);
}

Bstream cert_name(u_char certtype, Bstream certstr)
{
	switch(certtype) {
		case SKIP_CERT_X509: {
			X509SkipCert cert;
			cert.decode(certstr);
			return cert.skip_name();
		}
		case SKIP_DH_PUBLIC: {
			HashCert cert;
			cert.decode(certstr);
			return cert.skip_name();
		}
		default: {
			Bstream nullstr;
			return nullstr;
		}
	}
}

/*
 * Hash a string of bytes.  Uses the CRC-32 polynomial, preset
 * to -1, non-invert.  Used to reduce userID collisions and to
 * create a fake keyID for unparseable keys.
 *
 * CRC-32 polynomial in little-endian order:
 *   1+x+x^2+x^4+x^5+x^7+x^8+x^10+x^11+x^12+x^16+x^22+x^23+x^26+x^32
 *               1   1   2   2   2   3
 *   0   4   8   2   6   0   4   8   2
 * = 111011011011100010000011001000001
 * =    e   d   b   8   8   3   2   0
 * = 0xedb88320
 */
#define CRCPOLY 0xedb88320

u_long 
crc32(Bstream bbuf)
{
	u_long crc;
	int i, j;
	static u_long crctable[256];
	byte const *buf;
	size_t len;
	buf = bbuf.getdatap();
	len = bbuf.getlength();
	if (!crctable[255]) {
		/* crctable[0] is already 0 */
		crctable[128] = crc = CRCPOLY;
		i = 64;
		do {
			crc = crc>>1 ^ (crc & 1 ? CRCPOLY : 0);
			for (j = 0; j < 256; j += 2*i)
				crctable[i+j] = crc ^ crctable[j];
		} while (i >>= 1);
	}

	crc = 0xffffffff;
	while (len--)
		crc = (crc >> 8) ^ crctable[(crc ^ *buf++) & 255];
	return crc;
}

int
fcopy(int in, int out)
{
	char buf[1024];
	int n;

	do {
		n = read(in, buf, 1024);
		if (n > 0)
			write(out, buf, n);
	} while (n > 0);
	return 0;
}

void
skip_add_cert(int argc, char *argv[])
{
	int c; 
	Bstream certstr;
	Bstream castr, nullstr;
	u_char nsid;
	u_char certtype;
	char tmp[255];
	int quiet = 0;

	nsid = certtype = 0;

	argc--;
	while ((c=getopt(argc,&argv[1],"n:t:Vq"))!= -1) {
		 switch (c) {
			case 'n':
				nsid = atoi(optarg);
				break;
			case 't':
				certtype = atoi(optarg);
				if (certtype ==0) 
					certtype=Certname_to_num(optarg);
				break;
			case 'q':
				quiet = 1;
				break;
			default:
				add_usage();
		}
	}
	if (optind >= argc) 
		add_usage();

	optind++;
	if (nsid == 0 || certtype == 0)
		add_usage();

	if (strcmp(argv[optind],"-") == 0)  {
		sprintf(tmp,"/tmp/dbXXXXXX");
		mktemp(tmp);
		int f;
		f = open(tmp, O_CREAT|O_WRONLY, 0600);
		fcopy(0, f);
		close(f);
		certstr=File_to_Bstr(tmp);
		unlink(tmp);
	} else {
		certstr=File_to_Bstr(argv[optind]);
	}

	if (certstr.getlength()==0 )
		add_usage();

	SkipCert *s = NULL;

	if (SkipCert::decode(certtype, certstr, s)) {
		printf(" certificate decode failure: %s\n", argv[optind]);
		exit(1);
	}

	CertInfo ci;

	ci.nsid = nsid;
	ci.mkid = cert_name(certtype, certstr);
	ci.cert = s;

	if (db_put_certinfo(ci)) {
		fprintf(stderr, "database store failure\n");
		exit(1);
	}

	String temp;
	temp = ci.mkid.gethexstr();
	if (!quiet)
		printf("added %s to database\n", (const char *) temp);
}

extern int nsid_sizes[];


void
del_usage()
{
	fprintf(stderr, "skipdb rm [-h handle] -n <nsid> <hostid1>\n");
	exit(1);
}

void
extract_usage()
{
	fprintf(stderr, "skipdb extract [-h handle] -n <nsid> <hostid1>\n");
	exit(1);
}
	 
void
skip_db_ok(int argc, char **argv)
{
	struct stat s;
	if (stat(certs_base, &s))
		exit(1);
	else
		exit(0);
}

void 
skip_extract_cert(int argc, char **argv)
{
	u_char nsid = 0;
	u_long handle = 0;
	Bstream boxid;
	int c;
	int certtype = 0;

	argc--;
	while ((c=getopt(argc,&argv[1],"h:n:t:"))!= -1) {
                switch (c) {
                        case 'n':
                                nsid = atoi(optarg);
                                break;
			case 'h':
				sscanf(optarg, "%ul", &handle);
				break;
			case 't':
				certtype = atoi(optarg);
				if (certtype ==0)
					certtype=Certname_to_num(optarg);
				break;
			default:
				extract_usage();
		}
	}
	optind++;
	if (optind > argc || nsid == 0) 
		extract_usage();

	Bstream_atob(boxid, argv[optind]);

	CertInfoList out;
	db_get_certinfo_list(nsid,  boxid, out);
	for (CertInfo *i=out.first(); i; i=out.next()) {
		if (certtype && i->cert->CertType() != certtype)
			continue;
		if (handle == 0 || (handle == crc32(i->encode()))) {
			Bstream result;
			result = i->cert->encode();
			fwrite(result.getdatap(), result.getlength(), 1, 
				stdout);
			exit(0);
		}
	}
	fprintf(stderr,"Certificate Not found!\n");
	exit(1);
}

void
skip_del_cert( int argc, char **argv)
{
	Bstream certstr;
	if (argc < 4)
		del_usage();
	int retval;
	Bstream boxid;
	u_char nsid=0;
	int c;
	u_long handle = 0;

	argc--;
	while ((c=getopt(argc,&argv[1],"h:n:"))!= -1) {

                switch (c) {
                        case 'n':
                                nsid = atoi(optarg);
                                break;
			case 'h':
				sscanf(optarg, "%ul", &handle);
				break;
			default:
				del_usage();
		}

	}
	optind++;
	if (optind > argc || nsid == 0) 
		del_usage();

	Bstream_atob(boxid, argv[optind]);
	if (boxid.getlength()==0)
		del_usage();
	if (boxid.getlength()!=nsid_sizes[nsid]) {
		fprintf(stderr,
		        "Length of keyid %d != keylength for NSID(%d) %d \n",
			boxid.getlength(), nsid, nsid_sizes[nsid]);
		exit(1);
	}

	CertInfoList cil;
	if (db_get_certinfo_list(nsid, boxid, cil)) {
		fprintf(stderr,"Certificate Not found\n");
		exit(1);
	}
	if (handle) {
		int modified = 0;
		for (CertInfo *ci=cil.first(); ci!= NULL ; ci=cil.next()){
			Bstream m;
			m=ci->encode();
			if (handle == crc32(m)){
				modified = 1;
				cil.delete_current();
			}
		}
		if (modified)
			retval = db_put_certinfo_list(nsid, boxid, cil);
		else
			retval = 1;
	} else
		retval = skip_del_dh_cert(boxid, nsid);
	if (retval) {
		fprintf(stderr, "Error deleting certificate for boxid",
			(const char *)(argv[optind]));
		exit(1);
	}
	printf("Deleted %s hostid from certificate database\n", 
		(const char *)(argv[optind]));
	exit(0);
}

void
init_usage(char *s)
{
	fprintf(stderr,"usage: %s init [-d dbname] [-q] [-o]\n", s);
	exit(1);
}

void
skip_init_certbase(int nargc, char *nargv[])
{
	DB *db;
        extern HASHINFO *db_info();
	struct stat s;
	char *dbname = NULL;
	int quiet = 0;
	int c;
	int overwrite = 0;
 
	while ((c=getopt(nargc-1,&nargv[1],"oqd:"))!= -1) {
		switch(c) {
			case 'q':
				quiet = 1;
				break;
			case 'd':
				dbname = optarg;
				break;
			case 'o':
				overwrite = 1;
				break;
			default:
				init_usage(nargv[0]);
		}
	}

	if (dbname == NULL)
		dbname = certs_base;

	if (!stat(dbname, &s)) {
		if (!overwrite) {
			if (!quiet)
				fprintf(stderr,"won't overwrite current db\n");
			exit(2);
		}
	}
	db = dbopen(dbname, O_RDWR | O_CREAT | O_TRUNC, 0600,
		DB_HASH, db_info());
 
        if (db == NULL) {
		if (!quiet)
			fprintf(stderr, "create of %s failed, error=%s\n",
					certs_base, strerror(errno));
		exit(1);
	}
 
	(db->close)(db);
	if (!quiet)
		printf("Successfully initialized certificate database in %s\n",
			dbname);
 
	exit(0);
}

void
skip_dump_certs(int nargc, char *nargv[])
{
	extern HASHINFO *db_info();
	DB *db;
	u_char wantnsid = 0;
	int cnt = 0;
	u_char	nsid;
	DBT db_key, db_data;
	Bstream keystr, certstr;
	char *name;
	int ret;
	int verbose;
	int machprt;
	int expiration;
	u_long handle = 0;
	int c;

	machprt = expiration = verbose = 0;

	while ((c = getopt(nargc-1, &nargv[1], "h:Vven:")) != EOF)
               switch (c) {
			case 'v':
				verbose = 1;
				break;
			case 'V':
				machprt = 1;
				break;		
			case 'e':
				expiration = 1;
				break;
			case 'n':
				wantnsid = atoi(optarg);
				break;
			case 'h':
				sscanf(optarg, "%ul", &handle);
				break;
			case '?':
				fprintf(stderr,"Invalid flag %c\n", c);
				break;
		}

	optind++;
	if (optind >= nargc) {
		name = NULL;
	} else {
		name = nargv[optind];
		if (name == NULL) {
			fprintf(stderr,"Invalid Regular Expression\n");
			exit(1);
		}
	}	
	skip_db_lock();

	db = dbopen(certs_base, O_RDWR, 0600, DB_HASH, db_info());

	if (db == NULL)  {
		fprintf(stderr, "open of %s failed, error=%s\n",
					certs_base, strerror(errno));
		exit(1);
	}

	while (1) {
		ret = (db->seq)(db, &db_key, &db_data, R_NEXT);

		if (ret < 0)  {
			fprintf(stderr, "cant sequence database %s\n",
						certs_base);
			exit(1);
		}

		if (ret == 1)
			break;

		CertInfoList ll;

		keystr = Bstream(db_key.size, db_key.data);
		certstr = Bstream(db_data.size, db_data.data);

		keystr.fetchbyte(nsid);
		fflush(stdout);

		if (wantnsid != 0 && nsid != wantnsid ) 
			continue;

		String temp;
		if (name != NULL) {
			temp = keystr.gethexstr();
			if (strcmp(name, (const char *)temp)!=0)
				continue;
		}
		if (CertInfoList::decode(certstr, ll)) {
			fprintf(stderr, "certificate decoding failed\n");
			continue;
		}

		for (CertInfo *p=ll.first();p; p=ll.next()) {
			if ( handle ) {
				Bstream m;
				u_long c;
				m = p->encode();
				c = crc32(m);
				if (handle != c )
					continue;
			}
			if (verbose) {
				if (machprt)
					p->cert->machprint();
				else
					p->cert->print();
			} else {
				if (machprt) {
					Bstream res = p->encode();
					printf("handle=%lu ", crc32(res));
					printf("certtype=%s ",
						num_to_Certname(p->cert->CertType()));
					printf("nsid=%d mkid=%s ",
				 	    p->nsid, p->mkid.get_info_str());

				} else {
					printf("Type: %s NSID: %d MKID: %s\n",
					       num_to_Certname(p->cert->CertType()),
					       p->nsid, p->mkid.get_info_str());
				}
				if (expiration) {
				   if (machprt) {
				        printf("invalidbefore=%lu ",
				   		p->cert->skip_notvalidbefore());
				        printf("invalidafter=%lu",
					        p->cert->skip_notvalidafter());
				   } else {
				      long t;
				      t=NTP_TO_UNIX(
						p->cert->skip_notvalidbefore());
				      printf("\tNot Valid Before: %s",
				      ctime(&t));
				      t=NTP_TO_UNIX(
						p->cert->skip_notvalidafter());
				      printf("\tNot Valid After: %s", 
						ctime(&t));
				  }
			 	}
			}
			cnt++;
                       if (machprt)  {
                                if (verbose)
                                        printf("%%%%%%%%%%");
                                printf("\n");
			}
                        fflush(stdout);
		}
	}

	if (verbose) {
		printf("Total of %d certificate(s) in %s database\n", 
				cnt, certs_base);
	}

	(db->close)(db);
	skip_db_unlock();
	exit(0);
}


main(int argc, char *argv[])
{
	auth.load(SKIP_DIR);		// XXX check return value

	if (argc < 2) 
		usage(argv[0]);
	if (strcmp(argv[1],"add")==0)
		skip_add_cert(argc, argv);
	else if (strcmp(argv[1],"rm")==0 || strcmp(argv[1],"del")==0)
		skip_del_cert(argc, argv);
	else if (strcmp(argv[1],"list")==0)
		skip_dump_certs(argc, argv);
	else if (strcmp(argv[1],"init")==0)
		skip_init_certbase(argc, argv);
	else if (strcmp(argv[1],"extract")==0)
		skip_extract_cert(argc, argv);
	else if (strcmp(argv[1],"ispresent")==0)
		skip_db_ok(argc,argv);
	else 
		usage(argv[0]);
	exit(0);
}
