/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skip_keystore.c	1.36 96/06/20 Sun Microsystems"

/*
 * System includes
 */
#include <skip_os.h>

/*
 * SKIP includes
 */
#include <skip_proto.h>
#include <skip_types.h>
#include <skip_keymgrmsgs.h>
#include <skip_crypt.h>
#include <skip_key.h>
#include <skip_ah.h>
#include <skip_ioctl.h>

/*
 * SKIP bulk data cryptor.  Implements skip_encrypt()/skip_decrypt() for
 * kernel clients.
 */


/*
 * hash table information
 */
static MUTEX_T		*skip_encrypt_hashlocks, *skip_decrypt_hashlocks;
static skip_keycb_t	**skip_encrypt_hashtable, **skip_decrypt_hashtable;

/*
 * max number of packets to hold while waiting for a key, max wait time (secs)
 */
int			skip_key_max_pkts = SKIP_KEY_MAX_PKTS;
int			skip_key_max_km_wait = SKIP_KEY_MAX_KM_WAIT;

/*
 * maximum number of bytes to transmit before changing key
 */
int			skip_key_max_bytes = SKIP_KEY_MAX_BYTES;

/*
 * delete an unused key control block after this time (secs)
 */
int			skip_key_max_idle = SKIP_KEY_MAX_IDLE;

/*
 * maximum time to use an encrypt key before changing it
 */
int			skip_key_max_use = SKIP_KEY_MAX_USE;

/*
 * maximum number of cipher states to have open at one time
 */
static const unsigned long	skip_key_max_entries = SKIP_KEY_MAX_ENTRIES;

/*
 *  Ioctl stats interface
 */
ioctl_key_stat_t	skip_keys_stats = { 0 };
int			skip_params_debug = 0;

/*
 * Supported Kij algorithms by the key manager
 */
boolean_t	skip_kmgr_kij_alg[SKIP_MAXCRYPTORS] = { 0 };
boolean_t	skip_kmgr_kij_alg_v1[SKIP_MAXCRYPTORS] = { 0 };


/*
 * Forward declarations for static functions
 */
static int	skip_key_destroy(skip_keycb_t *, void *);
static skip_keycb_t *
		skip_encrypt_hash_locate(skip_param_t *, unsigned int); 
static skip_keycb_t *
		skip_decrypt_hash_locate(skip_param_t *, char *,
					int, unsigned int); 
static skip_keycb_t *
		skip_encrypt_hash_add(unsigned int);
static skip_keycb_t *
		skip_decrypt_hash_add(unsigned int);
static void
		skip_hash_remove(skip_keycb_t *, skip_keycb_t **,
					unsigned long *);
static int	queue_pkt(skip_keycb_t *, SKIP_BUF_T *, SKIP_BUF_T *,
					skip_client_t *, void *);

/* skip_key_initstore()
 *
 * Initialise the key store
 *
 * Returns: none
 */
void
skip_key_initstore()
{
	register int	i;

	skip_encrypt_hashtable = (skip_keycb_t **)
		SYSMEM_ALLOC(sizeof (skip_keycb_t *) * SKIP_HASH_TABLESZ);
	if (skip_encrypt_hashtable == NULL) {
		PANIC("skip: no encrypt hash table memory");
	}
	BZERO(skip_encrypt_hashtable,
			sizeof (skip_keycb_t *) * SKIP_HASH_TABLESZ);

	skip_decrypt_hashtable = (skip_keycb_t **)
		SYSMEM_ALLOC(sizeof (skip_keycb_t *) * SKIP_HASH_TABLESZ);
	if (skip_decrypt_hashtable == NULL) {
		PANIC("skip: no decrypt hash table memory");
	}
	BZERO(skip_decrypt_hashtable,
			sizeof (skip_keycb_t *) * SKIP_HASH_TABLESZ);

	skip_encrypt_hashlocks = (MUTEX_T *)
		SYSMEM_ALLOC(sizeof (MUTEX_T) * SKIP_HASH_TABLESZ);
	if (skip_encrypt_hashlocks == NULL) {
		PANIC("skip: no encrypt lock table memory");
	}

	skip_decrypt_hashlocks = (MUTEX_T *)
		SYSMEM_ALLOC(sizeof (MUTEX_T) * SKIP_HASH_TABLESZ);
	if (skip_decrypt_hashlocks == NULL) {
		PANIC("skip: no decrypt lock table memory");
	}

	for (i = 0; i < SKIP_HASH_TABLESZ; i++) {
		MUTEX_INIT(&skip_encrypt_hashlocks[i], "SKIP encrypt hash");
		MUTEX_INIT(&skip_decrypt_hashlocks[i], "SKIP decrypt hash");
	}

	skip_local_keyid_init();
}

/* skip_key_uninitstore()
 *
 * Uninitialise the key store
 *
 * Returns: none
 */
void
skip_key_uninitstore()
{
	register int	i;

	skip_key_iterate(skip_key_destroy, NULL);

	for (i = 0; i < SKIP_HASH_TABLESZ; i++) {
		MUTEX_DESTROY(&skip_encrypt_hashlocks[i]);
		MUTEX_DESTROY(&skip_decrypt_hashlocks[i]);
	}

	SYSMEM_FREE(skip_encrypt_hashlocks,
		sizeof (MUTEX_T) * SKIP_HASH_TABLESZ);
	skip_encrypt_hashlocks = NULL;

	SYSMEM_FREE(skip_decrypt_hashlocks,
		sizeof (MUTEX_T) * SKIP_HASH_TABLESZ);
	skip_decrypt_hashlocks = NULL;

	SYSMEM_FREE(skip_encrypt_hashtable, sizeof (skip_keycb_t *) * 
		SKIP_HASH_TABLESZ);
	skip_encrypt_hashtable = NULL;

	SYSMEM_FREE(skip_decrypt_hashtable, sizeof (skip_keycb_t *) * 
		SKIP_HASH_TABLESZ);
	skip_decrypt_hashtable = NULL;

	skip_local_keyid_uninit();
}

/* skip_open()
 *
 * Give a SKIP client access to the bulk data crypt service
 *
 * Returns: skip descriptor
 */
void *
skip_open(
	void *client_data,
	void (*encrypt_done)(
		void *, SKIP_BUF_T *, SKIP_BUF_T *, void *, skip_arg_t *),
	void (*decrypt_done)(
		void *, SKIP_BUF_T *, SKIP_BUF_T *, void *, skip_arg_t *)
)
{
	register skip_client_t	*sd;

	if (encrypt_done == NULL) {
		PANIC("skip_open: NULL encrypt_done");
	}

	if (decrypt_done == NULL) {
		PANIC("skip_open: NULL decrypt_done");
	}

	sd = (skip_client_t *) SYSMEM_ALLOC(sizeof(skip_client_t));
	if (sd == NULL) {
		return (NULL);
	}

	sd->client_data = client_data;
	sd->encrypt_done = encrypt_done;
	sd->decrypt_done = decrypt_done;

	return(sd);
}

/* skip_client_close()
 *
 * Remove all references to a SKIP client who is closing
 *
 * Returns: none
 */
static int
skip_client_close(skip_keycb_t *cp, void *sd)
{
	skip_queue_t		*callback;

	for (callback = cp->queue; callback; callback = callback->next) {
		if (callback->client == sd) {
			/*
			 * callback for this client - zap it
			 */
			return (1);
		}
	}
	return (0);
}

/* skip_close()
 *
 * Remove bulk data crypt user
 *
 * Returns: none
 */
void
skip_close(void *sd)
{
	skip_key_iterate(skip_client_close, sd);
	SYSMEM_FREE(sd, sizeof(skip_client_t));
}

/*
 * Get the IP protocol from original buffer
 * and the data offset.
 */
static unsigned int 
skip_get_proto(skip_param_t *params, SKIP_BUF_T *in)
{
	int unsigned len = 0;

	if ((params->version != SKIP_V2) && (params->version != SKIP_RAW)) {
		return(0);
	}

	if (params->ip_mode == SKIP_TRS_ON) {
		/*
		 * SKIP transport mode
		 */
		SKIP_GET_PROTO(params, in, len);
	} else {
		/*
	 	 * SKIP tunneling mode
	 	 */
		params->payload_type = IPPROTO_ENCAP;
	}
	/*
	 * Return offset of data to decrypt
	 */
	return(len);
}

/* skip_encryptpkt()
 *
 * Encrypt a packet using the supplied key information.
 * The original packet is unchanged.
 *
 * Returns: 0 on success, non-zero on error
 */
static int
skip_encryptpkt(skip_keycb_t *cp, SKIP_BUF_T *in, SKIP_BUF_T *out)
{
	unsigned int		msize;
	register skip_cryptor_t	*alg = NULL;
	register skip_mac_t	*mac = NULL;
	skip_param_t		*params = &cp->hdr.params;
	unsigned int		hdr_offset = 0;
	unsigned int		payload_offset;
	unsigned int		ah_size = 0;
	unsigned int		in_offset =  0;
	unsigned char		*ah_start = NULL;
	unsigned char		*in_start, *ip_start;


	/*
	 * Keep address of the original buffer 
	 */
	in_start = BUFSTART(in);

	/*
	 * Get the IP protocol from original buffer
	 * and the offset of the data
	 */
	in_offset = skip_get_proto(params, in);

	in_start += in_offset;
	msize = BUFLEN(in) - in_offset;

	/*
	 * Keep the address of the IP Header
	 */
	ip_start = BUFSTART(out);

	/*
	 * Insert the SKIP header
	 */
	hdr_offset = skip_hdr_encode(&cp->hdr, BUFEND(out));
	BUFGROW(out, hdr_offset);

	if (params->mac_alg) {

		/*
		 * Keep the address of the begining of the AH Header
		 */
		ah_start = BUFEND(out);

		/*
		 * Keep space for the AH Header
		 */
		mac = GETMACMOD(params->mac_alg);
		if (!mac) {
			return (EINVAL);
		}
		ah_size = SKIP_AH_HDR_SZ + mac->data_len;
		BUFGROW(out, ah_size);
	}


	if (params->kp_alg) {

		/*
		 * The packet must be encrypted...
		 */
		alg = GETCRYPTMOD(params->version, params->kp_alg);

		payload_offset = skip_iv_encode(&cp->hdr, BUFEND(out));
		BUFGROW(out, payload_offset);

		/*
		 * note how many bytes we have sent this key
		 */
		cp->obytes += msize;
	
		/*
		 * encrypt the packet...
		 */
		(*alg->encrypt)(cp->cs,
				(char *) in_start,
				&msize,
				(char *) BUFEND(out),
				cp->hdr.mid.buf,
				cp->hdr.mid.len,
				&params->payload_type);

		if (msize) {
			/*
			 * set the encrypted packet's size and destination
			 */
			BUFGROW(out, msize);
			alg->encrypts++;
		} else {
			/*
			 * XXX something went wrong... trap to management
			 */
			if (alg->encrypterrs % 5 == 0) {
				SKIP_DEBUG1(
					"skip_encryptpkt: %s encryption failed",
					alg->name);
			}
			alg->encrypterrs++;
			return (EINVAL);
		}

	} else {
		/*
		 * Packet content is sent in clear...
		 */
		BCOPY(in_start, BUFEND(out), msize);
		BUFGROW(out, msize);
	}

	if (params->mac_alg) {
		/*
		 * now, fill in the AH header
		 */
		if (skip_sign(cp, params, ip_start, ah_start, BUFLEN(out))) {
			return (EINVAL);
		}
	}

	return (0);
}

/* skip_update_encrypt_key()
 *
 * The key manager has responded with a key. Update the associated key block 
 * and encrypt any waiting packets.
 *
 * Returns: none
 */
void
skip_update_encrypt_key(struct Encrypted_key_rsp *er)
{
	register skip_keycb_t	*cp;
	register skip_queue_t	*callback = NULL, *nextcallback;
	register unsigned int	h;
	skip_param_t		*params;
	register skip_cryptor_t	*alg;

	SKIP_DEBUG("skip_update_encrypt_key: updating key control block\n");

	params = &(er->params);
	/*
	 * lock the corresponding entry in the hash table
	 */
	h = SKIP_ENCRYPT_HASH(params->r_mkeyid.buf);

	MUTEX_ENTER(&skip_encrypt_hashlocks[h]);

	cp = skip_encrypt_hash_locate(params, h);

	if (cp) {

		/*
		 * initialise E_kp/A_kp in the key control block
		 */
		KEYVARSET(cp->E_kp, er->E_kp);
		KEYVARSET(cp->A_kp, er->A_kp);

		/*
		 * initialise the message indicator in the key control block
		 * to the initialisation vector supplied by the key manager 
		 */
		KEYVARSET(cp->hdr.mid, er->iv);

		/*
		 * initialise ekp in the key control block
		 */
		KEYVARSET(cp->hdr.ekp, er->ekp);

		/*
		 * ESP/AH mode
		 */
		if (params->version == SKIP_V2) {

			cp->hdr.params.counter_n	= params->counter_n;
			cp->hdr.params.ttl		= params->ttl;	
		}
		
		/*
		 * Pre-compute the fixed part of SKIP header
		 */
		skip_hdr_init(cp);

		/*
		 * Packet has, at least, to be encrypted
		 */
		if (params->kp_alg) {

			alg = GETCRYPTMOD(params->version, 
						cp->hdr.params.kp_alg);
			if (cp->cs) {
				/*
				 * close existing key handle
				 */
				(alg->close)(cp->cs);
			}

			/*
			 * open new handle to crypt engine
			 */
			cp->cs = (alg->open)(
					cp->E_kp.buf,
					cp->E_kp.len,	
					(char *) er->iv.buf,
					er->iv.len);
	
			if (cp->cs == NULL) {
				/*
				 * open of crypto algorithm failed
				 */
				MUTEX_EXIT(&skip_encrypt_hashlocks[h]);
				return;
			}
		}

		/*
		 * reset the bytes transmitted
		 */
		cp->max_obytes 	= skip_key_max_bytes;
		cp->obytes	= 0;
		cp->last_obytes	= 0;

		/*
		 * reset the encrypt key's time to live value
		 */
		if (params->version == SKIP_V2) {
			cp->ttl = skip_key_max_idle > params->ttl ? 
					params->ttl : skip_key_max_idle;
		} else {
			cp->ttl = skip_key_max_idle;
		}

		/*
		 * Update the CB status...
		 */
		cp->init = B_TRUE;

		/*
		 * encrypt any packets which were waiting for the 
		 * key information. We queue them so that we can send them
		 * all in one go without holding any locks.
		 */
		SKIP_DEBUG("skip_update_encrypt_key: encrypting queued pkts");

		/*
		 * make a note of head of encrypted packet list
		 */

		callback = cp->queue;

		for (; cp->queue; cp->queue = cp->queue->next) {
			/*
			 * encrypt the packet
			 */
			cp->queue->res.rc = skip_encryptpkt(cp,
							cp->queue->in,
							cp->queue->out);
			/*
			 * Save SKIP parameters for encrypt_done()
			 */
			PARAMSCOPY(&cp->hdr.params, &cp->queue->res.params);
			cp->queue->res.sync = B_FALSE;
		}
	}
	MUTEX_EXIT(&skip_encrypt_hashlocks[h]);

	/*
	 *  send the encrypted packets
	 */
	for (; callback; callback = nextcallback) {

		nextcallback = callback->next;

		/*
		 * send encrypted packet on its way...
		 */
		(*callback->client->encrypt_done)(callback->client->client_data,
						callback->in,
						callback->out,
						callback->arg,
						&callback->res);
		SYSMEM_FREE(callback, sizeof (*callback));
		SKIP_DEBUG(".");
	}
	SKIP_DEBUG("\n");
}

/* skip_encrypt()
 *
 * Find a key control block  given a key id and kp_alg then
 * encrypt the packet. "plain" contains the original packet
 * "encrypted" is the buffer to write the encrypted packet into
 */
int
skip_encrypt(void *sd,
		SKIP_BUF_T	*plain,
		SKIP_BUF_T	*encrypted,
		skip_param_t	*skip_parms,
		void		*arg
)
{
	register skip_client_t	*client = sd;
	register skip_keycb_t	*cp;
	skip_arg_t		res;
	register unsigned int	h;

	skip_keys_stats.skip_key_lookups++;

	/*
	 * lock the corresponding entry in the hash table
	 */
	h = SKIP_ENCRYPT_HASH(skip_parms->r_mkeyid.buf);

	MUTEX_ENTER(&skip_encrypt_hashlocks[h]);

	cp = skip_encrypt_hash_locate(skip_parms, h);

	/*
	 * Set default mode to synchronous and save
	 * SKIP parameters for decrypt_done() callback
	 */
	res.sync = B_TRUE;
	PARAMSCOPY(skip_parms, &res.params);

	/*
	 * We have the key info..
	 */
	if (cp && cp->init) {

		/*
		 * holding the key information - encrypt and go...
		 */
		res.rc = skip_encryptpkt(cp, plain, encrypted);

		MUTEX_EXIT(&skip_encrypt_hashlocks[h]);

		/*
		 * send encrypted packet on its way...
		 */
		(*client->encrypt_done)(client->client_data,
					plain,
					encrypted,
					arg,
					&res);
		return(res.rc);
	}

	/*
	 * cp may or may not be NULL depending on whether we have
	 * already hit the key manager yet for this key
	 */
	if (cp == NULL) {

		cp = skip_encrypt_hash_add(h);

		if (cp == NULL) {

			MUTEX_EXIT(&skip_encrypt_hashlocks[h]);

			res.rc = ENOMEM;

			(*client->encrypt_done)(client->client_data,
						plain,
						encrypted,
						arg,
						&res);
			return(res.rc);
		}

		cp->hash	= h;
		cp->max_obytes	= SKIP_IGNORE_VAL;
		PARAMSCOPY(skip_parms, &cp->hdr.params);

		/*
		 * this allows the key manager skip_key_max_km_wait seconds
		 * to respond, after which the request is deleted
		 */
		cp->ttl	= skip_key_max_km_wait;

		/*
		 * hit the key manager for the key
		 */
		skip_key_tellkm(SKIP_ENCRYPTED_PKT_KEY_REQ, cp);
	}

	/*
	 * queue the packets up to a defined maximum
	 */
	res.rc = queue_pkt(cp, plain, encrypted, client, arg);
	MUTEX_EXIT(&skip_encrypt_hashlocks[h]);

	if (res.rc) {
		/*
		 * failed to queue the packet to encrypt
		 */
		(*client->encrypt_done)(client->client_data,
						plain,
						encrypted,
						arg,
						&res);
		return(res.rc);
	}

	return(EINPROGRESS);
}

/* skip_decryptpkt()
 *
 * decrypt a packet using the supplied key information. Returns the 
 * decrypted packet on sucess or NULL on failure. The original packet is
 * unchanged.
 */
static int
skip_decryptpkt(skip_keycb_t *cp,
		SKIP_BUF_T *in,
		SKIP_BUF_T *out,
		int hdr_offset,
		skip_arg_t *res
)
{
	register int		payload_offset = 0;
	register int		iv_size = 0;
	register int		ah_size = 0;
	unsigned int		msize = 0;
	register skip_param_t	*params;
	register skip_cryptor_t	*alg;

	params = &cp->hdr.params;

	res->rc		= 0;
	res->modes	= params->ip_mode;
	res->offset	= hdr_offset;

	/*
	 * key was used - reset the time to live
	 */
	cp->ttl = skip_key_max_idle;

	if (params->mac_alg) {
		/*
		 * The packet was authenticated...
		 */
		res->modes |= SKIP_MAC_ON;
		ah_size = skip_auth(cp,
					params,
					BUFSTART(in),
					BUFSTART(in) + hdr_offset,
					BUFLEN(in));
		if (ah_size < 0) {
			/*
			 * Authentication failed - we must drop the packet
			 */
			res->rc = EINVAL;
			return(EINVAL);
		}
		res->offset += ah_size;
	}

	if (!params->kp_alg) {
		/*
		 * The packet was sent in clear...
		 */
		res->proto = params->payload_type;
		return(0);
	}

	/*
	 * We must also decrypt the packet...
	 */
	alg = GETCRYPTMOD(params->version, params->kp_alg);
	res->modes |= SKIP_CRYPT_ON;

	/*
	 * Get the MID/IV value as well as the payload offset
	 */
	if (params->version == SKIP_V1) {
		iv_size = skip_iv_v1_decode(&cp->hdr,
					BUFSTART(in) + hdr_offset + ah_size,
					BUFEND(in));
	} else {
		iv_size = skip_iv_esp_decode(&cp->hdr,
					BUFSTART(in) + hdr_offset + ah_size,
					BUFEND(in));
	}

	if (iv_size < 0) {
		res->rc = EPROTO;
		return(EPROTO);
	}

	payload_offset = hdr_offset + ah_size + iv_size;

	msize = BUFLEN(in) - payload_offset;

	/*
	 * decrypt the packet
	 */
	(*alg->decrypt)(cp->cs, 
			(char *) BUFSTART(in) + payload_offset,
			&msize,
			(char *) BUFSTART(out),
			cp->hdr.mid.buf,
			cp->hdr.mid.len,
			&params->payload_type);

	if (msize) {
		BUFGROW(out, msize);
		alg->decrypts++;
	} else {
		/*
		 * XXX something went wrong... trap to management
		 */
		if (alg->decrypterrs % 1000 == 0) {
			SKIP_DEBUG1(
				"skip_decryptpkt: %s decryption failed",
				alg->name);
		}
		alg->decrypterrs++;
		res->rc = EINVAL;
		return(EINVAL);
	}

	res->proto = params->payload_type;

	return (0);
}

/* skip_update_decrypt_key()
 *
 * The key manager has responded with a key. Update the associated key block 
 * and decrypt any waiting packets.
 *
 * Returns: none
 */
void
skip_update_decrypt_key(struct Decrypted_key_rsp *dr)
{
	register skip_keycb_t	*cp;
	register skip_queue_t	*callback = NULL, *nextcallback;
	register unsigned int	h;
	skip_param_t		*params;
	register skip_cryptor_t	*alg;

	params = &(dr->params);

	h = dr->ekp.len ? SKIP_DECRYPT_HASH(dr->ekp.buf) : 0;

	MUTEX_ENTER(&skip_decrypt_hashlocks[h]);

	cp = skip_decrypt_hash_locate(params, dr->ekp.buf, dr->ekp.len, h);

	if (cp) {
		/*
		 * found it...initialise E_kp/A_kp in the key control block
		 */
		KEYVARSET(cp->E_kp, dr->E_kp);
		KEYVARSET(cp->A_kp, dr->A_kp);

		if (params->version == SKIP_V2) {
			/*
			 * IPSP mode only..
			 */
			cp->hdr.params.counter_n	= params->counter_n;
			cp->hdr.params.ttl		= params->ttl;	
		}

		if (cp->hdr.params.kp_alg) {

			alg = GETCRYPTMOD(cp->hdr.params.version,
						cp->hdr.params.kp_alg);
			if (cp->cs) {
				/*
				 * close existing key handle
				 */
				(alg->close)(cp->cs);
			}

			/*
		 	 * open the crypto algorithm for decrypt
		 	 */
			cp->cs = (alg->open)(
					cp->E_kp.buf, cp->E_kp.len,	
					cp->hdr.mid.buf, cp->hdr.mid.len);
	
			if (cp->cs == NULL) {
				/*
				 * open failed
				 */
				SKIP_DEBUG1("skip_update_decrypt_key: crypto "
					"algorithm %d open failed\n",
						cp->hdr.params.kp_alg);
	
				MUTEX_EXIT(&skip_decrypt_hashlocks[h]);
				return;
			}

		}

		/*
		 * Update the CB status
		 */
		cp->init = B_TRUE;

		/*
		 * initialise the key's time to live value
		 */
		cp->ttl = skip_key_max_idle;

		/*
		 * decrypt any packets which were waiting for the 
		 * key information. We queue them so that we can send them
		 * all in one go without holding the mutex.
		 */
		callback = cp->queue;

		for (; cp->queue; cp->queue = cp->queue->next) {

			cp->queue->res.rc = skip_decryptpkt(cp,
						cp->queue->in,
						cp->queue->out,
						cp->data_offset,
						&cp->queue->res);

			/*
			 * Save SKIP parameters for decrypt_done()
			 */
			PARAMSCOPY(&cp->hdr.params, &cp->queue->res.params);
			cp->queue->res.sync = B_FALSE;
		}

	}

	MUTEX_EXIT(&skip_decrypt_hashlocks[h]);

	/*
	 *  send the decrypted packets
	 */
	for (; callback; callback = nextcallback) {

		nextcallback = callback->next;

		/*
		 * send decrypted packet on its way...
		 */
		(*callback->client->decrypt_done)(
						callback->client->client_data,
						callback->in,
						callback->out,
						callback->arg,
						&callback->res);
		SYSMEM_FREE(callback, sizeof(*callback));
		SKIP_DEBUG(".");
	}
	SKIP_DEBUG("\n");
}

/* skip_decrypt()
 *
 * Find a key control block and decrypt the packet. 
 *
 */
int
skip_decrypt(void * sd,
	     SKIP_BUF_T *encrypted,
	     SKIP_BUF_T *plain,
	     skip_hdr_t	*hdr,
	     int	data_offset,
	     void 	*arg)
{
	register skip_client_t	*client = sd;
	register skip_keycb_t	*cp;
	skip_arg_t		res;
	register unsigned int	h;

	skip_keys_stats.skip_key_lookups++;

	/*
	 * lock the corresponding entry in the hash table
	 */
	h = hdr->ekp.len ? SKIP_DECRYPT_HASH(hdr->ekp.buf) : 0;

	MUTEX_ENTER(&skip_decrypt_hashlocks[h]);

	cp = skip_decrypt_hash_locate(&hdr->params,
					hdr->ekp.buf, hdr->ekp.len, h);

	/*
	 * Set default mode to synchronous and save
	 * SKIP parameters for decrypt_done() callback
	 */
	res.sync = B_TRUE;
	PARAMSCOPY(&hdr->params, &res.params);

	/*
	 * We hold the key info...
	 */
	if (cp && (cp->init == B_TRUE)) {

		/*
		 * update header information
		 */
		HDRCOPY(hdr, &cp->hdr);
		cp->data_offset = data_offset;

		/*
		 * holding valid key information 
		 * authencticate and/or decrypt and go.
		 */
		skip_decryptpkt(cp, encrypted,
					plain, data_offset, &res);

		MUTEX_EXIT(&skip_decrypt_hashlocks[h]);
	
		/*
		 * send decrypted packet on its way...
		 */
		(*client->decrypt_done)(client->client_data,
						encrypted,
						plain,
						arg,
						&res);
		return(res.rc);
	}

	/*
	 * not currently holding valid decrypt key information
	 */
	if (cp == NULL) {
		/*
		 * This is the first time we have had to decrypt a packet
		 * like this.  Make a new key control block.
		 */
		cp = skip_decrypt_hash_add(h);

		if (cp == NULL) {

			MUTEX_EXIT(&skip_decrypt_hashlocks[h]);
			/*
			 * memory allocation failure?
			 */
			res.rc		= ENOMEM;

			(*client->decrypt_done)(
					client->client_data,
					encrypted,
					plain,
					arg,
					&res);

			return(res.rc);
		}

		cp->hash	= h;
		cp->data_offset = data_offset;
		cp->max_obytes	= SKIP_IGNORE_VAL;
		HDRCOPY(hdr, &cp->hdr);

		/*
		 * this allows the key manager skip_key_max_km_wait seconds
		 * to respond, after which the request is deleted
		 */
		cp->ttl		= skip_key_max_km_wait;

		/*
		 * hit the key manager for the key
		 */
		skip_key_tellkm(SKIP_DECRYPTED_PKT_KEY_REQ, cp);
	}

	/*
	 * queue the packet up to a defined maximum
	 */
	res.rc = queue_pkt(cp, encrypted, plain, client, arg);
	MUTEX_EXIT(&skip_decrypt_hashlocks[h]);

	if (res.rc) {
		/*
		 * failed to queue the packet to decrypt
		 */
		(*client->decrypt_done)(
				client->client_data,
				encrypted,
				plain,
				arg,
				&res);
		return(res.rc);
	}

	return(EINPROGRESS);
}

/* skip_key_check()
 *
 * When iterated over the key list, decides which keys need attention -
 * either they should be removed or they need updated.
 *
 * Returns: 1 if entry should be removed, 0 otherwise
 */
int
skip_key_check(skip_keycb_t *cp, void *arg)
{
	if (cp->max_obytes != SKIP_IGNORE_VAL) {
		/*
		 * check to see if we need to change encryption key
		 */
		if ((cp->obytes >= (unsigned) cp->max_obytes) ||
			((cp->last_obytes != cp->obytes) &&
			 (cp->ttl <= skip_key_tick))) {
			/*
			 * time to change encryption keys. Ask for it
			 * now and give the key manager a short time to
			 * respond
			 */
			cp->max_obytes	= SKIP_IGNORE_VAL;
			cp->ttl		= skip_key_max_km_wait;
			skip_key_tellkm(SKIP_ENCRYPTED_PKT_KEY_REQ, cp);
			return (0);
		}
		cp->last_obytes = cp->obytes;
	}

	/*
	 * check to see if key has expired
	 */
	if ((cp->ttl -= skip_key_tick) <= 0) {
		return (1);
	}

	return (0);
}

/* skip_key_destroy()
 *
 * When iterated over the key list, destroys all keys.
 *
 * Returns: 1
 */
/*ARGSUSED*/
static int
skip_key_destroy(skip_keycb_t *cp, void *arg)
{
	return (1);
}

/* skip_key_iterate()
 *
 * apply a function to all entries. The function may specify the removal of
 * the current element by returning a non-zero return code.
 *
 * Returns: none
 */
void
skip_key_iterate(int (*f)(), void *arg)
{
	skip_keycb_t	*cp, *ncp;
	int		i, rc;

	for (i = 0; i < SKIP_HASH_TABLESZ; i++) {

		/*
		 * iterate over the encryption key control blocks
		 */
		MUTEX_ENTER(&skip_encrypt_hashlocks[i]);
		for (cp = skip_encrypt_hashtable[i]; cp; cp = ncp) {
			ncp = cp->next;

			rc = (*f)(cp, arg);

			if (rc) {
				skip_hash_remove(cp, skip_encrypt_hashtable,
				&skip_keys_stats.skip_encrypt_keys_active);
			}
		}
		MUTEX_EXIT(&skip_encrypt_hashlocks[i]);
	}
	for (i = 0; i < SKIP_HASH_TABLESZ; i++) {

		/*
		 * iterate over the decryption key control blocks
		 */
		MUTEX_ENTER(&skip_decrypt_hashlocks[i]);
		for (cp = skip_decrypt_hashtable[i]; cp; cp = ncp) {
			ncp = cp->next;

			rc = (*f)(cp);

			if (rc) {
				skip_hash_remove(cp, skip_decrypt_hashtable,
				&skip_keys_stats.skip_decrypt_keys_active);
			}
		}
		MUTEX_EXIT(&skip_decrypt_hashlocks[i]);
	}
}

/* queue_pkt()
 *
 * add a pair of {in, out} packets plus details of who to call back
 * to the key control block (up to the specified limit)
 */
static int
queue_pkt(skip_keycb_t *cp,
		SKIP_BUF_T *in,
		SKIP_BUF_T *out,
		skip_client_t *client,
		void *arg)
{
	register int			pktcnt;
	register skip_queue_t		*queue, *prevqueue;

	pktcnt = 0;
	prevqueue = NULL;
	for (queue = cp->queue; queue; queue = queue->next) {
		prevqueue = queue;
		pktcnt++;
	}

	if (pktcnt >= skip_key_max_pkts) {
		/*
		 * holding queue overflow
		 */
		return(ENOSPC);
	}

	queue = (skip_queue_t *) SYSMEM_ALLOC(sizeof (*queue));

	if (queue == NULL) {
		/*
		 * No memory
		 */
		return(ENOMEM);
	}

#ifdef KERNEL
	/*
	 * XXX 4.x only
	 * we have knowledge that arg is a pointer to a struct sockaddr which
	 * needs to be preserved across function calls
	 */
	if (arg) {
		queue->dst = * (struct sockaddr *) arg;
		arg = &queue->dst;
	}
#endif
	 
	queue->next	= NULL;
	queue->in	= in;
	queue->out	= out;
	queue->client	= client;
	queue->arg	= arg;

	if (prevqueue == NULL) {
		cp->queue = queue;
	} else {
		prevqueue->next = queue;
	}

	return(0);
}

/* skip_encrypt_hash_locate()
 *
 * Find a hash table entry for an encrypt key
 *
 * Returns: the entry or NULL if not found
 */
static skip_keycb_t *
skip_encrypt_hash_locate(skip_param_t *params, unsigned int hash)
{
	register skip_keycb_t		*cp;

	cp = skip_encrypt_hashtable[hash];

	while (cp) {
		/*
		 * check collision entries...
		 */
		if (PARAMSMATCH(params, &cp->hdr.params)) {
			/*
			 * found it...
			 */
			break;
		} else {
			SKIP_DIFFS("skip_encrypt_hash_locate",
					params, &cp->hdr.params);
		}
		cp = cp->next;
	}
	return (cp);
}

/* skip_encrypt_hash_add()
 *
 * Add a hash table entry for an encryt key
 *
 * Returns: the entry or NULL on error
 */
static skip_keycb_t *
skip_encrypt_hash_add(unsigned int hash)
{
	register skip_keycb_t	*cp;

	if (skip_keys_stats.skip_encrypt_keys_active == skip_key_max_entries) {
		return (NULL);
	}

        cp = (skip_keycb_t *) SYSMEM_ALLOC(sizeof(*cp));

	if (cp == NULL) {
		return (NULL);
	}
	BZERO(cp, sizeof(*cp));

	skip_keys_stats.skip_encrypt_keys_active++;

	if (skip_encrypt_hashtable[hash] == NULL) {
		skip_encrypt_hashtable[hash] = cp;
	} else {
		/*
		 * hash collision found - insert in collision chain...
		 */
		skip_keys_stats.skip_hash_collisions++;
		cp->next = skip_encrypt_hashtable[hash];
		skip_encrypt_hashtable[hash]->prev = cp; 
		skip_encrypt_hashtable[hash] = cp; 
	}
	return(cp);
}

/* skip_decrypt_hash_locate()
 *
 * Find a hash table entry for a decrypt key
 *
 * Returns: the entry or NULL if not found
 */
static skip_keycb_t *
skip_decrypt_hash_locate(skip_param_t *params, 
				char *ekpbuf,
				int ekplen,
				unsigned int hash
)
{
	register skip_keycb_t		*cp;

	cp = skip_decrypt_hashtable[hash];

	while (cp) {
		/*
		 * check collision entries...
		 */
		if (((!ekplen && !cp->hdr.ekp.len)  || 
				!BCMP(ekpbuf, cp->hdr.ekp.buf, ekplen)) &&
				PARAMSMATCH(params, &cp->hdr.params)) {
			/*
			 * found it...
			 */
			break;
		}
		cp = cp->next;
	}

	return (cp);
}

/* skip_decrypt_hash_add()
 *
 * Add a hash table entry for a decryt key
 *
 * Returns: the entry or NULL on error
 */
static skip_keycb_t *
skip_decrypt_hash_add(unsigned int hash)
{
	register skip_keycb_t	*cp;

	if (skip_keys_stats.skip_decrypt_keys_active == skip_key_max_entries) {
		return (NULL);
	}

        cp = (skip_keycb_t *) SYSMEM_ALLOC(sizeof(*cp));

	if (cp == NULL) {
		return (NULL);
	}
	BZERO(cp, sizeof(*cp));

	skip_keys_stats.skip_decrypt_keys_active++;

	if (skip_decrypt_hashtable[hash] == NULL) {
		skip_decrypt_hashtable[hash] = cp;
	} else {
		/*
		 * hash collision found - insert in collision chain...
		 */
		skip_keys_stats.skip_hash_collisions++;
		cp->next = skip_decrypt_hashtable[hash];
		skip_decrypt_hashtable[hash]->prev = cp; 
		skip_decrypt_hashtable[hash] = cp; 
	}
	return(cp);
}


/* skip_hash_remove()
 *
 * Returns: remove a decryption hash table entry
 */
static void
skip_hash_remove(skip_keycb_t * cp, skip_keycb_t *table[], 
			     unsigned long *countp)
{
	register skip_queue_t	*callback, *nextcallback;
	register skip_cryptor_t	*alg;

	if (cp->next != NULL) {
		cp->next->prev = cp->prev;
	}
	if (cp->prev != NULL) {
		cp->prev->next = cp->next;
	} else {
		table[cp->hash] = cp->next;
	}

	/*
	 * drop any waiting packets
	 */
	for (callback = cp->queue; callback; callback = nextcallback) {

		nextcallback = callback->next;

		BUFFREE(callback->in);
		if (callback->out) {
			BUFFREE(callback->out);
		}
		SYSMEM_FREE(callback, sizeof(*callback));

	}

	if (cp->cs) {
		/*
		 * close key handle
		 */
		alg = GETCRYPTMOD(cp->hdr.params.version,
					cp->hdr.params.kp_alg);
		(alg->close)(cp->cs);
	}

	/*
	 * XXX anything else that should be zapped?
	 */

	SYSMEM_FREE(cp, sizeof(*cp));

	(*countp)--;

}


/*
 * skip_get_key_status()
 *
 *  fill in key status ioctl
 */
int
skip_get_key_status(ioctl_key_status_t *p_status)
{

	register skip_keycb_t	*cp;
	register unsigned int	hash;

	p_status->count_out = 0;
	p_status->count_in = 0;


	/*
	 * Get encrypt keys status
	 */
	for (hash = 0; hash < SKIP_HASH_TABLESZ; hash++) {

		MUTEX_ENTER(&skip_encrypt_hashlocks[hash]);

		cp = skip_encrypt_hashtable[hash];

		while (cp) {
			if ((p_status->params.r_nsid ==
					cp->hdr.params.r_nsid) &&
				KEYVAREQUAL(p_status->params.r_mkeyid,
					cp->hdr.params.r_mkeyid) &&
				!BCMP((caddr_t) &p_status->params.ip_addr,
					(caddr_t) &cp->hdr.params.ip_addr,
					sizeof(struct in_addr))) {
				/*
				 * found it...
				 */
				PARAMSCOPY(&cp->hdr.params, &p_status->params);
				p_status->count_out	= 1;
				p_status->obytes	= cp->obytes;
				p_status->max_obytes	= cp->max_obytes;
	
				p_status->ekp_len = cp->hdr.ekp.len;
				KEYTOBUF(cp->hdr.ekp, p_status->ekp_buf);
				break;
			}
			cp = cp->next;
		}

		MUTEX_EXIT(&skip_encrypt_hashlocks[hash]);
	}


	/*
	 * Get decrypt keys status
	 */
	for (hash = 0; hash < SKIP_HASH_TABLESZ; hash++) {

		MUTEX_ENTER(&skip_decrypt_hashlocks[hash]);

		cp = skip_decrypt_hashtable[hash];
	
		while (cp) {
			if ((p_status->params.r_nsid ==
					cp->hdr.params.r_nsid) &&
				KEYVAREQUAL(p_status->params.r_mkeyid,
					cp->hdr.params.r_mkeyid) &&
				!BCMP((caddr_t) &p_status->params.ip_addr,
					(caddr_t) &cp->hdr.params.ip_addr,
					sizeof(struct in_addr))) {
				/*
				 * found it...
				 */
				p_status->count_in++;
			}
			cp = cp->next;
		}

		MUTEX_EXIT(&skip_decrypt_hashlocks[hash]);
	}
	return(0);
}

#ifdef SKIP_PARAMS_DEBUG
/*
 * skip_diffs()
 *
 * debugging function for printing diffs of two skip_param_t structs
 */
int
skip_diffs(char *who, skip_param_t *in, skip_param_t *ref)
{
	int i;
	int s = skip_key_debug;

	if (!(skip_params_debug & SKIP_DBG_PARAMS_DIFFS)) {
		return(0);
	}

	skip_key_debug = 1;

	SKIP_DEBUG1("%s: PARAMS mismatch\n", who);

	if (in->version != ref->version) {
		SKIP_DEBUG2("in->version=%d, ref->version=%d\n",
					in->version, ref->version);
	}
	if (in->r_nsid != ref->r_nsid) {
		SKIP_DEBUG2("in->r_nsid=%d, ref->r_nsid=%d\n",
					in->r_nsid, ref->r_nsid);
	}
	if (in->s_nsid != ref->s_nsid) {
		SKIP_DEBUG2("in->s_nsid=%d, ref->s_nsid=%d\n",
					in->s_nsid, ref->s_nsid);
	}
	if (in->kij_alg != ref->kij_alg) {
		SKIP_DEBUG2("in->kij_alg=%d, ref->kij_alg=%d\n",
					in->kij_alg, ref->kij_alg);
	}
	if (in->kp_alg != ref->kp_alg) {
		SKIP_DEBUG2("in->kp_alg=%d, ref->kp_alg=%d\n",
					in->kp_alg, ref->kp_alg);
	}
	if (in->mac_alg != ref->mac_alg) {
		SKIP_DEBUG2("in->mac_alg=%d, ref->mac_alg=%d\n",
					in->mac_alg, ref->mac_alg);
	}
	if (in->comp_alg != ref->comp_alg) {
		SKIP_DEBUG2("in->comp_alg=%d, ref->comp_alg=%d\n",
					in->comp_alg, ref->comp_alg);
	}
	if (in->r_mkeyid.len != ref->r_mkeyid.len) {
		SKIP_DEBUG2("in->r_mkeyid.len=%d, ref->r_mkeyid.len=%d\n",
					in->r_mkeyid.len, ref->r_mkeyid.len);
	}

	if (in->s_mkeyid.len != ref->s_mkeyid.len) {
		SKIP_DEBUG2("in->s_mkeyid.len=%d, ref->s_mkeyid.len=%d\n",
					in->s_mkeyid.len, ref->s_mkeyid.len);
	}

	if (!KEYVAREQUAL(in->r_mkeyid, ref->r_mkeyid)) {
		SKIP_DEBUG("Master key in  [");
		for (i = 0; i < 4; i++ ) {
			SKIP_DEBUG1("0x%x ",
				(unsigned char) in->r_mkeyid.buf[i]);

		}
		SKIP_DEBUG("]\nMaster key ref [");
		for (i = 0; i < 4; i++ ) {
			SKIP_DEBUG1("0x%x ",
				(unsigned char) ref->r_mkeyid.buf[i]);
		}
		SKIP_DEBUG("]\n");
	}

	if (!KEYVAREQUAL(in->s_mkeyid, ref->s_mkeyid)) {
		SKIP_DEBUG("Master key in  [");
		for (i = 0; i < 4; i++ ) {
			SKIP_DEBUG1("0x%x ",
				(unsigned char) in->s_mkeyid.buf[i]);

		}
		SKIP_DEBUG("]\nMaster key ref [");
		for (i = 0; i < 4; i++ ) {
			SKIP_DEBUG1("0x%x ",
				(unsigned char) ref->s_mkeyid.buf[i]);
		}
		SKIP_DEBUG("]\n");
	}

	if (BCMP((caddr_t) &in->ip_addr, (caddr_t) &ref->ip_addr,
						sizeof(in->ip_addr))) {
		SKIP_DEBUG2("in->ip_addr=0x%x, ref->ip_addr=0x%x\n",
					(unsigned int) in->ip_addr.s_addr,
					(unsigned int) ref->ip_addr.s_addr);
	}

	skip_key_debug = s;

	return(0);
}

/*
 * skip_print()
 *
 * debugging function for printing skip_param_t struct content
 */
int
skip_print(char *who, skip_param_t *params)
{
	int i;
	int s = skip_key_debug;

	if (!(skip_params_debug & SKIP_DBG_PARAMS_PRINT)) {
		return(0);
	}

	skip_key_debug = 1;

	SKIP_DEBUG2(">> %s: SKIP (version=%d) parameters:\n",
					who, params->version);

	SKIP_DEBUG2("r_nsid=%d, s_nsid=%d\n",
			params->r_nsid, params->s_nsid);

	SKIP_DEBUG2("kij_alg=%d, kp_alg=%d\n",
			params->kij_alg, params->kp_alg);

	SKIP_DEBUG2(" mac_alg=%d, comp_alg=%d\n",
			params->mac_alg, params->comp_alg);

	SKIP_DEBUG2("r_mkeyid.len=%d, s_mkeyid.len=%d\n",
			params->r_mkeyid.len, params->s_mkeyid.len);

	SKIP_DEBUG("Receiver Master Key ID [");
	for (i = 0; i < 4; i++ ) {
		SKIP_DEBUG1("0x%x ",
			(unsigned char) params->r_mkeyid.buf[i]);

	}

	SKIP_DEBUG("]\nSender Master Key ID [");
	for (i = 0; i < 4; i++ ) {
		SKIP_DEBUG1("0x%x ",
			(unsigned char) params->s_mkeyid.buf[i]);

	}
	SKIP_DEBUG("]\n");

	SKIP_DEBUG1("ip_addr=0x%x\n>>\n",
		(unsigned int) params->ip_addr.s_addr);

	skip_key_debug = s;

	return(0);
}
#endif
