/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skiphost.c	1.11 95/11/15 Sun Microsystems"

#include <stdio.h>
#include <fcntl.h>
#include <stropts.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <string.h>
#include <netdb.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>

#include <skip_proto.h>
#include <skip_types.h>
#include <skip_ioctl.h>
#include <skip_lib.h>

extern int	errno;

extern char	*optarg;
extern int	optind;

static char	*progname;
static char	*interface;
static char	*hostname;
static char	*kij_alg;
static char	*kp_alg;
static char	*mac_alg;
static char	*comp_alg;
static char	*r_nsid;
static char	*s_nsid;
static char	*r_mkeyid;
static char	*skip_version;
static char	*skip_mode;

static int	opt, opt_cpt;
static int	opt_if, opt_unpb;
static int	opt_add, opt_del;
static int	opt_r_nsid, opt_s_nsid;
static int	opt_r_keyid;
static int	opt_kij, opt_crypt;
static int	opt_mac, opt_comp;
static int	opt_mode, opt_vers;
static int	opt_prt, opt_sel;

#define 	UNKNOWN		"<unknown>"

/*
 * List of all supported kij 
 */
static 
void kij_list()
{
	extern char		*skip_kij_algs[];
	extern boolean_t	skip_supported_kij_algs[];
	extern char		*skip_kij_algs_v1[];
	extern boolean_t	skip_supported_kij_algs_v1[];
	int			idx;

	printf("SKIP Kij algorithms (SunScreen compatibility):\n");

	for (idx = 0; idx <SKIP_MAXCRYPTORS; idx++) {
		if (skip_supported_kij_algs_v1[idx]) {
			printf("\t%s\n", skip_kij_algs_v1[idx] ?
					skip_kij_algs_v1[idx] : UNKNOWN);
		}
	}	
	printf("SKIP Kij algorithms:\n");

	for (idx = 0; idx <SKIP_MAXCRYPTORS; idx++) {
		if (skip_supported_kij_algs[idx]) {
			printf("\t%s\n", skip_kij_algs[idx] ?
					skip_kij_algs[idx] : UNKNOWN);
		}
	}	

}

/*
 * List of all supported Kp 
 */
static 
void kp_list()
{
	extern char		*skip_kp_algs[];
	extern boolean_t	skip_supported_kp_algs[];
	extern char		*skip_kp_algs_v1[];
	extern boolean_t	skip_supported_kp_algs_v1[];
	int			idx;

	printf("SKIP Kij algorithms (SunScreen compatibility):\n");

	for (idx = 0; idx <SKIP_MAXCRYPTORS; idx++) {
		if (skip_supported_kp_algs_v1[idx]) {
			printf("\t%s\n", skip_kp_algs_v1[idx] ?
					skip_kp_algs_v1[idx] : UNKNOWN);
		}
	}	
	printf("SKIP Kij algorithms:\n");

	for (idx = 0; idx <SKIP_MAXCRYPTORS; idx++) {
		if (skip_supported_kp_algs[idx]) {
			printf("\t%s\n", skip_kp_algs[idx] ?
					skip_kp_algs[idx] : UNKNOWN);
		}
	}	

}

/*
 * List of all NSID  numbers
 */
static 
void nsid_list()
{
	int idx;
	
	printf("SKIP NSID list:\n");

	for (idx = 1; idx < skip_max_nsids; idx++) {
		printf("\t%d\t(%d bytes):\t%s\n",
			idx, skip_nsids[idx].len, 
			skip_nsids[idx].name ? skip_nsids[idx].name : UNKNOWN);
	}

}

#define MAXHOSTNAME	80
/*
 * Print an ACL entry with "skiphost" command format
 */
static void
skip_cmd_print(skip_param_t *params)
{
	char			host[MAXHOSTNAME];
	char			keyid[MAXVARSZ*2];
	extern char		*skip_kij_algs[];
	extern char		*skip_kp_algs[];
	extern char		*skip_kij_algs_v1[];
	extern char		*skip_kp_algs_v1[];

	if (skip_addr_to_host(&params->ip_addr, host) < 0) {
		printf("Get host name failed!\n");
		return;
	}

	printf("skiphost -i %s -a %s", interface, host);

	switch (params->version) {

	case SKIP_NONE:
		break;

	case SKIP_V1:
		printf(" -v 1 -k %s -t %s ",
			(skip_kij_algs_v1[params->kij_alg] ?
				skip_kij_algs_v1[params->kij_alg] : UNKNOWN),
			(skip_kp_algs_v1[params->kp_alg] ?
				skip_kp_algs_v1[params->kp_alg] : UNKNOWN));

		if (params->r_nsid == SKIP_NSID_IPV4) {
			if (skip_keyid_to_s(&params->r_mkeyid,
						params->r_nsid, keyid) < 0) {
				printf("\t*** Error in key id\n");
			} else {
				printf("-R %s ", keyid);
			}
		} else {
			printf("*** : Invalid NSID/node id\n");
		}
		break;

	case SKIP_V2 :
		printf(" -v 2 ");

		if (params->kij_alg) {
			printf("-k %s ",
				skip_kij_algs[params->kij_alg] ?
				skip_kij_algs[params->kij_alg] : UNKNOWN);
		}

		if (params->kp_alg) {
			printf("-t %s ",
				skip_kp_algs[params->kp_alg] ?
				skip_kp_algs[params->kp_alg] : UNKNOWN);
		}

		/*
		 * XXX MAC and Comp algs...
		 */

		if (params->r_nsid) {
			printf("-r %d ", params->r_nsid);
		}

		if (params->r_nsid && params->r_mkeyid.len) {
			if (skip_keyid_to_s(&params->r_mkeyid,
						params->r_nsid, keyid) < 0) {
				printf("\t*** Error in key id\n");
			} else {
				printf("-R %s ", keyid);
			}
		}

		if (params->s_nsid) {
			printf("-s %d ", params->s_nsid);
		}

		break;

	default:
		printf("%s:\t\tbad version %d\n", host, params->version);

	}

	printf("\n");

}

/*
 * Display current status for given interface
 */
static void
summary(char *interface)
{
	int		rc;

	if (!interface) {
		printf("summary: bad interface name %s!\n", interface);
		return;
	}

	if (skip_get_mode(interface) == SkipAccessControlOff) {
		printf("%s: access control disabled, any host can connect\n",
								interface);
	} else {
		printf("%s: access control enabled, only authorized SKIP hosts "
						"can connect\n", interface);
	}
	rc = skip_list_hosts(interface, (void (*)()) skip_host_print);
	if (rc < 0) {
		if (errno == ENODEV) {
			fprintf(stderr, "%s: not in secure mode.\n", interface);
		} else {
			fprintf(stderr, "%s.\n", skip_errmsg);
		}
		exit(1);
	}
}

/*
 * Print a "skiphost" like file
 */
static void
print_hosts(char *interface)
{
	int		rc;

	rc = skip_list_hosts(interface, (void (*)()) skip_cmd_print);
	if (rc < 0) {
		if (errno == ENODEV) {
			fprintf(stderr, "%s: not in secure mode.\n", interface);
		} else {
			fprintf(stderr, "%s.\n", skip_errmsg);
		}
		exit(1);
	}

	if (skip_get_mode(interface) == SkipAccessControlOff) {
		printf("skiphost -i %s -o off\n", interface);
	} else {
		printf("skiphost -i %s -o on\n", interface);
	}
}

/*
 * Plumb or un-plumb the SKIP driver for a give interface
 */
static void
interface_plumb(char *name, int secure)
{
	boolean_t	error = B_FALSE;
	char		*mode;

	mode = secure ? "secure" : "unsecure";

	printf("%s: saving interface state\n", name);
	if (skip_if_state(name, Save) < 0) {
		fprintf(stderr, "%s\n", skip_errmsg);
		exit(1);
	}
	printf("%s: bringing interface down\n", name);
	skip_if_unplumb(name);
	printf("%s: starting %s mode\n", name, mode);

	if (skip_if_plumb(name, secure) < 0) {
		error = B_TRUE;
		if (secure) {
			fprintf(stderr, "%s: secure mode FAILED - trying to "
				"use unsecure mode\n", name);

			if (skip_if_plumb(name, 0) < 0) {
				fprintf(stderr, "%s: unsecure mode FAILED\n",
					name);
				fprintf(stderr, "%s: replumb FAILED\n", name);
				exit(1);
			}
			mode = "unsecure";
			secure = 0;
		} else {
			fprintf(stderr, "%s: unsecure mode FAILED\n", name);
			exit(1);
		}
	}
	printf("%s: restoring interface state\n", name);
	skip_if_state(name, Restore);
	skip_if_set_flags(name, secure);
	printf("%s: now in %s mode\n", name, mode);
	if (error) {
		exit(1);
	}
}


/*
 * Change the access control mode On/Off
 */
static void
set_secure(char *interface)
{

	if (!strcmp(skip_mode, "on")) {
		skip_access_ctrl_on(interface);
	} else {
		if (!strcmp(skip_mode, "off")) {
			skip_access_ctrl_off(interface);
		} else {
			fprintf(stderr,
				"%s: invalid access control mode \"%s\".\n",
							progname, skip_mode);
			usage();
		}
	}

	summary(interface);

}


/*
 * Add a host into ACL
 */
static void
add_host(char *interface)
{
	skip_param_t	parms = { 0 };
	int		alg;
	char		*keyid;

	/*
	 * Get the IP address of this host...
	 */
	if (skip_host_to_addr(hostname, &parms.ip_addr) < 0) {
		fprintf(stderr, "Hostname to IP address conversion failed.\n");
		exit(1);
	}

	/*
	 * Check if we add a cleartext host...
	 */
	opt_sel = opt_r_nsid + opt_s_nsid + opt_r_keyid;
	opt_sel += opt_kij + opt_crypt + opt_mac + opt_comp;
	opt_sel += opt_vers;

	if (!opt_sel) {
		/*
		 * Adding a cleartext host...
		 */
		printf("Adding %s (cleartext)...", hostname);
		fflush(stdout);
		if (skip_allow_host(interface, &parms) < 0) {
			fprintf(stderr, "\n%s.\n", skip_errmsg);
			exit(1);
		}
		printf("done.\n");
		return;
	}

	/*
	 * Get the SKIP version 
	 */
	if (opt_vers) {
		parms.version = atoi(skip_version);;
		if ((parms.version != SKIP_V1) && (parms.version != SKIP_V2)) {
			fprintf(stderr, "%s: invalid SKIP version \"%s\".\n",
						progname, skip_version);
			fprintf(stderr, "\tvalid versions : 1 or 2\n");
			usage();
		}
	} else {
		/*
		 * Default is IPSP mode 
		 */
		parms.version = SKIP_V2;
	}

	/*
	 * Check for un-supported options in SunScreen mode...
	 */
	if (parms.version  != SKIP_V2) {
		if (opt_mac) {
			/* 
			 * Forbid AH mode for SunScreen mode...
			 */
			fprintf(stderr, "%s: "
				"authentication not supported for Version 1.\n",
				progname);
			usage();
		}
			
		if (opt_comp) {
			/* 
			 * Forbid Compression mode for SunScreen mode...
			 */
			fprintf(stderr, "%s: "
				"compression not supported for Version 1.\n",
				progname);
			usage();
		}

	}

	/*
	 * Check if at least this host uses encryption or authentication...
	 */
	if (parms.version == SKIP_V2) {
		opt_sel = opt_crypt + opt_mac + opt_comp;
	} else {
		/*
		 * AH and compression were not supported with SunScreen..
		 */
		opt_sel = opt_crypt;
	}

	if (!opt_sel) {
		/*
		 * Mismatch in options...
		 */
		fprintf(stderr, "%s: "
			"no encryption/authentication algorithms specified.\n",
			progname);
		usage();
	}

	/*
	 * we must select a key encryption algorithm for traffic encryption
	 */
	if (opt_crypt && !opt_kij) {
		fprintf(stderr, "%s: no key encryption algorithm specified.\n",
			progname);
		usage();
	}
 
	/*
	 * we must also select a key encryption algorithm for authentication
	 */
	if (opt_mac && !opt_kij) {
		fprintf(stderr, "%s: no key encryption algorithm specified.\n",
			progname);
		usage();
	}

	/*
	 *  Check for supported Kij algorithms
	 */
	if ((alg =  skip_name_to_kij_alg(kij_alg, parms.version)) <0) {
		fprintf(stderr, "%s: unsupported kij algorithm \"%s\".\n",
						progname, kij_alg);
		kij_list();
		usage();
	} else {
		parms.kij_alg = (unsigned char) alg;
	}

	/*
	 *  Check for supported Crypt algorithms
	 */
	if (opt_crypt) {
		if ((alg = skip_name_to_kp_alg(kp_alg, parms.version)) < 0) {
			fprintf(stderr,
				"%s: unsupported Kp algorithm \"%s\".\n",
				progname, kp_alg);
			kp_list();
			usage();
		} else {
			parms.kp_alg = (unsigned char) alg;
		}
	}

	/*
	 *  XXX Check for MAC algorithms (Nor supported yet!)
	 */
	if (opt_mac) {
		fprintf(stderr, "%s: unsupported MAC algorithm \"%s\".\n",
						progname, mac_alg);
		usage();
	}

	/*
	 *  XXX Check for Compression algorithms (Nor supported yet!)
	 */
	if (opt_comp) {
		fprintf(stderr, "%s: unsupported Comp algorithm \"%s\".\n",
						progname, comp_alg);
		usage();
	}

	/*
	 * Process the Receiver NSID and Master Key Id...
	 */
	keyid = r_mkeyid;

	if (opt_r_nsid) {
		/*
		 * Convert NSID string to ID
		 */
		parms.r_nsid = atoi(r_nsid);
		if ((int) parms.r_nsid >= skip_max_nsids) {
			/*
			 * NSID string is not a valid one
			 */
			fprintf(stderr, "%s: invalid NSID \"%s\".\n",
							progname, r_nsid);
			nsid_list();
			usage();
		}
	} else {
		/*
		 * Default NSID is NONE for IPSP and IPV4 for SunScreen
		 */
		if (parms.version == SKIP_V2) {
			parms.r_nsid = SKIP_NSID_NONE;
		} else {
			parms.r_nsid = SKIP_NSID_IPV4;
		}
	}

	if (!opt_r_keyid) {
		switch (parms.r_nsid) {
	
		case SKIP_NSID_NONE:
			/* 
			 * XXXX should we use remote IP address as Key Id ?
			 */
			keyid = NULL;
			break;
	
		case SKIP_NSID_IPV4:
			/* 
			 * We use remote IP address as Key Id
			 */
			printf("%s: set remote IP address as key id.\n",
								progname);
			keyid = hostname;
			break;
	
		default:
			/*
			 * The Key Id is missing...
			 */
			fprintf(stderr, "%s: missing receiver NSID/key id.\n",
								progname);
			usage();
			break;
		}
	}

	if ((parms.version == SKIP_V1) && (keyid == NULL)) {
		/*
		 * node ID is mandatory for SunScreen mode (nodeid)
		 */
		fprintf(stderr, "%s: missing node ID for SunScreen mode.\n",
								progname);
		usage();
	}


	if (keyid) {
		if (skip_s_to_keyid(keyid, parms.r_nsid, &parms.r_mkeyid) < 0) {
			fprintf(stderr, "%s: Invalid receiver NSID/key id.\n",
								progname);
			usage();
		}
	}

	/*
	 * Process the Sender NSID
	 */
	if (opt_s_nsid) {
		parms.s_nsid = atoi(s_nsid);
		if ((int) parms.s_nsid >= skip_max_nsids) {
			fprintf(stderr, "%s: Invalid sender NSID \"%s\".\n",
							progname, s_nsid);
		}
	} else {
		/*
		 * Default NSID is none
		 */
		parms.s_nsid = SKIP_NSID_NONE;
	}

	if ((parms.version == SKIP_V1) && (parms.s_nsid != SKIP_NSID_NONE)) {
		/*
		 * Sender Key ID is forbiden for SunScreen mode...
		 */
		fprintf(stderr, 
			"%s: sender NSID not used for SunScreen mode.\n",
								progname);
		usage();
	}

	
	printf("Adding ");
	skip_host_print(&parms);
	printf("...");
	fflush(stdout);

	/*
	 * Add this host into the ACL...
	 */
	if (skip_allow_host(interface, &parms) < 0) {
		fprintf(stderr, "\n%s.\n", skip_errmsg);
		exit(1);
	}
	printf("done.\n");

	return;
}


/*
 * Remove a host from ACL
 */
static void
rem_host(char *interface)
{
	skip_param_t	parms = { 0 };

	/*
	 * Get the IP address of this host...
	 */
	if (skip_host_to_addr(hostname, &parms.ip_addr) < 0) {
		fprintf(stderr, "Hostnmae to IP address failed\n");
		exit(1);
	}

	printf("%s: %s being removed from SKIP access control list...\n",
					interface, hostname);
	fflush(stdout);

	if (skip_disallow_host(interface, &parms) < 0) {
		fprintf(stderr, "%s\n", skip_errmsg);
		exit(1);
	}
	printf("done.\n");
}

/*
 * Format error/help message for valid options
 */
static
usage()
{

	fprintf(stderr, 
		"usage:\n"
		"\t%s [-i <interface>]\n"
		"\t%s [-i <interface>] -p\n"
		"\t%s [-i <interface>] -o <on/off>\n"
		"\t%s [-i <interface>] -u\n"
		"\t%s [-i <interface>] -d <hostname>\n"
		"\t%s [-i <interface>] -a <hostname>...\n"
		"\t\t[-k <kij algorithm>] [-t <crypt algorithm>]...\n"
		"\t\t[-m <mac algorithm>] [-c <comp algorithm>]...\n"
		"\t\t[-r <receiver NSID>] [-s <sender NSID>]...\n"
		"\t\t[-R <receiver key Id>] [-v <SKIP version>]\n",
		progname, progname, progname,
		progname, progname, progname);

	fprintf(stderr, "\n");
	exit(1);
}

/*
 * Check that an options has been used only once
 */
static void
skip_one(int *skip_opt)
{
	if (*skip_opt) {
		usage();
	}
	(*skip_opt)++;
}

/*
 * skiphost tool
 *
 */
main(int argc, char *argv[])
{

	opt = opt_cpt = 0;
	opt_if = opt_unpb = 0;
	opt_add = opt_del = 0;
	opt_r_nsid = opt_s_nsid = 0;
	opt_r_keyid = opt_prt = 0;
	opt_kij = opt_crypt = 0;
	opt_mac = opt_comp = 0;
	opt_mode = opt_vers = 0;

	interface = skip_default_if();

	progname = argv[0];

	/*
	 * Command line parsing
	 */
	while ((opt = getopt(argc, argv,
			"pha:d:o:ui:s:r:R:k:t:m:c:v:")) != -1) {

		switch (opt) {
	
		/*
		 * Allowed skiphost actions
		 */
		case 'u':
			skip_one(&opt_unpb);
			break;

		case 'p':
			skip_one(&opt_prt);
			break;

		case 'a':
			skip_one(&opt_add);
			hostname = optarg;
			break;

		case 'd':
			skip_one(&opt_del);
			hostname = optarg;
			break;

		case 'o':
			skip_one(&opt_mode);
			skip_mode = optarg;
			break;

		/*
		 * Algorithms selection
		 */
		case 'k':
			skip_one(&opt_kij);
			kij_alg = optarg;
			break;

		case 't':
			skip_one(&opt_crypt);
			kp_alg = optarg;
			break;

		case 'm':
			skip_one(&opt_mac);
			mac_alg = optarg;
			break;

		case 'c':
			skip_one(&opt_comp);
			comp_alg = optarg;
			break;

		/*
		 * NSID and Master Key IDs
		 */
		case 's':
			skip_one(&opt_s_nsid);
			s_nsid = optarg;
			break;

		case 'r':
			skip_one(&opt_r_nsid);
			r_nsid = optarg;
			break;

		case 'R':
			skip_one(&opt_r_keyid);
			r_mkeyid = optarg;
			break;

		/*
		 * Misc...
		 */
		case 'i':
			skip_one(&opt_if);
			interface = optarg;
			break;

		case 'v':
			skip_one(&opt_vers);
			skip_version = optarg;
			break;

		case 'h':
		default:
			usage();
			

		}

		if (opt != 'i') {
			opt_cpt++;
		}
	}

	/*
	 * Useful pre-checks
	 */
	if (skip_pre_checks(interface)) {
		switch (errno) {
		case EACCES:
			fprintf(stderr, "%s: you must be root to run this "
				"program\n", progname);
			break;

		case ENODEV:
			fprintf(stderr, "%s\n", skip_errmsg);
			fprintf(stderr, "Check that SKIP is correctly "
				"installed on this system\n", progname);
			break;
		default:
			fprintf(stderr, "%s\n", skip_errmsg);
		}
		exit(1);
	}

	/*
	 * Check if SKIP driver is loaded
	 */
	if (!opt_unpb && !skip_if_module_present(interface)) {
		interface_plumb(interface, 1);
		skip_access_ctrl_off(interface);
	}

	/*
	 * Get SKIP driver info for Kij algs,...
	 */
	if (!opt_unpb && skip_var_init(progname)) {
		fprintf(stderr, "%s\n", skip_errmsg);
		exit(1);
	}

	/*
	 * Check for default action 
	 */
	if (!opt_cpt) {
		/*
	 	 * Default action is summarize...
	 	 */
		summary(interface);
		return(0);
	}

	/*
	 * Check that no more than one action is selected...
	 */
	if ((opt_unpb + opt_add + opt_del + opt_mode + opt_prt) != 1) {
		usage();
	}

	/*
	 * Process each command line option
	 */

	if (opt_unpb) {

		/*
		 * Check for invalid options...
		 */
		opt_sel = opt_r_nsid + opt_s_nsid + opt_r_keyid;
		opt_sel += opt_kij + opt_crypt + opt_mac + opt_comp;
		opt_sel += opt_mode + opt_vers;

		if (opt_sel) {
			fprintf(stderr, "%s -u does not take options\n",
								progname);
			usage();
		}

		/*
		 * un-plumb the SKIP driver
		 */
		interface_plumb(interface, 0);
		exit(0);
	}

	if (opt_prt) {

		/*
		 * Check for invalid options...
		 */
		opt_sel = opt_r_nsid + opt_s_nsid + opt_r_keyid;
		opt_sel += opt_kij + opt_crypt + opt_mac + opt_comp;
		opt_sel += opt_mode + opt_vers;

		if (opt_sel) {
			fprintf(stderr, "%s -p does not take options\n",
								progname);
			usage();
		}

		/*
		 * print each entry in a "skiphost" command format
		 */
		print_hosts(interface);
		exit(0);
	}

	if (opt_mode) {
		/*
		 * Check for invalid options...
		 */
		opt_sel = opt_r_nsid + opt_s_nsid + opt_r_keyid;
		opt_sel += opt_kij + opt_crypt + opt_mac + opt_comp;
		opt_sel += opt_vers;

		if (opt_sel) {
			usage();
		}

		/*
		 * Set secure mode on/off
		 */
		set_secure(interface);
		exit(0);
	}

	if (opt_add) {
		/*
		 * Check for invalid options...
		 */
		if (opt_mode) {
			usage();
		}

		/*
		 * Add a host in the ACL
		 */
		add_host(interface);
		exit(0);
	}

	if (opt_del) {
		/*
		 * Check for invalid options...
		 */
		opt_sel = opt_r_nsid + opt_s_nsid + opt_r_keyid;
		opt_sel += opt_kij + opt_crypt + opt_mac + opt_comp;
		opt_sel += opt_mode + opt_vers;

		if (opt_sel) {
			usage();
		}
		/*
		 * Remove a host from ACL
		 */
		rem_host(interface);
		exit(0);
	}

	/*
	 * Never reached !
	 */
	return (0);
}
