/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skip_sys.c	1.17 95/11/21 Sun Microsystems"

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <netdb.h>
#include <signal.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/sockio.h>
#include <sys/stropts.h>
#ifdef SYSV
#include <sys/dlpi.h>
#include <sys/systeminfo.h>
#endif SYSV

#include <netinet/in.h>
#include <net/if.h>
#include <netinet/if_ether.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <arpa/inet.h>


#include <skip_types.h>
#include <skip_proto.h>
#include <skip_ioctl.h>
#include <skip_lib.h>

#define BUFSIZE		10000

#ifdef SYSV
#define	DEVDIR		"/dev"
#ifndef	ARP_MOD_NAME
#define	ARP_MOD_NAME	"arp"
#endif
#ifndef	IP_DEV_NAME
#define	IP_DEV_NAME	"/dev/ip"
#endif
#ifndef	IP_MOD_NAME
#define	IP_MOD_NAME	"ip"
#endif
#define	DLPI_TIMEOUT	10

#endif /* SYSV */

extern char *sys_errlist[];

static int		_skip_lib_debug = 0;

/* _skip_error()
 *
 * Set skip_errmsg string
 */
static void
_skip_error(char *errstr)
{
	sprintf(skip_errmsg, "%s: %s: %s", SKIPNAME, errstr, sys_errlist[errno]);
}

/* _skip_iferror()
 *
 * Set skip_errmsg string with interface name information
 */
static void
_skip_iferror(char *ifname, char *errstr)
{
	sprintf(skip_errmsg, "%s: %s: %s: %s",
		SKIPNAME, ifname, errstr, sys_errlist[errno]);
}

/* _skip_invalid_ifname()
 *
 * Make sure an interface name is not too long
 */
static int
_skip_invalid_ifname(char *ifname)
{
	if (strlen(ifname) > (size_t) (IFNAMSIZ - 1)) {
		sprintf(skip_errmsg, "%s: interface name too long", SKIPNAME);
		return (1);
	}
	return (0);
}

/*
 * open a connection to the SKIP kernel
 */
#ifdef SYSV
int
_skip_open(char *name)
{
	int		skip_fd;
	char		pathname[MAXPATHLEN], *devname();

	(void) ifnametotype2device(name, pathname);

	skip_fd = open(pathname, O_RDWR);
	if (skip_fd < 0) {
		sprintf(skip_errmsg, "%s: cannot open device %s: %s",
				SKIPNAME, pathname, sys_errlist[errno]);
		return (-1);
	}
	if (ioctl(skip_fd, I_PUSH, SKIP_ES) < 0) {
		sprintf(skip_errmsg, "%s: cannot push %s module: %s",
				SKIPNAME, SKIP_ES, sys_errlist[errno]);
		return (-1);
	}
	return (skip_fd);
}

#else

int
_skip_open(char *name)
{
	int		skip_fd, minor = 1;
	char		device[sizeof SKIPDEV "999"];

	do {
		(void) sprintf(device, SKIPDEV "%d", minor++);
		skip_fd = open(device, O_RDWR);
	} while (skip_fd < 0 && errno == EBUSY);

	if (skip_fd < 0) {
		sprintf(skip_errmsg, "%s: cannot open device %s: %s",
				SKIPNAME, SKIPDEV, sys_errlist[errno]);
		return (-1);
	}
	return (skip_fd);
}
#endif

/*
 * close a connection to the SKIP kernel
 */
void
_skip_close(int skip_fd)
{
	(void) close(skip_fd);
}

/* _skip_open_ip()
 *
 * Returns: file descriptor on success, -1 otherwise.
 */
static int
_skip_open_ip()
{
	int			ip_fd;

#ifdef SYSV
	ip_fd = open(IP_DEV_NAME, O_RDWR);
	if (ip_fd < 0) {
		_skip_error("skip_open_ip open IP_DEV_NAME failed");
		return(-1);
	}
#else
	ip_fd = socket(AF_INET, SOCK_DGRAM, 0);
	if (ip_fd < 0) {
		_skip_error("skip_open_ip socket() failed");
		return(-1);
	}
#endif
	return (ip_fd);
}

/* _skip_gethostbyname()
 *
 * Non-name service based gethostbyname() - eats /etc/hosts
 *
 * Returns: as gethostbyname()
*/
static struct hostent *
_skip_gethostbyname(char *hostname)
{
#define	ETC_HOSTS		"/etc/hosts"
#define	MAXLINESZ		128
	static long		addr;
	static char		line[MAXLINESZ], *s, *addrs[1];
	static struct hostent	h, *hp = &h;
	FILE			*fp;

	fp = fopen(ETC_HOSTS, "r");
	if (fp == NULL) {
		return (NULL);
	}
	
	memset((char *) hp, 0, sizeof (*hp));

	while (!feof(fp)) {
		if (fgets(line, MAXLINESZ, fp) == NULL) {
			break;
		}
		s = strtok(line, " \t\n");
		if (s == NULL) {
			continue;
		}
		if (s[0] == '#') {
			continue;
		}
		if ((addr = inet_addr(s)) == -1) {
			continue;
		}
		for (; s; s = strtok(NULL, " \t\n")) {
			if (strchr(s, '#')) {
				break;
			}
			if (!strcmp(s, hostname)) {
				/*
				 * found the host
				 */
				addrs[0] = (char *) &addr;
				hp->h_name = s;
				hp->h_addrtype = AF_INET;
				hp->h_length = sizeof(addr);
				hp->h_addr_list = addrs;
				break;
			}
		}
		if (hp->h_name) {
			break;
		}
	}
	fclose(fp);
	return (hp->h_name ? hp : NULL);
}

/* skip_default_if()
 *
 * Return the name of the default network interface
 *
 * Returns: interface name
 */
char *
skip_default_if()
{
#define	SKIP_DEFAULT_IF		"le0"
	static char		devname[MAXPATHLEN];
	char			cbuf[sizeof(struct ifreq) * 16];
	struct ifconf		ifc;
	struct ifreq		*ifr, ifflags;
	struct hostent		*hp;
	struct sockaddr_in	*sa;
	char			hostname[MAXHOSTNAMELEN];
	int			i, s, n;

	strcpy(devname, SKIP_DEFAULT_IF);

#ifdef SYSV
	(void) sysinfo(SI_HOSTNAME, hostname, MAXHOSTNAMELEN);
#else
	(void) gethostname(hostname, MAXHOSTNAMELEN);
#endif

	hp = _skip_gethostbyname(hostname);

	if (hp == NULL) {
		return(devname);
	}

	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		return (devname);
	}

	ifc.ifc_len = sizeof (cbuf);
	ifc.ifc_buf = cbuf;
	if (ioctl(s, SIOCGIFCONF, (char *)&ifc) < 0) {
		goto out;
	}

	n = ifc.ifc_len / sizeof (struct ifreq);
	ifr = ifc.ifc_req;

	/*
	 * scan list trying to find the principal interface, based on
	 * primary host address.  Note any up interfaces in passing,
	 * just in case the primary can't be found.
	 * 
	 */
	for (; n > 0; n--, ifr++) {
		strcpy(ifflags.ifr_name, ifr->ifr_name);
		if (ioctl(s, SIOCGIFFLAGS, (char *) &ifflags) < 0) {
			goto out;
		}
		if ((ifflags.ifr_flags & (IFF_LOOPBACK|IFF_UP|IFF_RUNNING)) ==
					(IFF_UP|IFF_RUNNING)) {
			/*
			 * note the interface name just in case
			 */
			strcpy(devname, ifr->ifr_name);

			/*
			 * but try to find an exact match
			 */
			sa = (struct sockaddr_in *) &ifr->ifr_addr;
			if (memcmp(hp->h_addr_list[0], (void *) &sa->sin_addr,
							hp->h_length) == 0) {
					break;
			}
		}
	}
out:
 	(void) close(s);
	return (devname);
}
 

/* skip_pre_checks()
 *
 * check that permissions and software installation is correct
 *
 * Returns: 0 on success, non-zero otherwise
 */
int
skip_pre_checks(char *name)
{
	struct ifreq	flags;
	int		fd;

	if (_skip_invalid_ifname(name)) {
		return (-1);
	}

	fd = _skip_open_ip();
	if (fd < 0) {
		_skip_error("cannot connect to TCP/IP on this system");
		return(-1);
	}

	strcpy(flags.ifr_name, name);
	if (ioctl(fd, SIOCGIFFLAGS, (caddr_t) &flags) < 0) {
		_skip_iferror(name, "cannot configure interface");
		close(fd);
		return(-1);
	}
	close(fd);

	fd = _skip_open(name);
	if (fd < 0) {
		return (-1);
	}
	close(fd);

#ifdef SYSV
	fd = dlpi_open_attach(name);
	if (fd < 0) {
		_skip_iferror(name, "open of network interface failed");
		return (-1);
	}
	if (ioctl(fd, I_PUSH, SKIP_ES) < 0) {
		_skip_error("pushing SKIP endsystem module failed");
		close(fd);
		return(-1);
	}
	close(fd);
#endif
	return (0);
}

/* skip_if_state()
 *
 * save/restore if state
 *
 * Returns: 0 on success, -1 otherwise
 */
int
skip_if_state(char *name, int mode)
{
	static struct ifreq	addr, broadaddr, dstaddr, netmask, flags, mtu;
	static int		saved = 0;
	int			ip_fd;

	if (_skip_invalid_ifname(name)) {
		return (-1);
	}

	if ((ip_fd = _skip_open_ip()) < 0) {
		return(-1);
	}

	switch (mode) {
	case Save:
		/*
		 * save flags state
		 */
		strcpy(flags.ifr_name, name);
		if (ioctl(ip_fd, SIOCGIFFLAGS, (caddr_t)&flags) < 0) {
			_skip_iferror(name, "SIOCGIFFLAGS");
			close(ip_fd);
			return(-1);
		}
		/*
		 * save addr state
		 */
		strcpy(addr.ifr_name, name);
		if (ioctl(ip_fd, SIOCGIFADDR, (caddr_t)&addr) < 0) {
			_skip_iferror(name, "SIOCGIFADDR");
			close(ip_fd);
			return(-1);
		}
		if (flags.ifr_flags & IFF_POINTOPOINT) {
			strcpy(dstaddr.ifr_name, name);
			if (ioctl(ip_fd, SIOCGIFDSTADDR, (caddr_t)&dstaddr)
									< 0) {
				_skip_iferror(name, "SIOCGIFDSTADDR");
				close(ip_fd);
				return(-1);
			}
		}
		/*
		 * save broadcast address state
		 */
		if (flags.ifr_flags & IFF_BROADCAST) {
			strcpy(broadaddr.ifr_name, name);
			if (ioctl(ip_fd, SIOCGIFBRDADDR, (caddr_t)&broadaddr)	
									< 0) {
				_skip_iferror(name, "SIOCGIFBRDADDR");
				close(ip_fd);
				return(-1);
			}
		}
		/*
		 * save netmask state
		 */
		strcpy(netmask.ifr_name, name);
		if (ioctl(ip_fd, SIOCGIFNETMASK, (caddr_t)&netmask) < 0) {
			_skip_iferror(name, "SIOCGIFNETMASK");
			close(ip_fd);
			return(-1);
		}
#ifdef SYSV
		/*
		 * save mtu state (only if interface is not already using skip)
		 * this is so we can detect a user configured mtu.
		 *
 		 * Most SunOS 4.x interfaces don't support this :0(
		 */
		if (flags.ifr_flags & IFF_PRIVATE) {
			/*
			 * SKIP present - ignore the MTU
			 */
			mtu.ifr_metric = 0;
		} else {
			strcpy(mtu.ifr_name, name);
			if (ioctl(ip_fd, SIOCGIFMTU, (caddr_t)&mtu) < 0) {
				_skip_iferror(name, "SIOCGIFMTU");
				close(ip_fd);
				return(-1);
			}
		}
#endif
		saved = 1;
		break;

	case Restore:
		if (saved == 0) {
			break;
		}
		/*
		 * restore addr state
		 */
		if (ioctl(ip_fd, SIOCSIFADDR, (caddr_t)&addr) < 0) {
			_skip_iferror(name, "SIOCSIFADDR");
			close(ip_fd);
			return(-1);
		}
		if (flags.ifr_flags & IFF_POINTOPOINT) {
			if (ioctl(ip_fd, SIOCSIFDSTADDR, (caddr_t)&dstaddr)
									< 0) {
				_skip_iferror(name, "SIOCSIFDSTADDR");
				close(ip_fd);
				return(-1);
			}
		}
		/*
		 * restore broadcast address state
		 */
		if (flags.ifr_flags & IFF_BROADCAST) {
			if (ioctl(ip_fd, SIOCSIFBRDADDR, (caddr_t)&broadaddr)
									< 0) {
				_skip_iferror(name, "SIOCSIFBRDADDR");
				close(ip_fd);
				return(-1);
			}
		}
		/*
		 * restore netmask state
		 */
		if (ioctl(ip_fd, SIOCSIFNETMASK, (caddr_t)&netmask) < 0) {
			_skip_iferror(name, "SIOCSIFNETMASK");
			close(ip_fd);
			return(-1);
		}
		/*
		 * restore flags state
		 */
		if (ioctl(ip_fd, SIOCSIFFLAGS, (caddr_t)&flags) < 0) {
			_skip_iferror(name, "SIOCSIFFLAGS");
			close(ip_fd);
			return(-1);
		}
#ifdef notdef
		/*
		 * restore mtu
		 */
		if (mtu.ifr_metric != 0) {
			mtu.ifr_metric -= SKIP_HDR_SZ;
			if (ioctl(ip_fd, SIOCSIFMTU, (caddr_t)&mtu) < 0) {
				_skip_iferror(name, "SIOCSIFMTU");
				close(ip_fd);
				return(-1);
			}
		}
#endif
		break;

	default:
		_skip_iferror(name, "bad command (internal error)");
		close(ip_fd);
		return(-1);
	}
	close(ip_fd);
	return (0);
}

int
skip_if_set_mtu(char * name, int val)
{
	static struct ifreq	mtu;
	int			ip_fd;

	if (_skip_invalid_ifname(name)) {
		return (-1);
	}

	if ((ip_fd = _skip_open_ip()) < 0) {
		return (-1);
	}

	strcpy(mtu.ifr_name, name);
	mtu.ifr_metric = val;
	if (ioctl(ip_fd, SIOCSIFMTU, (caddr_t)&mtu) < 0) {
		_skip_iferror(name, "SIOCSIFFLAGS");
		close(ip_fd);
		return(-1);
	}
	close(ip_fd);
	return (0);
}

int
skip_if_get_mtu(char * name)
{
	static struct ifreq	mtu;
	int			ip_fd;

	if (_skip_invalid_ifname(name)) {
		return (-1);
	}

	if ((ip_fd = _skip_open_ip()) < 0) {
		return (-1);
	}

	strcpy(mtu.ifr_name, name);
	if (ioctl(ip_fd, SIOCGIFMTU, (caddr_t)&mtu) < 0) {
		_skip_iferror(name, "SIOCGIFMTU");
		close(ip_fd);
		return(-1);
	}
	close(ip_fd);
	return (mtu.ifr_metric);
}

int
skip_if_set_flags(char *name, int secure)
{
	static struct ifreq	f;
	int			ip_fd;

	if (_skip_invalid_ifname(name)) {
		return (-1);
	}

	if ((ip_fd = _skip_open_ip()) < 0) {
		return (-1);
	}

	strcpy(f.ifr_name, name);
	if (ioctl(ip_fd, SIOCGIFFLAGS, (caddr_t)&f) < 0) {
		_skip_iferror(name, "SIOCGIFFLAGS");
		close(ip_fd);
		return(-1);
	}
	if (secure) {
		f.ifr_flags |= IFF_PRIVATE;
	} else {
		f.ifr_flags &= ~IFF_PRIVATE;
	}
	if (ioctl(ip_fd, SIOCSIFFLAGS, (caddr_t)&f) < 0) {
		_skip_iferror(name, "SIOCSIFFLAGS");
		close(ip_fd);
		return(-1);
	}
	close(ip_fd);
	return (0);
}

int
skip_if_get_broadcast(char *name, struct in_addr *addr)
{
	static struct ifreq	f;
	struct sockaddr_in	*sin;
	int			ip_fd;

	if (_skip_invalid_ifname(name)) {
		return (-1);
	}

	if ((ip_fd = _skip_open_ip()) < 0) {
		return (-1);
	}

	strcpy(f.ifr_name, name);
	if (ioctl(ip_fd, SIOCGIFFLAGS, (caddr_t)&f) < 0) {
		_skip_iferror(name, "SIOCGIFFLAGS");
		close(ip_fd);
		return(-1);
	}
	if (f.ifr_flags & IFF_BROADCAST) {
		if (ioctl(ip_fd, SIOCGIFBRDADDR, (caddr_t) &f)	< 0) {
			_skip_iferror(name, "SIOCGIFBRDADDR");
			close(ip_fd);
			return(-1);
		}
		close(ip_fd);
		sin = (struct sockaddr_in *) &f.ifr_broadaddr;
		*addr = sin->sin_addr;
		return (0);
	}
	_skip_iferror(name, "non-broadcast interface");
	close(ip_fd);
	return (-1);
}

int
skip_if_module_present(char *name)
{
	return (skip_get_mode(name) != -1);
}

#ifdef SYSV
static int
plumb_one_device (name, dev_name, ppa, mux_fd, secure)
	char	* name;
	char	* dev_name;
	int	ppa;
	int	mux_fd;
	char	*secure;
{
	struct ifreq	ifr;
	int		arp_fd = -1;
	int		ip_fd = -1;
	int		arp_muxid, ip_muxid;
	skip_es_req_t	req;

	/* Open the device and push IP */
	if ((ip_fd = open(dev_name, O_RDWR)) == -1) {
		_skip_error("open device");
		return (-1);
	}

	if (secure) {
		/*
		 * config skip end system module
		 */
		if (ioctl(ip_fd, I_PUSH, SKIP_ES) == -1) {
			_skip_error("I_PUSH SKIP_ES");
			close(ip_fd);
			return (-1);
		}
	      	strcpy(req.if_name, name); 
		if (_skip_ioctl(ip_fd, SKIP_ES_ADD_IF, (char *) &req,
						sizeof(req.if_name))) {
			_skip_error("SKIP_ES_ADD_IF");
			close(ip_fd);
			return (-1);
		}
	}

	if (ioctl(ip_fd, I_PUSH, IP_MOD_NAME) == -1) {
		_skip_error("I_PUSH IP_MOD_NAME");
		close(ip_fd);
		return (-1);
	}

	if (ioctl(ip_fd, IF_UNITSEL, (char *)&ppa) == -1) {
		_skip_error("I_UNITSEL for ip");
		close(ip_fd);
		return (-1);
	}

	/* Check if IFF_NOARP is set */
	strncpy(ifr.ifr_name, name, sizeof ifr.ifr_name);
	if (ioctl(ip_fd, SIOCGIFFLAGS, (caddr_t)&ifr) < 0) {
		_skip_error("SIOCGIFFLAGS");
		close(ip_fd);
		return (-1);
	}

	if (!(ifr.ifr_flags & IFF_NOARP)) {
		if ((arp_fd = open(dev_name, O_RDWR)) == -1) {
			_skip_error("open arp_fd");
			close(ip_fd);
			return (-1);
		}

		if (ioctl(arp_fd, I_PUSH, ARP_MOD_NAME) == -1) {
			_skip_error("I_PUSH ARP_MOD_NAME");
			close(arp_fd);
			close(ip_fd);
			return (-1);
		}

		if (ioctl(ip_fd, I_PUSH, ARP_MOD_NAME) == -1) {
			_skip_error("I_PUSH ARP_MOD_NAME");
			close(arp_fd);
			close(ip_fd);
			return (-1);
		}

		/* Note that arp has no support for transparent ioctls */
		if (_skip_ioctl(arp_fd, IF_UNITSEL, (char *)&ppa, 
			     sizeof(ppa)) == -1) {
			_skip_error("I_UNITSEL for arp");
			close(arp_fd);
			close(ip_fd);
			return (-1);
		}

		if ((arp_muxid = ioctl(mux_fd, I_PLINK, arp_fd)) == -1) {
			_skip_error("I_PLINK for arp");
			close(arp_fd);
			close(ip_fd);
			return (-1);
		}

		ifr.ifr_arp_muxid = arp_muxid;
		if (_skip_lib_debug)
			printf("arp muxid = %d\n", arp_muxid);
	}
	if ((ip_muxid = ioctl(mux_fd, I_PLINK, ip_fd)) == -1) {
		if (!(ifr.ifr_flags & IFF_NOARP))
			(void)ioctl(mux_fd, I_PUNLINK, ip_muxid);
			_skip_error("I_PLINK for ip");
			close(arp_fd);
			close(ip_fd);
			return (-1);
	}

	ifr.ifr_ip_muxid = ip_muxid;
	if (_skip_lib_debug) {
		printf("ip muxid = %d\n", ip_muxid);
	}

	if (ioctl(mux_fd, SIOCSIFMUXID, (caddr_t)&ifr) < 0) {
		_skip_error("SIOCSIFMUXID");
		close(arp_fd);
		close(ip_fd);
		return (-1);
	}

	close(arp_fd);
	close(ip_fd);
	return(0);
}
#endif

/* skip_if_plumb()
 *
 * If secure == 1, insert SKIP between IP and the specified network interface
 * otherwise remove it.
 *
 * Returns: 0 on success, -1 on error
 */
int
skip_if_plumb(char *name, int secure)
{
	char		path[MAXPATHLEN];
	int		style;
	int		ppa;
	char 		mux_fd;
	int		fd, rc = 0;
	skip_es_req_t	req;

	if (_skip_invalid_ifname(name)) {
		return (-1);
	}

#ifdef SYSV
	/* Verify that ppa is valid */
	fd = dlpi_open_attach(name);
	if (fd < 0) {
		/* Not found */
		_skip_error("plumb");
		return (-1);
	}

	if (_skip_lib_debug)
		printf("opened & attached %s\n", name);
	(void) close(fd);

	style = ifname2device_ppa(name, path, &ppa);
	if (style < 0) {
		/* Not found */
		_skip_error("ifname2device_ppa");
		return (-1);
	}
	mux_fd = open(IP_DEV_NAME, 2);
	if (mux_fd == -1) {
		_skip_error("open IP_DEV_NAME");
		return (-1);
	}
	rc = plumb_one_device(name, path, ppa, mux_fd, secure);
	(void) close(mux_fd);
#else
	fd = _skip_open(name);
	if (fd < 0) {
		sprintf(skip_errmsg, "%s: failed to open %s: %s",
			SKIPNAME, SKIPDEV, sys_errlist[errno]);
		return (-1);
	}
	strcpy(req.if_name, name); 
	if (secure) {
		if (_skip_ioctl(fd, SKIP_ES_ADD_IF, (char *) &req,
						sizeof(req.if_name)) < 0) {
			_skip_error("SKIP_ES_ADD_IF");
			close(fd);
			return (-1);
		}
	} else {
		if (_skip_ioctl(fd, SKIP_ES_DEL_IF, (char *) &req,
						sizeof(req.if_name)) < 0) {
			if (errno != ENXIO) {
				/*
				 * silently allow the removal of
				 * an already removed interface
				 */
				_skip_error("SKIP_ES_DEL_IF");
				close(fd);
				return (-1);
			}
		}
	}
	close(fd);
#endif
	return (rc);
}

/* skip_if_unplumb()
 *
 * Remove network interface from under IP (SYSV only)
 *
 * Returns: 0 on success, -1 on error
 */
/*ARGSUSED*/
int 
skip_if_unplumb(char *name)
{
	struct ifreq	ifr;
	int		ip_muxid, arp_muxid;
	int		mux_fd, fd;
	int		arp;

#ifdef SYSV
	mux_fd = open(IP_DEV_NAME, 2);
	if (mux_fd == -1) {
		_skip_error("open IP_DEV_NAME");
		return (-1);
	}
	strncpy(ifr.ifr_name, name, sizeof ifr.ifr_name);
	if (ioctl(mux_fd, SIOCGIFFLAGS, (caddr_t)&ifr) < 0) {
		_skip_error("SIOCGIFFLAGS");
		close(mux_fd);
		return (-1);
	}
	arp = !(ifr.ifr_flags & IFF_NOARP);
	if (ioctl(mux_fd, SIOCGIFMUXID, (caddr_t)&ifr) < 0) {
		_skip_error("SIOCGIFMUXID");
		close(mux_fd);
		return (-1);
	}
	if (arp) {
		arp_muxid = ifr.ifr_arp_muxid;
		if (_skip_lib_debug) printf("arp_muxid %d\n", arp_muxid);
		if (ioctl(mux_fd, I_PUNLINK, arp_muxid) < 0) {
			_skip_error("I_PUNLINK for arp");
			close(mux_fd);
			return (-1);
		}
	}
	ip_muxid = ifr.ifr_ip_muxid;
	if (_skip_lib_debug) {
		printf("ip_muxid %d\n", ip_muxid);
	}
	if (ioctl(mux_fd, I_PUNLINK, ip_muxid) < 0) {
		_skip_error("I_PUNLINK for ip");
		close(mux_fd);
		return (-1);
	}
	(void) close(mux_fd);
#endif
	return (0);
}


/* _skip_ioctl()
 *
 * Returns: -1 on error, bytes returned by ioctl()
 */
int
_skip_ioctl(s, cmd, buf, buflen)
	int s;
	int cmd;
	char *buf;
	int buflen;
{
	skip_io_t	ioc;
	int		rc;

	memset((char *)&ioc, 0, sizeof(ioc));
	ioc.ic_cmd = cmd;
	ioc.ic_timout = 0;
	ioc.ic_len = buflen;
	ioc.ic_dp = buf;
#ifdef SYSV
	rc = ioctl(s, I_STR, (char *)&ioc);
#else
	rc = ioctl(s, SKIPIOACL, &ioc);
#endif
	return (rc < 0 ? rc : ioc.ic_len);
}

#ifdef SYSV

/*ARGSUSED*/
static void
sigalarm_attach(int xxx)
{
}
 

/* Get ppa and device from BSD style interface name assuming it is a
 * DLPI type 2 interface.
 */
static int
ifnametotype2device(ifname, path)
	char	*ifname;
	char 	*path;
{
	int	i;
	ulong	p = 0;
	int	m = 1;

	/* device name length has already been checked
	 */
	(void) sprintf(path, "%s/%s", DEVDIR, ifname);
	i = strlen(path) - 1;
	while (i >= 0 && '0' <= path[i] && path[i] <= '9') {
		p += (path[i] - '0')*m;
		m *= 10;
		i--;
	}
	path[i + 1] = '\0';
	return(p);
}


/* Get ppa and device from BSD style interface name assuming it is a
 * DLPI type 1 interface. Always returns -1 for the ppa signalling that no 
 * attach is needed.
 */
static int
ifnametotype1device(ifname, path)
	char	*ifname;
	char 	*path;
{
	int	i;
	ulong	p = 0;
	int	m = 1;

	/* device name length has already been checked
	 */
	(void) sprintf(path, "%s/%s", DEVDIR, ifname);
	i = strlen(path) - 1;
	while (i >= 0 && '0' <= path[i] && path[i] <= '9') {
		p += (path[i] - '0')*m;
		m *= 10;
		i--;
	}
	return(p);
}


/*
 * Get ppa, device, and style of device from BSD style interface name.
 * Return the device name in path and the ppa in *ppap
 * Returns 2 for a style 2 device name.
 * Returns 1 for a style 1 device name (signalling that no attach is needed.)
 * Returns -1 on errors.
 */
static int
ifname2device_ppa(name, path, ppap)
	char	*name;
	char 	*path;
	int	*ppap;
{
	struct stat st;
	int ppa;

	ppa = ifnametotype2device(name, path);
	if (stat(path, &st) >= 0) {
		*ppap = ppa;
		return (2);
	}
	if (errno != ENOENT)
		return (-1);
		
	ppa = ifnametotype1device(name, path);
	if (stat(path, &st) >= 0) {
		*ppap = ppa;
		return (1);
	}
	return (-1);
}

static int
dlpi_open_attach(ifname)
	char	*ifname;
{
	int			fd;
	char			path[MAXPATHLEN];
	union DL_primitives	*dlp;
	char			*buf;
	struct strbuf		ctl;
	int			style;
	int			ppa, flags;
	int			tmp;

	style = ifname2device_ppa(ifname, path, &ppa);
	if (style < 0) {
		/* Not found */
		errno = ENXIO;
		return (-1);
	}

	if (_skip_lib_debug) {
		fprintf(stderr, "device %s, ppa %d\n", path, ppa);
	}

	/* Open the datalink provider.
	 */
	if ((fd = open(path, O_RDWR)) < 0) {
		return(-1);
	}

	/* Allocate required buffers
	 */
	if ((buf = malloc(BUFSIZE)) == NULL) {
		_skip_error("malloc() failed");
		(void) close(fd);
		return(-1);
	}

	if (style == 2) {
		/* Issue DL_ATTACH_REQ
		 */
		dlp = (union DL_primitives*)buf;
		dlp->attach_req.dl_primitive = DL_ATTACH_REQ;
		dlp->attach_req.dl_ppa = ppa;
		ctl.buf = (char *)dlp;
		ctl.len = DL_ATTACH_REQ_SIZE;
		if (putmsg(fd, &ctl, NULL, 0) < 0) {
			_skip_error("putmsg");
			(void) close(fd);
			free(buf);
			return(-1);
		}
		
		/* start timeout for DL_OK_ACK reply
		 */
		(void) signal(SIGALRM, sigalarm_attach);
		(void) alarm(DLPI_TIMEOUT);
		
		/* read reply
		 */
		ctl.buf = (char *)dlp;
		ctl.len = 0;
		ctl.maxlen = BUFSIZE;
		flags = 0;
		if ((tmp = getmsg(fd, &ctl, NULL, &flags)) < 0) {
			_skip_error("getmsg");
			(void) close(fd);
			free(buf);
			return(-1);
		}
		if (_skip_lib_debug) {
			fprintf(stderr,
				"ok_ack: ret[%d] ctl.len[%d] flags[%d]\n",
				tmp, ctl.len, flags);
		}
		
		/* got reply - turn off alarm
		 */
		(void) alarm(0);
		(void) signal(SIGALRM, SIG_DFL);
		
		/* Validate DL_OK_ACK reply.
		 */
		if (ctl.len < sizeof(ulong)) {
			sprintf(skip_errmsg, "attach failed: short reply to "
					"attach request");
			free(buf);
			return(-1);
		}
		
		if (dlp->dl_primitive == DL_ERROR_ACK) {
			if (_skip_lib_debug) {
				fprintf(stderr,
					"attach failed: dl_errno %d "
					"unix_errno %d\n",
					dlp->error_ack.dl_errno,
					dlp->error_ack.dl_unix_errno);
			}
			(void) close(fd);
			free(buf);
			errno = ENXIO;
			_skip_error("attach failed");
			return(-1);
		}
		if (dlp->dl_primitive != DL_OK_ACK) {
			sprintf(skip_errmsg,
				"attach failed:  unrecognizable "
				"dl_primitive %d received",
				dlp->dl_primitive);
			(void) close(fd);
			free(buf);
			return(-1);
		}
		if (ctl.len < DL_OK_ACK_SIZE) {
			sprintf(skip_errmsg,
				"attach failed: short attach "
				"acknowledgement received");
			(void) close(fd);
			free(buf);
			return(-1);
		}
		if (dlp->ok_ack.dl_correct_primitive != DL_ATTACH_REQ) {
			sprintf(skip_errmsg,
				"attach failed: returned prim %d != "
				"requested prim %d\n",
				dlp->ok_ack.dl_correct_primitive,
				DL_ATTACH_REQ);
			(void) close(fd);
			free(buf);
			return(-1);
		}
		
		if (_skip_lib_debug)
			fprintf(stderr, "attach done\n");
	}
	free(buf);
	return (fd);
}
#endif
