/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skip_hdr.c	1.11	95/11/16 Sun Microsystems"

/*
 * System includes
 */
#ifdef SYSV
#include <skip_sysv.h>
#endif
#ifdef SUNOS
#include <skip_sunos.h>
#endif
#if !defined(SYSV) && !defined(SUNOS)
#include <skip_os.h>
#endif

/*
 * SKIP includes
 */
#include <bdcmod.h>
#include <skip_types.h>
#include <skip_proto.h>
#include <skip_keymgrmsgs.h>
#include <skip_key.h>
#include <skip_crypt.h>
#include <skip_ioctl.h>

/*
 * skip header statistics
 */
ioctl_hdr_stat_t	skip_hdrs_stats = { 0 };


/* skip_hdr_init()
 * 
 * Initialise a skip_hdr_t structure for use with skip_hdr_encode()
 *
 */
void
skip_hdr_init(skip_keycb_t *cp)
{
	skip_hdr_t		*hdr = &cp->hdr;
	skip_param_t		*params = &hdr->params;
	unsigned char		*p = hdr->info;
	unsigned int		i;

	bzero((caddr_t) p, SKIP_INIT_HDR_LG);

	if (params->version != SKIP_V2) {

		/*
		 * SunScreen Mode
		 */
		*p++ = 0x18;	/* Version = 1 and E = 1	*/
		*p++ = 0x80;	/* N = 1			*/
		p += 2;		/* skip reserved bytes 		*/
		/*
		 * Algorithm information field
		 */
		*p++ = params->kij_alg;
		*p++ = params->kp_alg;
		p += 2;		/* skip reserved bytes 		*/
	
		/*
		 * Set Local node ID  
		 */
		KEYTOBUF(skip_local_keys[SKIP_NSID_IPV4], p);
		p += skip_local_keys[SKIP_NSID_IPV4].len;

		/*
		 * Insert ekp 
		 */
		KEYTOBUF(hdr->ekp, p);
		p += hdr->ekp.len;

	} else {

		/*
		 * IPSP Mode
		 */
		*p++ = 0x10;	/* Version = 1 	*/
	
		/*
		 * NSID fields
		 */
		*p++ = params->s_nsid;
		*p++ = params->r_nsid;
	
		/*
		 * Next Header
		 */
		*p++ = SKIP_NEXT_ESP;
	
		/*
		 * n counter 
		 */
		*((unsigned long *)p) = params->counter_n;
		p += 4;

		/*
		 * Algorithm information field
		 */
		*p++ = params->kij_alg;
		*p++ = params->kp_alg;
		*p++ = params->mac_alg;
		*p++ = params->comp_alg;
	
		/*
		 * encrypted kp (ekp) in kijn field
		 */
		for (i = 0; i < hdr->ekp.len; i++) {
			*p++ = hdr->ekp.buf[i];
		}
		
		/*
		 * Sender  Master key ID
		 */
		if ((int) params->s_nsid < SKIP_MAX_NSID) {
			KEYTOBUF(skip_local_keys[hdr->params.s_nsid], p); 
			p += skip_local_keys[hdr->params.s_nsid].len;
		}
		
		/*
		 * Receiver  Master key ID
		 */
		if (params->r_nsid) {
			KEYTOBUF(params->r_mkeyid, p);
			p += params->r_mkeyid.len;
		}
	
	}

	hdr->hdrlen = p - hdr->info;
}

/* skip_hdr_encode()
 * 
 * Encode a skip header in a packet
 *
 * Returns: the length of the header written in bytes
 */
int
skip_hdr_encode(skip_hdr_t *hdr, unsigned char *p)
{

	bcopy((caddr_t) hdr->info, (caddr_t) p, hdr->hdrlen);
	
	if (hdr->params.version == SKIP_V2) {
		skip_hdrs_stats.skip_ipsp_encodes++;
	} else {
		skip_hdrs_stats.skip_hdr_encodes++;
	}

	return (hdr->hdrlen);
}

/*
 * skip_iv_encode() 
 *
 * Write MID/IV value to output buffer
 */
int
skip_iv_encode(skip_hdr_t * hdr, unsigned char *start) 
{
	register unsigned int	i;
	unsigned char		*p = start;
	skip_param_t		*params = &(hdr->params);

	if (params->version == SKIP_V2) {
		*((unsigned long *)p) = SKIP_SPI;
		p += 4;
	}

	/*
 	 * copy the Initialization Vector (IV)
 	 */
	for (i = 0; i < hdr->mid.len; i++) {
		*p++ = hdr->mid.buf[i];
	}

	return (p - start);
}

/* skip_hdr_v1_decode()
 * 
 * Decode a skip header in a packet
 *
 * Returns: the offset of the start of real data in the packet, or
 * -1 if the skip header was not valid.
 */
int
skip_hdr_v1_decode(unsigned char *start, unsigned char *end, 
			struct in_addr *ip_addr, skip_hdr_t *hdr)
{
	register unsigned char	*c, *p = start;
	register unsigned int	i;
	unsigned char		version;
	skip_cryptor_t		*alg;

	hdr->params.version = SKIP_V1;

	/*
	 * a minimum skip header is 8 bytes
	 */
	if ((end - p) < 8) {
		skip_hdrs_stats.skip_hdr_runts++;
		return (-1);
	}

	/*
	 * Decode the version/SAID field
	 */
	version = (*p >> 4) & 0x0f;

	if (version != SKIP_VERSION) {
		skip_hdrs_stats.skip_hdr_bad_versions++;
		return(-1);
	}
	p++;

	/*
	 * Decode the node id bit
	 */
	hdr->params.r_nsid = (*p >> 7) & 0x01;
	hdr->params.s_nsid = 0;

	/*
	 * The node id must be present
	 */
	if (!hdr->params.r_nsid) {
		skip_hdrs_stats.skip_hdr_short_nodeids++;
		return (-1);
	}
	p++;

	/*
	 * skip remainder of SAID field
	 */
	p += 2;

	hdr->params.kij_alg 	= *p++;
	hdr->params.kp_alg 	= *p++;

	/*
	 * skip the skip reserved field
	 */
	p += 2;

	/*
	 * check that this is a supported kp alg
	 */
	alg = GETCRYPTMOD(SKIP_V1, hdr->params.kp_alg);

	if (alg->mid_len < 0) {
		skip_hdrs_stats.skip_hdr_bad_kp_algs++;
		return (-1);
	}
	/*
	 * nodeid field
	 */
	hdr->params.r_mkeyid.len = 4;
	if ((end - p) < hdr->params.r_mkeyid.len) {
		skip_hdrs_stats.skip_hdr_short_nodeids++;
		return (-1);
	}
	c = (unsigned char *) &hdr->params.r_mkeyid.buf;
	*c++ = *p++;
	*c++ = *p++;
	*c++ = *p++;
	*c++ = *p++;

	/*
	 * encrypted kp (ekp) field
	 */
	if (alg->ekp_len < 0) {
		skip_hdrs_stats.skip_hdr_bad_kp_algs++;
		return (-1);
	}

	hdr->ekp.len = alg->ekp_len;
	
	if ((int) (end - p) < hdr->ekp.len) {
		skip_hdrs_stats.skip_hdr_short_ekps++;
		return (-1);
	}

	/*
	 * do this with a for-loop to avoid alignment problems and the
	 * overhead of calling bcopy()
	 */
	for (i = 0; i < hdr->ekp.len; i++) {
		hdr->ekp.buf[i] = *p++;
	}
	
	skip_hdrs_stats.skip_hdr_decodes++;

	/*
	 * Keep the original IP address
	 */
	bcopy((caddr_t) ip_addr,
			(caddr_t) &(hdr->params.ip_addr),
			sizeof(ip_addr));


	return (p - start);
}

/*
 * skip_v1_iv_decode() 
 *
 * Retrieve the value of the IV 
 */
int
skip_iv_v1_decode(skip_hdr_t * hdr, unsigned char *start, unsigned char *end) 
{
	unsigned char	*p = start;
	skip_cryptor_t	*kp_crypt;
	unsigned int	i;

	kp_crypt = GETCRYPTMOD(SKIP_V1, hdr->params.kp_alg);

	hdr->mid.len = kp_crypt->mid_len;

	/*
	 * copy the MID/Initialization Vector (IV)
	 */
	if ((int) (end - p) < hdr->mid.len) {
		skip_hdrs_stats.skip_hdr_short_mids++;
		return (-1);
	}

	for (i = 0; i < hdr->mid.len; i++) {
		hdr->mid.buf[i] = *p++;
	}

	SKIP_ESP_DEBUG1("skip_iv_v1_decode: IV length=%d\n", (p - start));

	return (p - start);

}
