/*
 * Copyright (c) 1992, 1993, 1995
 *      The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software donated to Berkeley by
 * Jan-Simon Pendry.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      @(#)cryptfs_vfsops.c     8.2 (Berkeley) 1/21/94
 *
 * @(#)lofs_vfsops.c    1.2 (Berkeley) 6/18/92
 * $Id: vfs.c,v 1.1.1.1 1998/11/05 21:06:28 ezk Exp $
 */

/*
 * Cryptfs Layer
 * (See cryptfs_vnops.c for a description of what this does.)
 */

#include "opt_debug.h"

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/proc.h>
#include <sys/malloc.h>
#include <sys/vnode.h>
#include <sys/mount.h>
#include <sys/namei.h>
#include <cryptfs.h>

static MALLOC_DEFINE(M_CRYPTFSMNT, "CRYPTFS mount", "CRYPTFS mount structure");

static int cryptfs_fhtovp __P((struct mount *mp, struct fid * fidp,
			      struct sockaddr * nam, struct vnode ** vpp,
			      int *exflagsp, struct ucred ** credanonp));
static int cryptfs_mount __P((struct mount *mp, char *path, caddr_t data,
			     struct nameidata * ndp, struct proc * p));
static int cryptfs_quotactl __P((struct mount *mp, int cmd, uid_t uid,
				caddr_t arg, struct proc * p));
static int cryptfs_root __P((struct mount *mp, struct vnode ** vpp));
static int cryptfs_start __P((struct mount *mp, int flags, struct proc * p));
static int cryptfs_statfs __P((struct mount *mp, struct statfs * sbp,
			      struct proc * p));
static int cryptfs_sync __P((struct mount *mp, int waitfor,
			    struct ucred * cred, struct proc * p));
static int cryptfs_unmount __P((struct mount *mp, int mntflags,
			       struct proc * p));
static int cryptfs_vget __P((struct mount *mp, ino_t ino,
			    struct vnode ** vpp));
static int cryptfs_vptofh __P((struct vnode * vp, struct fid * fhp));

BF_KEY fixed_key;
static unsigned char cbc_key[16] =
	{0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
	0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22};

/*
 * Mount cryptfs layer
 */
static int
cryptfs_mount(mp, path, data, ndp, p)
     struct mount *mp;
     char *path;
     caddr_t data;
     struct nameidata *ndp;
     struct proc *p;
{
  int error = 0;
  struct cryptfs_args args;
  struct vnode *lowerrootvp, *vp;
  struct vnode *cryptfsm_rootvp;
  struct cryptfs_mount *xmp;
  u_int size;
  int isvnunlocked = 0;

#ifdef CRYPTFS_DIAGNOSTIC
  printf("cryptfs_mount(mp = %p)\n", mp);
#endif

  /*
   * Update is a no-op
   */
  if (mp->mnt_flag & MNT_UPDATE) {
    return (EOPNOTSUPP);
    /* return VFS_MOUNT(MOUNTTOCRYPTFSMOUNT(mp)->cryptfsm_vfs, path, data, ndp,
     * p); */
  }
  /*
   * Get argument
   */
  error = copyin(data, (caddr_t) & args, sizeof(struct cryptfs_args));
  if (error)
    return (error);

  /*
   * Unlock lower node to avoid deadlock.
   * (XXX) VOP_ISLOCKED is needed?
   */
  if ((mp->mnt_vnodecovered->v_op == cryptfs_vnodeop_p) &&
      VOP_ISLOCKED(mp->mnt_vnodecovered)) {
    VOP_UNLOCK(mp->mnt_vnodecovered, 0, p);
    isvnunlocked = 1;
  }
  /*
   * Find lower node
   */
  NDINIT(ndp, LOOKUP, FOLLOW | WANTPARENT | LOCKLEAF,
	 UIO_USERSPACE, args.target, p);
  error = namei(ndp);
  /*
   * Re-lock vnode.
   */
  if (isvnunlocked && !VOP_ISLOCKED(mp->mnt_vnodecovered))
    vn_lock(mp->mnt_vnodecovered, LK_EXCLUSIVE | LK_RETRY, p);

  if (error)
    return (error);

  /*
   * Sanity check on lower vnode
   */
  lowerrootvp = ndp->ni_vp;

  vrele(ndp->ni_dvp);
  ndp->ni_dvp = NULLVP;

  /*
   * Check multi cryptfs mount to avoid `lock against myself' panic.
   */
  if (lowerrootvp == VP_TO_CRYPTFS(mp->mnt_vnodecovered)->cryptfs_lowervp) {
#ifdef DIAGNOSTIC
    printf("cryptfs_mount: multi cryptfs mount?\n");
#endif
    return (EDEADLK);
  }
  xmp = (struct cryptfs_mount *) malloc(sizeof(struct cryptfs_mount),
				       M_CRYPTFSMNT, M_WAITOK);	/* XXX */

  /*
   * Save reference to underlying FS
   */
  xmp->cryptfsm_vfs = lowerrootvp->v_mount;

  /*
   * Save reference.  Each mount also holds
   * a reference on the root vnode.
   */
  error = cryptfs_node_create(mp, lowerrootvp, &vp);
  /*
   * Unlock the node (either the lower or the alias)
   */
  VOP_UNLOCK(vp, 0, p);
  /*
   * Make sure the node alias worked
   */
  if (error) {
    vrele(lowerrootvp);
    free(xmp, M_CRYPTFSMNT);	/* XXX */
    return (error);
  }
  /*
   * Keep a held reference to the root vnode.
   * It is vrele'd in cryptfs_unmount.
   */
  cryptfsm_rootvp = vp;
  cryptfsm_rootvp->v_flag |= VROOT;
  xmp->cryptfsm_rootvp = cryptfsm_rootvp;
  if (CRYPTFS_VP_TO_LOWERVP(cryptfsm_rootvp)->v_mount->mnt_flag & MNT_LOCAL)
    mp->mnt_flag |= MNT_LOCAL;
  mp->mnt_data = (qaddr_t) xmp;
  vfs_getnewfsid(mp);

  (void) copyinstr(path, mp->mnt_stat.f_mntonname, MNAMELEN - 1, &size);
  bzero(mp->mnt_stat.f_mntonname + size, MNAMELEN - size);
  (void) copyinstr(args.target, mp->mnt_stat.f_mntfromname, MNAMELEN - 1,
		   &size);
  bzero(mp->mnt_stat.f_mntfromname + size, MNAMELEN - size);
  (void) cryptfs_statfs(mp, &mp->mnt_stat, p);
#ifdef CRYPTFS_DIAGNOSTIC
  printf("cryptfs_mount: lower %s, alias at %s\n",
	 mp->mnt_stat.f_mntfromname, mp->mnt_stat.f_mntonname);
#endif

  /* finally, setup the Blowfish key */
  BF_set_key(&fixed_key, 16, cbc_key);

  return (0);
}

/*
 * VFS start.  Nothing needed here - the start routine
 * on the underlying filesystem will have been called
 * when that filesystem was mounted.
 */
static int
cryptfs_start(mp, flags, p)
     struct mount *mp;
     int flags;
     struct proc *p;
{
  return (0);
  /* return VFS_START(MOUNTTOCRYPTFSMOUNT(mp)->cryptfsm_vfs, flags, p); */
}

/*
 * Free reference to cryptfs layer
 */
static int
cryptfs_unmount(mp, mntflags, p)
     struct mount *mp;
     int mntflags;
     struct proc *p;
{
  struct vnode *cryptfsm_rootvp = MOUNT_TO_CRYPTFS_MOUNT(mp)->cryptfsm_rootvp;
  int error;
  int flags = 0;

#ifdef CRYPTFS_DIAGNOSTIC
  printf("cryptfs_unmount(mp = %p)\n", mp);
#endif

  if (mntflags & MNT_FORCE)
    flags |= FORCECLOSE;

  /*
   * Clear out buffer cache.  I don't think we
   * ever get anything cached at this level at the
   * moment, but who knows...
   */
#if 0
  mntflushbuf(mp, 0);
  if (mntinvalbuf(mp, 1))
    return (EBUSY);
#endif
  if (cryptfsm_rootvp->v_usecount > 1)
    return (EBUSY);
  error = vflush(mp, cryptfsm_rootvp, flags);
  if (error)
    return (error);

#ifdef CRYPTFS_DIAGNOSTIC
  vprint("alias root of lower", cryptfsm_rootvp);
#endif
  /*
   * Release reference on underlying root vnode
   */
  vrele(cryptfsm_rootvp);
  /*
   * And blow it away for future re-use
   */
  vgone(cryptfsm_rootvp);
  /*
   * Finally, throw away the cryptfs_mount structure
   */
  free(mp->mnt_data, M_CRYPTFSMNT);	/* XXX */
  mp->mnt_data = 0;
  return 0;
}

static int
cryptfs_root(mp, vpp)
     struct mount *mp;
     struct vnode **vpp;
{
  struct proc *p = curproc;	/* XXX */
  struct vnode *vp;

#ifdef CRYPTFS_DIAGNOSTIC
  printf("cryptfs_root(mp = %p, vp = %x->%x)\n", mp,
	 MOUNTTOCRYPTFSMOUNT(mp)->cryptfsm_rootvp,
	 CRYPTFSVPTOLOWERVP(MOUNTTOCRYPTFSMOUNT(mp)->cryptfsm_rootvp)
    );
#endif

  /*
   * Return locked reference to root.
   */
  vp = MOUNT_TO_CRYPTFS_MOUNT(mp)->cryptfsm_rootvp;
  VREF(vp);
  if (VOP_ISLOCKED(vp)) {
    /*
     * XXX
     * Should we check type of node?
     */
#ifdef DIAGNOSTIC
    printf("cryptfs_root: multi cryptfs mount?\n");
#endif
    vrele(vp);
    return (EDEADLK);
  } else
    vn_lock(vp, LK_EXCLUSIVE | LK_RETRY, p);
  *vpp = vp;
  return 0;
}

static int
cryptfs_quotactl(mp, cmd, uid, arg, p)
     struct mount *mp;
     int cmd;
     uid_t uid;
     caddr_t arg;
     struct proc *p;
{
  return VFS_QUOTACTL(MOUNT_TO_CRYPTFS_MOUNT(mp)->cryptfsm_vfs, cmd, uid, arg, p);
}

static int
cryptfs_statfs(mp, sbp, p)
     struct mount *mp;
     struct statfs *sbp;
     struct proc *p;
{
  int error;
  struct statfs mstat;

#ifdef CRYPTFS_DIAGNOSTIC
  printf("cryptfs_statfs(mp = %p, vp = %x->%x)\n", mp,
	 MOUNTTOCRYPTFSMOUNT(mp)->cryptfsm_rootvp,
	 CRYPTFSVPTOLOWERVP(MOUNTTOCRYPTFSMOUNT(mp)->cryptfsm_rootvp)
    );
#endif

  bzero(&mstat, sizeof(mstat));

  error = VFS_STATFS(MOUNT_TO_CRYPTFS_MOUNT(mp)->cryptfsm_vfs, &mstat, p);
  if (error)
    return (error);

  /* now copy across the "interesting" information and fake the rest */
  sbp->f_type = mstat.f_type;
  sbp->f_flags = mstat.f_flags;
  sbp->f_bsize = mstat.f_bsize;
  sbp->f_iosize = mstat.f_iosize;
  sbp->f_blocks = mstat.f_blocks;
  sbp->f_bfree = mstat.f_bfree;
  sbp->f_bavail = mstat.f_bavail;
  sbp->f_files = mstat.f_files;
  sbp->f_ffree = mstat.f_ffree;
  if (sbp != &mp->mnt_stat) {
    bcopy(&mp->mnt_stat.f_fsid, &sbp->f_fsid, sizeof(sbp->f_fsid));
    bcopy(mp->mnt_stat.f_mntonname, sbp->f_mntonname, MNAMELEN);
    bcopy(mp->mnt_stat.f_mntfromname, sbp->f_mntfromname, MNAMELEN);
  }
  return (0);
}

static int
cryptfs_sync(mp, waitfor, cred, p)
     struct mount *mp;
     int waitfor;
     struct ucred *cred;
     struct proc *p;
{
  /*
   * XXX - Assumes no data cached at cryptfs layer.
   */
  fist_dprint(4, "CRYPTFS_SYNC: vfs=0x%x\n", (int) mp);
  return (0);
}

static int
cryptfs_vget(mp, ino, vpp)
     struct mount *mp;
     ino_t ino;
     struct vnode **vpp;
{

  return VFS_VGET(MOUNT_TO_CRYPTFS_MOUNT(mp)->cryptfsm_vfs, ino, vpp);
}

static int
cryptfs_fhtovp(mp, fidp, nam, vpp, exflagsp, credanonp)
     struct mount *mp;
     struct fid *fidp;
     struct sockaddr *nam;
     struct vnode **vpp;
     int *exflagsp;
     struct ucred **credanonp;
{

  return VFS_FHTOVP(MOUNT_TO_CRYPTFS_MOUNT(mp)->cryptfsm_vfs, fidp, nam,
		    vpp, exflagsp, credanonp);
}

static int
cryptfs_vptofh(vp, fhp)
     struct vnode *vp;
     struct fid *fhp;
{
  return VFS_VPTOFH(CRYPTFS_VP_TO_LOWERVP(vp), fhp);
}

static struct vfsops cryptfs_vfsops =
{
  cryptfs_mount,
  cryptfs_start,
  cryptfs_unmount,
  cryptfs_root,
  cryptfs_quotactl,
  cryptfs_statfs,
  cryptfs_sync,
  cryptfs_vget,
  cryptfs_fhtovp,
  cryptfs_vptofh,
  cryptfs_init,
};

VFS_SET(cryptfs_vfsops, cryptfs, VFCF_LOOPBACK);
