/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skip_es.c	1.17 95/11/21 Sun Microsystems"

/*
 * System includes
 */
#include <skip_sunos.h>

/*
 * SKIP includes
 */
#include <skip_types.h>
#include <skip_proto.h>
#include <skip_keymgrmsgs.h>
#include <skip_key.h>
#include <skip_ioctl.h>
#include <skip_es.h>
#include <skip_if.h>
#include <skip_crypt.h>

/*
 * global SKIP interface information
 */
char			skip_module_name[] 	= "skip v1.17";
int			skip_busy 		= 0;
int			skip_key_debug 		= 0;
const int		skip_key_tick		= SKIP_KEY_TICK;
int			skip_alignment		= 4; /* 32 bit */
static struct protosw	*inetsw_default		= NULL;
static unsigned int	inetsw_size		= 0;
skip_es_t		*skip_es_ifs		= NULL;
static int		skip_inited 		= 0;
static unsigned short	skip_pktid;
static skip_softc_t	skip_softc[SKIP_MAX_OPENS];

/*
 * statistics
 */
static unsigned long	skip_bufallocfail	= 0;
extern ioctl_key_stat_t	skip_keys_stats;
MUTEX_T			skip_es_lock;


/*
 * crypto modules available...
 */
extern skip_cryptor_t	simplecrypt_module;
extern const int	simplecrypt_id;
extern skip_cryptor_t	des_cbc_module_v1;
extern skip_cryptor_t	des_cbc_module;
extern const int	des_cbc_id;
extern skip_cryptor_t	des_ede_k2_module;
extern skip_cryptor_t	des_ede_k2_module_v1;
extern const int	des_ede_k2_id;
extern skip_cryptor_t	des_ede_k3_module;
extern skip_cryptor_t	des_ede_k3_module_v1;
extern const int	des_ede_k3_id;

extern skip_es_t	* skip_es_find_if(char *);
static skip_es_t	* skip_es_find_ifnet(struct ifnet *);

/* skip_init()
 *
 * Install SKIP in system
 *
 * Returns: 0 on success, errno otherwise.
 */
int
skip_init()
{
	register int		s;
	register struct protosw	*pr;

	/*
	 *  one-off initialisations
	 */
	if (!skip_inited) {

		skip_inited = 1;
		skip_key_initstore();

		/*
		 * Initialize the crypto system
		 */
		skip_key_initcryptors();

		/*
		 * SunScreen Mode cryptors
		 */
		skip_install_cryptmod(&simplecrypt_module,
						SKIP_SIMPLECRYPT, SKIP_V1);
		skip_install_cryptmod(&des_cbc_module_v1,
						SKIP_DES_CBC, SKIP_V1);
		skip_install_cryptmod(&des_ede_k2_module_v1,
						SKIP_DES_EDE_K2, SKIP_V1);
		skip_install_cryptmod(&des_ede_k3_module_v1,
						SKIP_DES_EDE_K3, SKIP_V1);

		/*
		 * IPSP Mode cryptors
		 */
		skip_install_cryptmod(&simplecrypt_module,
					SKIP_CRYPT_SIMPLECRYPT, SKIP_V2);
		skip_install_cryptmod(&des_cbc_module,
					SKIP_CRYPT_DES_CBC, SKIP_V2);
		skip_install_cryptmod(&des_ede_k3_module,
					SKIP_CRYPT_DES_EDE_K3, SKIP_V2);
		skip_inittimers();
		skip_pktid = htons(lbolt);

		s = splimp();
		/*
		 * save current inetsw
		 */
		inetsw_size = (inetdomain.dom_protoswNPROTOSW -
			inetdomain.dom_protosw) * sizeof(struct protosw);

		inetsw_default = KMEM_ALLOC(inetsw_size, KMEM_NOSLEEP);
		if (inetsw_default== NULL) {
			return (ENOMEM);
		}
		bcopy ((caddr_t) inetsw, (caddr_t) inetsw_default, inetsw_size);

		/*
		 * redirect protocol input routines to skip
		 */
		s = splimp();
		for (pr = inetdomain.dom_protosw;
			pr < inetdomain.dom_protoswNPROTOSW; pr++) {

			pr->pr_input = skip_ifinput;
		}
		splx(s);
	}
	return (0);
}
 
/* skip_uninit()
 *
 * Uninstall SKIP in system
 *
 * Returns: 0 on success, errno otherwise.
 */
int
skip_uninit()
{
	int		s;

	if (skip_busy || skip_keys_stats.skip_encrypt_keys_active 
			|| skip_keys_stats.skip_decrypt_keys_active) {
		return (EBUSY);
	}

	if (skip_inited) {
		/*
		 * restore inetsw
		 */
		s = splimp();
		bcopy((caddr_t) inetsw_default, (caddr_t) inetsw, inetsw_size);
		splx(s);
		KMEM_FREE(inetsw_default, inetsw_size);
		inetsw_size = 0;

		skip_uninittimers();
		skip_key_uninitcryptors();
		skip_key_uninitstore();
		skip_inited = 0;
	}
	return (0);
}
 
/* skip_ifinput()
 *
 * Receive a packet from ipintr(), apply access control and possibly
 * decrypt it if required.
 *
 * Returns: 0 on success, errno otherwise.
 */
static int
skip_ifinput(m, ifp)
	struct mbuf		*m;
	struct ifnet		*ifp;
{
	register struct ip	*ip = mtod(m, struct ip *);
	register skip_es_t	*skip_if;
	register int		iphlen, hdrlen = 0;
	struct mbuf		*decryptbuf;
	extern u_char		ip_protox[];
	int			rc;
	skip_param_t		params;
	skip_hdr_t		skip_hdr;

	bzero((caddr_t) &params, sizeof(skip_param_t));
	/* cleartext by default */
	bzero((caddr_t) &skip_hdr, sizeof(skip_hdr_t));
	
	if (!SKIP_IF(ifp)) {
		/*
		 * SKIP not present on this interface, pass the packet
		 */
		goto pass;
	}

	skip_if = skip_es_find_ifnet(ifp);
	if (skip_if == NULL) {
		panic("skip: failed to find interface in skip_ifinput()\n");
	}

	/*
	 * check for tag which indicates the packet has already been
	 * through SKIP
	 */
	if (SKIP_DECRYPTED(m, skip_if)) {
		goto pass;
	}

	skip_if->stats.skip_if_ipkts++;

	SKIP_DEBUG("^");

	if (skip_if->if_mode == SkipAccessControlOff) {
		goto pass;
	}

	/*
	 * check the IP protocol field
	 */
	iphlen = ip->ip_hl << 2;

	switch (ip->ip_p) {

	case SKIP_PROTO_V1:
	case SKIP_IPSP:
		/*
		 * SKIP packet - decode the header to verify the
		 * algorithm information
		 */
		m = skip_bufalign(m, 0);
		if (m == NULL) {
			skip_if->stats.skip_if_drops++;
			return (ENOBUFS);
		}
		ip = mtod(m, struct ip *);

		if (ip->ip_p == SKIP_PROTO_V1) {
			/*
			 * We are in SunScreen mode...
			 */
			hdrlen = skip_hdr_v1_decode(
					(unsigned char *) ip + iphlen,
				 	(unsigned char *) ip + m->m_len,
					&ip->ip_src,
				 	&skip_hdr	
					);
		} else {
			/*
			 * We are in IPSP mode...
			 */
			hdrlen = skip_hdr_ipsp_decode(
					(unsigned char *) ip + iphlen,
				 	(unsigned char *) ip + m->m_len,
					&ip->ip_src,
				 	&skip_hdr	
					);
		}

		if (hdrlen < 0) {
			/*
			 * bad SKIP header
			 */
			skip_if->stats.skip_if_drops++;
			m_freem(m);
			SKIP_ESP_DEBUG("skip_ifinput: bad header length\n");
			return (EPROTO);
		}
		break;

	case IPPROTO_UDP:
		/*
		 * make sure that the UDP header is present for any
		 * potential bypass test.
		 */
		if (m->m_len < (iphlen + sizeof (struct udphdr))) {
			m = m_pullup(m, iphlen + sizeof (struct udphdr));
			if (m == NULL) {
				skip_if->stats.skip_if_drops++;
				return (ENOBUFS);
			}
			ip = mtod(m, struct ip *);
		}
		break;
	}

	/*
	 * check if we've heard of this host
	 */
	MUTEX_ENTER(&skip_es_lock);

	if (!skip_es_ok(skip_if,
			&ip->ip_src,
			ip,
			m->m_len,
			&params
			)) {

		/*
		 * host is not configured - tell someone about it.
		 */
		if (hdrlen == 0) {
			bcopy((caddr_t) &ip->ip_src,
					(caddr_t) &skip_hdr.params.ip_addr,
					sizeof(skip_hdr.params.ip_addr));
		}
		skip_notify(skip_if,
				ip,
				&skip_hdr.params,
				SkipUnknownSrc
				);
		skip_if->stats.skip_if_drops++;
		m_freem(m);
		MUTEX_EXIT(&skip_es_lock);
		return (EACCES);

	}

	if (!ACLMATCH(&params, &(skip_hdr.params))) {
		/*
		 * The remote is not using the correct parameters
		 */
		skip_diffs("skip_ifinput", &params, &(skip_hdr.params));
		if (hdrlen == 0) {
			bcopy((caddr_t) &ip->ip_src,
					(caddr_t) &skip_hdr.params.ip_addr,
					sizeof(skip_hdr.params.ip_addr));
		}
		skip_notify(skip_if,
				ip,
				&skip_hdr.params,
				SkipBadParams
				);
		skip_if->stats.skip_if_drops++;
		m_freem(m);
		MUTEX_EXIT(&skip_es_lock);
		return (EACCES);
	}

	MUTEX_EXIT(&skip_es_lock);

	if (!skip_hdr.params.kp_alg && !skip_hdr.params.mac_alg &&
			!skip_hdr.params.comp_alg) {
		/*
		 * host allowed to communicate in the clear - send the
		 * packet and go...
		 */
		goto pass;
	}

	if (skip_hdr.params.comp_alg) {
		/*
		 * Compressed packet...drop it!
		 */
		skip_if->stats.skip_if_drops++;
		SKIP_ESP_DEBUG("skip_ifinput: received IPSP compressed!\n");
		m_freem(m);
		return (EPROTO);
	}
	
	if (skip_hdr.params.mac_alg) {
		/*
		 * AH packet...drop it!
		 */
		skip_if->stats.skip_if_drops++;
		SKIP_ESP_DEBUG("skip_ifinput: received AH packet!\n");
		m_freem(m);
		return (EPROTO);
	}

	if (skip_hdr.params.kp_alg) {
		/*
	 	 * The packet was encrypted.
		 * Allocate a buffer to decrypt it.
		 */
		SKIP_ESP_DEBUG1("skip_ifinput: allocate size=%d\n", m->m_len);
		decryptbuf = skip_bufalloc(m->m_len);
		if (decryptbuf == NULL) {
			m_freem(m);
			skip_bufallocfail++;
			return(ENOBUFS);
		}
	
		SKIP_ESP_DEBUG2("skip_ifinput: decryptbuf m_len=%d m_off=%d\n",
					decryptbuf->m_len, decryptbuf->m_off);
		/*
		 * and give to the decryptor...
		 */
		skip_decrypt(skip_if->sd, m, decryptbuf, 
					&skip_hdr, (iphlen + hdrlen), NULL);
		skip_if->stats.skip_if_decrypts++;
		return (0);
	} 

pass:
	(*inetsw_default[ip_protox[ip->ip_p]].pr_input)(m, ifp);
	return (0);
}


/* skip_ifoutput()
 *
 * Receive a packet from IP, apply access control & optionally encrypt
 * before transmission.
 *
 * Returns: 0
 */
int
skip_ifoutput(ifp, m, dst)
	struct ifnet		*ifp;
	struct mbuf		*m;
	struct sockaddr		*dst;
{
	register skip_es_t		*skip_if;
	register struct ip		*ip, *newip;
	register int			iphlen;
	register struct mbuf		*encryptbuf;
	skip_param_t			params;
	
	bzero((caddr_t) &params, sizeof(skip_param_t));

	skip_if = skip_es_find_ifnet(ifp);
	if (skip_if == NULL) {
		panic("skip: failed to find interface in skip_ifoutput()\n");
	}

	if (dst->sa_family != AF_INET) {
		return (*skip_if->if_output)(ifp, m, dst);
	}

	SKIP_DEBUG(".");

	if (skip_if->if_mode == SkipAccessControlOff) {
		return (*skip_if->if_output)(ifp, m, dst);
	}
		
	skip_if->stats.skip_if_opkts++;

	/*
	 * check if destination requires encryption
	 */
	ip = mtod(m, struct ip *);
	iphlen = ip->ip_hl << 2;

	if (ip->ip_p == IPPROTO_UDP) {
		/*
		 * make sure that the UDP header is present for any
		 * potential bypass test.
		 */
		if (m->m_len < (iphlen + sizeof (struct udphdr))) {
			m = m_pullup(m, iphlen + sizeof (struct udphdr));
			if (m == NULL) {
				skip_if->stats.skip_if_drops++;
				return (0);
			}
			ip = mtod(m, struct ip *);
		}
	}

	MUTEX_ENTER(&skip_es_lock);

	if (!skip_es_ok(skip_if, &ip->ip_dst, ip, m->m_len, &params)) {
		/*
		 * never heard of this host
		 */
		bcopy((caddr_t) &ip->ip_dst, (caddr_t) &params.ip_addr, 
						sizeof(params.ip_addr));
		skip_notify(skip_if,
				ip,
				&params,
				SkipUnknownDst
				);
		MUTEX_EXIT(&skip_es_lock);
		m_freem(m);
		skip_if->stats.skip_if_drops++;
		return (EACCES);
	}

	MUTEX_EXIT(&skip_es_lock);

	if (params.version != SKIP_V2) {

		/*
		 * We are in SunScreen mode..
		 */
		if (params.kp_alg == 0) {
			/*
			 * configured to send in the clear
			 */
			return (*skip_if->if_output)(ifp, m, dst);
		}
	
		/*
		 * packet is to be encrypted - make sure source is aligned and
		 * contiguous.
		 */
		m = skip_bufalign(m, SKIP_MAX_PAD);
		if (m == NULL) {
			return (ENOBUFS);
		}
	
		/*
		 * prepare a buffer for the encryptor
		 */
		encryptbuf = skip_bufalloc(m->m_len + SKIP_HDR_SZ);
		if (encryptbuf == NULL) {
			m_freem(m);
			skip_bufallocfail++;
			return (ENOBUFS);
		}
	
		/*
		 * insert new IP header in encryptbuf
		 */
		newip = mtod(encryptbuf, struct ip *);
	
		bcopy((caddr_t) ip, (caddr_t) newip, sizeof(struct ip));
	
		newip->ip_p = SKIP_PROTO_V1;
	
		/*
		 * Mark the SKIP packet as unfragmented, even though the payload
		 * may have been a fragment.
		 */
		newip->ip_off = 0;
	
		/*
		 * insert a new packet id
		 */
		newip->ip_id = htons(skip_pktid++);
	
		encryptbuf->m_len += sizeof (struct ip);
	
		bcopy((caddr_t) &ip->ip_dst, (caddr_t) &params.ip_addr,
						sizeof(params.ip_addr));
		/*
		 * handle bundle to encryptor
		 */
		skip_encrypt(skip_if->sd,	/* SKIP client handle */
				m,		/* plaintext packet */
				encryptbuf,	/* buffer for ciphertext */
				&params,	/* XXX - new SKIP V2 */
				dst);

		skip_if->stats.skip_if_encrypts++;
		return (0);
	}

	/*
	 * From now, IPSP mode...
	 */
	if (params.comp_alg) {
		/*
		 * XXX - The packet should be compressed - drop it!
		 */
		m_freem(m);
		SKIP_ESP_DEBUG1(
			"skip_ifoutput: should compress ESP, alg=%d !\n",
			params.comp_alg);
		skip_if->stats.skip_if_drops++;
		return (EPROTO);
	}

	if (!params.kp_alg && !params.mac_alg) {
		/*
		 * configured to send in the clear
		 */
		return (*skip_if->if_output)(ifp, m, dst);
	}

	/*
	 * packet is to be encrypted - make sure source is aligned and
	 * contiguous.
	 */
	m = skip_bufalign(m, SKIP_MAX_PAD);
	if (m == NULL) {
		skip_if->stats.skip_if_drops++;
		return (ENOBUFS);
	}

	/*
	 * prepare a buffer for the encryptor
	 */
	encryptbuf = skip_bufalloc(m->m_len + SKIP_HDR_SZ);
	if (encryptbuf == NULL) {
		m_freem(m);
		skip_bufallocfail++;
		return (ENOBUFS);
	}

	/*
	 * insert new IP header in encryptbuf
	 */
	newip = mtod(encryptbuf, struct ip *);

	bcopy((caddr_t) ip, (caddr_t) newip, sizeof(struct ip));

	newip->ip_p = SKIP_IPSP;

	/*
	 * Mark the SKIP packet as unfragmented, even though the payload
	 * may have been a fragment.
	 */
	newip->ip_off = 0;

	/*
	 * insert a new packet id
	 */
	newip->ip_id = htons(skip_pktid++);

	encryptbuf->m_len += sizeof (struct ip);

	bcopy((caddr_t) &ip->ip_dst, (caddr_t) &params.ip_addr,
						sizeof(params.ip_addr));

	SKIP_ESP_DEBUG1("ifoutput: ip_len = %d\n", ntohs(ip->ip_len));
	/*
	 * handle bundle to encryptor
	 */
	skip_encrypt(skip_if->sd,	/* SKIP client handle */
			m,		/* plaintext packet */
			encryptbuf,	/* buffer for ciphertext */
			&params,	/* XXX - new SKIP V2 */
			dst);

	skip_if->stats.skip_if_encrypts++;
	return (0);
}

/* skip_decrypt_done()
 *
 * receive a decrypted packet from SKIP and send to IP
 *
 * Returns: none
 */
static void
skip_decrypt_done(client_data, original, m, dst, rc)
	void		*client_data;
	struct mbuf	*original;
	struct mbuf	*m;
	struct sockaddr	*dst;
	int		rc;
{
	register skip_es_t	*skip_if = client_data;
	register struct ip	*ip;
	skip_param_t		params;
	int			iphlen, s;
	extern struct ifqueue	ipintrq;
	struct mbuf		*new_hdr;

	m_freem(original);

	if (rc) {
		SKIP_ESP_DEBUG("skip_decrypt_done: error\n");
		m_freem(m);
		return;
	}

	/*
	 * check we want to talk to the now-decrypted remote host
	 */
	ip = mtod(m, struct ip *);

	SKIP_ESP_DEBUG2("skip_decrypt_done: loopk up for 0x%x, size=%d\n",
		ip->ip_src.s_addr, m->m_len - sizeof(struct ifnet *));

	MUTEX_ENTER(&skip_es_lock);
	if (!skip_es_ok(skip_if,
			&ip->ip_src,
			ip,
			m->m_len,
			&params
			)) {
		/*
		 * Interesting.  We believed this was a host we wanted to
		 * talk to (based on the outer IP address).  The inside
		 * address we don't like.
		 *
		 * XXX Should look at original.
		 */
		SKIP_ESP_DEBUG("skip_decrypt_done: SkipAuthFailed\n");
		skip_notify(skip_if, ip, NULL, SkipAuthFailed);
		MUTEX_EXIT(&skip_es_lock);
		m_freem(m);
		skip_if->stats.skip_if_drops++;
		return;
	}
	MUTEX_EXIT(&skip_es_lock);

	iphlen = ip->ip_hl << 2;
	if (m->m_len < iphlen) {
		/*
		 * short packet?
		 */
		skip_if->stats.skip_if_drops++;
		m_freem(m);
		return;
	}

	SKIP_ESP_DEBUG2("skip_decrypt_done:  ip_len=%d, m->m_off=%d\n",
						ntohs(ip->ip_len), m->m_off);

	/*
	 * XXX could have re-used the original
	 */
	MGET(new_hdr, M_DONTWAIT, MT_DATA);
	if (new_hdr == NULL) {
		m_freem(m);
		skip_bufallocfail++;
		return;
	}

	/*
	 * tag the start of the header buffer so SKIP can recognise it
	 */
	* mtod(new_hdr, skip_es_t **) = skip_if;
	new_hdr->m_off += sizeof(skip_es_t *);

	/*
	 * add the ifnet pointer
	 */
	* mtod(new_hdr, struct ifnet **) = skip_if->ifp;

	/*
	 * pull up the ip header (which would have been done anyway by
	 * ipintr()
	 */
	bcopy(mtod(m, caddr_t), mtod(new_hdr, caddr_t) + sizeof(struct ifnet *),
									iphlen);
		
	new_hdr->m_len = sizeof(struct ifnet *) + iphlen;

	m->m_off += iphlen;
	m->m_len -= iphlen;
	
	SKIP_ESP_DEBUG1("skip_decrypt_done:  final m->m_len=%d\n",  m->m_len);

	/*
	 * send the decrypted packet back to IP for regular processing
	 */
	s = splimp();
	new_hdr->m_next = m;
	if (IF_QFULL(&ipintrq)) {
                IF_DROP(&ipintrq);
                m_freem(new_hdr);
                (void) splx(s);
                return;
        }
	if (ipintrq.ifq_len == 0) {
		schednetisr(NETISR_IP);
	}
        IF_ENQUEUE(&ipintrq, new_hdr);
        (void) splx(s);
}

/* skip_encrypt_done()
 *
 * complete an encrypted packet from SKIP and send down to the network
 *
 * Returns: none
 */
static void
skip_encrypt_done(client_data, original, m, dst, rc)
	void		*client_data;
	struct mbuf	*original;
	struct mbuf	*m;
	struct sockaddr	*dst;
	int		rc;
{
	skip_es_t		*skip_if = client_data;
	unsigned short		len, *sump;
	unsigned int		sum;
	struct ip		*hptr;
	
	m_freem(original);

	if (rc) {
		m_freem(m);
		return;
	}
	SKIP_ESP_DEBUG1("skip_encrypt_done: len=%d\n", m->m_len);
	/*
	 * compute checksum of encrypted packet
	 */
	len = m->m_len;
	sump = mtod(m, unsigned short *);
	hptr = (struct ip *) sump;
	hptr->ip_len = htons(len); /* total length */

	/* compute new checksum */
	sum = sump[0] + sump[1] + sump[2] + sump[3] + sump[4] +
		sump[6] + sump[7] + sump[8] + sump[9];
	sum = (sum & 0xFFFF) + (sum >> 16);
	sum = ~(sum + (sum >> 16)) & 0xFFFF;
	if (sum == 0xFFFF) {
		sum = 0;
	}

	hptr->ip_sum = (unsigned short)sum; 

	(*skip_if->if_output)(skip_if->ifp, m, dst);
}


/* skip_ifopen()
 *
 * Common open routine for communication with user space.
 *
 * Returns: 0 on success, errno otherwise
 */
/*ARGSUSED*/
int
skip_ifopen(dev, flags)
	dev_t		dev;	/* maj/min device number */
	int		flags;	/* flags */
{
	skip_softc_t	*sp;

	SKIP_DEBUG2("skip_ifopen(): DEVICE major=%d, flags = %d\n",
						major(dev), flags);

	if (!suser()) {
		return (EPERM);
	}

	if (minor(dev) >= SKIP_MAX_OPENS) {
		return (ENXIO);
	}

	sp = &skip_softc[minor(dev)];

	if (sp->sp_flags & SKIP_BUSY) {
		return (EBUSY);
	}
	sp->sp_flags |= SKIP_BUSY;

	if (flags & FNDELAY) {
		sp->sp_flags |= SKIP_NDELAY;
	}
	bzero((caddr_t) &sp->q, sizeof (struct ifqueue));
	sp->q.ifq_maxlen = SKIP_KEY_MAX_PKTS;

	skip_busy++;
	return (0);
}

/* skip_ifclose()
 *
 * Common close routine for communication with user space.
 *
 * Returns: 0
 */
int
skip_ifclose(dev, flags)
	dev_t		dev;	/* maj/min device number */
	int		flags;	/* flags */
{
	register skip_softc_t	*sp = &skip_softc[minor(dev)];
	register struct mbuf	*m;
	register skip_es_t	*skip_if;
	int			s;

	SKIP_DEBUG1("skip_ifclose(%d)\n", minor(dev));

	s = splimp();
	for (;;) {
		IF_DEQUEUE(&sp->q, m); 
		if (m) {
			m_freem(m);
		} else {
			break;
		}
	}
	splx(s);

	/*
	 * if this pseudo-device was used to notify events on an interface
	 * then remove it
	 */
	for (skip_if = skip_es_ifs; skip_if; skip_if = skip_if->next) {
		if (skip_if->notifier == minor(dev)) {
			skip_if->notifier = 0;
		}
	}

	bzero((caddr_t) sp, sizeof (skip_softc_t));

	skip_busy--;
	return (0);
}

/* skip_ifwakeup()
 *
 * Wakeup a sleeping user-process
 *
 * Returns: none.
 */
static void
skip_ifwakeup(sp)
	skip_softc_t	*sp;
{
	if (sp->sp_rsel) {
		/*
		 * deal with pending select()
		 */
		selwakeup(sp->sp_rsel, sp->sp_flags & SKIP_RCOLL);
		sp->sp_flags &= ~SKIP_RCOLL;
		sp->sp_rsel = NULL;
	}
	wakeup((caddr_t) sp);
}

/* skip_ifselect()
 *
 * pseudo-device select function.
 *
 * Returns: 0 if no data available, 1 otherwise
 */
int
skip_ifselect(dev, rw)
	dev_t		dev;
	int		rw;
{
	register skip_softc_t	*sp = &skip_softc[minor(dev)];
	register int		s;

	if (rw == FWRITE) {
		return (1);
	}
	s = splimp();
	if (sp->q.ifq_len > 0) {
		splx(s);
		return (1);
	}

	if (sp->sp_rsel && (sp->sp_rsel->p_wchan == (caddr_t) &selwait)) {
		sp->sp_flags |= SKIP_RCOLL;
	} else {
		 sp->sp_rsel= u.u_procp;
	}
		
	splx(s);
	return(0);
}

/* skip_ifread()
 *
 * pseudo-device read function.
 *
 * Returns: 0
 */
int
skip_ifread(dev, uio)
	dev_t		dev;
	struct uio	*uio;
{
	register skip_softc_t	*sp = &skip_softc[minor(dev)];
	register struct mbuf	*m;
	register int		s, st, rc, len;

	if (minor(dev) >= SKIP_MAX_OPENS) {
		return (ENXIO);
	}

	/*
	 * read request from the user space process
	 */
	s = splimp();
	for (;;) {
		IF_DEQUEUE(&sp->q, m); 

		if (m == NULL) {
			/*
			 * no messages at the moment
			 */
			if (sp->sp_flags & SKIP_NDELAY) {
				splx(s);
				return (EWOULDBLOCK);
			}
			
			rc = sleep((caddr_t) sp, SKIP_PRI | PCATCH);
			if (rc) {
				splx(s);
				return (rc);
			}
		} else {
			break;
		}
	}
	splx(s);

	len = m->m_len;

	rc = uiomove(mtod(m, caddr_t), len, UIO_READ, uio);

	m_freem(m);
	return (0);
}

/* skip_ifwrite()
 *
 * pseudo-device write function.
 *
 * Returns: 0
 */
int
skip_ifwrite(dev, uio)
	dev_t		dev;
	struct uio	*uio;
{
	register skip_softc_t	*sp = &skip_softc[minor(dev)];
	register struct mbuf	*m;
	register int		rc;

	if (minor(dev) != SKIP_KEYMGR_MINOR) {
		return (ENXIO);
	}

	/*
	 * process the message from user-space
	 */
	m = skip_bufalloc(uio->uio_resid);
	if (m == NULL) {
		skip_bufallocfail++;
		return (ENOBUFS);
	}

	m->m_len = uio->uio_resid;

	rc = uiomove(mtod(m, caddr_t), uio->uio_resid, UIO_WRITE, uio);
	if (rc) {
		m_freem(m);
		return (rc);
	}
	skip_key_fromkm(mtod(m, union skip_messages *), m->m_len);
	m_freem(m);
	return (0);
}

/* skip_free_buf()
 *
 * free loaned mbuf buffer
 *
 * Returns: none
 */
static int
skip_free_buf(arg)
	int		arg;
{
	caddr_t		buf = (caddr_t) arg;
	int		len;

	buf -= sizeof(int);
	len = * (int *) buf;

	KMEM_FREE(buf, len);
	return (0);
}

/* skip_bufalloc()
 *
 * Allocate a contiguous buffer of the specified size.
 *
 * The returned mbuf's length is marked as zero but the storage is available.
 *
 * Returns: mbuf on success, NULL otherwise.
 */
static struct mbuf *
skip_bufalloc(len)
	int		len;
{
	register struct mbuf	*m;
	caddr_t			buf;

	MGET(m, M_DONTWAIT, MT_DATA);
	if (m == NULL) {
		return (NULL);
	}
	if (len <= MLEN) {
		/*
		 * a small mbuf will suffice
		 */
		goto ok; 
	}
	if (len <= MCLBYTES) {
		if (mclget(m)) {
			/*
			 * fits into a cluster mbuf
			 */
			goto ok;
		}
		m_free(m);
		return (NULL);
	}
	/*
	 * need to create a loaned mbuf
	 */

	buf = KMEM_ALLOC(len + sizeof(int), KMEM_NOSLEEP);
	if (buf == NULL) {
		m_free(m);
		return (NULL);
	}

	/*
	 * stuff the length at the start to make it easy to free
	 */

	 * (int *) buf = len + sizeof(int);
	buf += sizeof(int);

	/*
	 * transform the mbuf into a loaned mbuf
	 */
	m->m_off = (int) buf - (int)m;
	m->m_cltype = MCL_LOANED;
	m->m_clfun = skip_free_buf;
	m->m_clarg = (int) buf;
	m->m_clswp = NULL;
ok:
	m->m_len = 0;
	return (m);
}

/* skip_bufextend()
 * 
 * test if space exists at the end of the buffer to extend it
 */
static int
skip_bufextend(m, pad)
struct mbuf	*m;
int		pad;
{
	if (M_HASCL(m)) {
		if ((m->m_cltype == MCL_STATIC)) {
			return ((MCLBYTES - m->m_len) >= pad);
		} else {
			return(0);
		}
	}
	return ((MLEN - m->m_len) >= pad);
}

/* skip_bufalign()
 *
 * concatenate and align buffers for the bulk data cryptor
 *
 * An optional extend can be specified
 *
 */
static struct mbuf *
skip_bufalign(m, pad)
	struct mbuf	*m;
	int		pad;
{
	register struct mbuf	*m0, *nextm;
	register int		len, n;

	/*
	 * test for already concatenated and aligned
	 */
	if ((m->m_next == NULL) && ALIGNED(mtod(m, caddr_t)) &&
						skip_bufextend(m, pad)) {
		return (m);
	}

	/*
	 * work out total size of mbuf
	 */
	for (len = 0, m0 = m; m0; m0 = m0->m_next) {
		len += m0->m_len;
	}

	/*
	 * allocate new contiguous and aligned mbuf
	 */
	m0 = skip_bufalloc(len + pad);
	if (m0 == NULL) {
		m_freem(m);
		return (NULL);
	}

	/*
	 * copy data to contiguous and aligned buffer
	 */
	m_cpytoc(m, 0, len, mtod(m0, caddr_t));
	m0->m_len = len;
	m_freem(m);
	return (m0);
}
 
/* skip_es_find_ifnet()
 *
 * Given an ifnet pointer, find the corresponding skip interface structure
 *
 * Returns: 0 on success, errno otherwise
 */
static skip_es_t *
skip_es_find_ifnet(ifp)
	struct ifnet		*ifp;
{
	register skip_es_t	*skip_if;

	for (skip_if = skip_es_ifs; skip_if; skip_if = skip_if->next) {
		if (skip_if->ifp == ifp) {
			break;
		}
	}
	return (skip_if);
}

/* skip_add_interface()
 *
 * Add a new skip interface
 *
 * Returns: 0 on success, errno otherwise
 */
static int
skip_add_interface(if_name)
	char 		*if_name;
{
	register skip_es_t	*skip_if;
	register struct ifnet	*ifp;
	register void		*sd;
	register int		s;

	/*
	 * find specified network interface in system
	 */
	if ((ifp = ifunit(if_name, strlen(if_name))) == NULL) {
		return (ENODEV);
	}

	if (ifp->if_mtu < SKIP_MIN_MTU) {
		log("skip: %s: interface mtu of %d is too small\n", 
						if_name, ifp->if_mtu);
		return (EINVAL);
	}

	skip_if = skip_es_find_if(if_name);

	if (skip_if) {
		/*
		 * already exists... 
		 */
		return (EEXIST);
	}

	skip_if = KMEM_ZALLOC(sizeof(skip_es_t), KMEM_NOSLEEP);

	if (skip_if == NULL) {
		/*
		 * no more space... 
		 */
		return (ENOSPC);
	}

	SKIP_DEBUG1("adding skip to %s\n", if_name);

	/*
	 * declare this interface as a SKIP client
	 */
	sd = skip_open(skip_if, skip_encrypt_done, skip_decrypt_done);

	if (sd == NULL) {
		KMEM_FREE(skip_if, sizeof (skip_es_t));
		return (ENOSPC);
	}

	/*
	 * create the hash table for the access control information
	 */
	skip_if->hashtable = KMEM_ZALLOC(
					sizeof (skip_es_hash_t *) *
					SKIP_ES_HASH_TABLESZ,
					KMEM_NOSLEEP);
	if (skip_if->hashtable == NULL) {
		KMEM_FREE(skip_if, sizeof (skip_es_t));
		skip_close(sd);
		return (ENOSPC);
	}

	skip_if->sd		= sd;
	skip_if->ifp		= ifp;
	skip_if->if_mode	= SkipAccessControlOn;

	strncpy(skip_if->if_name, if_name, sizeof (skip_if->if_name));

	/*
	 * add to list
	 */
	s = splimp();
	skip_busy++;
	skip_if->next = skip_es_ifs;
	skip_es_ifs = skip_if;

	/*
	 * and redirect network interface traffic to SKIP
	 */
	skip_if->if_output	= ifp->if_output;
	ifp->if_output		= skip_ifoutput;

	/*
	 * drop interface mtu to allow for SKIP header
	 */
	ifp->if_mtu 		-= SKIP_HDR_SZ;

	splx(s);

	return (0);
}

/* skip_del_interface()
 *
 * Remove a skip interface
 *
 * Returns: 0 on success, errno otherwise
 */
static int
skip_del_interface(if_name)
	char 		*if_name;
{
	register skip_es_t		*skip_if, **prevskip_if;
	register skip_es_hash_t		*hp, *nhp;
	register int			h;
	register int			s;
	
	s = splimp();

	/*
	 * remove the interface from the list
	 */
	for (prevskip_if = &skip_es_ifs; (skip_if = *prevskip_if);
						prevskip_if = &skip_if->next) {
		if (strcmp(skip_if->if_name, if_name) == 0) {
			break;
		}
	}

	if (skip_if != NULL) {
		*prevskip_if = skip_if->next;
	}

	if (skip_if == NULL) {
		splx(s);
		return (ENXIO);
	}

	SKIP_DEBUG1("removing skip from %s\n", if_name);

	skip_close(skip_if->sd);

	/*
	 * restore normal network interface output
	 */
	skip_if->ifp->if_output = skip_if->if_output;

	/*
	 * restore original mtu
	 */
	skip_if->ifp->if_mtu 	+= SKIP_HDR_SZ;
	skip_busy--;

	splx(s);

	if (skip_if->hashtable) {
		for (h = 0; h < SKIP_ES_HASH_TABLESZ; h++) {
			for (hp = skip_if->hashtable[h]; hp; hp = nhp) {
				nhp = hp->next;
				KMEM_FREE(hp, sizeof (*hp));
			}
		}
		KMEM_FREE(skip_if->hashtable, sizeof (skip_es_hash_t *) *
						SKIP_ES_HASH_TABLESZ);
	}

	KMEM_FREE(skip_if, sizeof(skip_es_t));
	return (0);
}


/* skip_notify()
 *
 * Tell someone about network events
 *
 * Returns: none
 */
static void
skip_notify(skip_if, ip, params, what)
	skip_es_t	*skip_if;
	struct ip	*ip;
	skip_param_t	*params;
	int		what;
{
	register skip_softc_t		*sp;
	register int			s;
	register struct mbuf		*m;
	register skip_es_notification_t	*np;

	if (skip_if->if_mode != SkipInteractive) {
		return;
	}

	if (skip_if->notifier == 0) {
		return;
	}

	sp = &skip_softc[skip_if->notifier];

	s = splimp();
	if (IF_QFULL(&sp->q)) {
		SKIP_DEBUG("skip_notify: too many requests\n");
		IF_DROP(&sp->q);
		splx(s);
		return;
	}
	splx(s);

	/*
	 * create an mbuf for the message
	 */
	m = skip_bufalloc(sizeof (union skip_messages) + MAXVARSZ);
	if (m == NULL) {
		SKIP_DEBUG("skip_notify: mbuf alloc failed\n");
		skip_bufallocfail++;
		return;
	}

	np = mtod(m, skip_es_notification_t *);

	m->m_len = sizeof (skip_es_notification_t);

	bzero((caddr_t) np, sizeof(*np));

	bcopy(skip_if->if_name, np->if_name, sizeof(skip_if->if_name));
	np->what = what;
	bcopy((caddr_t) ip, (caddr_t) &np->iphdr, sizeof(*ip));

	if (params) {
		bcopy((caddr_t) params, (caddr_t) &np->params,
							sizeof (*params));
		/*
		 * the kernel internally initialises r_mkeyid even for NSID 0
		 * This is confusing for user-space tools so zap it.
		 */
		if (np->params.r_nsid == 0) {
			np->params.r_mkeyid.len = 0;
		}
	}

	s = splimp();
	IF_ENQUEUE(&sp->q, m);
	splx(s);
	skip_ifwakeup(sp);
}



/* skip_ifioctl()
 *
 * handle ioctls from user space
 */
int
skip_ifioctl(dev, cmd, data, flag)
	dev_t		dev;
	int		cmd;
	caddr_t		data;
	int		flag;
{
	register skip_es_req_t		*reqp;
	register skip_ctrl_stats_t	*statsp;
	register ioctl_alg_list_t	*algp;
	register ioctl_crypt_stat_t	*cstatp;
	register ioctl_skip_sym_t	*symbp;
	register ioctl_key_status_t	*kstatp;
	register caddr_t		iobuf;
	register int			rc = EINVAL;
	int				len;
	skip_io_t			*iocp = (skip_io_t *) data;

	
	if (iocp->ic_len < SKIPSZ || iocp->ic_len > SKIP_MAX_IOSZ) {
		SKIP_DEBUG("skip_ifioctl: invalid IOCTL size\n");
		return (EINVAL);
	}

	iobuf = KMEM_ALLOC(iocp->ic_len, KMEM_NOSLEEP);
	if (iobuf == NULL) {
		return (ENOMEM);
	}
	len = iocp->ic_len;

	rc = copyin(iocp->ic_dp, (caddr_t) iobuf, iocp->ic_len);
	if (rc) {
		KMEM_FREE(iobuf, iocp->ic_len);
		return (rc);
	}

	reqp = (skip_es_req_t *) iobuf;
	statsp = (skip_ctrl_stats_t *) iobuf;


	switch (iocp->ic_cmd) {

	case SKIP_ES_ADD_IF:

		if (iocp->ic_len != sizeof(reqp->if_name)) {
			 goto out;
		}
		rc = skip_add_interface(reqp->if_name);
		break;

	case SKIP_ES_DEL_IF:

		if (iocp->ic_len != sizeof(reqp->if_name)) {
			 goto out;
		}
		rc = skip_del_interface(reqp->if_name);
		break;

	case SKIP_ES_ADD_HOST:

		rc = skip_es_add_host(reqp, iocp->ic_len);
		break;

	case SKIP_ES_DEL_HOST:

		rc = skip_es_del_host(reqp, iocp->ic_len);
		break;

	case SKIP_ES_GET_HOST:

		rc = skip_es_get_host(reqp, iocp->ic_len);

		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);		}
		break;

	case SKIP_ES_SET_MODE:

		if (iocp->ic_len != sizeof(skip_es_mode_t)) {
			goto out;
		}
		rc = skip_es_set_mode(minor(dev),reqp);
		break;

	case SKIP_ES_GET_MODE:

		if (iocp->ic_len != sizeof(skip_es_mode_t)) {
			goto out;
		}
		rc = skip_es_get_mode(reqp);
		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp,
					sizeof(skip_es_mode_t));
		}
		break;

	case SKIP_ES_LIST_HOSTS:

		rc = skip_es_list_hosts(reqp, &iocp->ic_len);
		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		}
		break;

	case SKIP_ES_GET_IF_STATS:

		if (iocp->ic_len != sizeof(skip_ctrl_stats_t)) {
			rc = EINVAL;
			goto out;
		}

		rc = skip_get_if_stats(statsp);
		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		}
		break;

	case SKIP_ES_GET_KEY_STATS:

		if (iocp->ic_len != sizeof(skip_ctrl_stats_t)) {
			rc = EINVAL;
			goto out;
		}

		rc = skip_get_key_stats(statsp);
		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		}
		break;

	case SKIP_ES_GET_HDR_STATS:

		if (iocp->ic_len != sizeof(skip_ctrl_stats_t)) {
			rc = EINVAL;
			goto out;
		}

		rc = skip_get_hdr_stats(statsp);
		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		}
		break;

	case SKIP_ES_GET_CRYPTORS:

		if (iocp->ic_len < sizeof(ioctl_alg_list_t)) {
			rc = EINVAL;
			goto out;
		}

		algp = (ioctl_alg_list_t *) iobuf;

		skip_get_kp_alg_list(algp);
		rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		break;

	case SKIP_ES_GET_KIJ_ALGS:

		if (iocp->ic_len < sizeof(ioctl_alg_list_t)) {
			rc = EINVAL;
			goto out;
		}

		algp = (ioctl_alg_list_t *) iobuf;

		skip_get_kij_alg_list(algp);
		rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		break;

	case SKIP_ES_GET_CPT_STATS:

		if (iocp->ic_len < sizeof(ioctl_crypt_stat_t)) {
			rc = EINVAL;
			goto out;
		}

		cstatp = (ioctl_crypt_stat_t *) iobuf;

		skip_cryptor_stats(cstatp);
		rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		break;

	case SKIP_ES_GET_VAR:

		if (iocp->ic_len < sizeof(ioctl_skip_sym_t)) {
			rc = EINVAL;
			goto out;
		}

		symbp = (ioctl_skip_sym_t *) iobuf;
		rc = skip_get_var(symbp);
		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		}
		break;

	case SKIP_ES_SET_VAR:

		if (iocp->ic_len < sizeof(ioctl_skip_sym_t)) {
			rc = EINVAL;
			goto out;
		}

		symbp = (ioctl_skip_sym_t *) iobuf;
		rc = skip_set_var(symbp);
		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		}
		break;

	case SKIP_ES_KEY_STATUS:

		if (iocp->ic_len < sizeof(ioctl_key_status_t)) {
			rc = EINVAL;
			goto out;
		}

		kstatp = (ioctl_key_status_t *) iobuf;
		rc = skip_get_key_status(kstatp);
		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		}
		break;
	default:
		break;
	}

out:
	KMEM_FREE(iobuf, len);
	return (rc);
}

/* skip_key_tellkm()
 *
 * send a message to the key manager daemon
 *
 * Returns: none
 */
void
skip_key_tellkm(msg, cp)
	enum skip_msgs	msg;
	skip_keycb_t	*cp;
{
	register skip_softc_t		*sp = &skip_softc[SKIP_KEYMGR_MINOR];
	register struct mbuf		*m;
	union skip_messages		*sm;
	register int			s;

	if ((sp->sp_flags & SKIP_BUSY) == 0) {
		SKIP_DEBUG("skip_key_tellkm: no key manager\n");
		return;
	}

	s = splimp();
	if (IF_QFULL(&sp->q)) {
		SKIP_DEBUG("skip_key_tellkm: too many requests\n");
		IF_DROP(&sp->q);
		splx(s);
		return;
	}
	splx(s);

	/*
	 * create an mbuf for the message
	 */
	m = skip_bufalloc(sizeof (union skip_messages) + MAXVARSZ);
	if (m == NULL) {
		SKIP_DEBUG("skip_key_tellkm: mbuf alloc failed\n");
		skip_bufallocfail++;
		return;
	}

	sm = mtod(m, union skip_messages *);

	sm->msgtype   = msg;

	/*
	 * Now, format the key manager message
	 */
	m->m_len = skip_fmt_kmgr(sm, cp);

	s = splimp();
	IF_ENQUEUE(&sp->q, m);
	splx(s);
	skip_keys_stats.skip_keymgr_requests++;
	skip_ifwakeup(sp);
}


/* skip_inittimers()
 *
 * start timing operations
 *
 * Returns: None
 */
static void
skip_inittimers()
{
	timeout(skip_timer, NULL, skip_key_tick * hz);
}
 
/* skip_uninittimers()
 *
 * stop timing operations
 *
 * Returns: None
 */
static void
skip_uninittimers()
{
	untimeout(skip_timer, NULL);
}
 
/* skip_timer()
 *
 * scan the current key list, trying to find those which have
 * aged or exceeded their transmission quota
 *
 * Returns: none
 */
/*ARGSUSED*/
static void
skip_timer(arg)
	caddr_t		arg;
{
	/*
	 * run through the key store
	 */
	skip_key_iterate(skip_key_check, NULL);
	timeout(skip_timer, NULL, skip_key_tick * hz);
}

/*
 * mbuf debug
 */
static int
pr_mbuf(struct mbuf *m, skip_es_t *skip_if)
{
	for (; m; m = m->m_next) {
		if (SKIP_DECRYPTED(m, skip_if)) {
			printf("from skip: ");
		}
		printf("m = 0x%x m_len = %d m_off = 0x%x",
			m, m->m_len, m->m_off);
		if (m->m_next) {
			printf(" -> \n");
		} else {
			printf("\n\n");
		}
	}
	return (0);
}
