#include <sys/param.h>
#include <sys/systm.h>
#include <sys/errno.h>
#include <sys/vnode.h>
#include <sys/vfs.h>
#include <sys/uio.h>
#include <sys/cred.h>
#include <sys/pathname.h>
#include <sys/dirent.h>
#include <sys/debug.h>
#include <sys/sysmacros.h>
#include <sys/tiuser.h>
#include <sys/cmn_err.h>
#include <sys/stat.h>
#include <sys/mode.h>
#include <rpc/types.h>
#include <rpc/auth.h>
#include <rpc/clnt.h>
#include <sys/fs_subr.h>
#include <sys/mman.h>
#include <sys/vm.h>
#include <vm/as.h>
#include <vm/pvn.h>
#include <vm/seg_vn.h>
#include <vm/seg_kp.h>
#include <vm/seg_map.h>
#include <vm/page.h>

#include <fist.h>
#include <cryptfs.h>

/*
 *  Vnode ops for fist_cryptfs
 */
static int fist_crypt_open(vnode_t **, int, cred_t *);
static int fist_crypt_close(vnode_t *, int, int, offset_t, cred_t *);
static int fist_crypt_getattr(vnode_t *, vattr_t *, int, cred_t *);
static int fist_crypt_access(vnode_t *, int, int, cred_t *);
static int fist_crypt_lookup(vnode_t *, char *, vnode_t **,
			     pathname_t *, int, vnode_t *, cred_t *);
static int fist_crypt_remove(vnode_t *, char *, cred_t *);
static int fist_crypt_rename(vnode_t *, char *, vnode_t *, char *, cred_t *);
static int fist_crypt_mkdir(vnode_t *, char *, vattr_t *, vnode_t **, cred_t *);
static int fist_crypt_rmdir(vnode_t *, char *, vnode_t *, cred_t *);
static int fist_crypt_readdir(vnode_t *, uio_t *, cred_t *, int *);
static int fist_crypt_symlink(vnode_t *, char *, vattr_t *, char *, cred_t *);
static int fist_crypt_fsync(vnode_t *, int, cred_t *);
static void fist_crypt_inactive(vnode_t *, cred_t *);
static void fist_crypt_rwlock(vnode_t *, int);
static void fist_crypt_rwunlock(vnode_t * vp, int);
static int fist_crypt_seek(vnode_t * vp, offset_t, offset_t *);
static int fist_crypt_cmp(vnode_t *, vnode_t *);
static int fist_crypt_read(vnode_t *, uio_t *, int, cred_t *);
static int fist_crypt_write(vnode_t *, uio_t *, int, cred_t *);
static int fist_crypt_ioctl(vnode_t *, int, int, int, cred_t *, int *);
static int fist_crypt_setfl(vnode_t *, int, int, cred_t *);
static int fist_crypt_setattr(vnode_t *, vattr_t *, int, cred_t *);
static int fist_crypt_create(vnode_t *, char *, vattr_t *, vcexcl_t, int,
			     vnode_t **, cred_t *, int);
static int fist_crypt_link(vnode_t *, vnode_t *, char *, cred_t *);
static int fist_crypt_readlink(vnode_t *, uio_t *, cred_t *);
static int fist_crypt_fid(vnode_t *, fid_t *);
static int fist_crypt_frlock(vnode_t *, int, struct flock64 *, int, offset_t, cred_t *);
static int fist_crypt_space(vnode_t *, int, struct flock64 *, int, offset_t, cred_t *);
static int fist_crypt_realvp(vnode_t *, vnode_t **);

extern int fist_crypt_getpage(vnode_t *, offset_t, u_int, u_int *, page_t **,
		       u_int, struct seg *, caddr_t, enum seg_rw, cred_t *);
extern int fist_crypt_getapage(vnode_t *, u_int, u_int, u_int *, page_t **,
		       u_int, struct seg *, caddr_t, enum seg_rw, cred_t *);
extern int fist_crypt_putpage(vnode_t *, offset_t, u_int, int, cred_t *);
extern int fist_crypt_putapage(struct vnode *vp, page_t * pp, u_offset_t * offp, u_int * lenp, int flags, struct cred *cr);
extern int fist_crypt_map(vnode_t *, offset_t, struct as *, caddr_t *, u_int,
			  u_char, u_char, u_int, cred_t *);
extern int fist_crypt_addmap(vnode_t *, offset_t, struct as *, caddr_t, u_int,
			     u_char, u_char, u_int, cred_t *);
extern int fist_crypt_delmap(vnode_t *, offset_t, struct as *, caddr_t, u_int,
			     u_int, u_int, u_int, cred_t *);

static int fist_crypt_poll(vnode_t *, short, int, short *, pollhead_t **);
static int fist_crypt_dump(vnode_t *, caddr_t, int, int);
static int fist_crypt_pathconf(vnode_t *, int, u_long *, cred_t *);
static int fist_crypt_pageio(vnode_t *, page_t *, u_offset_t, u_int, int, cred_t *);
static int fist_crypt_dumpctl(vnode_t *, int);
static void fist_crypt_dispose(vnode_t *, page_t *, int, int, cred_t *);
static int fist_crypt_setsecattr(vnode_t *, vsecattr_t *, int, cred_t *);
static int fist_crypt_getsecattr(vnode_t *, vsecattr_t *, int, cred_t *);

vnodeops_t fist_crypt_vnodeops =
{
  fist_crypt_open,		/* open */
  fist_crypt_close,		/* close */
  fist_crypt_read,		/* read */
  fist_crypt_write,		/* write */
  fist_crypt_ioctl,		/* ioctl */
  fist_crypt_setfl,		/* setfl */
  fist_crypt_getattr,		/* getattr */
  fist_crypt_setattr,		/* setattr */
  fist_crypt_access,		/* access */
  fist_crypt_lookup,		/* lookup */
  fist_crypt_create,		/* create */
  fist_crypt_remove,		/* remove */
  fist_crypt_link,		/* link */
  fist_crypt_rename,		/* rename */
  fist_crypt_mkdir,		/* mkdir */
  fist_crypt_rmdir,		/* rmdir */
  fist_crypt_readdir,		/* readdir */
  fist_crypt_symlink,		/* symlink */
  fist_crypt_readlink,		/* readlink */
  fist_crypt_fsync,		/* fsync */
  fist_crypt_inactive,		/* inactive */
  fist_crypt_fid,		/* fid */
  fist_crypt_rwlock,		/* rwlock */
  fist_crypt_rwunlock,		/* rwunlock */
  fist_crypt_seek,		/* seek */
  fist_crypt_cmp,		/* cmp */
  fist_crypt_frlock,		/* frlock */
  fist_crypt_space,		/* space */
  fist_crypt_realvp,		/* realvp */
  fist_crypt_getpage,		/* getpage */
  fist_crypt_putpage,		/* putpage */
  fist_crypt_map,		/* map */
  fist_crypt_addmap,		/* addmap */
  fist_crypt_delmap,		/* delmap */
  fist_crypt_poll,		/* poll */
  fist_crypt_dump,		/* dump */
  fist_crypt_pathconf,		/* pathconf */
  fist_crypt_pageio,		/* pageio */
  fist_crypt_dumpctl,		/* dumpctl */
  fist_crypt_dispose,		/* dispose */
  fist_crypt_setsecattr,	/* setsecattr */
  fist_crypt_getsecattr		/* getsecattr */
};

#define	NEED_MOUNT(vp)	(vntoan(vp)->an_mntflags & MF_MNTPNT)
#define	SPECIAL_PATH(pnp) ((pnp)->pn_path[pn_pathleft(pnp) - 1] == ' ')
#define	MOUNTED_ON(vp) (vntoan(vp)->an_mntflags & MF_MOUNTED)


/* interpose on an old vnode and return new one */
vnode_t *
fist_crypt_interpose(vnode_t * hidden_vp, vfs_t * this_vfsp)
{
  vnode_t *fw_vp;
  fist_cryptnode_t *fwnp;
  struct fist_cryptinfo *fwip = NULL;

  mutex_enter(&vfstofwi(this_vfsp)->fwi_ht_lock);

  /* look if entry exists in HT */
  fw_vp = fist_ht_find_vp(hidden_vp, this_vfsp);
  if (fw_vp) {
    fist_dprint(6, "CRYPT_INTERPOSE found vp in HT!\n");
    VN_HOLD(fw_vp);		/* maybe VN_RELE(hidden_vp) */
    VN_RELE(hidden_vp);
    goto out;
  }
  /* allocate new vnode */
  fw_vp = kmem_zalloc(sizeof(vnode_t), KM_SLEEP);
  /* XXX: need VN_INIT2 that will reuse lock var of interposed vnode */
  VN_INIT(fw_vp, this_vfsp, hidden_vp->v_type, (dev_t) NULL);

  /* allocate and fill in fist_cryptnode_t */
  fwnp = (fist_cryptnode_t *)
    kmem_zalloc(sizeof(fist_cryptnode_t), KM_SLEEP);
  if (!fwnp) {
    kmem_free(fw_vp, sizeof(vnode_t));
    fw_vp = NULL;
    goto out;
  }

  /* init fwnp */
  fwnp->fwn_mapcnt = 0;		/* no mapped pages */
  mutex_init(&fwnp->fwn_lock, "fist_cryptfs private data lock",
	     MUTEX_DEFAULT, NULL);
  /* store new vnode */
  fwnp->fwn_vnodep = hidden_vp;
  fw_vp->v_data = (caddr_t) fwnp;

  /* operations are fist_crypt vnode ops */
  fw_vp->v_op = &fist_crypt_vnodeops;

  /* set rest of fields to NULL */
  fw_vp->v_vfsmountedhere = NULL;
  fw_vp->v_filocks = NULL;

  /* don't do this one for now */
  /* fw_vp->v_cv = NULL; */

  /* increment interposed vnodes counter */
  fwip = vfstofwi(this_vfsp);
  fwip->fwi_num_vnodes++;

  /* insert into HT */
  fist_ht_insert_vp(hidden_vp, fw_vp);
  /* VN_RELE(hidden_vp); */

#if 0
  /* hold our vnode */
  VN_HOLD(fw_vp);
#endif

out:
  mutex_exit(&vfstofwi(this_vfsp)->fwi_ht_lock);

  print_location();

  /* finally  return vnode to this newly created one */
  return (fw_vp);
}


static int
fist_crypt_open(
		 vnode_t ** vpp,
		 int flag,
		 cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp, *new_vp;

  fist_dprint(4, "fist_crypt_open vpp %x, flag 0x%x\n", vpp, flag);
  if (flag & FAPPEND) {
    fist_dprint(6, "***WARNING*** file is opened in append-only mode!!!\n");
    flag &= ~FAPPEND;		/* turn off FAPPEND flag */
  }

  new_vp = hidden_vp = vntofwn(*vpp)->fwn_vnodep;
  /* we NEED read access */
  flag |= FREAD;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_OPEN(&hidden_vp, flag, cr);

  /* check if a new vnode was returned (cloned) */
  if (!error && new_vp != hidden_vp) {
    fist_dprint(6, "CRYPT_OPEN1: hidden_vp->v_count %d\n",
		hidden_vp->v_count);
    /*
     * yes: need to allocate a new fist_crypt vnode,
     * initialize it, store interposed vnode in it, and
     * finally return the fist_crypt vnode back.
     */
    *vpp = fist_crypt_interpose(hidden_vp, (*vpp)->v_vfsp);
    fist_dprint(6, "CRYPT_OPEN2: vpp->v_count %d, hidden_vp->v_count %d\n",
		(*vpp)->v_count, hidden_vp->v_count);
  }
  print_location();
  return (error);
}


static int
fist_crypt_close(
		  vnode_t * vp,
		  int flag,
		  int count,
		  offset_t offset,
		  cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_close vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  fist_dprint(6, "CRYPT_CLOSE1: vp->v_count %d, hidden_vp->v_count %d\n",
	      vp->v_count, hidden_vp->v_count);

  /* pass operation to hidden filesystem, and return status */
  error = VOP_CLOSE(hidden_vp, flag, count, offset, cr);

  fist_dprint(6, "CRYPT_CLOSE2: vp->v_count %d, hidden_vp->v_count %d\n",
	      vp->v_count, hidden_vp->v_count);
  print_location();
  return (error);
}

static int
fist_crypt_read(
		 vnode_t * vp,
		 uio_t * uiop,
		 int ioflag,
		 cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;
  uio_t temp_uio;
  iovec_t *temp_iovec;
  caddr_t current_base;
  int i, bytes_read;
  int num_pages, resid;
  long long start_loffset, end_loffset;
  long long cleartext_start_loffset, cleartext_end_loffset, current_loffset;

  fist_dprint(4, "fist_crypt_read vp %x\n", vp);

#ifdef FIST_DEBUG
  fist_print_uios("fist_crypt_read", uiop);
#endif /* FIST_DEBUG */

  if (fist_get_userpass(vp->v_vfsp, cr) == NULL)
    return EACCES;

  cleartext_start_loffset = uiop->uio_loffset;
  cleartext_end_loffset = uiop->uio_loffset + uiop->uio_resid;
  start_loffset = cleartext_start_loffset & ~(PAGESIZE - 1);
  end_loffset = cleartext_end_loffset & ~(PAGESIZE - 1);
  /* if not multiple of PAGESIZE, then the above formula loses one page.
   * adjust for it */
  if (cleartext_end_loffset > end_loffset)
    end_loffset += PAGESIZE;
  resid = end_loffset - start_loffset;
  num_pages = resid >> PAGESHIFT;

  fist_dprint(6,
	      "READ: so=%d eo=%d cs=%d es=%d res=%d np=%d ps=%d\n",
	      (int) start_loffset,
	      (int) end_loffset,
	      (int) cleartext_start_loffset,
	      (int) cleartext_end_loffset,
	      resid,
	      num_pages,
	      PAGESIZE);

  temp_iovec = kmem_zalloc(num_pages * sizeof(iovec_t), KM_SLEEP);
  for (i = 0; i < num_pages; i++) {
    temp_iovec[i].iov_len = PAGESIZE;
    temp_iovec[i].iov_base = kmem_zalloc(PAGESIZE, KM_SLEEP);
    fist_dprint(6, "READ allocated %d address 0x%x\n",
		i, temp_iovec[i].iov_base);
  }

  temp_uio.uio_iov = temp_iovec;
  temp_uio.uio_iovcnt = num_pages;
  temp_uio.uio_loffset = start_loffset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_fmode = uiop->uio_fmode;
  temp_uio.uio_llimit = uiop->uio_llimit;
  temp_uio.uio_resid = resid;

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /*
   * pass operation to hidden filesystem, and return status
   */

  error = VOP_READ(hidden_vp, &temp_uio, ioflag, cr);

  if (error) {
    fist_dprint(4, "VOP_READ in read returned error - not good\n");
    /* XXX to be checked */
    goto out_free;
  }

  current_loffset = start_loffset;
  for (i = 0; i < num_pages; i++) {
    bytes_read = PAGESIZE - temp_iovec[i].iov_len;
    if (bytes_read == 0)
      break;

    temp_iovec[i].iov_base -= bytes_read;
    current_base = temp_iovec[i].iov_base;

    /* decrypt the page/block */

    crypt_decode_block(__FUNCTION__,__LINE__, current_base, bytes_read, vp, cr);

    /*
     * save the original size, for kmem_free.
     * no need for it w/ cryptfs; size is always PAGESIZE, hence this line
     * is commented out:
     *		temp_iovec[i].iov_len = uiop->uio_iov[i].iov_len;
     */
    /* treat first and last iovec separately, not all data in them is needed */
    if (current_loffset + bytes_read > cleartext_end_loffset) {
      bytes_read = cleartext_end_loffset - current_loffset;
    }
    if (i == 0) {
      bytes_read -= cleartext_start_loffset - start_loffset;
      current_loffset += cleartext_start_loffset - start_loffset;
      current_base += cleartext_start_loffset - start_loffset;
    }
    if ((error = uiomove(current_base, bytes_read, UIO_READ, uiop)))
      /*
       * XXX: we have to see the exact semantics of returning with an
       * EFAULT from read
       */
      break;
    current_loffset += bytes_read;
  }

out_free:
  for (i = 0; i < num_pages; i++) {
    fist_dprint(6, "READ freeing %d address 0x%x\n",
		i, temp_iovec[i].iov_base);
    kmem_free(temp_iovec[i].iov_base, PAGESIZE);
  }
  kmem_free(temp_iovec, num_pages * sizeof(iovec_t));

#ifdef FIST_DEBUG
  fist_print_uios("fist_crypt_read (END)", uiop);
#endif /* FIST_DEBUG */

  print_location();
  return (error);
}

static int
fist_crypt_write(
		  vnode_t * vp,
		  uio_t * uiop,
		  int ioflag,
		  cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;
  vattr_t va;
  uio_t temp_uio;
  iovec_t *temp_iovec;
  iovec_t *free_iovec;		/* for freeing allocated memory */
  int i;
  caddr_t current_base;
  int resid, bytes_read, num_pages, first_page_bytes, real_first_page;
  long long start_loffset, end_loffset, real_start_loffset;
  long long cleartext_start_loffset, cleartext_end_loffset, current_loffset;
  int hidden_ioflag = (ioflag & ~FAPPEND);

  fist_dprint(4, "fist_crypt_write vp %x, ioflag 0x5x\n", vp, ioflag);

  if (fist_get_userpass(vp->v_vfsp, cr) == NULL)
    return EACCES;

#ifdef FIST_DEBUG
  fist_print_uios("fist_crypt_write (START)", uiop);
#endif /* FIST_DEBUG */

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* we don't want anybody to do updates while we write, so lock the vnode */
  mutex_enter(&vp->v_lock);

  /* get the attributes, length is necessary for correct updates */
  va.va_mask = AT_SIZE;
  if ((error = VOP_GETATTR(hidden_vp, &va, 0, cr))) {
    fist_dprint(4, "VOP_GETATTR returned error - not good\n");
    /* XXX to be checked */
    goto out;
  }

  /* just in case someone tries to pull a fast one */
  if (uiop->uio_resid == 0) {
    error = 0;
    goto out;
  }

  cleartext_start_loffset = uiop->uio_loffset;
  cleartext_end_loffset = uiop->uio_loffset + uiop->uio_resid;

  if (ioflag & FAPPEND) {
    fist_dprint(6, "WRITE: turning off append flag\n");
    cleartext_start_loffset += va.va_size;
    cleartext_end_loffset += va.va_size;
  }

  start_loffset = MIN(cleartext_start_loffset, va.va_size) & ~(PAGESIZE - 1);
  real_start_loffset = cleartext_start_loffset & ~(PAGESIZE - 1);
  first_page_bytes = MIN(cleartext_start_loffset, va.va_size) - start_loffset;
  /* must use this to avoid shifting a quad w/ gcc */
  real_first_page = (int)(real_start_loffset - start_loffset) >> PAGESHIFT;
  end_loffset = cleartext_end_loffset & ~(PAGESIZE - 1);
  ASSERT(first_page_bytes <= PAGESIZE);
  /*
   * if not multiple of PAGESIZE, then the above formula loses one page.
   * adjust for it
   */
  if (cleartext_end_loffset > end_loffset)
    end_loffset += PAGESIZE;
  resid = end_loffset - start_loffset;
  num_pages = resid >> PAGESHIFT;

  if (num_pages == 1)
    first_page_bytes = PAGESIZE;

  temp_iovec = kmem_zalloc(num_pages * sizeof(iovec_t), KM_SLEEP);
  free_iovec = kmem_zalloc(num_pages * sizeof(iovec_t), KM_SLEEP);
  for (i = 0; i < num_pages; i++) {
    temp_iovec[i].iov_len = free_iovec[i].iov_len = PAGESIZE;
    /* we need the pages to be zeroed out */
    temp_iovec[i].iov_base = free_iovec[i].iov_base = kmem_zalloc(PAGESIZE, KM_SLEEP);
  }

  fist_dprint(6,
	      "WRITE: so=%d eo=%d cso=%d ceo=%d rso=%d res=%d np=%d rfp=%d\n",
	      (int) start_loffset,
	      (int) end_loffset,
	      (int) cleartext_start_loffset,
	      (int) cleartext_end_loffset,
	      (int) real_start_loffset,
	      resid,
	      num_pages,
	      real_first_page
	      );

  current_loffset = start_loffset;

  /* read first block XXX check length of file */
  temp_uio.uio_iov = temp_iovec;
  temp_uio.uio_iovcnt = 1;
  temp_uio.uio_loffset = start_loffset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_fmode = FREAD;
  temp_uio.uio_llimit = uiop->uio_llimit;
  temp_uio.uio_resid = first_page_bytes;
  fist_print_uios("WRITE (before VOP_READ 1)", &temp_uio);
  error = VOP_READ(hidden_vp, &temp_uio, hidden_ioflag, cr);
  if (error) {
    fist_dprint(5, "VOP_READ returned error - not good\n");
    /* XXX to be checked */
    goto out_free;
  }
  fist_print_uios("WRITE (after VOP_READ 1)", &temp_uio);
  bytes_read = PAGESIZE - temp_iovec[0].iov_len;
  temp_iovec[0].iov_base -= bytes_read;
  temp_iovec[0].iov_len = PAGESIZE;
  /* decrypt block read */
  crypt_decode_block(__FUNCTION__,__LINE__,
		     temp_iovec[0].iov_base, bytes_read, vp, cr);

  /*
   * if num_pages == 1, we already read the page... don't clobber it
   * if num_pages > 1, then we must read the last page, and decrypt it
   * completely, before clobbering it.
   * XXX: if end offset is on page boundary, we don't have to do this.
   */
  if (num_pages > 1) {
    /* read last block XXX check length of file */
    temp_uio.uio_iov = temp_iovec + (num_pages - 1);
    temp_uio.uio_iovcnt = 1;
    temp_uio.uio_loffset = end_loffset - PAGESIZE;
    temp_uio.uio_segflg = UIO_SYSSPACE;
    temp_uio.uio_fmode = FREAD;
    temp_uio.uio_llimit = uiop->uio_llimit;
    temp_uio.uio_resid = PAGESIZE;

    fist_print_uios("WRITE (before VOP_READ 2)", &temp_uio);
    error = VOP_READ(hidden_vp, &temp_uio, hidden_ioflag, cr);
    fist_print_uios("WRITE (after VOP_READ 3)", &temp_uio);
    if (error) {
      fist_dprint(4, "VOP_READ returned error - not good\n");
      /* XXX to be checked */
      goto out_free;
    }
    bytes_read = PAGESIZE - temp_iovec[num_pages - 1].iov_len;
    temp_iovec[num_pages - 1].iov_base -= bytes_read;
    temp_iovec[num_pages - 1].iov_len = PAGESIZE;
    /* decrypt block read */
    crypt_decode_block(__FUNCTION__,__LINE__,
		       temp_iovec[num_pages-1].iov_base, bytes_read, vp, cr);
  }
  /*
   * Now we are ready to write the bytes within the start/end
   * cleartext offsets in the buffers we allocated.
   */
  for (i = 0; i < num_pages; i++) {
    if (i >= real_first_page) {
      bytes_read = PAGESIZE;
      current_base = temp_iovec[i].iov_base;
      if (i == real_first_page) {
#define real_first_page_offset (cleartext_start_loffset - real_start_loffset)
	bytes_read -= real_first_page_offset;
	current_loffset += real_first_page_offset;
	current_base += real_first_page_offset;
#undef real_first_page_offset
      }
      if (current_loffset + bytes_read > cleartext_end_loffset) {
	bytes_read = cleartext_end_loffset - current_loffset;
      }
      if ((error = uiomove(current_base, bytes_read, UIO_WRITE, uiop)))
	break;
    }
    /* encode block before writing */
    crypt_encode_block(__FUNCTION__,__LINE__,temp_iovec[i].iov_base, PAGESIZE, vp, cr);
    current_loffset += bytes_read;
  }
  fist_print_uios("WRITE (after for loop 4)", &temp_uio);

  if (va.va_size < end_loffset) {
    if (va.va_size < cleartext_end_loffset)
      resid -= end_loffset - cleartext_end_loffset;
    else
      resid -= end_loffset - va.va_size;
  }

  /* XXX: no need for full initialization here */
  temp_uio.uio_iov = temp_iovec;
  temp_uio.uio_iovcnt = num_pages;
  temp_uio.uio_loffset = start_loffset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_fmode = uiop->uio_fmode;
  temp_uio.uio_llimit = uiop->uio_llimit;
  temp_uio.uio_resid = resid;

  /*
   * pass operation to hidden filesystem, and return status
   */
  fist_print_uios("WRITE (before write)", &temp_uio);
  error = VOP_WRITE(hidden_vp, &temp_uio, hidden_ioflag, cr);
  fist_print_uios("WRITE (after write)", &temp_uio);

  if (temp_uio.uio_loffset < cleartext_end_loffset) {
    /* incomplete write: this case is an error and should not happen */
    uiop->uio_loffset = temp_uio.uio_loffset;
    uiop->uio_resid = cleartext_end_loffset - temp_uio.uio_loffset;
  } else {
    /* we may have written more than what was asked of us to preserve the
     * encryption/encoding over a whole page
     */
    uiop->uio_loffset = cleartext_end_loffset;
    uiop->uio_resid = 0;
  }
  /* if FAPPEND was used, return offset of 0 to upper level */
  if (ioflag & FAPPEND) {
    uiop->uio_loffset = 0;
  }

out_free:
  for (i = 0; i < num_pages; i++) {
    fist_dprint(6, "PRINT_BASE1 %d: 0x%x (len=%d)\n", i,
		temp_iovec[i].iov_base,
		temp_iovec[i].iov_len);
    fist_dprint(6, "PRINT_BASE2 %d: 0x%x (len=%d)\n", i,
		free_iovec[i].iov_base,
		free_iovec[i].iov_len);
    kmem_free(free_iovec[i].iov_base, PAGESIZE);
  }
  kmem_free(free_iovec, num_pages * sizeof(iovec_t));
  kmem_free(temp_iovec, num_pages * sizeof(iovec_t));

out:
  mutex_exit(&vp->v_lock);

#ifdef FIST_DEBUG
  fist_print_uios("fist_crypt_write (END)", uiop);
#endif /* FIST_DEBUG */

  print_location();
  return (error);
}


static int
fist_crypt_ioctl(
		  vnode_t * vp,
		  int cmd,
		  int arg,
		  int flag,
		  cred_t * cr,
		  int *rvalp
)
{
  int error = EPERM;
  vnode_t *hidden_vp;
  int val = 0;
  unsigned char cbc_key[16];

  fist_dprint(4, "fist_crypt_ioctl vp %x cmd %d\n", vp,cmd);

  /* check if asked for local commands */
  switch (cmd) {
  case FIST_IOCTL_GET_DEBUG_VALUE:
    if (copyin((caddr_t) arg, (caddr_t) & val, sizeof(int))) {
      error = EFAULT;
      goto out;
    }
    val = fist_get_debug_value();
    if (copyout((caddr_t) & val, (caddr_t) arg, sizeof(int))) {
      error = EFAULT;
      goto out;
    }
    error = 0;
    goto out;
    break;

  case FIST_IOCTL_SET_DEBUG_VALUE:
    val = (int) arg;
    if (val < 0 || val > 20) {
      error = EINVAL;
      goto out;
    }
    fist_dprint(6, "IOCTL: got arg %d\n", val);
    fist_set_debug_value(val);
    error = 0;
    goto out;
    break;

  case FIST_IOCTL_SET_KEY:
    fist_dprint(8, "IOCTL: in set key\n");
    if (copyin((caddr_t) arg, (caddr_t) cbc_key, 16)) {
      error = EFAULT;
      goto out;
    }
    fist_set_userpass(vp->v_vfsp, cbc_key, cr);
    error = 0;
    goto out;

  }

  hidden_vp = vntofwn(vp)->fwn_vnodep;
  /* pass operation to hidden filesystem, and return status */
  error = VOP_IOCTL(hidden_vp, cmd, arg, flag, cr, rvalp);

out:
  print_location();
  return (error);
}


static int
fist_crypt_setfl(
		  vnode_t * vp,
		  int oflags,
		  int nflags,
		  cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_setfl vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_SETFL(hidden_vp, oflags, nflags, cr);

  print_location();
  return (error);
}


static int
fist_crypt_getattr(
		    vnode_t * vp,
		    vattr_t * vap,
		    int flags,
		    cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_getattr vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_GETATTR(hidden_vp, vap, flags, cr);

  if (error) {
    fist_dprint(4, "ERROR: fist_crypt_getattr %d\n", error);
  }
  print_location();
  return (error);
}


static int
fist_crypt_setattr(
		    vnode_t * vp,
		    vattr_t * vap,
		    int flags,
		    cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_setattr vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_SETATTR(hidden_vp, vap, flags, cr);

  print_location();
  return (error);
}


static int
fist_crypt_access(
		   vnode_t * vp,
		   int mode,
		   int flags,
		   cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_access vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_ACCESS(hidden_vp, mode, flags, cr);

  print_location();
  return (error);
}


static int
fist_crypt_lookup(
		   vnode_t * dvp,
		   char *name,
		   vnode_t ** vpp,
		   pathname_t * pnp,
		   int flags,
		   vnode_t * rdir,
		   cred_t * cr
)
{
  int error = EPERM;
  char *encoded_name;
  int encoded_length;
  vnode_t *hidden_dvp;

  fist_dprint(4, "fist_crypt_lookup dvp %x, rdir %x, name \"%s\"\n",
	      dvp, rdir, (name ? name : "Null"));

  if (fist_get_userpass(dvp->v_vfsp, cr) == NULL)
    return EACCES;

  hidden_dvp = vntofwn(dvp)->fwn_vnodep;

  fist_crypt_encodefilename(dvp->v_vfsp, name, &encoded_name, &encoded_length, SKIP_DOTS, cr);

  /* pass operation to hidden filesystem, and return status */
  error = VOP_LOOKUP(hidden_dvp, encoded_name, vpp, pnp, flags, rdir, cr);
  /* if no error, interpose vnode */
  if (!error) {
    fist_dprint(6, "CRYPT_LOOKUP1: hidden_vp->v_count=%d, vpp->v_type=%d\n",
		(*vpp)->v_count, (*vpp)->v_type);
    *vpp = fist_crypt_interpose(*vpp, dvp->v_vfsp);
    fist_dprint(6, "CRYPT_LOOKUP2: vpp->v_count=%d, hidden_vp->v_count=%d, vpp->v_type=%d\n",
		(*vpp)->v_count, vntofwn(*vpp)->fwn_vnodep->v_count,
		(*vpp)->v_type);
  }
  kmem_free(encoded_name, encoded_length);

  if (error) {
    fist_dprint(4, "ERROR: fist_crypt_lookup %d\n", error);
  }

  print_location();
  return (error);
}

static int
fist_crypt_create(
		   vnode_t * dvp,
		   char *name,
		   vattr_t * vap,
		   vcexcl_t excl,
		   int mode,
		   vnode_t ** vpp,
		   cred_t * cr,
		   int flag  /* XXX: EZK: new in 2.6, why? 64-bit support? */
)
{
  int error = EPERM;
  vnode_t *hidden_vp;
  char *encoded_name;
  int encoded_length;

  fist_dprint(4, "fist_crypt_create vp=%x name=\"%s\" mode=%x\n", dvp, name, mode);

  if (fist_get_userpass(dvp->v_vfsp, cr) == NULL)
    return EACCES;

  hidden_vp = vntofwn(dvp)->fwn_vnodep;

  fist_crypt_encodefilename(dvp->v_vfsp, name, &encoded_name, &encoded_length, SKIP_DOTS, cr);

  /* pass operation to hidden filesystem, and return status */
  error = VOP_CREATE(hidden_vp, encoded_name, vap, excl, mode, vpp, cr, flag);

  /* if no error, interpose vnode */
  if (!error) {
    fist_dprint(6, "CRYPT_CREATE1: hidden_vp->v_count %d\n", (*vpp)->v_count);
    *vpp = fist_crypt_interpose(*vpp, dvp->v_vfsp);
    fist_dprint(6, "CRYPT_CREATE2: vpp->v_count %d, hidden_vp->v_count %d\n",
		(*vpp)->v_count, vntofwn(*vpp)->fwn_vnodep->v_count);
  }
  kmem_free(encoded_name, encoded_length);

  print_location();
  return (error);
}


static int
fist_crypt_remove(
		   vnode_t * vp,
		   char *name,
		   cred_t * cr
)
{
  int error = EPERM;
  char *encoded_name;
  int encoded_length;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_remove vp %x\n", vp);

  if (fist_get_userpass(vp->v_vfsp, cr) == NULL)
    return EACCES;

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  fist_crypt_encodefilename(vp->v_vfsp, name, &encoded_name, &encoded_length, SKIP_DOTS, cr);

  /* pass operation to hidden filesystem, and return status */
  error = VOP_REMOVE(hidden_vp, encoded_name, cr);

  kmem_free(encoded_name, encoded_length);

  print_location();
  return (error);
}


static int
fist_crypt_link(
		 vnode_t * tdvp,
		 vnode_t * svp,
		 char *name,
		 cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_tdvp, *hidden_svp;
  char *encoded_name;
  int encoded_length;

  fist_dprint(4, "fist_crypt_link tdvp %x\n", tdvp);

  /* MUST make sure we only hard link into our own file system! */
  if (svp->v_op != &fist_crypt_vnodeops) {
    printk("HARDLINK NOT ALLOWED to %x\n", svp);
    return EXDEV;
  }

  if (fist_get_userpass(svp->v_vfsp, cr) == NULL)
    return EACCES;

  hidden_tdvp = vntofwn(tdvp)->fwn_vnodep;
  hidden_svp = vntofwn(svp)->fwn_vnodep;

  fist_crypt_encodefilename(tdvp->v_vfsp, name, &encoded_name, &encoded_length, SKIP_DOTS, cr);

  /* pass operation to hidden filesystem, and return status */
  error = VOP_LINK(hidden_tdvp, hidden_svp, encoded_name, cr);

  kmem_free(encoded_name, encoded_length);

  print_location();
  return (error);
}


static int
fist_crypt_rename(
		   vnode_t * sdvp,
		   char *snm,
		   vnode_t * tdvp,
		   char *tnm,
		   cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_sdvp, *hidden_tdvp;
  char *source_encoded_name, *target_encoded_name;
  int source_encoded_length, target_encoded_length;

  fist_dprint(4, "fist_crypt_rename sdvp %x, tdvp %x\n", sdvp, tdvp);

  if (fist_get_userpass(sdvp->v_vfsp, cr) == NULL)
    return EACCES;

  hidden_sdvp = vntofwn(sdvp)->fwn_vnodep;
  hidden_tdvp = vntofwn(tdvp)->fwn_vnodep;

  fist_crypt_encodefilename(sdvp->v_vfsp, snm, &source_encoded_name, &source_encoded_length, SKIP_DOTS, cr);
  fist_crypt_encodefilename(tdvp->v_vfsp, tnm, &target_encoded_name, &target_encoded_length, SKIP_DOTS, cr);

  /* pass operation to hidden filesystem, and return status */
  error = VOP_RENAME(hidden_sdvp, source_encoded_name, hidden_tdvp, target_encoded_name, cr);

  kmem_free(source_encoded_name, source_encoded_length);
  kmem_free(target_encoded_name, target_encoded_length);

  print_location();
  return (error);
}


static int
fist_crypt_mkdir(
		  vnode_t * dvp,
		  char *name,
		  vattr_t * vap,
		  vnode_t ** vpp,
		  cred_t * cr
)
{
  int error = EPERM;
  char *encoded_name;
  int encoded_length;
  vnode_t *hidden_dvp;

  fist_dprint(4, "fist_crypt_mkdir vp %x\n", dvp);

  if (fist_get_userpass(dvp->v_vfsp, cr) == NULL)
    return EACCES;

  hidden_dvp = vntofwn(dvp)->fwn_vnodep;

  /* this code encodes the name and uuencodes it */
  fist_crypt_encodefilename(dvp->v_vfsp, name, &encoded_name, &encoded_length, SKIP_DOTS, cr);

  /* pass operation to hidden filesystem, and return status */
  error = VOP_MKDIR(hidden_dvp, encoded_name, vap, vpp, cr);
  fist_dprint(6, "mkdir: encoded name is %s\n", encoded_name);
  /* if no error, interpose vnode */
  if (!error) {
    fist_dprint(6, "CRYPT_MKDIR1: hidden_vp->v_count %d\n", (*vpp)->v_count);
    *vpp = fist_crypt_interpose(*vpp, dvp->v_vfsp);
  }
  kmem_free(encoded_name, encoded_length);

  print_location();
  return (error);
}


static int
fist_crypt_rmdir(
		  vnode_t * vp,
		  char *name,
		  vnode_t * cdir,
		  cred_t * cr
)
{
  int error = EPERM;
  char *encoded_name;
  int encoded_length;
  vnode_t *hidden_vp, *hidden_cdir;

  fist_dprint(4, "fist_crypt_rmdir vp %x\n", vp);

  if (fist_get_userpass(vp->v_vfsp, cr) == NULL)
    return EACCES;

  hidden_vp = vntofwn(vp)->fwn_vnodep;
  hidden_cdir = vntofwn(cdir)->fwn_vnodep;

  fist_crypt_encodefilename(vp->v_vfsp, name, &encoded_name, &encoded_length, SKIP_DOTS, cr);

  /* pass operation to hidden filesystem, and return status */
  error = VOP_RMDIR(hidden_vp, encoded_name, hidden_cdir, cr);
  fist_dprint(6, "rmdir: encoded name is %s\n", encoded_name);

  kmem_free(encoded_name, encoded_length);

  print_location();
  return (error);
}


static int
fist_crypt_readdir(
		    vnode_t * vp,
		    uio_t * uiop,
		    cred_t * cr,
		    int *eofp
)
{
  int error = EPERM;
  vnode_t *hidden_vp;
  uio_t temp_uio;
  iovec_t temp_iovec;
  int aux, bytes_read, length, temp_length, tmp, old_reclen;
  static char temp_name[MAXNAMLEN];

  fist_dprint(4, "fist_crypt_readdir vp %x\n", vp);

  if (fist_get_userpass(vp->v_vfsp, cr) == NULL)
    return EACCES;

  ASSERT(uiop->uio_iovcnt == 1);

#ifdef FIST_DEBUG
  fist_print_uios("fist_crypt_readdir (START)", uiop);
#endif /* FIST_DEBUG */

  temp_iovec.iov_len = uiop->uio_resid;
  temp_iovec.iov_base = kmem_zalloc(uiop->uio_resid, KM_SLEEP);

  temp_uio.uio_iov = &temp_iovec;
  temp_uio.uio_iovcnt = 1;
  temp_uio.uio_loffset = uiop->uio_loffset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_fmode = uiop->uio_fmode;
  temp_uio.uio_llimit = uiop->uio_llimit;
  temp_uio.uio_resid = uiop->uio_resid;

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_READDIR(hidden_vp, &temp_uio, cr, eofp);

  bytes_read = uiop->uio_resid - temp_uio.uio_resid;
  temp_iovec.iov_base -= bytes_read;
  temp_iovec.iov_len += bytes_read;

  if (error)
    goto clean_up;

#define crt_dirent ((struct dirent *)(temp_iovec.iov_base + aux))
  for (aux = 0; aux < bytes_read; aux += old_reclen) {
    old_reclen = crt_dirent->d_reclen;
    fist_dprint(5, "RD: old_reclen %d\n", old_reclen);
    if (fist_crypt_decodefilename(vp->v_vfsp,
				  crt_dirent->d_name,
				  crt_dirent->d_reclen - sizeof(struct dirent) + 2,
				  temp_name,
				  &temp_length, /* includes terminating null */
				  SKIP_DOTS, cr) == 0) {
      /*
       * We copy the dirent to userspace only if the csum matched
       */
      strcpy(crt_dirent->d_name, temp_name); /* ok, b/c of terminating null */
      length = temp_length - 2 + sizeof(struct dirent);
      fist_dprint(5, "RD: length calculated to %d, temp_length %d, struct dirent: %d\n", length,temp_length,sizeof(struct dirent));
      if ((tmp = length & 3))
	length += 4 - tmp;
      crt_dirent->d_reclen = length;
//      kmem_free(temp_name, temp_length);
#ifdef FIST_CRYPTDEBUG
      fist_dprint(5, "RD: adding entry \"%s\" of length %d\n",
		  crt_dirent->d_name, crt_dirent->d_reclen);
#endif
      error = uiomove(temp_iovec.iov_base + aux, crt_dirent->d_reclen, UIO_READ, uiop);
      if (error)
	goto clean_up;
    }
  }
  uiop->uio_offset = temp_uio.uio_offset;

#ifdef FIST_DEBUG
  fist_print_uios("fist_crypt_readdir (END)", uiop);
#endif /* FIST_DEBUG */

clean_up:
  kmem_free(temp_iovec.iov_base, temp_iovec.iov_len);

  if (error) {
    fist_dprint(4, "ERROR: fist_crypt_readdir %d\n", error);
  }
  print_location();
  return (error);
}


static int
fist_crypt_symlink(
		    vnode_t * vp,
		    char *linkname,
		    vattr_t * vap,
		    char *target,
		    cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;
  char *encoded_linkname, *encoded_target;
  int encoded_linkname_length, encoded_target_length;

  fist_dprint(4, "fist_crypt_symlink vp %x\n", vp);

  if (fist_get_userpass(vp->v_vfsp, cr) == NULL)
    return EACCES;

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  fist_crypt_encodefilename(vp->v_vfsp,
			    linkname,
			    &encoded_linkname,
			    &encoded_linkname_length,
			    SKIP_DOTS, cr);
  fist_crypt_encodefilename(vp->v_vfsp,
			    target,
			    &encoded_target,
			    &encoded_target_length,
			    DO_DOTS, cr);

  /* pass operation to hidden filesystem, and return status */
  error = VOP_SYMLINK(hidden_vp, encoded_linkname, vap, encoded_target, cr);

  kmem_free(encoded_linkname, encoded_linkname_length);
  kmem_free(encoded_target, encoded_target_length);

  print_location();
  return (error);
}


static int
fist_crypt_readlink(
		     vnode_t * vp,
		     uio_t * uiop,
		     cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;
  uio_t temp_uio;
  iovec_t temp_iovec;
  caddr_t temp_addr2free;
  int bytes_read;
  int temp_length, target_real_length;
  static char temp_name[MAXNAMLEN];

  fist_dprint(4, "fist_crypt_readlink vp %x\n", vp);

  if (fist_get_userpass(vp->v_vfsp, cr) == NULL)
    return EACCES;

  fist_print_uios("fist_crypt_readlink (START)", uiop);

  temp_iovec.iov_len = PAGESIZE;
  temp_iovec.iov_base = temp_addr2free = kmem_zalloc(PAGESIZE, KM_SLEEP);
  if (!temp_iovec.iov_base) {
    printk("no more memory in readlink\n");
    error = ENOMEM;
    goto out;
  }
  temp_uio.uio_iov = &temp_iovec;
  temp_uio.uio_iovcnt = 1;
  temp_uio.uio_loffset = 0;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_fmode = uiop->uio_fmode;
  temp_uio.uio_llimit = uiop->uio_llimit;
  temp_uio.uio_resid = uiop->uio_resid;

  hidden_vp = vntofwn(vp)->fwn_vnodep;
  /* pass operation to hidden filesystem, and return status */

  error = VOP_READLINK(hidden_vp, &temp_uio, cr);
  bytes_read = PAGESIZE - temp_iovec.iov_len;

  if (fist_crypt_decodefilename(vp->v_vfsp, temp_iovec.iov_base - bytes_read,
				bytes_read, temp_name, &temp_length, DO_DOTS, cr) != 0) {
    /* a checksum error had occured: skip entry */
    cmn_err(CE_PANIC, "symlink value encrypted with different key");
  }
  /* must find real string length, which is guaranteed null terminated here */
  target_real_length = strlen(temp_name) + 1;
  fist_dprint(4, "fist_crypt_readlink DECODE len=%d, real_len=%d, bytes_read=%d, name=\"%s\"",
	      temp_length, target_real_length, bytes_read, temp_name);

  uiomove(temp_name, target_real_length, UIO_READ, uiop);
  /* already OK: uiop->uio_resid and uiop->uio_loffset */
//  kmem_free(temp_name, temp_length);
  kmem_free(temp_addr2free, PAGESIZE);

  fist_print_uios("fist_crypt_readlink (END)", uiop);

out:
  if (error) {
    fist_dprint(4, "ERROR: fist_crypt_readlink %d\n", error);
  }
  print_location();
  return (error);
}


static int
fist_crypt_fsync(
		  vnode_t * vp,
		  int syncflag,
		  cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_fsync vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_FSYNC(hidden_vp, syncflag, cr);

  print_location();

  return (error);
}


/*
 * Free up a vnode once the last reference count to it
 * has been released (via vn_rele()).
 */
static void
fist_crypt_inactive(
		     vnode_t * vp,
		     cred_t * cr
)
{
  vnode_t *hidden_vp;
  struct fist_cryptinfo *fwip = NULL;
  int ret;

  fist_dprint(4, "fist_crypt_inactive vp %x\n", vp);

  /* flush all pages if need to */
  if (vp->v_pages) {
    /*
     * using B_INVAL will instruct *_dispose() to remove the
     * mappings of the page from the vnode and the page hash.
     */
    ret = pvn_vplist_dirty(vp, 0, fist_crypt_putapage, B_INVAL, cr);
    if (ret) {
      fist_dprint(6, "CRYPT_INACTIVE, pvn_vnlist_dirty returns %d.\n", ret);
    }
  }

  hidden_vp = vntofwn(vp)->fwn_vnodep;

#if 0
  /*
   * XXX: EZK experimental.
   * flush hidden_vp's pages.
   * don't do it: too slow
   */
  if (hidden_vp->v_pages) {
    printk("INACTIVE: hidden_vp has pages to flush...\n");
    ret = VOP_PUTPAGE(hidden_vp, (offset_t) 0, 0, B_INVAL, cr);
    printk("INACTIVE: hidden PUTPAGE returned %d\n", ret);
  }
#endif

  /*
   * This is a tricky function.  We need to first perform the action to the
   * interposed vnode, and then to this vnode.  This call can only have come
   * normally from VN_RELE() which called vn_rele().
   *
   * XXX: should I call VN_RELE on the interposed vnode instead?
   */

  /*
   * XXX: Is this right?  Should I call inactive on the interposed vode as
   * well?  If I do it means that the reference count for an interposed and
   * interposing vnodes are always the same.  That cannot always be true
   * because I can access the interposed filesystem from the original path.
   * It might be best if I ensure that the refcount is exactly 1 here.
   * I must investigate this bug!!! -Erez.
   */

  /* pass operation to hidden filesystem, and return status */
  /* VOP_INACTIVE(hidden_vp, cr); */
  fist_dprint(6, "CRYPT_INACTIVE1: hidden_vp->v_count %d\n", hidden_vp->v_count);
  VN_RELE(hidden_vp);
  fist_dprint(6, "CRYPT_INACTIVE2: hidden_vp->v_count %d\n", hidden_vp->v_count);

  mutex_enter(&vfstofwi(vp->v_vfsp)->fwi_ht_lock);
  fist_ht_del_vp(hidden_vp, vp->v_vfsp);

  /* free space used by opaque v_data field */
  if (vp->v_data) {
    mutex_destroy(&vntofwn(vp)->fwn_lock);
    kmem_free(vp->v_data, sizeof(fist_cryptnode_t));
    vp->v_data = NULL;
  }
  /* free actual vnode */
  kmem_free(vp, sizeof(vnode_t));
  /* XXX: should I reset this vnode? */
  /* vp = NULL; */

  /* decrement interposed vnodes counter */
  fwip = vfstofwi(vp->v_vfsp);
  fwip->fwi_num_vnodes--;

  mutex_exit(&vfstofwi(vp->v_vfsp)->fwi_ht_lock);

  print_location();
  return;
}


static int
fist_crypt_fid(
		vnode_t * vp,
		fid_t * fidp
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_fid vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_FID(hidden_vp, fidp);

  print_location();
  return (error);
}


static void
fist_crypt_rwlock(
		   vnode_t * vp,
		   int write_lock
)
{
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_rwlock vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  VOP_RWLOCK(hidden_vp, write_lock);

  print_location();
  return;
}


static void
fist_crypt_rwunlock(
		     vnode_t * vp,
		     int write_lock
)
{
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_rwunlock vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  VOP_RWUNLOCK(hidden_vp, write_lock);

  print_location();
  return;
}


static int
fist_crypt_seek(
		 vnode_t * vp,
		 offset_t offset,
		 offset_t * offsetp
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_seek vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_SEEK(hidden_vp, offset, offsetp);

  print_location();
  return (error);
}


static int
fist_crypt_cmp(
		vnode_t * vp1,
		vnode_t * vp2
)
{
  int error = EPERM;
  vnode_t *hidden_vp1;
  vnode_t *hidden_vp2;

  fist_dprint(4, "fist_crypt_cmp vp1 %x, vp2\n", vp1, vp2);

  hidden_vp1 = vntofwn(vp1)->fwn_vnodep;
  hidden_vp2 = vntofwn(vp2)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_CMP(hidden_vp1, hidden_vp2);

  print_location();
  return (error);
}


static int
fist_crypt_frlock(
		   vnode_t * vp,
		   int cmd,
		   struct flock64 * bfp,
		   int flag,
		   offset_t offset,
		   cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_frlock vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_FRLOCK(hidden_vp, cmd, bfp, flag, offset, cr);

  print_location();
  return (error);
}


static int
fist_crypt_space(
		  vnode_t * vp,
		  int cmd,
		  struct flock64 * bfp,
		  int flag,
		  offset_t offset,
		  cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_space vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_SPACE(hidden_vp, cmd, bfp, flag, offset, cr);

  print_location();
  return (error);
}


static int
fist_crypt_realvp(
		   vnode_t * vp,
		   vnode_t ** vpp
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_realvp vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_REALVP(hidden_vp, vpp);

  /* if no error, interpose vnode */
  if (!error) {
    /* XXX: is this right? Is the vfs type passed to _interpose right? */
    *vpp = fist_crypt_interpose(*vpp, vp->v_vfsp);
  }
  print_location();
  return (error);
}

/* mmap functions were here and were moved to fist_crypt_mmap.c */


static int
fist_crypt_poll(
		 vnode_t * vp,
		 short events,
		 int anyyet,
		 short *reventsp,
		 pollhead_t ** phpp
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_poll vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_POLL(hidden_vp, events, anyyet, reventsp, phpp);

  print_location();
  return (error);
}


static int
fist_crypt_dump(
		 vnode_t * vp,
		 caddr_t addr,
		 int ldbn,
		 int dblks
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_dump vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_DUMP(hidden_vp, addr, ldbn, dblks);

  print_location();
  return (error);
}


static int
fist_crypt_pathconf(
		     vnode_t * vp,
		     int cmd,
		     u_long * valp,
		     cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_pathconf vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_PATHCONF(hidden_vp, cmd, valp, cr);

  print_location();
  return (error);
}


static int
fist_crypt_pageio(
		   vnode_t * vp,
		   page_t * pp,
		   u_offset_t io_off,
		   u_int io_len,
		   int flags,
		   cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_pageio vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_PAGEIO(hidden_vp, pp, io_off, io_len, flags, cr);

  print_location();
  return (error);
}


static int
fist_crypt_dumpctl(
		    vnode_t * vp,
		    int free
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_dumpctl vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_DUMPCTL(hidden_vp, free);

  print_location();
  return (error);
}


static void
fist_crypt_dispose(
		    vnode_t * vp,
		    page_t * pp,
		    int fl,
		    int dn,
		    cred_t * cr
)
{
  fist_dprint(4, "fist_crypt_DISPOSE vp 0x%x, page 0x%x fl=0x%x, dn=0x%x\n", vp, pp, fl, dn);
  /* fist_crypt_DISPOSE vp 0x600cf2b0, page 0x104f59c0 fl=0x8000, dn=0x2000 */

  fs_dispose(vp, pp, fl, dn, cr);
  return;
}


static int
fist_crypt_setsecattr(
		       vnode_t * vp,
		       vsecattr_t * vsap,
		       int flag,
		       cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_setsecattr vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_SETSECATTR(hidden_vp, vsap, flag, cr);

  print_location();
  return (error);
}


static int
fist_crypt_getsecattr(
		       vnode_t * vp,
		       vsecattr_t * vsap,
		       int flag,
		       cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_crypt_getsecattr vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_GETSECATTR(hidden_vp, vsap, flag, cr);

  print_location();
  return (error);
}
