/*  Hey Emacs, switch to -*-C-*-, please 
------------

Copyright (C) 1997 Tage Stabell-Kul (tage@acm.org).  All rights
reserved.

This is a package, containing a library, that makes it possible for
programmers to include security functionality in their programs, in a
format compatible with PGP.

This library is free for commercial and non-commercial use as long as
the following conditions are aheared to:

Copyright remains Tage Stabell-Kul's, and as such any Copyright
notices in the code are not to be removed.
If this package is used in a product, Tage Stabell-Kul should be given
attribution as the author of the parts of the library used.
This can be in the form of a textual message at program startup or
in documentation (online or textual) provided with the package.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:
   "This product includes software written by
    Tage Stabell-Kul (tage@acm.org)"

THIS SOFTWARE IS PROVIDED BY TAGE STABELL-KUL ``AS IS'' AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The licence and distribution terms for any publically available
version or derivative of this code cannot be changed.  i.e. this code
cannot simply be copied and put under another distribution licence
[including the GNU Public Licence.]

The reason behind this being stated in this direct manner is past
experience in code simply being copied and the attribution removed
from it and then being distributed as part of other packages. This
implementation was a non-trivial and unpaid effort.
*/

#include "pgplib.h"
#include "pgp_m.h"

/*
 * Find the real packetlength of a PGP packet.  p must point to the CTB.  
 * We return the length of the _entire_ packet.
 * We access p and 1, 2 or 4 more bytes depending of the length; don't 
 * use this to find out how large a packet will become based on the
 * CTB !
 */
int
packet_length(u_char *p)
{
    int 	i, ret;
    u_short 	us;
    u_long	ul;

    i = ctb_llength(*p);
    p++; 
    switch (i) {
    case 1:
	ret = 1 + i + (int)*p;
	break;
    case 2:
	memcpy(&us, p, 2);
	ret = 1 + i + (int) ntohs(us);;
	break;
    case 4:
	memcpy(&ul, p, 4);
	ret =  1 + i + (int) ntohl(ul);
	break;
    default:
	assert(0);
    }

    return(ret);
}

/* 
 * The length of the length of a PGP packet depends on the lenght
 * of the data, and the type.  Take a type, a length, and a "force"
 * (which forces the use of 4 bytes length) and a buffer (at least 5
 * bytes long) and write a CTB and the length (in Network Byte Order)
 * into the buffer.
 * Return the number of bytes written into the buffer.
 * The idea was taken from write_ctb_len in PGP/crypto.c
 * 
 */
int
make_ctb(u_char ctb_type, u_long length, int force, u_char *buf)
{
    int		llength, llenb, ret;
    u_short	us;
    u_long	ul;

    if (force || (length > 0xffffL)) {
	llength = 4;
	llenb = 2;
    } 
    else if ( (u_short)length > 0xff) {
	llength = 2;
	llenb = 1;
    } 
    else {
	llength = 1;
	llenb = 0;
    }
	
    buf[0] = CTB_BYTE(ctb_type, llenb);
    /* convert length to external byteorder */
    switch ( llength) {
    case 1:
	buf[1] = (u_char)length;
	ret = 2;
	break;
    case 2:
	us = htons( (u_short)length);
	memcpy(buf+1, &us, 2);
	ret = 3;
	break;
    case 4:
	ul = htonl(length);
	memcpy(buf+1, &ul, 4);
	ret = 5;
	break;
    default:
	assert(0);
    }
    return(ret);
} 


/*
 * Copy at most length-1 bytes from buffer into line, stopping after
 * the first newline has been found or buflen bytes has been
 * copied.  In any case, a NULL is appended to the string.
 * If some parameter is unreasonable, 0 is returned, if not, the 
 * number of bytes consumed from buf is returned.
 * If a one-byte line is passed in, 0 will obviously be returned.
 */

int
bufgets(u_char	*line,
	int 	linelen, 
	u_char 	*buf, 
	int 	buflen)
{
    int		i;

    if ( line == NULL || linelen < 1 || buf == NULL || buflen < 1)
	return(0);

    i = 0;
    while ( i < linelen && i < buflen) {
	line[i] = buf[i];
	if ( line[i] == '\n' )
	    if ( i+1 == linelen ) {
		line[i] = NULL;
		return(i);
	    }
	    else {
		line[i+1] = NULL;
		return(i+1);
	    }
	i += 1;
	continue;
    }
    if ( i == linelen ) {
	line[i-1] = NULL;
	/* The last char was overwritten, not consumed */
	return(i-1);
    }
    if ( i == buflen ) {
	if ( i == linelen ) {
	    line[i-1] = NULL;
	    return(i-1);
	}
	else {
	    line[i] = NULL;
	    return(i);
	}
    }
    if ( i == linelen ) {
	line[i-1] = NULL;
	return(i-1);
    }
    return(0);
}

/* 
 * This code was written by Peter Bosch <Peter.Bosch@huygens.com>
 * to overcome the missing functionality in FreeBSD.
 * 
 */

#ifdef __FreeBSD__
unsigned long long
atoq(const char *s)
{
    unsigned long long rv;

    if (s == NULL) 
	return 0;

    /* Parse spaces */
    while (*s && isspace(*s))
	s++;
		
    /* Parse 0x or 0X */
    if (*s != '0') 
	return 0;
    s++;
    if (*s != 'x' && *s != 'X') 
	return 0;
    s++;

    rv = 0;
    while (*s) {
	if (*s == '\0' ||
	    ((*s < '0' || *s > '9') &&
	     (*s < 'a' || *s > 'f') &&
	     (*s < 'A' || *s > 'F'))) 
	    return rv;
	rv *= 16;
	rv += (*s >= '0' && *s <= '9')? *s - '0':
	    ((*s >= 'a' || *s <= 'f')? *s - 'a' + 10:
	     *s - 'A' + 10);
	s++;
    }
    return rv;
}
#endif


/* 
 * Take 8 bytes (in network order) and make a quad 
 */

u_quad_t
to_quad(u_char 	*buf)
{
    int 	i;
    u_quad_t	q;
    
    q = 0LL;
    for(i=0; i < 8; i++)
	q |= ((u_quad_t) buf[i] << 8*(7-i));

    return(q);
}

/* 
 * Copy a quad into a buffer, in network order.
 */

void
copy_quad(u_quad_t q, u_char *buf)
{
    int	i,j;

    for(j=0, i=7; i>=0; i--, j++)
        buf[j] =  (u_char) (q >> 8*i) & 0xff;
}

/* 
 * This is how PGP generates "checksum" of a buffer.  It is only used 
 * to add two bytes to a DEK in order to be able to verify that the 
 * correct key  has been used.  I've read the sources for PGP to find 
 * this out, in particular the file mpiio.c
 */
u_short 
checksum(u_char	*buf, 
	 int 	count)
{
    u_short cksum;

    assert(count >= 0);
    assert(buf != NULL);

    for(cksum = 0; count >= 0; count--)
	cksum += buf[count];

    return (cksum);
}

/* 
 * A buffer, containing a PGP MPI, is converted into a SSL 
 * BIGNUM. On error we return 0, else the number of bytes 
 * consumed.
 */

int
mpi2bignum(u_char	*buf,
	   int		buflen,
	   BIGNUM	**N)
{
    u_short	us;
    int		length, numbit;
    int		sofar;
    /* scratch */
    u_char	*p;

    sofar = 0;

    /* Length of N */
    memcpy( &us, buf, 2);
    numbit = (int)ntohs(us);
    length = numbit / NBBY;
    if ( numbit % NBBY  != 0 )
	length += 1;
    p = calloc( length, 1);
    if ( p == NULL ) {
	fprintf(stderr, "Out of memory\n");
	return(0);
    }
    sofar += 2;

    if ( buflen - (sofar + length) < 0) {
	free(p);
	return(0);
    }
    /* Read N */
    memcpy( p, buf+sofar, length);
    sofar += length;    
    /* Turn N into a BIGNUM */
    *N = BN_bin2bn(p, length, NULL);
    free(p);    
    if (*N == NULL)
	return(0);
    else
	return(sofar);
}
