/* 
 * The code below was taken from PGP, armor.c in PGP 2.6.3.a.  
 * There, the following note can be found

   (c) Copyright 1990-1996 by Philip Zimmermann.  All rights reserved.
   The author assumes no liability for damages resulting from the use
   of this software, even if the damage results from defects in this
   software.  No warranty is expressed or implied.

   Note that while most PGP source modules bear Philip Zimmermann's
   copyright notice, many of them have been revised or entirely written
   by contributors who frequently failed to put their names in their
   code.  Code that has been incorporated into PGP from other authors
   was either originally published in the public domain or is used with
   permission from the various authors.

   PGP is available for free to the public under certain restrictions.
   See the PGP User's Guide (included in the release package) for
   important information about licensing, patent restrictions on
   certain algorithms, trademarks, copyrights, and export controls.

 * Modefied by Tage Stabell-Kul to work with buffers rather than 
 * with FILEs.
 * Also, cleaned up usage of char rather than unsigned char.
 */

/*====================BEGIN armor.c from PGP==========*/

/* Begin ASCII armor routines.
   This converts a binary file into printable ASCII characters, in a
   radix-64 form mostly compatible with the MIME format.
   This makes it easier to send encrypted files over a 7-bit channel.
 */

/* Index this array by a 6 bit value to get the character corresponding
 * to that value.
 */
static
unsigned char bintoasc[] = 
"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

/* Index this array by a 7 bit value to get the 6-bit binary field
 * corresponding to that value.  Any illegal characters return high bit set.
 */
static
unsigned char asctobin[] =
{
    0200, 0200, 0200, 0200, 0200, 0200, 0200, 0200,
    0200, 0200, 0200, 0200, 0200, 0200, 0200, 0200,
    0200, 0200, 0200, 0200, 0200, 0200, 0200, 0200,
    0200, 0200, 0200, 0200, 0200, 0200, 0200, 0200,
    0200, 0200, 0200, 0200, 0200, 0200, 0200, 0200,
    0200, 0200, 0200, 0076, 0200, 0200, 0200, 0077,
    0064, 0065, 0066, 0067, 0070, 0071, 0072, 0073,
    0074, 0075, 0200, 0200, 0200, 0200, 0200, 0200,
    0200, 0000, 0001, 0002, 0003, 0004, 0005, 0006,
    0007, 0010, 0011, 0012, 0013, 0014, 0015, 0016,
    0017, 0020, 0021, 0022, 0023, 0024, 0025, 0026,
    0027, 0030, 0031, 0200, 0200, 0200, 0200, 0200,
    0200, 0032, 0033, 0034, 0035, 0036, 0037, 0040,
    0041, 0042, 0043, 0044, 0045, 0046, 0047, 0050,
    0051, 0052, 0053, 0054, 0055, 0056, 0057, 0060,
    0061, 0062, 0063, 0200, 0200, 0200, 0200, 0200
};

/************************************************************************
 * CRC Routines.
 *    These CRC functions are derived from code in chapter 19 of the book 
 *    "C Programmer's Guide to Serial Communications", by Joe Campbell.
 *    Generalized to any CRC width by Philip Zimmermann.
 */

#define	crcword		unsigned long

#define CRCBITS 	24		/* may be 16, 24, or 32 */
#define maskcrc(crc)	((crc) & 0xffffffL) /* if CRCBITS is 24 */
#define CRCHIBIT 	((crcword) (1L<<(CRCBITS-1))) /* 0x8000 if CRCBITS is 16 */
#define CRCSHIFTS 	(CRCBITS-8)

/*
 * Notes on making a good 24-bit CRC--
 * The primitive irreducible polynomial of degree 23 over GF(2),
 * 040435651 (octal), comes from Appendix C of "Error Correcting Codes,
 * 2nd edition" by Peterson and Weldon, page 490.  This polynomial was
 * chosen for its uniform density of ones and zeros, which has better
 * error detection properties than polynomials with a minimal number of
 * nonzero terms.  Multiplying this primitive degree-23 polynomial by
 * the polynomial x+1 yields the additional property of detecting any
 * odd number of bits in error, which means it adds parity.  This 
 * approach was recommended by Neal Glover.
 *
 * To multiply the polynomial 040435651 by x+1, shift it left 1 bit and
 * bitwise add (xor) the unshifted version back in.  Dropping the unused 
 * upper bit (bit 24) produces a CRC-24 generator bitmask of 041446373 
 * octal, or 0x864cfb hex.  
 *
 * You can detect spurious leading zeros or framing errors in the 
 * message by initializing the CRC accumulator to some agreed-upon 
 * nonzero value, but the value used is a bit nonstandard.  
 */

#define CCITTCRC 0x1021			/* CCITT's 16-bit CRC generator polynomial */
#define PRZCRC	0x864cfbL		/* PRZ's 24-bit CRC generator polynomial */
#define CRCINIT	0xB704CEL		/* Init value for CRC accumulator */

static crcword crctable[256];		/* Table for speeding up CRC's */

/*
 * mk_crctbl derives a CRC lookup table from the CRC polynomial.
 * The table is used later by the crcbytes function given below.
 * mk_crctbl only needs to be called once at the dawn of time.
 *
 * The theory behind mk_crctbl is that table[i] is initialized
 * with the CRC of i, and this is related to the CRC of i>>1,
 * so the CRC of i>>1 (pointed to by p) can be used to derive
 * the CRC of i (pointed to by q).
 */

static void
mk_crctbl(crcword poly)
{
    int 	i;
    crcword 	t, *p, *q;
    
    p = q = crctable;
    *q++ = 0;
    *q++ = poly;
    for (i = 1; i < 128; i++) {
	t = *++p;
	if (t & CRCHIBIT) {
	    t <<= 1;
	    *q++ = t ^ poly;
	    *q++ = t;
	} else {
	    t <<= 1;
	    *q++ = t;
	    *q++ = t ^ poly;
	}
    }
}


/*
 * Accumulate a buffer's worth of bytes into a CRC accumulator,
 * returning the new CRC value.
 */
crcword
crcbytes(u_char		*buf, 
	 u_long		len, 
	 u_long	 	accum)
{
    do {
	accum = accum << 8 ^ crctable[(u_char) (accum >> CRCSHIFTS) ^ *buf++];
    } while (--len);
    return maskcrc(accum);
}				/* crcbytes */

/* Initialize the CRC table using our codes */
void
init_crc(void)
{
    mk_crctbl(PRZCRC);
}


/* ENC is the basic 1 character encoding function to make a char printing */
#define ENC(c) 	((int)bintoasc[ (c) ])
#define PAD	'='

/*
 * Output one group of up to 3 bytes, pointed at by in, to out.
 * If fewer than 3 are present, the 1 or two extras must be zeros.
 * Tage Stabell-Kul removed several redundant &'s.
 */
static void
outdec(char 	*out, 
       u_char	*in, 
       int 	count)
{
    int c1, c2, c3, c4;

    c1 = in[0] >> 2;
    c2 = ((in[0] << 4) & 0x30) | (in[1] >> 4);
    c3 = ((in[1] << 2) & 0x3C) | (in[2] >> 6);
    c4 = in[2] & 0x3F;
    out[0] = ENC(c1);
    out[1] = ENC(c2);
    if (count == 1) {
	out[2] = PAD;
	out[3] = PAD;
    } else {
	out[2] = ENC(c3);
	if (count == 2)
	    out[3]=PAD;
	else
	    out[3] = ENC(c4);
    }
}				/* outdec */

/*
 * ASCII armor decode routines.
 */
static int
darmor_line(char	*inbuf, 
	    u_char 	*outbuf, 
	    int 	*outlength)
{
    u_char	*bp;
    int 	length;
    u_int 	c1, c2, c3, c4;
    int 	j;

    length = 0;
    bp = (u_char *) inbuf;

    /* FOUR input characters go into each THREE output charcters */

    while (*bp != '\0') {
	if (*bp & 0x80 || (c1 = asctobin[*bp]) & 0x80)
	    return -1;
	++bp;
	if (*bp & 0x80 || (c2 = asctobin[*bp]) & 0x80)
	    return -1;
	if (*++bp == PAD) {
	    c3 = c4 = 0;
	    length += 1;
	    if (c2 & 15)
		return -1;
	    if (strcmp((char *) bp, "==") == 0)
		bp += 1;
	    else if (strcmp((char *) bp, "=3D=3D") == 0)
		bp += 5;
	    else
		return -1;
	} else 
	    if (*bp & 0x80 || (c3 = asctobin[*bp]) & 0x80) {
		return -1;
	    } else {
		if (*++bp == PAD) {
		    c4 = 0;
		    length += 2;
		    if (c3 & 3)
			return -1;
		    if (strcmp((char *) bp, "=") == 0)
			;	/* All is well */
		    else 
			if (strcmp((char *) bp, "=3D") == 0)
			    bp += 2;
			else
			    return -1;
		} else 
		    if (*bp & 0x80 || (c4 = asctobin[*bp]) & 0x80) {
			return -1;
		    } 
		    else {
			length += 3;
		    }
	    }
	++bp;
	j = (c1 << 2) | (c2 >> 4);
	*outbuf++ = j;
	j = (c2 << 4) | (c3 >> 2);
	*outbuf++ = j;
	j = (c3 << 6) | c4;
	*outbuf++ = j;
    }

    *outlength = length;
    return 0;			/* normal return */
}				/* darmor_line */

/*====================END armor.c from PGP==========*/
