/*
 *
 *                         DES SOFTWARE PACKAGE
 *                             Version 2.2
 *
 *                                        _
 * Copyright (c) 1990,1991,1992,1993 Stig Ostholm.
 * All Rights Reserved
 *
 *
 * The author takes no responsibility of actions caused by the use of this
 * software package and does not guarantee the correctness of the functions.
 *
 * This software package may be freely distributed for non-commercial purpose
 * as long as the copyright notice is kept. Any changes made should be
 * accompanied by a comment indicating who made the change, when it was made
 * and what was changed.
 *
 * This software package, or any parts of it, may not be used or in any way
 * re-distributed for commercial purpose without the authors permission.
 * The author keeps the right to decide between of what is commercial and
 * what is non-commercial purpose.
 *
 * Restrictions due to national laws governing the use, import or export of
 * cryptographic software is the responsibility of the software user/importer/
 * exporter to follow.
 *
 *
 *                                              _
 *                                         Stig Ostholm
 *                                         Chalmers University of Technology
 *                                         Department of Computer Engineering
 *                                         S-412 96 Gothenburg
 *                                         Sweden
 *                                         ----------------------------------
 *                                         Email: ostholm@ce.chalmers.se
 *                                         Phone: +46 31 772 1703
 *                                         Fax:   +46 31 772 3663
 */

#include	"local_def.h"
#include	"des.h"
#include	"version.h"

#ifdef HAS_GETTIMEOFDAY
# include	<sys/time.h>
#else  /* HAS_GETTIMEOFDAY */
# ifdef HAS_TIMEB
#  include	<sys/types.h>
#  include	<sys/time.h>
#  include	<sys/timeb.h>
# else /* HAS_TIMEB */
#  include	<sys/types.h>
#  include	<sys/time.h>
# endif /* HAS_TIMEB */
#endif /* HAS_GETTIMEOFDAY */


/*
 * des_random_cblock
 *
 *	The routine returns a random generated DES 64-bit block.
 *
 *	This function is my attempt to make a secure random generator.
 *	Secure and random implies that the genrator should have the
 *	following properties:
 *
 *		1) The prediction of the next value produced should
 *		   not be reduced to any proper subset of values.
 *	
 *		2) The returned value should be evenly distributed for
 *		   any subset of bits at any point in time.
 *
 *
 *	The basic idea behind this algorithm is to, based on cryptographic
 *	methods, generate a set of random bits from smaller set of "real"
 *	random bits. 
 *
 *	The algorithm is based on two DES encryptions used in feed-back
 *	mode combined with a random external disturbance from.
 *
 *		        ext-var------+
 *				     |
 *		      ext-const--+   |		   input block
 *				 |   |			|
 *			 ########|###|##############	|
 *			 #	 V   V		   #	|
 *			 #	 +->XOR--+ 	   #	|
 *	       +->KEY--+-------+ |	 |	   #	|
 *	       |       | #     | +-------+	   #	|
 *	       +--XOR<-+ #     V	 V	   #	V
 *		   ^	 # +->DES--+--->XOR--+-------->XOR
 *		   |	 # | 	   |	     V	   #	|
 *		   |	 # +--XOR<-+        KEY    # 	|
 *		   |	 #     ^	     V	   #	V
 *		   |	 #     |	 +->DES--+---->XOR
 *		   |	 #     |         |	 | #	|
 *		   +-----------+---------+-------+ #	|
 *			 ###########################	|
 *							V
 *						   output block
 *
 *	The use of DES as psedo-random generators fullfills the property 2).
 *	Or, at least, as close as DES manages.
 *
 *	The property 1) is fulfilled by:
 *
 *		1. Starting with an unknown internal state.
 *
 *		2. Hiding the internal state of the generators.
 *
 *		3. Adding a random disturbance to the state.
 *
 *	The unknown initial state is produced by running the algorithm
 *	64 rounds before any output value is produced. If, at least, one of
 *	the external bits is random. DES "distributes" this bit	througout
 *	the whole block, which finally results in <= 64 random bits. The
 *	number of random bits may in fact be > 64 if more than one of the
 *	external bits are random. NOTE: I do not have any proof of this
 *	bevaior. KEY0 is changed for each call of the function by x-oring
 *	with previous value from the seconds DES.
 *
 *	The internal state of the psedu-random generator is hidden by
 *	the final x-or of the input block.
 *
 *	The unpredictable bits causes the alogorithm to produce different
 *	values even from the same previous internal state.
 *
 *
 *	The external bits consists of a combination of unitialized variables
 *	variables, current time & timezone and curernt process id.
 *	The external information is divided into those that are constant
 *	througout the whole function and those that may change. Constant
 *	values are x-ored into the process only once for each function call.
 *	The "chaning" values are x-ored into the process for each round.
 *
 *	The significant bits in the various values have priority according
 *	to the possibility to predict them. The least significant bit of
 *	micro-seconds has, as example, higher priority the the most
 *	significant of seconds. The different values are then combined
 *	togheter with the intention to preserve as many unpredictable bits
 *	with as high priority as possible.
 *	
 *
 *	After the generation of the initial value, the strength of this
 *	alogrithm is that of DES + number of unpredictable bits. It can,
 *	however, be strengthen by running it 64 rounds each time. Setting
 *	the define MAX_STRENGTH causes the algorithm to run 64 rounds for
 *	each call.
 *
 */

/* #define MAX_STRENGTH /**/

static des_cblock	seed0 = {
				0x7a, 0x54, 0xeb, 0xa6, 0x9c, 0x66, 0x14, 0xdd
			};

static des_cblock	seed1 = {
				0x61, 0xa6, 0x6b, 0x7c, 0x13, 0x96, 0x30, 0x0d
			};

static des_cblock	key0 = {
				0x13, 0x87, 0xad, 0x45, 0xea, 0x84, 0x4a, 0x4d
			};


#ifndef MAX_STRENGTH
static int		rounds = DES_BLOCK_BITS;
#endif  /* ! MAX_STRENGTH */


/*
 * External bit generation.
 *
 *	`EXTERNAL_DECL' contains declaration of variables (except the final
 *	64-nit block) used by `EXTERNAL_CONST' and `EXTERNAL_VAR'.
 *
 *	`EXTERNAL_CONST' inserts the call-constant values into the external
 *	bits.
 *
 *	`EXTERNAL_VAR' inserts the call-varibale values into the external
 *	bits.
 */

#define EXTERNAL_DECL register unsigned long	e1, e2;

#ifdef HAS_GETTIMEOFDAY

#define EXTERNAL_CONST { \
		register unsigned long	pid; \
		struct timeval		tv; \
		struct timezone		tz; \
\
		pid ^= (unsigned long) getpid(); /* [0 .. 32768) */ \
		(void) gettimeofday(&tv, &tz); \
		e1 ^= (pid << 17) ^ \
		      (((unsigned long) tz.tz_minuteswest) << 19); /* [0 .. 780] */ \
		e2 ^= pid << 20; \
	}

#define EXTERNAL_VAR(ext) { \
		struct timeval		tv; \
\
		(void) gettimeofday(&tv, (struct timezone *) 0); \
		e1 ^= (unsigned long) tv.tv_sec;	/* [0, 2**31) */ \
		e2 ^= (unsigned long) tv.tv_usec;	/* [0, 1000000) */ \
		LONG_TO_CHAR_8(ext, e1, e2); \
	}

#else  /* HAS_GETTIMEOFDAY */
# ifdef HAS_TIMEB

#define EXTERNAL_CONST { \
		register unsigned long	pid; \
		struct timeb		tb; \
\
		pid ^= (unsigned long) getpid(); /* [0 .. 32768) */ \
		(void) ftime(&tb); \
		e1 ^= pid << 17; \
		e2 ^= pid << 20 ^ \
		      (((unsigned long) tb.timezone) << 10) ^ /* [0 .. 780] */ \
	}

#define EXTERNAL_VAR(ext) { \
		struct timeb		tb; \
\
		(void) ftime(&tb); \
		e1 ^= (unsigned long) tb.time; /* [0, 2**31) */ \
		e2 ^= (unsigned long) tb.millit; /* [0, 1000) */ \
		LONG_TO_CHAR_8(ext, e1, e2); \
	}

# endif /* HAS_TIMEB */

#define EXTERNAL_CONST { \
		register unsigned long	pid; \
\
		pid ^= (unsigned long) getpid(); /* [0 .. 32768) */ \
		e1 ^= pid << 17; \
	}

#define EXTERNAL_VAR(ext) { \
		unsigned long		usec; \
\
		e1 ^= (unsigned long) time(NULL);	/* [0, 2**31) */ \
		e2 ^= usec; /* [0, 2**32) */ \
		LONG_TO_CHAR_8(ext, e1, e2); \
	}

#endif /* HAS_GETTIMEOFDAY */


int	des_random_cblock(
#ifdef __STDC__
	des_cblock *cblock)
#else
	cblock)
des_cblock	*cblock;
#endif
{
	register int		i;
	des_cblock		key1, ext;
	des_key_schedule	schedule;
	EXTERNAL_DECL


	EXTERNAL_CONST

	XOR_8(key0, seed1);
	(void) des_fixup_key_parity((des_cblock *) key0);

#ifdef MAX_STRENGTH
	for (i = 0; i < DES_BLOCK_BITS; i++) {
#else  /* MAX_STRENGTH */
	for (i = rounds; i > 0; i--) {
#endif /* MAX_STRENGTH */

		XOR_8(seed0, seed1);
		(void) des_key_sched((des_cblock *) key0, schedule);
		(void) des_dea((des_cblock *) seed0,
			       (des_cblock *) seed0,
			       schedule, DES_ENCRYPT);

		EXTERNAL_VAR(ext)
		XOR_8(seed0, ext);

		XOR_8(key1, seed0);
		(void) des_fixup_key_parity((des_cblock *) key1);
		(void) des_key_sched((des_cblock *) key1, schedule);
		(void) des_dea((des_cblock *) seed1,
			       (des_cblock *) seed1,
			       schedule, DES_ENCRYPT);
	}
#ifndef MAX_STRENGTH
	rounds = 1;
#endif  /* ! MAX_STRENGTH */

	XOR_8((*cblock), seed0);
	XOR_8((*cblock), seed1);

	return 0;
}

/*
 * des_init_random_number_generator
 *
 *	The routine sets the random number generator to a new value by
 *	doing a x-or with a new value.
 *
 *	NOTE:	This function will be removed or made a noop when
 *		when the random number algorithm has been strengthen.
 *
 */

int	des_init_random_number_generator(
#ifdef __STDC__
	des_cblock *key)
#else
	key)
des_cblock	*key;
#endif
{
	/* X-or the seed value with the key. */
	XOR2_8(seed1, seed1, (*key));

	return 0;
}
