/*
 * dnode.c - BSDI node functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dnode.c,v 1.5 97/10/23 12:16:23 abe Exp $";
#endif


#include "lsof.h"


#if	defined(HASFDESCFS)
_PROTOTYPE(static int lkup_dev_tty,(dev_t *dr, unsigned long *ir));
#endif	/* defined(HASFDESCFS) */

_PROTOTYPE(static void process_lock,(struct lockf *lp));

#if	defined(HASPROCFS)
_PROTOTYPE(static void getmemsz,(pid_t pid));


/*
 * getmemsz() - get memory size of a /proc/<n>/mem entry
 */

static void
getmemsz(pid)
	pid_t pid;
{
	int n;
	struct kinfo_proc *p;
	struct vmspace vm;

	for (n = 0, p = P; n < Np; n++, p++) {
		if (p->P_PID == pid) {
			if (!p->P_VMSPACE
			||  kread((KA_T)p->P_VMSPACE, (char *)&vm, sizeof(vm)))
				return;
			Lf->sz = (SZOFFTYPE)ctob(vm.vm_tsize + vm.vm_dsize
							     + vm.vm_ssize);
			Lf->sz_def = 1;
			return;
		}
	}
}
#endif	/* defined(HASPROCFS) */


#if	defined(HASFDESCFS)
/*
 * lkup_dev_tty() - look up /dev/tty
 */

static int
lkup_dev_tty(dr, ir)
	dev_t *dr;			/* place to return device number */
	unsigned long *ir;		/* place to return inode number */
{
	int i;

	readdev(0);

# if	defined(HASDCACHE)

lkup_dev_tty_again:

# endif	/* defined(HASDCACHE) */

	for (i = 0; i < Ndev; i++) {
	    if (strcmp(Devtp[i].name, "/dev/tty") == 0) {

# if	defined(HASDCACHE)
		if (DCunsafe && !Devtp[i].v && !vfy_dev(&Devtp[i]))
		    goto lkup_dev_tty_again;
# endif	/* defined(HASDCACHE) */

		*dr = Devtp[i].rdev;
		*ir = (unsigned long)Devtp[i].inode;
		return(1);
	    }
	}

# if	defined(HASDCACHE)
	if (DCunsafe) {
	    (void) rereaddev();
	    goto lkup_dev_tty_again;
	}
# endif	/* defined(HASDCACHE) */

	return(-1);
}
#endif	/* defined(HASFDESCFS) */


/*
 * process_lock() - process node lock information
 */

static void
process_lock(lp)
	struct lockf *lp;		/* kernel lockf structure */
{
	struct lockf lf, *lfp;
	int lt;

	if (!lp)
	    return;
/*
 * Determine the lock state.
 */
	lfp = lp;
	do {
	    if (kread((KA_T)lfp, (char *)&lf, sizeof(lf)))
		break;
	    lt = 0;
	    switch (lf.lf_flags & (F_FLOCK|F_POSIX)) {
	    case F_FLOCK:
		if (Cfp && (struct file *)lf.lf_id == Cfp)
		    lt = 1;
		break;
	    case F_POSIX:
		if ((KA_T)lf.lf_id == Kpa)
		    lt = 1;
		break;
	    }
	    if (!lt)
		continue;
	    if (lf.lf_start == (off_t)0 && lf.lf_end == 0xffffffffffffffffLL)
		lt = 1;
	    else
		lt = 0;
	    if (lf.lf_type == F_RDLCK)
		Lf->lock = lt ? 'R' : 'r';
	    else if (lf.lf_type == F_WRLCK)
		Lf->lock = lt ? 'W' : 'w';
	    else if (lf.lf_type == (F_RDLCK | F_WRLCK))
		Lf->lock = 'u';
	    return;
	} while ((lfp = lf.lf_next) && lfp != lp);
}


/*
 * process_node() - process vnode
 */

void
process_node(va)
	caddr_t va;			/* vnode kernel space address */
{
	struct iso_node cd;
	u_int cn;
	dev_t dev;
	struct denode dn;
	char dev_ch[32];
	int lt;
	struct inode i;
	struct mfsnode m;
	struct nfsnode n;
	enum nodetype {NONODE, INODE, MFSNODE, NFSNODE, CDFSNODE, PCFSNODE,
		FDESCNODE, PFSNODE} nty = NONODE;
	char *ty;
	enum vtype type;
	struct vnode *v, vb;
	struct l_vfs *vfs;

#if	defined(HASFDESCFS)
	struct fdescnode f;
	static dev_t f_tty_dev;
	static unsigned long f_tty_ino;
	static int f_tty_s = 0;
#endif	/* defined(HASFDESCFS) */

#if	defined(HASPROCFS)
	struct pfsnode p;
	struct procfsid *pfi;
#endif	/* defined(HASPROCFS) */

/*
 * Read the vnode.
 */
	if ( ! va) {
	    enter_nm("no vnode address");
	    return;
	}
	v = &vb;
	if (readvnode((caddr_t)va, v)) {
	    enter_nm(Namech);
	    return;
        }

# if	defined(HASNCACHE)
	Lf->id = v->v_id;
	Lf->na = (unsigned long)va;
# endif	/* defined(HASNCACHE) */

/*
 * Get the vnode type.
 */
	if (!v->v_mount)
	    vfs = (struct l_vfs *)NULL;
	else {
	    vfs = readvfs(v->v_mount);
	    if (vfs) {

#if	BSDIV<30000
		if (vfs->type == MOUNT_UFS || vfs->type == MOUNT_MFS)
		    Ntype = N_REGLR;
		else if (vfs->type == MOUNT_NFS)
		    Ntype = N_NFS;
		else if (vfs->type == MOUNT_CD9660)
		    Ntype = N_CDFS;
		else if (vfs->type == MOUNT_MSDOS)
		    Ntype = N_PCFS;
		else if (vfs->type == MOUNT_NONE) {
		    if (v->v_type != VNON && v->v_type != VBAD) {
			enter_nm("no file system");
			return;
		    }
		}

# if	defined(HASPROCFS)
		else if (vfs->type == MOUNT_PROCFS)
		    Ntype = N_PROC;
# endif	/* defined(HASPROCFS) */

		else {
		    (void) sprintf(Namech,
			"unsupported file system type: %d", vfs->type);
		    enter_nm(Namech);
		    return;
		}
#else	/* BSDIV>=30000 */
		if (strcasecmp(vfs->typnm, "ufs") == 0
		||  strcasecmp(vfs->typnm, "mfs") == 0)
		    Ntype = N_REGLR;
		else if (strcasecmp(vfs->typnm, "nfs") == 0)
		    Ntype = N_NFS;
		else if (strcasecmp(vfs->typnm, "cd9660") == 0)
		    Ntype = N_CDFS;
		else if (strcasecmp(vfs->typnm, "msdos") == 0)
		    Ntype = N_PCFS;

# if	defined(HASPROCFS)
		else if (strcasecmp(vfs->typnm, "procfs") == 0)
		    Ntype = N_PROC;
# endif	/* defined(HASPROCFS) */

		else {
		    if (vfs->typnm[0]
		    ||  (v->v_type != VNON && v->v_type != VBAD)) {
			(void) sprintf(Namech,
			    "unsupported file system type: %s", vfs->typnm);
			enter_nm(Namech);
			return;
		    }
		}
#endif	/* BSDIV>=30000 */

	    }
	}
	if (!vfs && v->v_type != VNON && v->v_type != VBAD) {
	    enter_nm("no file system");
	    return;
	}
	if (Ntype == N_REGLR) {
	    switch (v->v_type) {
	    case VFIFO:
		Ntype = N_FIFO;
		break;
	    }
	}
/*
 * Read the successor node.
 */
	switch (v->v_tag) {
	case VT_ISOFS:
	    if (kread((KA_T)v->v_data, (char *)&cd, sizeof(cd)) != 0) {
		(void) sprintf(Namech, "can't read cd9660_node at: %x",
		    v->v_data);
		enter_nm(Namech);
		return;
	    }
	    nty = CDFSNODE;
	    (void) process_lock(cd.i_lockf);
	    break;
	case VT_MFS:
	    if (kread((KA_T)v->v_data, (char *)&m, sizeof(m)) != 0) {
		(void) sprintf(Namech, "can't read mfsnode at: %x", v->v_data);
		enter_nm(Namech);
		return;
	    }
	    nty = MFSNODE;
	    break;
	case VT_MSDOSFS:
	    if (kread((KA_T)v->v_data, (char *)&dn, sizeof(dn)) != 0) {
		(void) sprintf(Namech, "can't read denode at: %x", v->v_data);
		enter_nm(Namech);
		return;
	    }
	    nty = PCFSNODE;
	    (void) process_lock(dn.de_lockf);
	    break;
	case VT_NFS:
	    if (kread((KA_T)v->v_data, (char *)&n, sizeof(n)) != 0) {
		(void) sprintf(Namech, "can't read nfsnode at: %x", v->v_data);
		enter_nm(Namech);
		return;
	    }
	    nty = NFSNODE;
	    break;

#if	defined(HASFDESCFS)
	case VT_FDESC:
		if (kread((KA_T)v->v_data, (char *)&f, sizeof(f)) != 0) {
			(void) sprintf(Namech, "can't read fdescnode at: %x",
				v->v_data);
			enter_nm(Namech);
			return;
		}
		nty = FDESCNODE;
		break;
#endif	/* defined(HASFDESCFS) */


#if	defined(HASPROCFS)
	case VT_PROCFS:
	    if (kread((KA_T)v->v_data, (char *)&p, sizeof(p)) != 0) {
		(void) sprintf(Namech, "can't read pfsnode at: %x", v->v_data);
		enter_nm(Namech);
		return;
	    }
	    nty = PFSNODE;
	    break;
#endif	/* defined(HASPROCFS) */

	case VT_UFS:
	    if (kread((KA_T)v->v_data, (char *)&i, sizeof(i)) != 0) {
		(void) sprintf(Namech, "can't read inode at: %x", v->v_data);
		enter_nm(Namech);
		return;
	    }
	    nty = INODE;
	    (void) process_lock(i.i_lockf);
	    break;
	}
/*
 * Get device and type for printing.
 */
	type = v->v_type;
	switch (nty) {
	case INODE:
	    if (type == VCHR || type == VBLK)
		dev = i.i_din.di_rdev;
	    else
		dev = i.i_dev;
	    break;
	case NFSNODE:
	    dev = n.n_vattr.va_fsid;
	    break;
	case CDFSNODE:
	    dev = cd.i_dev;
	    break;
	case PCFSNODE:
	    dev = dn.de_dev;
	}
/*
 * Obtain the inode number.
 */
	switch (nty) {

#if	defined(HASFDESCFS)
	case FDESCNODE:
		if (f.fd_link
		&&  kread((KA_T)f.fd_link, Namech, sizeof(Namech) - 1) == 0)
		{
		    Namech[sizeof(Namech) - 1] = '\0';
		    break;
		}
		if (f.fd_type == Fctty) {
		    if (f_tty_s == 0)
			f_tty_s = lkup_dev_tty(&f_tty_dev, &f_tty_ino);
		    if (f_tty_s == 1) {
			dev = f_tty_dev;
			Lf->inode = f_tty_ino;
			Lf->inp_ty = 1;
		    }
		}
		break;
#endif	/* defined(HASFDESCFS) */

	case INODE:
	    if (type != VBLK) {
		Lf->inode = (unsigned long)i.i_number;
		Lf->inp_ty = 1;
	    }
	    break;
	case NFSNODE:
	    Lf->inode = (unsigned long)n.n_vattr.va_fileid;
	    Lf->inp_ty = 1;
	    break;
	case CDFSNODE:
	    Lf->inode = (unsigned long)cd.i_number;
	    Lf->inp_ty = 1;
	    break;
	case PCFSNODE:
	    if (dn.de_Attributes & ATTR_DIRECTORY) {
		if ((cn = dn.de_StartCluster) == MSDOSFSROOT)
		    cn = 1;
	    } else {
		if ((cn = dn.de_dirclust) == MSDOSFSROOT)
		    cn = 1;
		cn = (cn << 16)

#if	BSDIV<30000
		   | (dn.de_diroffset & 0xffff);
#else	/* BSDIV>=30000 */
		   | ((dn.de_diroffset / sizeof(struct direntry)) & 0xffff);
#endif	/* BSDIV<30000 */

	    }
	    Lf->inode = (unsigned long)cn;
	    Lf->inp_ty = 1;
	    break;

#if	defined(HASPROCFS)
	case PFSNODE:
	    Lf->inode = (unsigned long)p.pfs_fileno;
	    Lf->inp_ty = 1;
	    break;
#endif	/* defined(HASPROCFS) */

	}

/*
 * Obtain the file size.
 */
	if (Foffset)
	    Lf->off_def = 1;
	else {
	    switch (Ntype) {
	    case N_CDFS:
		if (nty == CDFSNODE) {
		    Lf->sz = (SZOFFTYPE)cd.i_size;
		    Lf->sz_def = 1;
		}
		break;
	    case N_FIFO:
		if (!Fsize)
		    Lf->off_def = 1;
		break;
	    case N_PCFS:
		if (nty == PCFSNODE) {
		    Lf->sz = (SZOFFTYPE)dn.de_FileSize;
		    Lf->sz_def = 1;
		}
		break;
	    case N_NFS:
		if (nty == NFSNODE) {
		    Lf->sz = (SZOFFTYPE)n.n_vattr.va_size;
		    Lf->sz_def = 1;
		}
		break;

#if	defined(HASPROCFS)
	    case N_PROC:
		if (nty == PFSNODE) {
		    switch (p.pfs_type) {
		    case Proot:
		    case Pproc:
			Lf->sz = (SZOFFTYPE)DEV_BSIZE;
			Lf->sz_def = 1;
			break;
		    case Pmem:
			(void) getmemsz(p.pfs_pid);
			break;
		    }
		}
		break;
#endif	/* defined(HASPROCFS) */

	    case N_REGLR:
		if (type == VREG || type == VDIR) {
		    if (nty == INODE) {
			Lf->sz = (SZOFFTYPE)i.i_din.di_size;
			Lf->sz_def = 1;
		    } else if (nty == MFSNODE) {
			Lf->sz = (SZOFFTYPE)m.mfs_size;
			Lf->sz_def = 1;
		    }
		} else if ((type == VCHR || type == VBLK) && !Fsize)
		    Lf->off_def = 1;
		break;
	    }
	}
/*
 * Record an NFS file selection.
 */
	if (Ntype == N_NFS && Fnfs)
	    Lf->sf |= SELNFS;
/*
 * Save the file system names.
 */
	if (vfs) {
	    Lf->fsdir = vfs->dir;
	    Lf->fsdev = vfs->fsname;
	}
/*
 * Format the vnode type, and possibly the device name.
 */
	switch (type) {

	case VNON:
	    ty ="VNON";
	    break;
	case VREG:
	case VDIR:
	    ty = (type == VREG) ? "VREG" : "VDIR";
	    Lf->dev_def = 1;
	    Lf->dev = dev;
	    break;
	case VBLK:
	    ty = "VBLK";
	    Lf->dev_def = 1;
	    Lf->dev = dev;
	    Ntype = N_BLK;
	    break;
	case VCHR:
	    ty = "VCHR";
	    Lf->dev = dev;
	    Lf->dev_def = 1;
	    Ntype = N_CHR;
	    break;
	case VLNK:
	    ty = "VLNK";
	    break;

#if	defined(VSOCK)
	case VSOCK:
	    ty = "SOCK";
	    break;
#endif	/* defined(VSOCK) */

	case VBAD:
	    ty = "VBAD";
	    break;
	case VFIFO:
	    Lf->dev = dev;
	    Lf->dev_def = 1;
	    ty = "FIFO";
	    break;
	default:
	    if (type > 9999)
		(void) sprintf(Lf->type, "*%03d", type % 1000);
	    else
		(void) sprintf(Lf->type, "%4d", type);
	    (void) strcpy(Namech, "unknown type");
	    ty = (char *)NULL;
	}
	if (ty)
	    (void) strcpy(Lf->type, ty);
	Lf->ntype = Ntype;
/*
 * Handle some special cases:
 *
 * 	ioctl(fd, TIOCNOTTY) files;
 *	memory node files;
 *	/proc files.
 */

	if (type == VBAD)
	    (void) strcpy(Namech, "(revoked)");
	else if (nty == MFSNODE) {
	    Lf->dev_def = 0;
	    (void) sprintf(Namech, "%#x", m.mfs_baseoff);
	    (void) strcpy(dev_ch, "    memory");
	    enter_dev_ch(dev_ch);
	}

#if	defined(HASPROCFS)
	else if (nty == PFSNODE) {
	    Lf->dev_def = 0;
	    ty = (char *)NULL;
	    (void) sprintf(Namech, "/%s", HASPROCFS);
	    switch (p.pfs_type) {
	    case Proot:
		ty = "PDIR";
		break;
	    case Pproc:
		(void) sprintf(endnm(), "/%d", p.pfs_pid);
		ty = "PDIR";
		break;
	    case Pfile:
		(void) sprintf(endnm(), "/%d/file", p.pfs_pid);
		ty = "PFIL";
		break;
	    case Pmem:
		(void) sprintf(endnm(), "/%d/mem", p.pfs_pid);
		ty = "PMEM";
		break;
	    case Pregs:
		(void) sprintf(endnm(), "/%d/regs", p.pfs_pid);
		ty = "PREG";
		break;
	    case Pfpregs:
		(void) sprintf(endnm(), "/%d/fpregs", p.pfs_pid);
		ty = "PFPR";
		break;
	    case Pctl:
		(void) sprintf(endnm(), "/%d/ctl", p.pfs_pid);
		ty = "PCTL";
		break;
	    case Pstatus:
		(void) sprintf(endnm(), "/%d/status", p.pfs_pid);
		ty = "PSTA";
		break;
	    case Pnote:
		(void) sprintf(endnm(), "/%d/note", p.pfs_pid);
		ty = "PNTF";
		break;
	    case Pnotepg:
		(void) sprintf(endnm(), "/%d/notepg", p.pfs_pid);
		ty = "PGRP";
		break;
	    }
	    if (ty)
		(void) strcpy(Lf->type, ty);
	    enter_nm(Namech);
	}
#endif	/* defined(HASPROCFS) */

#if	defined(HASBLKDEV)
/*
 * If this is a VBLK file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && type == VBLK && Lf->dev_def)
	    find_bl_ino();
#endif	/* defined(HASBLKDEV) */

/*
 * If this is a VCHR file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && type == VCHR && Lf->dev_def)
	    find_ch_ino();
/*
 * Test for specified file.
 */

#if	defined(HASPROCFS)
	if (Ntype == N_PROC) {
	    if (Procsrch)
		Lf->sf |= SELNM;
	    else if (nty == PFSNODE) {
		for (pfi = Procfsid; pfi; pfi = pfi->next) {

# if	defined(HASPINODEN)
		    if (pfi->inode == p.pfs_fileno)
# else	/* !defined(HASPINODEN) */
		    if (pfi->pid == p.pfs_pid)
# endif	/* defined(HASPINODEN) */

		    {
			Lf->sf |= SELNM;
			break;
		    }
		}
	    }
	} else
#endif	/* defined(HASPROCFS) */

	{
	    if (Sfile && is_file_named((char *)NULL, type))
		Lf->sf |= SELNM;
	}
/*
 * Enter name characters.
 */
	if (Namech[0])
	    enter_nm(Namech);
}
