/*
    Filter Manager - Management software for the Drawbridge package
    Copyright (C) 1993 David K. Hess, Douglas Lee Schales, David R. Safford

    Please see the file `COPYING' for the complete copyright notice.

    fm.c - Version 1.0 - 4/21/93
*/

/* fm - Filter Manager 1.0.
 *
 * Reads commands from ~/.fmrc
 *
 * Command set:
 *
 * set (verbose|target|key) <args>
 * load (network|classes|allow|reject) <filename>
 * show (networks|host|class|allow|reject|target|verbose|key) [<args>]
 * query (host|class|allow|reject|stats) [<args>]
 * upload (networks|classes|allow|reject)
 * write
 * quit
 * clear
 * reset
 * release (network|classes|allow|reject) <args>
 * reboot
 * ping
 * newkey <name>
 * makekey <name>
 *     (note that the above command does an implicit set key command)
 */
#include "fm.h"

static char conditions[] = "
		    GNU GENERAL PUBLIC LICENSE
   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION

  0. This License Agreement applies to any program or other work which
contains a notice placed by the copyright holder saying it may be
distributed under the terms of this General Public License.  The
\"Program\", below, refers to any such program or work, and a \"work based
on the Program\" means either the Program or any work containing the
Program or a portion of it, either verbatim or with modifications.  Each
licensee is addressed as \"you\".

  1. You may copy and distribute verbatim copies of the Program's source
code as you receive it, in any medium, provided that you conspicuously and
appropriately publish on each copy an appropriate copyright notice and
disclaimer of warranty; keep intact all the notices that refer to this
General Public License and to the absence of any warranty; and give any
other recipients of the Program a copy of this General Public License
along with the Program.  You may charge a fee for the physical act of
transferring a copy.

  2. You may modify your copy or copies of the Program or any portion of
it, and copy and distribute such modifications under the terms of Paragraph
1 above, provided that you also do the following:

    a) cause the modified files to carry prominent notices stating that
    you changed the files and the date of any change; and

    b) cause the whole of any work that you distribute or publish, that
    in whole or in part contains the Program or any part thereof, either
    with or without modifications, to be licensed at no charge to all
    third parties under the terms of this General Public License (except
    that you may choose to grant warranty protection to some or all
    third parties, at your option).

    c) If the modified program normally reads commands interactively when
    run, you must cause it, when started running for such interactive use
    in the simplest and most usual way, to print or display an
    announcement including an appropriate copyright notice and a notice
    that there is no warranty (or else, saying that you provide a
    warranty) and that users may redistribute the program under these
    conditions, and telling the user how to view a copy of this General
    Public License.

    d) You may charge a fee for the physical act of transferring a
    copy, and you may at your option offer warranty protection in
    exchange for a fee.

Mere aggregation of another independent work with the Program (or its
derivative) on a volume of a storage or distribution medium does not bring
the other work under the scope of these terms.

  3. You may copy and distribute the Program (or a portion or derivative of
it, under Paragraph 2) in object code or executable form under the terms of
Paragraphs 1 and 2 above provided that you also do one of the following:

    a) accompany it with the complete corresponding machine-readable
    source code, which must be distributed under the terms of
    Paragraphs 1 and 2 above; or,

    b) accompany it with a written offer, valid for at least three
    years, to give any third party free (except for a nominal charge
    for the cost of distribution) a complete machine-readable copy of the
    corresponding source code, to be distributed under the terms of
    Paragraphs 1 and 2 above; or,

    c) accompany it with the information you received as to where the
    corresponding source code may be obtained.  (This alternative is
    allowed only for noncommercial distribution and only if you
    received the program in object code or executable form alone.)

Source code for a work means the preferred form of the work for making
modifications to it.  For an executable file, complete source code means
all the source code for all modules it contains; but, as a special
exception, it need not include source code for modules which are standard
libraries that accompany the operating system on which the executable
file runs, or for standard header files or definitions files that
accompany that operating system.

  4. You may not copy, modify, sublicense, distribute or transfer the
Program except as expressly provided under this General Public License.
Any attempt otherwise to copy, modify, sublicense, distribute or transfer
the Program is void, and will automatically terminate your rights to use
the Program under this License.  However, parties who have received
copies, or rights to use copies, from you under this General Public
License will not have their licenses terminated so long as such parties
remain in full compliance.

  5. By copying, distributing or modifying the Program (or any work based
on the Program) you indicate your acceptance of this license to do so,
and all its terms and conditions.

  6. Each time you redistribute the Program (or any work based on the
Program), the recipient automatically receives a license from the original
licensor to copy, distribute or modify the Program subject to these
terms and conditions.  You may not impose any further restrictions on the
recipients' exercise of the rights granted herein.

  7. The Free Software Foundation may publish revised and/or new versions
of the General Public License from time to time.  Such new versions will
be similar in spirit to the present version, but may differ in detail to
address new problems or concerns.

Each version is given a distinguishing version number.  If the Program
specifies a version number of the license which applies to it and \"any
later version\", you have the option of following the terms and conditions
either of that version or of any later version published by the Free
Software Foundation.  If the Program does not specify a version number of
the license, you may choose any version ever published by the Free Software
Foundation.

  8. If you wish to incorporate parts of the Program into other free
programs whose distribution conditions are different, write to the author
to ask for permission.  For software which is copyrighted by the Free
Software Foundation, write to the Free Software Foundation; we sometimes
make exceptions for this.  Our decision will be guided by the two goals
of preserving the free status of all derivatives of our free software and
of promoting the sharing and reuse of software generally.

";

static char warranty[] = "
                            NO WARRANTY

  9. BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE LAW.  EXCEPT WHEN
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
PROVIDE THE PROGRAM \"AS IS\" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED
OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE ENTIRE RISK AS
TO THE QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH YOU.  SHOULD THE
PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY SERVICING,
REPAIR OR CORRECTION.

  10. IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES,
INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING
OUT OF THE USE OR INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED
TO LOSS OF DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY
YOU OR THIRD PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER
PROGRAMS), EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE
POSSIBILITY OF SUCH DAMAGES.

";

static char *version = "@(#)Filter Manager 1.0 - April 6, 1993";

int keySet = NO;
FILE *debug = stderr;
FILE *output = stdout;
FILE *input = stdin;
FILE *messages = stderr;

static unsigned char readBuffer[256];
static char currKeyFilename[256];
static char currAccessTableFilename[256];
static char currRejectTableFilename[256];
static char currAllowTableFilename[256];

static int uploadedAndNoWrite = NO;
static int verboseMode = NO;
static int accessTableLoaded = NO;
static int rejectTableLoaded = NO;
static int allowTableLoaded = NO;
static int numNetworks = 0;

/*
 * Statically allocate the tables for the filtering.
 */
static AddrTableEntry addrTable[MAX_NUM_NETWORKS];
static AccessListTableEntry in[MAX_NUM_ACCESS_LISTS * MAX_NUM_ACCESS_RANGES];
static AccessListTableEntry out[MAX_NUM_ACCESS_LISTS * MAX_NUM_ACCESS_RANGES];
static AccessListTableEntry source[MAX_NUM_ACCESS_LISTS * MAX_NUM_ACCESS_RANGES];
static AccessListTableEntry udp[MAX_NUM_ACCESS_LISTS * MAX_NUM_ACCESS_RANGES];
static RejectTableEntry rejectTable[MAX_NUM_REJECT_ENTRIES];
static AllowTableEntry allowTable[MAX_NUM_ALLOW_ENTRIES];

void handleSet(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    FILE *keyFile;
    struct ether_addr *newTarget;
    struct passwd *pwd;
    struct stat statBuf;
    int result;
    char *currTok;
    char newKeyFilename[256];
        
    currTok = strtok((char *) NULL," \t\n");
    result = tokenize(currTok);

    switch (result) {
	case FMVERBOSE:
	    currTok = strtok((char *) NULL," \t\n");

	    if (currTok != (char *) NULL) {
		if (strcasecmp(currTok,"no") == 0)
		    verboseMode = NO;
		else if (strcasecmp(currTok,"yes") == 0)
		    verboseMode = YES;
		else
		    fprintf(theMessages,"value for verbose must be 'yes' or 'no'\n");
	    }
	    else
		fprintf(theMessages,"value for verbose must be 'yes' or 'no'\n");

	    break;
	case FMTARGET:
	    currTok = strtok((char *) NULL," \t\n");

	    if (currTok && (newTarget = ether_aton(currTok)))
		currTarget = *newTarget;
	    else
		fprintf(theMessages,
			"value for target must be an ethernet address in xx:xx:xx:xx:xx:xx form\n");

	    break;
	case FMKEY:

#ifndef DES
	    fprintf(theMessages,"this installation of fm does not have DES support\n");
	    break;
#endif

	    currTok = strtok((char *) NULL," \t\n");

	    if (!currTok) {
		currKeyFilename[0] = '\0';
		keySet = NO;
		fprintf(theOutput,"switching to insecure mode (DES disabled)\n");
	    }
	    else {
		/*
		 * Build a path to the user's ~/.fmkey.*
		 */
		pwd = getpwuid(getuid());
		strcpy(newKeyFilename,pwd->pw_dir);
		strcat(newKeyFilename,"/.fmkey.");
		strcat(newKeyFilename,currTok);
		
		if (stat(newKeyFilename,&statBuf) == -1)
		    fprintf(theMessages,"%s: %s\n",newKeyFilename,sys_errlist[errno]);
		else {
		    /*
		     * Check the user's .fmkey.* and gripe if it is not in the correct mode.
		     */
		    if ((statBuf.st_mode & 0x1FF) != 0x100) {
			fprintf(theMessages,"the key file %s must be mode 400\n",newKeyFilename);
		    }
		    else {
			/*
			 * Load the key.
			 */ 
			keyFile = fopen(newKeyFilename,"r");
			if (!keyFile) {
			    fprintf(theMessages,"%s: Could not open file\n",newKeyFilename);
			}
			else {
			    if (fread(currKey,sizeof(currKey),1,keyFile) != 1)
				fprintf(theMessages,"%s: Could not read from file\n",newKeyFilename);
			    else {
				keySet = YES;
				strcpy(currKeyFilename,newKeyFilename);
			    }
			    fclose(keyFile);
			}
		    }
		}
	    }
	    break;
	case EMPTY:
	    fprintf(theMessages,"missing set key word\n");
	    break;
	case UNKNOWN:
	default:
	    fprintf(theMessages,"unknown set key word\n");
	    break;
    }
}

void handleShow(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    struct in_addr hostAddr;
    struct hostent *hostEntry;
    unsigned long host;
    unsigned long network;
    int result;
    int i;
    int j;
    int offset;
    int index;
    char *currTok;
        
    currTok = strtok((char *) NULL," \t\n");
    result = tokenize(currTok);

    switch (result) {
	case FMVERBOSE:
	    fprintf(theOutput,"verbose is %s\n",verboseMode ? "yes" : "no");
	    break;
	case FMTARGET:
	    fprintf(theOutput,"target is %s\n",ether_ntoa(&currTarget));
	    break;
	case FMKEY:

#ifndef DES
	    fprintf(theMessages,"this installation of fm does not have DES support\n");
	    break;
#endif

	    if (keySet == YES)
		fprintf(theOutput,"key is %s\n",currKeyFilename);
	    else
		fprintf(theOutput,"key is disabled (insecure mode)\n");
	    break;
	case FMHOST:
	    currTok = strtok((char *) NULL," \t\n");

	    if (!currTok)
		fprintf(theMessages,"missing host address/name\n");
	    else {
		hostEntry = gethostbyname(currTok);

		if (hostEntry != (struct hostent *) NULL)
		    hostAddr.s_addr = *(unsigned long *)hostEntry->h_addr_list[0];
		else {
		    hostAddr.s_addr = inet_addr(currTok);
		    if (hostAddr.s_addr == -1) {
			fprintf(theMessages,"%s: Invalid IP address\n",currTok);
			return;
		    }
		}

		/*
		 * Now that we have the address do the look up.
		 */
		if (IN_CLASSB(hostAddr.s_addr)) {
		    host = hostAddr.s_addr & IN_CLASSB_HOST;
		    network = hostAddr.s_addr & IN_CLASSB_NET;
		}
		else if (IN_CLASSC(hostAddr.s_addr)) {
		    host = hostAddr.s_addr & IN_CLASSC_HOST;
		    network = hostAddr.s_addr & IN_CLASSC_NET;
		}
		else {
		    fprintf(theMessages,"class A and D addresses are not supported\n");
		    return;
		}
		
		for (i = 0;i < numNetworks;++i)
		    if (addrTable[i].network.s_addr == network)
			break;

		if (i != numNetworks)
		    fprintf(theOutput,"host %s is using class %d\n",
			    currTok,
			    addrTable[i].hostTable[host]);
		else
		    fprintf(theMessages,"network %s is not loaded\n",
			    inet_ntoa(&network));
	    }
	    break;
	case FMCLASS:
	    currTok = strtok((char *) NULL," \t\n");

	    if (!currTok)
		fprintf(theMessages,"missing class number\n");
	    else {
		if (accessTableLoaded == NO) {
		    fprintf(theMessages,"no classes file has been loaded\n");
		    return;
		}

		index = atoi(currTok);

		if (index < 0 || index > MAX_NUM_ACCESS_LISTS - 1)
		    fprintf(theMessages,"index must be in range 0-255\n");
		else {
		    fprintf(theOutput,"class file is %s\n\n",currAccessTableFilename);
		    fprintf(theOutput,"    in\t\t    out\t\t  source\t    udp\n");
		    offset = index * MAX_NUM_ACCESS_RANGES;

		    while ((in[offset].begin     != 0 ||
			    out[offset].begin    != 0 ||
			    source[offset].begin != 0 ||
			    udp[offset].begin    != 0) &&
			   offset < (index + 1) * MAX_NUM_ACCESS_RANGES) {
			if (in[offset].begin != 0)
			    fprintf(theOutput,"%5d-%-5d\t",
				    in[offset].begin,
				    in[offset].end);
			else
			    fprintf(theOutput,"\t\t");

			if (out[offset].begin != 0)
			    fprintf(theOutput,"%5d-%-5d\t",
				    out[offset].begin,
				    out[offset].end);
			else
			    fprintf(theOutput,"\t\t");

			if (source[offset].begin != 0)
			    fprintf(theOutput,"%5d-%-5d\t",
				    source[offset].begin,
				    source[offset].end);
			else
			    fprintf(theOutput,"\t\t");

			if (udp[offset].begin != 0)
			    fprintf(theOutput,"%5d-%-5d\t",
				    udp[offset].begin,
				    udp[offset].end);
			else
			    fprintf(theOutput,"\t\t");

			fprintf(theOutput,"\n");
			++offset;
		    }
		}
	    }
	    break;
	case FMALLOW:
	    if (allowTableLoaded == NO) {
		fprintf(theMessages,"no allow file has been loaded\n");
		return;
	    }

	    fprintf(theOutput,"allow file is %s\n\n",
		    currAllowTableFilename);

	    i = 0;
	    while (allowTable[i].network.s_addr != 0 &&
		   i < MAX_NUM_ALLOW_ENTRIES) {

		fprintf(theOutput,"global allow to address %-15s",
			inet_ntoa(allowTable[i].network));
		fprintf(theOutput," with mask %-15s\n",
			inet_ntoa(&allowTable[i].mask));

		fprintf(theOutput,"   ports\n");

		j = 0;
		while (allowTable[i].access[j].begin != 0 &&
		       j < MAX_NUM_ACCESS_RANGES) {
		    fprintf(theOutput,"%5d-%-5d\n",
			    allowTable[i].access[j].begin,
			    allowTable[i].access[j].end);
		    ++j;
		}
		++i;
	    }
	    break;
	case FMREJECT:
	    if (rejectTableLoaded == NO) {
		fprintf(theMessages,"no reject file has been loaded\n");
		return;
	    }

	    fprintf(theOutput,"reject file is %s\n\n",
		    currRejectTableFilename);

	    i = 0;
	    while (rejectTable[i].network.s_addr != 0 &&
		   i < MAX_NUM_REJECT_ENTRIES) {
		fprintf(theOutput,"global reject from address %-15s",
			inet_ntoa(rejectTable[i].network));
		fprintf(theOutput," with mask %-15s\n",
			inet_ntoa(&rejectTable[i].mask));
		++i;
	    }
	    
	    break;
	case FMNETWORKS:
	    if (numNetworks == 0) {
		fprintf(theMessages,"no networks are loaded\n");
		return;
	    }

	    for (i = 0;i < numNetworks;++i)
		fprintf(theOutput,"network %s is loaded from file %s\n",
			inet_ntoa(addrTable[i].network),
			addrTable[i].networkFilename);
	    break;
	case EMPTY:
	    fprintf(theMessages,"missing show key word\n");
	    break;
	case UNKNOWN:
	default:
	    fprintf(theMessages,"unknown show key word\n");
	    break;
    }
}

void handleNewKey(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    FILE *keyFile;
    struct passwd *pwd;
    struct stat statBuf;
    int i;
    unsigned char *p;
    unsigned char newKey[128];
    unsigned char newKeyCopy[128];
    unsigned char newPassword[8];
    unsigned char tempBuf[8];
    char newKeyFilename[256];
    char *currTok;

#ifndef DES
    fprintf(theMessages,"this installation of fm does not have DES support\n");
    return;
#endif

    currTok = strtok((char *) NULL," \t\n");

    if (!currTok) {
	fprintf(theMessages,"missing name\n");
    }
    else {
	/* 
	 * If that succeeded then save the key into the key file.
	 *   Build a path to the user's ~/.fmkey.*
	 */
	pwd = getpwuid(getuid());
	strcpy(newKeyFilename,pwd->pw_dir);
	strcat(newKeyFilename,"/.fmkey.");
	strcat(newKeyFilename,currTok);
	
	if (stat(newKeyFilename,&statBuf) != -1) {
	    /*
	     * Check the user's .fmkey.* and gripe if it is not in the correct mode.
	     */
	    if ((statBuf.st_mode & 0777) != 0400) {
		fprintf(theMessages,"The key file %s must be mode 400\n",newKeyFilename);
		return;
	    }

	    /*
	     * CHMOD the file to 0600.
	     */
	    chmod(newKeyFilename,0600);
	}

	/*
	 * Else create the file.
	 */
	keyFile = fopen(newKeyFilename,"w");
	if (!keyFile) {
	    fprintf(theMessages,"%s: Could not open file\n",newKeyFilename);
	    chmod(newKeyFilename,0400);
	    return;
	}

	/*
	 * Generate the new key.
	 */
	fprintf(theOutput,"Generating new key...");
	while (1) {
	    for (i = 0;i < 8;++i)
		newPassword[i] = (rand() & 0xFF0000) >> 16;

	    if (genMethod((unsigned long *) newKey,newPassword) == 0)
		break;
	}
	fprintf(theOutput,"done\n");

	memcpy(newKeyCopy,newKey,sizeof(newKey));
	longSwap((unsigned long *) newKey,sizeof(newKey) / 4);

	/* 
	 * Now that we have the new key, install it into the filter.
	 *   If we are currently in an encrypted mode, then encrypt
	 *   the new key going across, else send it in plain sight.
	 */
	if (keySet == YES) {
	    for (p = newKey;p < newKey + sizeof(newKey);p += 8) {
		encrypt((unsigned char *) tempBuf,
			(unsigned long *) currKey,
			(unsigned char *) p);
		memcpy(p,tempBuf,8);
	    }
	}

	if (startTransaction() == 0) {
	    if (sendMessage(FM_M_NEWKEY,(void *) newKey,sizeof(newKey)) == (unsigned char *) NULL) {
		fprintf(theMessages,"installation of new key in filter failed\n");
		fclose(keyFile);
		chmod(newKeyFilename,0400);
		return;
	    }
	}
	else {
	    fprintf(theMessages,"installation of new key in filter failed\n");
	    fclose(keyFile);
	    chmod(newKeyFilename,0400);
	    return;
	}

	/*
	 * Copy the new key over to the current key storage.
	 */
	memcpy(currKey,newKeyCopy,sizeof(currKey));
	keySet = YES;
	strcpy(currKeyFilename,newKeyFilename);
	
	/*
	 * Write the key out to the file.
	 */
	if (fwrite(currKey,sizeof(currKey),1,keyFile) != 1)
	    fprintf(theMessages,"%s: Could not write key to file\n",newKeyFilename);

	fclose(keyFile);

	/*
	 * CHMOD the file back to 0400.
	 */
	chmod(newKeyFilename,0400);
    }
}

void handleGenKey(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    FILE *keyFile;
    FILE *filterKeyFile;
    struct passwd *pwd;
    struct stat statBuf;
    int i;
    unsigned char newKey[128];
    unsigned char newPassword[8];
    char *currTok;
    char newKeyFilename[256];
    char filterKeyFilename[256];
        
#ifndef DES
    fprintf(theMessages,"this installation of fm does not have DES support\n");
    return;
#endif

    currTok = strtok((char *) NULL," \t\n");

    if (!currTok)
	fprintf(theMessages,"missing name\n");
    else {
	/*
	 *   Build a path to the user's ~/.fmkey.*
	 */
	pwd = getpwuid(getuid());

	if (pwd == (struct passwd *) NULL) {
	    fprintf(theMessages,"who are you?\n");
	    return;
	}

	strcpy(newKeyFilename,pwd->pw_dir);
	strcat(newKeyFilename,"/.fmkey.");
	strcat(newKeyFilename,currTok);
	
	if (stat(newKeyFilename,&statBuf) != -1) {
	    /*
	     * Check the user's .fmkey.* and gripe if it is not in the correct mode.
	     */
	    if ((statBuf.st_mode & 0x1FF) != 0x100) {
		fprintf(theMessages,"the key file %s must be mode 400\n",newKeyFilename);
		return;
	    }

	    chmod(newKeyFilename,0600);
	}

	/*
	 * Build a path to ~/<name>.fmkey
	 */
	strcpy(filterKeyFilename,pwd->pw_dir);
	strcat(filterKeyFilename,"/des.key");
	if (stat(filterKeyFilename,&statBuf) != -1) {
	    /*
	     * Check the des.key and gripe if it is not in the correct mode.
	     */
	    if ((statBuf.st_mode & 0x1FF) != 0x100) {
		fprintf(theMessages,"The filter's key file %s must be mode 400\n",filterKeyFilename);
		chmod(newKeyFilename,0400);
		return;
	    }

	    chmod(filterKeyFilename,0600);
	}

	/*
	 * create the key file.
	 */
	keyFile = fopen(newKeyFilename,"w");
	if (!keyFile) {
	    fprintf(theMessages,"%s: Could not open file\n",newKeyFilename);
	    chmod(newKeyFilename,0400);
	    chmod(filterKeyFilename,0400);
	    return;
	}

	/*
	 * create the filter key file.
	 */
	filterKeyFile = fopen(filterKeyFilename,"w");
	if (!filterKeyFile) {
	    fprintf(theMessages,"%s: Could not open file\n",filterKeyFilename);
	    fclose(keyFile);
	    chmod(newKeyFilename,0400);
	    chmod(filterKeyFilename,0400);
	    return;
	}

	/*
	 * Generate the new key.
	 */
	fprintf(theOutput,"Generating new key...");
	while (1) {
	    for (i = 0;i < 8;++i)
		newPassword[i] = (rand() & 0xFF0000) >> 16;

	    if (genMethod((unsigned long *) newKey,newPassword) == 0)
		break;
	}
	fprintf(theOutput,"done\n");

	if (fwrite(newKey,sizeof(newKey),1,keyFile) != 1) {
	    fprintf(theMessages,"%s: Could not write key to file\n",newKeyFilename);
	}

	fclose(keyFile);
	chmod(newKeyFilename,0400);

	longSwap((unsigned long *) newKey,sizeof(newKey) / 4);
	
	if (fwrite(newKey,sizeof(newKey),1,filterKeyFile) != 1) {
	    fprintf(theMessages,"%s: Could not write key to file\n",filterKeyFilename);
	}

	fclose(filterKeyFile);
	chmod(filterKeyFilename,0400);

	fprintf(theOutput,"wrote the filter's DES key to '%s'\n",filterKeyFilename);
    }
}

char *tildeExpand(char *filename,FILE *theMessages)
{
    struct passwd *pwd;
    static char resultFilename[256];
    char username[256];
    char *slash;

    strcpy(resultFilename,filename);

    if (filename[0] == '~') {
	slash = strchr(filename + 1,'/');

	if (!slash) {
	    fprintf(theMessages,"invalid filename\n");
	    return (char *) NULL;
	}
	else {
	    strncpy(username,filename + 1,slash - (filename + 1));
	    username[slash - (filename + 1)] = '\0';

	    if (strlen(username) == 0) {
		/*
		 * Just '~'.
		 */
		pwd = getpwuid(getuid());

		if (pwd == (struct passwd *) NULL) {
		    fprintf(theMessages,"who are you?\n");
		    return (char *) NULL;
		}
	    }
	    else {
		/*
		 * '~<username>'
		 */
		pwd = getpwnam(username);

		if (pwd == (struct passwd *) NULL) {
		    fprintf(theMessages,"unknown user %s\n",username);
		    return (char *) NULL;
		}
	    }

	    strcpy(resultFilename,pwd->pw_dir);
	    strcat(resultFilename,slash);
	}
    }

    return resultFilename;
}

void handleLoad(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    FILE *dataFile;
    struct in_addr network;
    unsigned long size;
    int result;
    char *currTok;
    char *expandedFilename;
        
    currTok = strtok((char *) NULL," \t\n");
    result = tokenize(currTok);

    switch (result) {
	case FMNETWORK:
	    currTok = strtok((char *) NULL," \t\n");
	    if (!currTok)
		fprintf(theMessages,"load network requires a filename\n");
	    else {
		expandedFilename = tildeExpand(currTok,theMessages);

		if (expandedFilename) {

		    dataFile = fopen(expandedFilename,"r");

		    if (!dataFile) {
			fprintf(theMessages,"%s: Could not open network file\n",currTok);
			return;
		    }
		    
		    if (fread((char *)&network,sizeof(struct in_addr),1,dataFile) != 1) {
			fprintf(theMessages,"%s: Could not read network address from network file\n",currTok);
			fclose(dataFile);
			return;
		    }
		    
		    if (IN_CLASSB(network.s_addr)) {
			network.s_addr = network.s_addr & IN_CLASSB_NET;
			size = 0x10000;
		    }
		    else if (IN_CLASSC(network.s_addr)) {
			network.s_addr = network.s_addr & IN_CLASSC_NET;
			size = 0x100;
		    }
		    else {
			fprintf(theMessages,
				"version %s does not handle class A or D addresses\n",
				VERSION);
			fclose(dataFile);
			return;
		    }
		    
		    addrTable[numNetworks].network = network;
		    addrTable[numNetworks].hostTable = (unsigned char *) malloc(size);	

		    if (addrTable[numNetworks].hostTable == (unsigned char *) NULL) {
			fprintf(theMessages,"could not allocate memory for the network\n");
			fclose(dataFile);
			return;
		    }

		    strcpy(addrTable[numNetworks].networkFilename,currTok);

		    if (fread((char *)addrTable[numNetworks].hostTable,size,1,dataFile) != 1) {
			fprintf(theMessages,"%s: Could not read network table from network file\n",currTok);
			free(addrTable[numNetworks].hostTable);
			fclose(dataFile);
			return;
		    }
		    
		    ++numNetworks;
		    
		    fclose(dataFile);
		    fprintf(theOutput,"loaded network %s\n",inet_ntoa(network));
		}
	    }
	    break;
	case FMCLASSES:
	    currTok = strtok((char *) NULL," \t\n");
	    if (!currTok) {
		fprintf(theMessages,"load classes requires a filename\n");
	    }
	    else {
		expandedFilename = tildeExpand(currTok,theMessages);

		if (expandedFilename) {

		    dataFile = fopen(expandedFilename,"r");

		    if (!dataFile) {
			fprintf(theMessages,"%s: Could not open classes file\n",currTok);
			return;
		    }
	
		    currAccessTableFilename[0] = '\0';
		    accessTableLoaded = NO;

		    if (fread((char *) in,sizeof(in),1,dataFile) != 1) {
			fprintf(theMessages,"%s: Could not read in table from classes file\n",currTok);
			fclose(dataFile);
			return;
		    }

		    if (fread((char *) out,sizeof(out),1,dataFile) != 1) {
			fprintf(theMessages,"%s: Could not read out table from classes file\n",currTok);
			fclose(dataFile);
			return;
		    }

		    if (fread((char *) source,sizeof(source),1,dataFile) != 1) {
			fprintf(theMessages,"%s: Could not read source table from classes file\n",currTok);
			fclose(dataFile);
			return;
		    }
		    
		    if (fread((char *) udp,sizeof(udp),1,dataFile) != 1) {
			fprintf(theMessages,"%s: Could not read udp table from classes file\n",currTok);
			fclose(dataFile);
			return;
		    }
		    
		    fclose(dataFile);

		    accessTableLoaded = YES;
		    strcpy(currAccessTableFilename,currTok);
		}
	    }
	    break;
	case FMALLOW:
	    currTok = strtok((char *) NULL," \t\n");
	    if (!currTok) {
		fprintf(theMessages,"load allow requires a filename\n");
	    }
	    else {
		expandedFilename = tildeExpand(currTok,theMessages);

		if (expandedFilename) {

		    dataFile = fopen(expandedFilename,"r");

		    if (!dataFile) {
			fprintf(theMessages,"%s: Could not open allow file\n",currTok);
			return;
		    }

		    currAllowTableFilename[0] = '\0';
		    allowTableLoaded = NO;
		    memset((char *)allowTable,0,sizeof(allowTable));

		    if (fread((char *)allowTable,
			      sizeof(AllowTableEntry),
			      MAX_NUM_ALLOW_ENTRIES,
			      dataFile) <= 0) {
			fprintf(theMessages,"%s: Error reading allow table\n",currTok);
			fclose(dataFile);
			return;
		    }

		    fclose(dataFile);
		    allowTableLoaded = YES;
		    strcpy(currAllowTableFilename,currTok);
		}
	    }
	    break;
	case FMREJECT:
	    currTok = strtok((char *) NULL," \t\n");
	    if (!currTok) {
		fprintf(theMessages,"load reject requires a filename\n");
	    }
	    else {
		expandedFilename = tildeExpand(currTok,theMessages);

		if (expandedFilename) {

		    dataFile = fopen(expandedFilename,"r");
		    
		    if (!dataFile) {
			fprintf(theMessages,"%s: Could not open reject file\n",currTok);
			return;
		    }

		    currRejectTableFilename[0] = '\0';
		    rejectTableLoaded = NO;
		    memset((char *)rejectTable,0,sizeof(rejectTable));

		    if (fread((char *)rejectTable,
			      sizeof(RejectTableEntry),
			      MAX_NUM_REJECT_ENTRIES,
			      dataFile) <= 0) {
			fprintf(theMessages,"%s: Error reading reject table\n",currTok);
			fclose(dataFile);
			return;
		    }
		    
		    fclose(dataFile);
		    rejectTableLoaded = YES;
		    strcpy(currRejectTableFilename,currTok);
		}
	    }
	    break;
	case EMPTY:
	    fprintf(theMessages,"missing load key word\n");
	    break;
	case UNKNOWN:
	default:
	    fprintf(theMessages,"unknown load key word\n");
	    break;
    }
}

void handleQuery(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    QueryPacket query;
    AccessListTableEntry *filterIn;
    AccessListTableEntry *filterOut;
    AccessListTableEntry *filterSrc;
    AccessListTableEntry *filterUdp;
    AllowTableEntry *filterAllow;
    RejectTableEntry *filterReject;
    struct in_addr hostAddr;
    struct in_addr *networks;
    struct hostent *hostEntry;
    int result;
    int i;
    int j;
    int offset;
    int index;
    char *currTok;
    unsigned char *ack;
        
    currTok = strtok((char *) NULL," \t\n");
    result = tokenize(currTok);

    switch (result) {
	case FMHOST:
	    currTok = strtok((char *) NULL," \t\n");

	    if (!currTok) {
		fprintf(theMessages,"missing host address/name\n");
	    }
	    else {
		hostEntry = gethostbyname(currTok);

		if (hostEntry != (struct hostent *) NULL) {
		    hostAddr.s_addr = ntohl(*(unsigned long *)hostEntry->h_addr_list[0]);
		}
		else {
		    hostAddr.s_addr = inet_addr(currTok);
		    if (hostAddr.s_addr == -1) {
			fprintf(theMessages,"%s: Invalid IP address\n",currTok);
			return;
		    }
		}

		/*
		 * Now that we have the address do the look up.
		 */
		if (IN_CLASSA(hostAddr.s_addr) || IN_CLASSD(hostAddr.s_addr)) {
		    fprintf(theMessages,"class A and D addresses are not supported\n");
		    return;
		}
		
		/*
		 * Send the request over to the filter.
		 */
		memset((char *) &query,0,sizeof(query));

		query.type = FM_QUERY_HOST;
		query.queryValue.addr = hostAddr.s_addr;

		/*
		 * Swap the address around for the PC (barf!).
		 */
		longSwap((unsigned long *)&query.queryValue.addr,1);
		
		if (startTransaction() == 0) {

		    ack = sendMessage(FM_M_QUERY,(void *) &query,sizeof(query));

		    if (ack == (unsigned char *) NULL) {
			fprintf(theMessages,"query failed\n");
			return;
		    }
		}
		else {
		    fprintf(theMessages,"query failed\n");
		    return;
		}

		fprintf(theOutput,"host %s is using class %d\n",
			currTok,
			((QueryPacket *) (ack + sizeof(FiltHeader)))->queryResult.index);
	    }
	    break;
	case FMCLASS:
	    currTok = strtok((char *) NULL," \t\n");

	    if (!currTok) {
	        fprintf(theMessages,"missing class number\n");
	    }
	    else {
		index = atoi(currTok);
		
		if (index < 0 || index > MAX_NUM_ACCESS_LISTS - 1) {
		    fprintf(theMessages,
			    "index must be in range 0 - %d\n",
			    MAX_NUM_ACCESS_LISTS - 1);
		    return;
		}
		else {
		    memset((char *) &query,0,sizeof(query));

		    query.type = FM_QUERY_CLASS;
		    query.queryValue.index = index;
		    
		    if (startTransaction() == 0) {
			ack = sendMessage(FM_M_QUERY,(void *) &query,sizeof(query));

			if (ack == (unsigned char *) NULL) {
			    fprintf(theMessages,"query failed\n");
			    return;
			}
		    }
		    else {
			fprintf(theMessages,"query failed\n");
			return;
		    }
		    
		    /* 
		     * Make copies that are easier to reference and less expensive to
		     *   compute.
		     */    
		    filterIn = ((QueryPacket *) (ack + sizeof(FiltHeader)))->queryResult.accessList.in;
		    filterOut = ((QueryPacket *) (ack + sizeof(FiltHeader)))->queryResult.accessList.out;
		    filterSrc = ((QueryPacket *) (ack + sizeof(FiltHeader)))->queryResult.accessList.src;
		    filterUdp = ((QueryPacket *) (ack + sizeof(FiltHeader)))->queryResult.accessList.udp;

		    /* Byte swap the table. Since the tables are in
		     *    consecutive order in the packet, we can swap them all at the
		     *    same time. 
		     */    
		    shortSwap((unsigned short *)filterIn,
			      MAX_NUM_ACCESS_RANGES * 
			      sizeof(AccessListTableEntry) * 4);

		    fprintf(theOutput,"    in\t\t    out\t\t  source\t    udp\n");
		    offset = 0;
		    
		    while ((filterIn[offset].begin  != 0 ||
			    filterOut[offset].begin != 0 ||
			    filterSrc[offset].begin != 0 ||
			    filterUdp[offset].begin != 0) &&
			   offset < MAX_NUM_ACCESS_RANGES) {
			if (filterIn[offset].begin != 0)
			    fprintf(theOutput,"%5d-%-5d\t",
				    filterIn[offset].begin,
				    filterIn[offset].end);
			else
			    fprintf(theOutput,"\t\t");
			if (filterOut[offset].begin != 0)
			    fprintf(theOutput,"%5d-%-5d\t",
				    filterOut[offset].begin,
				    filterOut[offset].end);
			else
			    fprintf(theOutput,"\t\t");
			if (filterSrc[offset].begin != 0)
			    fprintf(theOutput,"%5d-%-5d\t",
				    filterSrc[offset].begin,
				    filterSrc[offset].end);
			else
			    fprintf(theOutput,"\t\t");
			if (filterUdp[offset].begin != 0)
			    fprintf(theOutput,"%5d-%-5d\t",
				    filterUdp[offset].begin,
				    filterUdp[offset].end);
			else
			    fprintf(theOutput,"\t\t");
			fprintf(theOutput,"\n");
			++offset;
		    }
		}
	    }
	    break;
	case FMALLOW:
	    memset((char *) &query,0,sizeof(query));

	    query.type = FM_QUERY_ALLOW;
	    
	    if (startTransaction() == 0) {
		ack = sendMessage(FM_M_QUERY,(void *) &query,sizeof(query));

		if (ack == (unsigned char *) NULL) {
		    fprintf(theMessages,"query failed\n");
		    return;
		}
	    }
	    else {
		fprintf(theMessages,"query failed\n");
		return;
	    }
	    
	    filterAllow = ((QueryPacket *) (ack + sizeof(FiltHeader)))->queryResult.allow;

	    i = 0;

	    while (filterAllow[i].network.s_addr != 0L &&
		   i < MAX_NUM_ALLOW_ENTRIES) {

		longSwap((unsigned long *)(filterAllow + i),2);

		fprintf(theOutput,"global allow to address %-15s",
			inet_ntoa(filterAllow[i].network));
		fprintf(theOutput," with mask %-15s\n",
			inet_ntoa(&filterAllow[i].mask));

		fprintf(theOutput,"   ports\n");

		j = 0;
		shortSwap((unsigned short *)filterAllow[i].access,
			  sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES / 2);

		while (filterAllow[i].access[j].begin != 0 &&
		       j < MAX_NUM_ACCESS_RANGES) {
		    fprintf(theOutput,"%5d-%-5d\n",
			    filterAllow[i].access[j].begin,
			    filterAllow[i].access[j].end);
		    ++j;
		}

		++i;
	    }
	    
	    break;
	case FMREJECT:
	    memset((char *) &query,0,sizeof(query));
	    query.type = FM_QUERY_REJECT;
	    
	    if (startTransaction() == 0) {
		ack = sendMessage(FM_M_QUERY,(void *) &query,sizeof(query));

		if (ack == (unsigned char *) NULL) {
		    fprintf(theMessages,"query failed\n");
		    return;
		}
	    }
	    else {
		fprintf(theMessages,"query failed\n");
		return;
	    }
	    
	    filterReject = ((QueryPacket *) (ack + sizeof(FiltHeader)))->queryResult.reject;

	    longSwap((unsigned long *) filterReject,
		     sizeof(RejectTableEntry) * MAX_NUM_REJECT_ENTRIES);

	    i = 0;
	    while (filterReject[i].network.s_addr != 0L &&
		   i < MAX_NUM_REJECT_ENTRIES) {
		fprintf(theOutput,"global reject from address %-15s",
			inet_ntoa(filterReject[i].network));
		fprintf(theOutput," with mask %-15s\n",
			inet_ntoa(&filterReject[i].mask));
		++i;
	    }
	    break;
	case FMNETWORKS:
	    memset((char *) &query,0,sizeof(query));
	    query.type = FM_QUERY_NETWORK;

	    /*
	     * Get the list from the filter.
	     */
	    if (startTransaction() == 0) {
		ack = sendMessage(FM_M_QUERY,(void *) &query,sizeof(query));

		if (ack == (unsigned char *) NULL) {
		    fprintf(theMessages,"query failed\n");
		    return;
		}
	    }
	    else {
		fprintf(theMessages,"query failed\n");
		return;
	    }

	    networks = ((QueryPacket *) (ack + sizeof(FiltHeader)))->queryResult.networks;
	    longSwap((unsigned long *) networks,MAX_NUM_NETWORKS);

	    i = 0;
	    while (networks[i].s_addr != 0L) {
		fprintf(theOutput,"network %15s is loaded\n",
			inet_ntoa(networks[i]));
		++i;
	    }

	    break;
	case EMPTY:
	    fprintf(theMessages,"missing query key word\n");
	    break;
	case UNKNOWN:
	default:
	    fprintf(theMessages,"unknown query key word\n");
	    break;
    }
}

void handleUpload(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    LoadPacket load;
    unsigned long j;
    unsigned long size;
    int result;
    int i;
    char *currTok;
        
    currTok = strtok((char *) NULL," \t\n");
    result = tokenize(currTok);

    switch (result) {
	case FMCLASSES:
	    if (accessTableLoaded == NO) {
		fprintf(theMessages,"no class table has been loaded\n");
		return;
	    }

	    /*
	     * Send the classes across one index at a time.
	     */
	    if (startTransaction() != 0) {
		fprintf(theMessages,"upload of classes failed\n");
	    }
	    else {

		load.type = FM_LOAD_CLASS;
		    
		for (i = 0;i < MAX_NUM_ACCESS_LISTS;++i) {
		    
		    /*
		     * Set the flags.
		     */
		    load.flags = 0;

		    if (i == 0)
			load.flags = FM_LOAD_FLAGS_BEGIN;

		    if (i == MAX_NUM_ACCESS_LISTS - 1)
			load.flags = FM_LOAD_FLAGS_END;

		    /*
		     * Set the index.
		     */
		    load.loadValue.index = i;

		    /*
		     * Copy the access lists to the packet.
		     */
		    memcpy((unsigned char *) (load.loadData.accessList.in),
			   (unsigned char *) (in + i * MAX_NUM_ACCESS_RANGES),
			   sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES);
		    memcpy((unsigned char *) load.loadData.accessList.out,
			   (unsigned char *) (out + i * MAX_NUM_ACCESS_RANGES),
			   sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES);
		    memcpy((unsigned char *) load.loadData.accessList.src,
			   (unsigned char *) (source + i * MAX_NUM_ACCESS_RANGES),
			   sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES);
		    memcpy((unsigned char *) load.loadData.accessList.udp,
			   (unsigned char *) (udp + i * MAX_NUM_ACCESS_RANGES),
			   sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES);

		    /*
		     * Byte swap 'em.
		     */
		    shortSwap((unsigned short *) load.loadData.accessList.in,
			      sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES * 4);

		    /*
		     * Send the packet.
		     */
		    if (sendMessage(FM_M_LOAD,(unsigned char *)&load,sizeof(load)) ==
			(unsigned char *) NULL) {
			fprintf(theMessages,"upload of classes table failed\n");
			break;
		    }
		}
		uploadedAndNoWrite = YES;
	    }

	    break;
	case FMALLOW:

	    if (allowTableLoaded == NO) {
		fprintf(theMessages,"no allow table has been loaded\n");
		return;
	    }

	    /*
	     * Send the allow table across.
	     */
	    if (startTransaction() != 0) {
		fprintf(theMessages,"upload of allow table failed\n");
	    }
	    else {
		load.type = FM_LOAD_ALLOW;
		    
		memcpy((unsigned char *) load.loadData.allow,
		       (unsigned char *) allowTable,
		       sizeof(allowTable));

		/* 
		 * Byte swap it.
		 */
		for (i = 0;i < MAX_NUM_ALLOW_ENTRIES;++i) {
		    longSwap((unsigned long *)&load.loadData.allow[i].network,2);
		    shortSwap((unsigned short *)load.loadData.allow[i].access,
			      sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES / 2);
		}

		if (sendMessage(FM_M_LOAD,(unsigned char *)&load,sizeof(load)) ==
		    (unsigned char *) NULL) {
		    fprintf(theMessages,"upload of allow table failed\n");
		}

		uploadedAndNoWrite = YES;
	    }

	    break;
	case FMREJECT:
	    if (rejectTableLoaded == NO) {
		fprintf(theMessages,"no reject table has been loaded\n");
		return;
	    }

	    /*
	     * Send the reject table across.
	     */
	    if (startTransaction() != 0) {
		fprintf(theMessages,"upload of reject table failed\n");
	    }
	    else {
		load.type = FM_LOAD_REJECT;
		    
		memcpy((unsigned char *) load.loadData.reject,
		       (unsigned char *) rejectTable,
		       sizeof(rejectTable));

		/*
		 * Byte swap it.
		 */
		longSwap((unsigned long *) load.loadData.reject,
			 sizeof(rejectTable) / 4);

		if (sendMessage(FM_M_LOAD,(unsigned char *)&load,sizeof(load)) ==
                    (unsigned char *) NULL) {
		    fprintf(theMessages,"upload of reject table failed\n");
		}
		
		uploadedAndNoWrite = YES;
	    }
	    
	    break;
	case FMNETWORKS:
	    if (numNetworks == 0) {
		fprintf(theMessages,"no networks are loaded\n");
		return;
	    }

	    /*
	     * Send all of the currently loaded networks across.
	     */
	    if (startTransaction() != 0) {
		fprintf(theMessages,"upload of networks failed\n");
	    }
	    else {
		load.type = FM_LOAD_NETWORK;
		    
		for (i = 0;i < numNetworks;++i) {

		    if (IN_CLASSC(addrTable[i].network.s_addr))
			size = 0x100;
		    else
			size = 0x10000;

		    for (j = 0;j < size;j += 1024) {
			
			/*
			 * Set the flags.
			 */
			load.flags = 0;
			    
			if (j == 0)
			    load.flags = FM_LOAD_FLAGS_BEGIN;
			
			if ((size == 0x10000 && j == size - 1024) ||
			    size == 0x100)
			    load.flags |= FM_LOAD_FLAGS_END;
			
			/*
			 * Set the network and offset.
			 */
			load.loadValue.networkBlock.network = addrTable[i].network;
			load.loadValue.networkBlock.offset = j;
			
			longSwap((unsigned long *) &load.loadValue.networkBlock.network,2);

			/*
			 * Copy the network block to the packet.
			 */
			memcpy((unsigned char *) (load.loadData.networkBlock),
			       (unsigned char *) (addrTable[i].hostTable + j),
			       1024);

			/*
			 * Send the packet.
			 */
			if (sendMessage(FM_M_LOAD,(unsigned char *)&load,sizeof(load)) ==
			    (unsigned char *) NULL) {
			    fprintf(theMessages,"upload of networks failed\n");
			    uploadedAndNoWrite = YES;
			    return;
			}
		    }
		    fprintf(theOutput,"uploaded network %s\n",
			    inet_ntoa(addrTable[i].network));
		}
		uploadedAndNoWrite = YES;
	    }

	    break;
	case EMPTY:
	    fprintf(theMessages,"missing upload key word\n");
	    break;
	case UNKNOWN:
	default:
	    fprintf(theMessages,"unknown upload key word\n");
	    break;
    }
}

void handleWrite(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    if (startTransaction() == 0) {
	if (sendMessage(FM_M_WRITE,(unsigned char *) NULL,0) ==
	    (unsigned char *) NULL) {
	    fprintf(theMessages,"write failed\n");
	}
	else
	    uploadedAndNoWrite = NO;
    }
    else {
	fprintf(theMessages,"write failed\n");
    }
}

void handleClear(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    int i;

    /* 
     * Clear memory.
     */
    accessTableLoaded = NO;
    rejectTableLoaded = NO;
    allowTableLoaded  = NO;

    for (i = 0;i < numNetworks;++i) {
	if (addrTable[i].network.s_addr != 0)
	    free(addrTable[i].hostTable);

	addrTable[i].network.s_addr = 0L;
    }

    numNetworks = 0;
}

void handleReset(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    int i;

    /* 
     * Reinitialize global variables.
     */
    accessTableLoaded  = NO;
    rejectTableLoaded  = NO;
    allowTableLoaded   = NO;
    uploadedAndNoWrite = NO;
    verboseMode        = NO;
    keySet             = NO;

    /* 
     * Clean out each loaded network.
     */
    for (i = 0;i < numNetworks;++i) {
	if (addrTable[i].network.s_addr != 0L)
	    free(addrTable[i].hostTable);

	addrTable[i].network.s_addr = 0L;
    }

    numNetworks = 0;

    /* 
     * Re-read ~/.fmrc
     */
    readRc();
}

void handleRelease(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    ReleasePacket release;
    struct in_addr network;
    int result;
    char *currTok;
        
    currTok = strtok((char *) NULL," \t\n");
    result = tokenize(currTok);

    switch (result) {
	case FMCLASSES:
	    /*
	     * Send the message to release the classes.
	     */
	    if (startTransaction() == 0) {
		release.type = FM_RELEASE_CLASSES;

		if (sendMessage(FM_M_RELEASE,(unsigned char *)&release,
			    sizeof(release)) == (unsigned char *) NULL) {
		    fprintf(theMessages,"release of classes table failed\n");
		}
	    }
	    else {
		fprintf(theMessages,"release of classes table failed\n");
	    }

	    break;
	case FMALLOW:
	    /*
	     * Send the allow table release across.
	     */
	    if (startTransaction() == 0) {
		release.type = FM_RELEASE_ALLOW;

		if (sendMessage(FM_M_RELEASE,(unsigned char *)&release,
			    sizeof(release)) == (unsigned char *) NULL) {
		    fprintf(theMessages,"release of allow table failed\n");
		}
	    }
	    else {
		fprintf(theMessages,"release of allow table failed\n");
	    }

	    break;
	case FMREJECT:
	    /*
	     * Send the reject table release across.
	     */
	    if (startTransaction() == 0) {
		release.type = FM_RELEASE_REJECT;

		if (sendMessage(FM_M_RELEASE,(unsigned char *)&release,
			    sizeof(release)) == (unsigned char *) NULL) {
		    fprintf(theMessages,"release of reject table failed\n");
		}
	    }
	    else {
		fprintf(theMessages,"release of reject table failed\n");
	    }

	    break;
	case FMNETWORK:
	    currTok = strtok((char *) NULL," \t\n");

	    if (!currTok) {
		fprintf(theMessages,"release network missing network argument\n");
	    }
	    else {
		network.s_addr = inet_addr(currTok);

		if (network.s_addr == -1) {
		    fprintf(theMessages,"%s: Invalid IP network\n",currTok);
		    break;
		}
		
		/*
		 * Now that we have the address do the look up.
		 */
		if (IN_CLASSA(network.s_addr) || IN_CLASSD(network.s_addr)) {
		    fprintf(theMessages,"class A and D addresses are not supported\n");
		    break;
		}
		
		/*
		 * Send the release request across.
		 */
		if (startTransaction() != 0) {
		    fprintf(theMessages,"release of network %s failed\n",currTok);
		}
		else {
		    release.type = FM_RELEASE_NETWORK;
		    release.network = network;
		    longSwap((unsigned long *) &release.network,1);
		    
		    if (sendMessage(FM_M_RELEASE,(unsigned char *)&release,
				sizeof(release)) == (unsigned char *) NULL) {
			fprintf(theMessages,"release of network %s failed\n",currTok);
		    }
		}
	    }

	    break;
	case EMPTY:
	    fprintf(theMessages,"missing release key word\n");
	    break;
	case UNKNOWN:
	default:
	    fprintf(theMessages,"unknown release key word\n");
	    break;
    }
}

void handleReboot(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    if (startTransaction() != 0)
	fprintf(theMessages,"reboot failed\n");
    else {
	if (sendMessage(FM_M_REBOOT,(void *) NULL,0) == (unsigned char *) NULL) {
	    fprintf(theMessages,"reboot failed\n");
	}
    }
}

void handleHelp(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    int result;
    char *currTok;
        
    currTok = strtok((char *) NULL," \t\n");
    result = tokenize(currTok);

    switch (result) {
	case FMSET:
	    fprintf(theOutput,"\tset (verbose|target|key) <args>\n");
	    fprintf(theOutput,"\nSets the specified variable with the given value.\n");
	    fprintf(theOutput,"Examples: set verbose yes\n");
	    fprintf(theOutput,"          set target 01:02:03:a0:b0:c0\n");
	    break;
	case FMLOAD:
	    fprintf(theOutput,"\tload (network|classes|allow|reject) <filename>\n");
	    fprintf(theOutput,"\nLoads the specified type of data from <filename>. Note\n");
	    fprintf(theOutput,"that multiple networks can be loaded but only one classes,\n");
	    fprintf(theOutput,"allow and reject table can be loaded at a time.\n");
	    fprintf(theOutput,"Examples: load network ~/fm/129.43.0.0\n");
	    fprintf(theOutput,"          load classes /networks/classes\n");
	    break;
	case FMSHOW:
	    fprintf(theOutput,"\tshow (host|class|allow|reject|target|verbose|key) [<args>]\n");
	    fprintf(theOutput,"\nPerforms look ups in the tables currently loaded into fm.\n");
	    fprintf(theOutput,"Also displays variable values.\n");
	    fprintf(theOutput,"Examples: show host athena.mit.edu\n");
	    fprintf(theOutput,"          show target\n");
	    fprintf(theOutput,"          show class 18\n");
	    break;
	case FMQUERY:
	    fprintf(theOutput,"\tquery (host|class|allow|reject|stats) [<args>]\n");
	    fprintf(theOutput,"\nPerforms look ups in the tables currently loaded into the\n");
	    fprintf(theOutput,"filter (pointed to by the target variable).\n");
	    fprintf(theOutput,"Examples: query host 193.12.45.71\n");
	    fprintf(theOutput,"          query reject\n");
	    break;
	case FMUPLOAD:
	    fprintf(theOutput,"\tupload (networks|classes|allow|reject)\n");
	    fprintf(theOutput,"\nUploads the specified data to the filter. Note that this\n");
	    fprintf(theOutput,"data gets loaded into memory only and takes effect immediately\n");
	    fprintf(theOutput,"(see the write command).\n");
	    fprintf(theOutput,"Examples: upload networks\n");
	    fprintf(theOutput,"          upload classes\n");
	    break;
	case FMWRITE:
	    fprintf(theOutput,"\twrite\n");
	    fprintf(theOutput,"\nTells the filter to write all of the newly loaded tables from\n");
	    fprintf(theOutput,"memory to disk. If you do not do this after loading data and quit,\n");
	    fprintf(theOutput,"fm will warn you that the data has not been written to disk.\n");
	    break;
	case FMRELEASE:
	    fprintf(theOutput,"\trelease (classes|allow|reject|network) [<args>]\n");
	    fprintf(theOutput,"\nTells the filter to remove a specified table from the filter's\n");
	    fprintf(theOutput,"memory and disk storage. This command takes effect immediately. The\n");
	    fprintf(theOutput,"filter will resume default behavior for the specified table.\n");
	    fprintf(theOutput,"Examples: release network 128.193.0.0\n");
	    fprintf(theOutput,"          release allow\n");
	    break;
	case FMPING:
	    fprintf(theOutput,"\tping\n");
	    fprintf(theOutput,"\nTries to establish sequence numbers with the filter. Can be used\n");
	    fprintf(theOutput,"to test connectivity and that the DES key and target address are\n");
	    fprintf(theOutput,"set properly.\n");
	    break;
	case FMREBOOT:
	    fprintf(theOutput,"\treboot\n");
	    fprintf(theOutput,"\nTells the filter to reboot the hardware (cold boot).\n");
	    break;
	case FMCLEAR:
	    fprintf(theOutput,"\tclear\n");
	    fprintf(theOutput,"\nClears all of the tables loaded into fm.\n");
	    break;
	case FMRESET:
	    fprintf(theOutput,"\treset\n");
	    fprintf(theOutput,"\nCompletely resets the internal state of fm. This includes rereading\n");
	    fprintf(theOutput,"the .fmrc file.\n");
	    break;
	case FMNEWKEY:
	    fprintf(theOutput,"\tnewkey <name>\n");
	    fprintf(theOutput,"\nGenerates a new DES key and installs it in the user's home directory\n");
	    fprintf(theOutput,"and in the filter. If a current DES key exists, the new one will be\n");
	    fprintf(theOutput,"encrypted with it before it is transmitted.\n");
	    break;
	case FMGENKEY:
	    fprintf(theOutput,"\tgenkey <name>\n");
	    fprintf(theOutput,"\nGenerates a new DES key and installs it in the user's home directory\n");
	    fprintf(theOutput,"and into a file suitable for copying to the filter machine by floppy.\n");
	    fprintf(theOutput,"Useful if you don't trust your net and want to boot strap the DES\n");
	    fprintf(theOutput,"encryption by hand (otherwise the first key will go across the net\n");
	    fprintf(theOutput,"unencrypted). Note that this does not do a \"set key\" for you.\n");
	    break;
	case FMQUIT:
	    fprintf(theOutput,"\tquit\n");
            fprintf(theOutput,"\nPretty self explanatory. fm will gripe if you quit and have upload'd\n");
            fprintf(theOutput,"tables into the filter without performing a write.\n");
	    break;
	case FMWARRANTY:
	    fprintf(theOutput,"\twarranty\n");
	    fprintf(theOutput,"\nThis command prints the GNU no warranty disclaimer.\n");
	    break;
	case FMCONDITIONS:
	    fprintf(theOutput,"\tconditions\n");
	    fprintf(theOutput,"\nThis command prints the GNU General Public License.\n");
	    break;
	case EMPTY:
	    fprintf(theOutput,"commands:\n");
	    fprintf(theOutput,"\tset (verbose|target|key) <args>\n");
	    fprintf(theOutput,"\tload (network|classes|allow|reject) <filename>\n");
	    fprintf(theOutput,"\tshow (host|class|allow|reject|target|verbose|key) [<args>]\n");
	    fprintf(theOutput,"\tquery (host|class|allow|reject|stats) [<args>]\n");
	    fprintf(theOutput,"\tupload (networks|classes|allow|reject)\n");
	    fprintf(theOutput,"\twrite\n");
	    fprintf(theOutput,"\trelease (classes|allow|reject|network) [<args>]\n");
	    fprintf(theOutput,"\tping\n");
	    fprintf(theOutput,"\treboot\n");
	    fprintf(theOutput,"\tclear\n");
	    fprintf(theOutput,"\treset\n");
	    fprintf(theOutput,"\tnewkey <name>\n");
	    fprintf(theOutput,"\tgenkey <name>\n");
	    fprintf(theOutput,"\twarranty\n");
	    fprintf(theOutput,"\tconditions\n");
	    fprintf(theOutput,"\tquit\n");
	    fprintf(theOutput,"'!' as the first character of a command passes the command to a shell\n");
	    fprintf(theOutput,"'#' is a comment\n");
	    break;
	case UNKNOWN:
	default:
	    fprintf(theMessages,"unknown help key word\n");
	    break;
    }
}

void handleInput(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    int result;
    char *currTok;
    char command[256];
    FILE *morePipe;
        
    fprintf(theOutput,"fm> ");
    fgets(readBuffer,sizeof(readBuffer),theInput);

    while (!feof(theInput)) {

	currTok = strtok(readBuffer," \t\n");
	result = tokenize(currTok);

	switch (result) {
	    case FMPING:
		if (startTransaction() == 0)
		    fprintf(theOutput,"the target is alive\n");
		break;
	    case FMSET:
		handleSet(theInput,theOutput,theMessages);
		break;
	    case FMLOAD:
		handleLoad(theInput,theOutput,theMessages);
		break;
	    case FMSHOW:
		handleShow(theInput,theOutput,theMessages);
		break;
	    case FMQUERY:
		handleQuery(theInput,theOutput,theMessages);
		break;
	    case FMUPLOAD:
		handleUpload(theInput,theOutput,theMessages);
		break;
	    case FMWRITE:
		handleWrite(theInput,theOutput,theMessages);
		break;
	    case FMQUIT:
		if (uploadedAndNoWrite)
		    fprintf(theMessages,"note: no write was performed\n");
		exit(0);
		break;
	    case FMCLEAR:
		handleClear(theInput,theOutput,theMessages);
		break;
	    case FMRESET:
		handleReset(theInput,theOutput,theMessages);
		break;
	    case FMRELEASE:
		handleRelease(theInput,theOutput,theMessages);
		break;
	    case FMREBOOT:
		handleReboot(theInput,theOutput,theMessages);
		break;
	    case FMNEWKEY:
		handleNewKey(theInput,theOutput,theMessages);
		break;
	    case FMGENKEY:
		handleGenKey(theInput,theOutput,theMessages);
		break;
	    case FMHELP:
		handleHelp(theInput,theOutput,theMessages);
		break;
	    case FMSHELL:
		strcpy(command,"/bin/csh -c \'");

		strcat(command,currTok + 1);
		while (currTok = strtok((char *) NULL," ")) {
		    strcat(command," ");
		    strcat(command,currTok);
		}
		strcat(command,"'");

		if (system(command) == 127)
		    fprintf(theMessages,"shell escape failed\n");
		break;
	    case FMCONDITIONS:
		morePipe = popen("/usr/ucb/more","w");
		fwrite(conditions,strlen(conditions),1,morePipe);
		pclose(morePipe);
		break;
	    case FMWARRANTY:
		morePipe = popen("/usr/ucb/more","w");
		fwrite(warranty,strlen(warranty),1,morePipe);
		pclose(morePipe);
		break;
	    case UNKNOWN:
		fprintf(theMessages,"unknown command\n");
		break;
	    case FMCOMMENT:
	    case EMPTY:
		/* Do nothing. Just read another line. */
		break;
	    default:
		fprintf(theMessages,"strange return value from lex\n");
		exit(1);
		break;
	}

        fprintf(theOutput,"fm> ");
        fgets(readBuffer,sizeof(readBuffer),theInput);
    }
}

void usage(void)
{
    fprintf(messages,"usage: fm [-i <interface>]\n");
    exit(1);
}

void readRc(void)
{
    struct passwd *pwd;
    struct stat statBuf;
    FILE *rcInput;
    FILE *nullOutput;
    char rcFilename[256];

    /* Get the users home directory.
     */
    pwd = getpwuid(getuid());

    if (pwd == (struct passwd *) NULL) {
        fprintf(messages,"Who are you?\n");
        exit(1);
    }

    /* Build a path to the user's ~/.fmrc
     */
    strcpy(rcFilename,pwd->pw_dir);
    strcat(rcFilename,"/.fmrc");

    if (stat(rcFilename,&statBuf) != -1) {
	/* Check the user's .fmrc and gripe if it is not in the correct mode.
	 */
        if ((statBuf.st_mode & 0x1FF) != 0x100) {
            fprintf(messages,"Your .fmrc must be mode 400\n");
            exit(1);
        }

	/* Process the .fmrc file.
	 */ 
        rcInput = fopen(rcFilename,"r");

        if (rcInput != (FILE *) NULL) {
            nullOutput = fopen("/dev/null","r+");
            handleInput(rcInput,nullOutput,messages);
        }
    }
}

int main(int argc,char *argv[])
{
    int i;
    int s;
    char *interface;
    char buf[128];
    struct ifconf ifc;

    if (geteuid() != 0) {
	fprintf(messages,"fm must be run by root or made setuid root\n");
	exit(1);
    }

    interface = (char *) NULL;

    /*
     * Get the command line arguments.
     */
    for (i = 1;i < argc;++i) {
	if (argv[i][0] == '-') {
	    switch (argv[i][1]) {
		case 'i':
		    ++i;
		    if (i < argc)
			interface = argv[i];
		    else
			usage();
		    break;
		default:
		    usage();
		    break;
	    }
	}
	else {
	    usage();
	}
    }

    /*
     * Find a likely interface to use if one was not specified.
     */
    if (interface == (char *) NULL) {

	    s = socket(AF_INET,SOCK_DGRAM,0);

	    if (s < 0) {
		fprintf(messages,"socket(): %s\n",sys_errlist[errno]);
		exit(1);
	    }

	    ifc.ifc_len = sizeof(buf);
	    ifc.ifc_buf = buf;

	    if (ioctl(s,SIOCGIFCONF,(char *) &ifc) < 0) {
		fprintf(messages,"ioctl(): %s\n",sys_errlist[errno]);
		exit(1);
	    }

	    interface = ifc.ifc_req->ifr_name;

	    close(s);
    }

    /*
     * Try to set up nit. This requires super user access.
     */
    initCommunications(interface);

    /*
     * Get rid of root access after grabbing the nit device.
     */
    setuid(getuid());

    /* Initialize the random number generator. This is used by the DES sequence number
     *   security code.
     */
    srand(time(NULL));

    /*
     * Set up the umask (in case we create any DES key files).
     */
    umask(0377);

    /*
     * Read in the ~/.fmrc
     */
    readRc();

    /*
     * Redirect the output to /dev/null if the stdin is not a tty.
     */
    if (!isatty(0)) {
	output = fopen("/dev/null","r+");
    }

    fprintf(output,"Filter Manager %s\n",VERSION);
    fprintf(output,"Copyright (C) 1993 David K. Hess, Douglas Lee Schales and David R. Safford\n");
    fprintf(output,"Filter Manager comes with ABSOLUTELY NO WARRANTY; type 'warranty' for details\n");
    fprintf(output,"This is free software, and you are welcome to redistribute it under certain\n");
    fprintf(output,"conditions; type 'conditions' for details\nUsing interface %s\n",interface);

    /*
     * Handle user input. Only returns when the user has entered an EOF.
     */
    handleInput(input,output,messages);

    fprintf(output,"\n");

    if (uploadedAndNoWrite)
	fprintf(messages,"note: no write was performed\n");

    return 0;
}
