/*
 * Copyright (c) 1993,1995
 *	Texas A&M University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Texas A&M University
 *	and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Developers:
 *             David K. Hess, Douglas Lee Schales, David R. Safford
 */
/* fm - Filter Manager
 *
 * Reads commands from ~/.fmrc
 *
 * Command set:
 *
 * set (verbose|target|password|retries|timeout) <args>
 * load (network|classes|allow|reject) <filename>
 * show (networks|host|class|allow|reject|target|verbose|password) [<args>]
 * query (host|class|allow|reject|stats) [<args>]
 * upload (networks|classes|allow|reject)
 * write
 * quit
 * clear
 * reset
 * release (network|classes|allow|reject) <args>
 * reboot
 * newpwd <password>
 *     (note that the above command does an implicit set password command)
 */
#include "fm.h"

int passwordSet = NO;
FILE *debug = stderr;
FILE *output = stdout;
FILE *input = stdin;
FILE *messages = stderr;
int targetSet = NO;

unsigned char password[PASSWORD_LENGTH];

static unsigned char readBuffer[256];
static char currAccessTableFilename[256];
static char currRejectTableFilename[256];
static char currAllowTableFilename[256];

static int uploadedAndNoWrite = NO;
static int verboseMode = NO;
static int accessTableLoaded = NO;
static int rejectTableLoaded = NO;
static int allowTableLoaded = NO;
static int numNetworks = 0;

int sendRetries = SEND_RETRIES;
int syncRetries = SYNC_RETRIES;
int sendTimeout = SEND_TIMEOUT;

/*
 * Statically allocate the tables for the filtering.
 */
static AddrTableEntry addrTable[MAX_NUM_NETWORKS];
static AccessListTableEntry in[MAX_NUM_ACCESS_LISTS * MAX_NUM_ACCESS_RANGES];
static AccessListTableEntry out[MAX_NUM_ACCESS_LISTS * MAX_NUM_ACCESS_RANGES];
static AccessListTableEntry source[MAX_NUM_ACCESS_LISTS * MAX_NUM_ACCESS_RANGES];
static AccessListTableEntry udp[MAX_NUM_ACCESS_LISTS * MAX_NUM_ACCESS_RANGES];
static RejectTableEntry rejectTable[MAX_NUM_REJECT_ENTRIES];
static AllowTableEntry allowTable[MAX_NUM_ALLOW_ENTRIES];

void handleSet(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    struct in_addr newTarget;
    struct sockaddr_in newAddr;
    int result;
    char *currTok;
        
    currTok = strtok((char *) NULL," \t\n");
    result = tokenize(currTok);

    switch (result) {
	case FMVERBOSE:
	    currTok = strtok((char *) NULL," \t\n");

	    if (currTok != (char *) NULL) {
		if (strcasecmp(currTok,"no") == 0)
		    verboseMode = NO;
		else if (strcasecmp(currTok,"yes") == 0)
		    verboseMode = YES;
		else
		    fprintf(theMessages,"value for verbose must be 'yes' or 'no'\n");
	    }
	    else
		fprintf(theMessages,"value for verbose must be 'yes' or 'no'\n");

	    break;
	case FMTARGET:
	    currTok = strtok((char *) NULL," \t\n");

	    if (!(currTok && (newTarget.s_addr = inet_addr(currTok)) != -1)) {
		fprintf(theMessages,
			"value for target must be an IP address\n");
		break;
	    }

	    /* A hack for Linux.
	    close(sockFd);
	    sockFd = socket(PF_INET,SOCK_DGRAM,0);

	    if (sockFd == -1) {
		fprintf(messages,"socket: %s\n",sys_errlist[errno]);
		exit(1);
	    }
	    End of hack for Linux. */

	    memset(&newAddr,0,sizeof(struct sockaddr_in));
	    newAddr.sin_family = AF_INET;
	    newAddr.sin_port   = htons(DEFAULT_PORT);
	    newAddr.sin_addr   = newTarget;

	    if (connect(sockFd,(struct sockaddr *) &newAddr,sizeof(struct sockaddr_in))) {
		perror("connect");
		return;
	    }

	    currTarget = newTarget;
	    targetSet = YES;
	    synced = NO;
	    break;

	case FMPASSWORD:
	    currTok = strtok((char *) NULL," \t\n");

	    if (!currTok) {
		passwordSet = NO;
		fprintf(theOutput,"switching to insecure mode (encryption disabled)\n");
	    }
	    else {
		strncpy(password,currTok,PASSWORD_LENGTH);
		passwordSet = YES;
		synced = NO;
	    }
	    break;
	case FMRETRIES:
	    currTok = strtok((char *) NULL," \t\n");

	    if (!currTok) {
		fprintf(theMessages,
			"value for retries must be a positive integer\n");
		break;
	    }
	    
	    result = atoi(currTok);
	    if (!result) {
		fprintf(theMessages,
			"value for retries must be a positive integer\n");
		break;
	    }
	    sendRetries = result;
	    break;
	case FMTIMEOUT:
	    currTok = strtok((char *) NULL," \t\n");

	    if (!currTok) {
		fprintf(theMessages,
			"value for timeout must be a positive integer\n");
		break;
	    }
	    
	    result = atoi(currTok);
	    if (!result) {
		fprintf(theMessages,
			"value for timeout must be a positive integer\n");
		break;
	    }
	    sendTimeout = result;
	    break;
	case EMPTY:
	    fprintf(theMessages,"missing set key word\n");
	    break;
	case UNKNOWN:
	default:
	    fprintf(theMessages,"unknown set key word\n");
	    break;
    }
}

void handleShow(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    struct in_addr hostAddr;
    struct hostent *hostEntry;
    unsigned long host;
    unsigned long network;
    int result;
    int i;
    int j;
    int offset;
    int index;
    char *currTok;
        
    currTok = strtok((char *) NULL," \t\n");
    result = tokenize(currTok);

    switch (result) {
	case FMVERBOSE:
	    fprintf(theOutput,"verbose is %s\n",verboseMode ? "yes" : "no");
	    break;
	case FMTARGET:
	    fprintf(theOutput,"target is %s\n",inet_ntoa(currTarget));
	    break;
	case FMPASSWORD:
	    if (passwordSet == YES)
		fprintf(theOutput,"password is %s\n",password);
	    else
		fprintf(theOutput,"password is disabled (insecure mode)\n");
	    break;
	case FMHOST:
	    currTok = strtok((char *) NULL," \t\n");

	    if (!currTok)
		fprintf(theMessages,"missing host address/name\n");
	    else {
		hostEntry = gethostbyname(currTok);

		if (hostEntry != (struct hostent *) NULL)
		    hostAddr.s_addr = *(unsigned long *)hostEntry->h_addr_list[0];
		else {
		    hostAddr.s_addr = inet_addr(currTok);
		    if (hostAddr.s_addr == -1) {
			fprintf(theMessages,"%s: Invalid IP address\n",currTok);
			return;
		    }
		}

		hostAddr.s_addr = ntohl(hostAddr.s_addr);

		/*
		 * Now that we have the address do the look up.
		 */
		if (IN_CLASSB(hostAddr.s_addr)) {
		    host = hostAddr.s_addr & IN_CLASSB_HOST;
		    network = hostAddr.s_addr & IN_CLASSB_NET;
		}
		else if (IN_CLASSC(hostAddr.s_addr)) {
		    host = hostAddr.s_addr & IN_CLASSC_HOST;
		    network = hostAddr.s_addr & IN_CLASSC_NET;
		}
		else {
		    fprintf(theMessages,"class A and D addresses are not supported\n");
		    return;
		}
		
		for (i = 0;i < numNetworks;++i)
		    if (addrTable[i].network.s_addr == network)
			break;

		if (i != numNetworks)
		    fprintf(theOutput,"host %s is using class %d\n",
			    currTok,
			    addrTable[i].hostTable[host]);
		else {
		    hostAddr.s_addr = htonl(hostAddr.s_addr);
		    fprintf(theMessages,"network %s is not loaded\n",
			    inet_ntoa(*(struct in_addr *)&network));
		}
	    }
	    break;
	case FMCLASS:
	    currTok = strtok((char *) NULL," \t\n");

	    if (!currTok)
		fprintf(theMessages,"missing class number\n");
	    else {
		if (accessTableLoaded == NO) {
		    fprintf(theMessages,"no classes file has been loaded\n");
		    return;
		}

		index = atoi(currTok);

		if (index < 0 || index > MAX_NUM_ACCESS_LISTS - 1)
		    fprintf(theMessages,"index must be in range 0-255\n");
		else {
		    fprintf(theOutput,"class file is %s\n\n",currAccessTableFilename);
		    fprintf(theOutput,"    in\t\t    out\t\t  source\t    udp\n");
		    offset = index * MAX_NUM_ACCESS_RANGES;

		    while ((in[offset].begin     != 0 ||
			    out[offset].begin    != 0 ||
			    source[offset].begin != 0 ||
			    udp[offset].begin    != 0) &&
			   offset < (index + 1) * MAX_NUM_ACCESS_RANGES) {
			if (in[offset].begin != 0)
			    fprintf(theOutput,"%5d-%-5d\t",
				    ntohs(in[offset].begin),
				    ntohs(in[offset].end));
			else
			    fprintf(theOutput,"\t\t");

			if (out[offset].begin != 0)
			    fprintf(theOutput,"%5d-%-5d\t",
				    ntohs(out[offset].begin),
				    ntohs(out[offset].end));
			else
			    fprintf(theOutput,"\t\t");

			if (source[offset].begin != 0)
			    fprintf(theOutput,"%5d-%-5d\t",
				    ntohs(source[offset].begin),
				    ntohs(source[offset].end));
			else
			    fprintf(theOutput,"\t\t");

			if (udp[offset].begin != 0)
			    fprintf(theOutput,"%5d-%-5d\t",
				    ntohs(udp[offset].begin),
				    ntohs(udp[offset].end));
			else
			    fprintf(theOutput,"\t\t");

			fprintf(theOutput,"\n");
			++offset;
		    }
		}
	    }
	    break;
	case FMALLOW:
	    if (allowTableLoaded == NO) {
		fprintf(theMessages,"no allow file has been loaded\n");
		return;
	    }

	    fprintf(theOutput,"allow file is %s\n\n",
		    currAllowTableFilename);

	    i = 0;
	    while (allowTable[i].network.s_addr != 0 &&
		   i < MAX_NUM_ALLOW_ENTRIES) {

		fprintf(theOutput,"global allow to address %-15s",
			inet_ntoa(allowTable[i].network));
		fprintf(theOutput," with mask %-15s\n",
			inet_ntoa(*(struct in_addr *) &allowTable[i].mask));

		fprintf(theOutput,"   ports\n");

		j = 0;
		while (allowTable[i].access[j].begin != 0 &&
		       j < MAX_NUM_ACCESS_RANGES) {
		    fprintf(theOutput,"%5d-%-5d\n",
			    ntohs(allowTable[i].access[j].begin),
			    ntohs(allowTable[i].access[j].end));
		    ++j;
		}
		++i;
	    }
	    break;
	case FMREJECT:
	    if (rejectTableLoaded == NO) {
		fprintf(theMessages,"no reject file has been loaded\n");
		return;
	    }

	    fprintf(theOutput,"reject file is %s\n\n",
		    currRejectTableFilename);

	    i = 0;
	    while (rejectTable[i].network.s_addr != 0 &&
		   i < MAX_NUM_REJECT_ENTRIES) {
		fprintf(theOutput,"global reject from address %-15s",
			inet_ntoa(rejectTable[i].network));
		fprintf(theOutput," with mask %-15s\n",
			inet_ntoa(*(struct in_addr *) &rejectTable[i].mask));
		++i;
	    }
	    
	    break;
	case FMNETWORKS:
	    if (numNetworks == 0) {
		fprintf(theMessages,"no networks are loaded\n");
		return;
	    }

	    for (i = 0;i < numNetworks;++i)
		fprintf(theOutput,"network %s is loaded from file %s\n",
			inet_ntoa(addrTable[i].network),
			addrTable[i].networkFilename);
	    break;
	case FMRETRIES:
	    fprintf(theOutput,"retries is %d\n",sendRetries);
	    break;
	case FMTIMEOUT:
	    fprintf(theOutput,"timeout is %d\n",sendTimeout);
	    break;
	case EMPTY:
	    fprintf(theMessages,"missing show key word\n");
	    break;
	case UNKNOWN:
	default:
	    fprintf(theMessages,"unknown show key word\n");
	    break;
    }
}

void handleNewpwd(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    unsigned char newPassword[PASSWORD_LENGTH];
    char *currTok;

    currTok = strtok((char *) NULL," \t\n");

    if (!currTok) {
	fprintf(theMessages,"missing name\n");
    }
    else {
	strncpy(newPassword,currTok,PASSWORD_LENGTH);

	/* 
	 * Now that we have the new key, install it into the filter.
	 */

	if (passwordSet == NO)
		fprintf(theMessages,"Warning: password being sent unencrypted\n");

        if (sendMessage(FM_M_NEWKEY,(void *) newPassword,PASSWORD_LENGTH) == (unsigned char *) NULL) {
	    fprintf(theMessages,"installation of new password in filter failed\n");
	    return;
        }

	fprintf(theMessages,"installation of new password in filter succeeded\n");

	/*
	 * Copy the new key over to the current key storage.
	 */
	memcpy(password,newPassword,PASSWORD_LENGTH);
	passwordSet = YES;
    }
}

char *tildeExpand(char *filename,FILE *theMessages)
{
    struct passwd *pwd;
    static char resultFilename[256];
    char username[256];
    char *slash;

    strcpy(resultFilename,filename);

    if (filename[0] == '~') {
	slash = strchr(filename + 1,'/');

	if (!slash) {
	    fprintf(theMessages,"invalid filename\n");
	    return (char *) NULL;
	}
	else {
	    strncpy(username,filename + 1,slash - (filename + 1));
	    username[slash - (filename + 1)] = '\0';

	    if (strlen(username) == 0) {
		/*
		 * Just '~'.
		 */
		pwd = getpwuid(getuid());

		if (pwd == (struct passwd *) NULL) {
		    fprintf(theMessages,"who are you?\n");
		    return (char *) NULL;
		}
	    }
	    else {
		/*
		 * '~<username>'
		 */
		pwd = getpwnam(username);

		if (pwd == (struct passwd *) NULL) {
		    fprintf(theMessages,"unknown user %s\n",username);
		    return (char *) NULL;
		}
	    }

	    strcpy(resultFilename,pwd->pw_dir);
	    strcat(resultFilename,slash);
	}
    }

    return resultFilename;
}

void handleLoad(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    FILE *dataFile;
    struct in_addr network;
    unsigned long size;
    int result;
    char *currTok;
    char *expandedFilename;
        
    currTok = strtok((char *) NULL," \t\n");
    result = tokenize(currTok);

    switch (result) {
	case FMNETWORK:
	    currTok = strtok((char *) NULL," \t\n");
	    if (!currTok)
		fprintf(theMessages,"load network requires a filename\n");
	    else {
		expandedFilename = tildeExpand(currTok,theMessages);

		if (expandedFilename) {

		    dataFile = fopen(expandedFilename,"r");

		    if (!dataFile) {
			fprintf(theMessages,"%s: Could not open network file\n",currTok);
			return;
		    }
		    
		    if (fread((char *)&network,sizeof(struct in_addr),1,dataFile) != 1) {
			fprintf(theMessages,"%s: Could not read network address from network file\n",currTok);
			fclose(dataFile);
			return;
		    }

		    network.s_addr = ntohl(network.s_addr);
		    
		    if (IN_CLASSB(network.s_addr)) {
			network.s_addr = network.s_addr & IN_CLASSB_NET;
			size = 0x10000;
		    }
		    else if (IN_CLASSC(network.s_addr)) {
			network.s_addr = network.s_addr & IN_CLASSC_NET;
			size = 0x100;
		    }
		    else {
			fprintf(theMessages,
				"version %s does not handle class A or D addresses\n",
				VERSION);
			fclose(dataFile);
			return;
		    }
		    
		    addrTable[numNetworks].network = network;
		    addrTable[numNetworks].hostTable = (unsigned char *) malloc(size);	

		    if (addrTable[numNetworks].hostTable == (unsigned char *) NULL) {
			fprintf(theMessages,"could not allocate memory for the network\n");
			fclose(dataFile);
			return;
		    }

		    strcpy(addrTable[numNetworks].networkFilename,currTok);

		    if (fread((char *)addrTable[numNetworks].hostTable,size,1,dataFile) != 1) {
			fprintf(theMessages,"%s: Could not read network table from network file\n",currTok);
			free(addrTable[numNetworks].hostTable);
			fclose(dataFile);
			return;
		    }
		    
		    ++numNetworks;
		    
		    fclose(dataFile);

		    network.s_addr = ntohl(network.s_addr);
		    fprintf(theOutput,"loaded network %s\n",inet_ntoa(network));
		}
	    }
	    break;
	case FMCLASSES:
	    currTok = strtok((char *) NULL," \t\n");
	    if (!currTok) {
		fprintf(theMessages,"load classes requires a filename\n");
	    }
	    else {
		expandedFilename = tildeExpand(currTok,theMessages);

		if (expandedFilename) {

		    dataFile = fopen(expandedFilename,"r");

		    if (!dataFile) {
			fprintf(theMessages,"%s: Could not open classes file\n",currTok);
			return;
		    }
	
		    currAccessTableFilename[0] = '\0';
		    accessTableLoaded = NO;

		    if (fread((char *) in,sizeof(in),1,dataFile) != 1) {
			fprintf(theMessages,"%s: Could not read in table from classes file\n",currTok);
			fclose(dataFile);
			return;
		    }

		    if (fread((char *) out,sizeof(out),1,dataFile) != 1) {
			fprintf(theMessages,"%s: Could not read out table from classes file\n",currTok);
			fclose(dataFile);
			return;
		    }

		    if (fread((char *) source,sizeof(source),1,dataFile) != 1) {
			fprintf(theMessages,"%s: Could not read source table from classes file\n",currTok);
			fclose(dataFile);
			return;
		    }
		    
		    if (fread((char *) udp,sizeof(udp),1,dataFile) != 1) {
			fprintf(theMessages,"%s: Could not read udp table from classes file\n",currTok);
			fclose(dataFile);
			return;
		    }
		    
		    fclose(dataFile);

		    accessTableLoaded = YES;
		    strcpy(currAccessTableFilename,currTok);
		}
	    }
	    break;
	case FMALLOW:
	    currTok = strtok((char *) NULL," \t\n");
	    if (!currTok) {
		fprintf(theMessages,"load allow requires a filename\n");
	    }
	    else {
		expandedFilename = tildeExpand(currTok,theMessages);

		if (expandedFilename) {

		    dataFile = fopen(expandedFilename,"r");

		    if (!dataFile) {
			fprintf(theMessages,"%s: Could not open allow file\n",currTok);
			return;
		    }

		    currAllowTableFilename[0] = '\0';
		    allowTableLoaded = NO;
		    memset((char *)allowTable,0,sizeof(allowTable));

		    if (fread((char *)allowTable,
			      sizeof(AllowTableEntry),
			      MAX_NUM_ALLOW_ENTRIES,
			      dataFile) <= 0) {
			fprintf(theMessages,"%s: Error reading allow table\n",currTok);
			fclose(dataFile);
			return;
		    }

		    fclose(dataFile);
		    allowTableLoaded = YES;
		    strcpy(currAllowTableFilename,currTok);
		}
	    }
	    break;
	case FMREJECT:
	    currTok = strtok((char *) NULL," \t\n");
	    if (!currTok) {
		fprintf(theMessages,"load reject requires a filename\n");
	    }
	    else {
		expandedFilename = tildeExpand(currTok,theMessages);

		if (expandedFilename) {

		    dataFile = fopen(expandedFilename,"r");
		    
		    if (!dataFile) {
			fprintf(theMessages,"%s: Could not open reject file\n",currTok);
			return;
		    }

		    currRejectTableFilename[0] = '\0';
		    rejectTableLoaded = NO;
		    memset((char *)rejectTable,0,sizeof(rejectTable));

		    if (fread((char *)rejectTable,
			      sizeof(RejectTableEntry),
			      MAX_NUM_REJECT_ENTRIES,
			      dataFile) <= 0) {
			fprintf(theMessages,"%s: Error reading reject table\n",currTok);
			fclose(dataFile);
			return;
		    }
		    
		    fclose(dataFile);
		    rejectTableLoaded = YES;
		    strcpy(currRejectTableFilename,currTok);
		}
	    }
	    break;
	case EMPTY:
	    fprintf(theMessages,"missing load key word\n");
	    break;
	case UNKNOWN:
	default:
	    fprintf(theMessages,"unknown load key word\n");
	    break;
    }
}

unsigned char *buildBuf(unsigned char *buf,unsigned long val)
{
        if (val == 0xFFFFFFFF) {
                sprintf(buf,"       N/A");
        }
        else {
                sprintf(buf,"%10lu",val);
        }
        return buf;
}

void handleQuery(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
	QueryPacket query;
	StatisticsPacket stats;
	AccessListTableEntry *filterIn;
	AccessListTableEntry *filterOut;
	AccessListTableEntry *filterSrc;
	AccessListTableEntry *filterUdp;
	AllowTableEntry *filterAllow;
	RejectTableEntry *filterReject;
	struct in_addr hostAddr;
	struct in_addr *networks;
/*	struct in_addr aNetwork; */
	struct hostent *hostEntry;
	int result;
	int i;
	int j;
	int offset;
	int index;
	char *currTok;
	unsigned char *ack;
	unsigned char buf1[15];
	unsigned char buf2[15];
	unsigned long upTime;
	unsigned long upDays;
	unsigned long upHours;
	unsigned long upMinutes;
	unsigned long upSeconds;
        
	currTok = strtok((char *) NULL," \t\n");
	result = tokenize(currTok);

	switch (result) {
		case FMSTATS:
			stats.type = FM_STATISTICS_QUERY;
			ack = sendMessage(FM_M_STATISTICS,(void *) &stats,sizeof(StatisticsPacket));

			if (ack == (unsigned char *) NULL) {
				fprintf(theMessages,"query failed\n");
				break;
			}
				
			/*
			fprintf(stderr,"size = %d\n",sizeof(StatisticsPacket));
			*/

			memcpy(&stats,
				ack + sizeof(FiltHeader),
				sizeof(StatisticsPacket));

			longSwap(stats.statistics,MAX_NUM_STATISTICS);

			fprintf(theOutput,"\n--- DRAWBRIDGE STATS ---\n");

			fprintf(theOutput,"                             Inside       Outside\n");
			fprintf(theOutput,"Packets filtered         %10lu    %10lu\n",
					ntohl(stats.statistics[FM_STAT_DB_PACKETS_FILTERED_INSIDE]),
					ntohl(stats.statistics[FM_STAT_DB_PACKETS_FILTERED_OUTSIDE]));
			fprintf(theOutput,"Packets received         %10lu    %10lu\n",
					ntohl(stats.statistics[FM_STAT_DB_PACKETS_RX_INSIDE]),
					ntohl(stats.statistics[FM_STAT_DB_PACKETS_RX_OUTSIDE]));
			fprintf(theOutput,"Packets transmitted      %10lu    %10lu\n",
					ntohl(stats.statistics[FM_STAT_DB_PACKETS_TX_INSIDE]),
					ntohl(stats.statistics[FM_STAT_DB_PACKETS_TX_OUTSIDE]));
			upTime = ntohl(stats.statistics[FM_STAT_UPTIME]);
			if (upTime) {
				upDays = upTime / (24 * 60 * 60);
				upTime -= upDays * 24 * 60 * 60;
				upHours = upTime / (60 * 60);
				upTime -= upHours * 60 * 60;
				upMinutes = upTime / 60;
				upSeconds = upTime - upMinutes * 60;
			}

			fprintf(theOutput,"Up for %lu days %lu hours %lu minutes %lu seconds\n",
				upDays,upHours,upMinutes,upSeconds);
			fprintf(theOutput,"Cache Accesses: %10lu  Cache Misses: %10lu  Hit Ratio: ",
					ntohl(stats.statistics[FM_STAT_DB_CACHE_ACCESSES]),
					ntohl(stats.statistics[FM_STAT_DB_CACHE_MISSES]));

			if (stats.statistics[FM_STAT_DB_CACHE_ACCESSES])
				fprintf(theOutput,"%3.0f%%\n",
					(ntohl(stats.statistics[FM_STAT_DB_CACHE_ACCESSES]) - 
					ntohl(stats.statistics[FM_STAT_DB_CACHE_MISSES])) * 100.0 / 
					ntohl(stats.statistics[FM_STAT_DB_CACHE_ACCESSES]));
			else
				fprintf(theOutput,"100%%\n");

			fprintf(theOutput,"Dropped packets due to lack of packet buffers: %10lu\n",
				ntohl(stats.statistics[FM_STAT_DB_DROPPED_PACKETS]));

			fprintf(theOutput,"\n--- CARD STATS ---\n");
			fprintf(theOutput,"                             Inside       Outside\n");

			fprintf(theOutput,"Rx Frames                %s    %s\n",
				buildBuf(buf1,ntohl(stats.statistics[FM_STAT_CARD_FRAMES_RX_INSIDE])),
				buildBuf(buf2,ntohl(stats.statistics[FM_STAT_CARD_FRAMES_RX_OUTSIDE])));
			fprintf(theOutput,"Rx Bytes                 %s    %s\n",
				buildBuf(buf1,ntohl(stats.statistics[FM_STAT_CARD_BYTES_RX_INSIDE])),
				buildBuf(buf2,ntohl(stats.statistics[FM_STAT_CARD_BYTES_RX_OUTSIDE])));
			fprintf(theOutput,"Rx Multicast             %s    %s\n",
				buildBuf(buf1,ntohl(stats.statistics[FM_STAT_CARD_MULTICAST_RX_INSIDE])),
				buildBuf(buf2,ntohl(stats.statistics[FM_STAT_CARD_MULTICAST_RX_OUTSIDE])));
			fprintf(theOutput,"Rx Broadcast             %s    %s\n",
				buildBuf(buf1,ntohl(stats.statistics[FM_STAT_CARD_BROADCAST_RX_INSIDE])),
				buildBuf(buf2,ntohl(stats.statistics[FM_STAT_CARD_BROADCAST_RX_OUTSIDE])));
			fprintf(theOutput,"Rx CRC Errors            %s    %s\n",
				buildBuf(buf1,ntohl(stats.statistics[FM_STAT_CARD_CRC_RX_INSIDE])),
				buildBuf(buf2,ntohl(stats.statistics[FM_STAT_CARD_CRC_RX_OUTSIDE])));
			fprintf(theOutput,"Rx Buffer Drops          %s    %s\n",
				buildBuf(buf1,ntohl(stats.statistics[FM_STAT_CARD_BUFFER_DROPS_RX_INSIDE])),
				buildBuf(buf2,ntohl(stats.statistics[FM_STAT_CARD_BUFFER_DROPS_RX_OUTSIDE])));
			fprintf(theOutput,"Rx Hardware Error Drops  %s    %s\n",
				buildBuf(buf1,ntohl(stats.statistics[FM_STAT_CARD_HARDWARE_DROPS_RX_INSIDE])),
				buildBuf(buf2,ntohl(stats.statistics[FM_STAT_CARD_HARDWARE_DROPS_RX_OUTSIDE])));
		       fprintf(theOutput,"Tx Frames                %s    %s\n",
				buildBuf(buf1,ntohl(stats.statistics[FM_STAT_CARD_FRAMES_TX_INSIDE])),
				buildBuf(buf2,ntohl(stats.statistics[FM_STAT_CARD_FRAMES_TX_OUTSIDE])));
			fprintf(theOutput,"Tx Bytes                 %s    %s\n",
				buildBuf(buf1,ntohl(stats.statistics[FM_STAT_CARD_BYTES_TX_INSIDE])),
				buildBuf(buf2,ntohl(stats.statistics[FM_STAT_CARD_BYTES_TX_OUTSIDE])));
			fprintf(theOutput,"Tx Multicast             %s    %s\n",
				buildBuf(buf1,ntohl(stats.statistics[FM_STAT_CARD_MULTICAST_TX_INSIDE])),
				buildBuf(buf2,ntohl(stats.statistics[FM_STAT_CARD_MULTICAST_TX_OUTSIDE])));
			fprintf(theOutput,"Tx Broadcast             %s    %s\n",
				buildBuf(buf1,ntohl(stats.statistics[FM_STAT_CARD_BROADCAST_TX_INSIDE])),
				buildBuf(buf2,ntohl(stats.statistics[FM_STAT_CARD_BROADCAST_TX_OUTSIDE])));
			fprintf(theOutput,"Tx Timeout Drops         %s    %s\n",
				buildBuf(buf1,ntohl(stats.statistics[FM_STAT_CARD_TIMEOUT_DROPS_TX_INSIDE])),
				buildBuf(buf2,ntohl(stats.statistics[FM_STAT_CARD_TIMEOUT_DROPS_TX_OUTSIDE])));
			fprintf(theOutput,"Tx Hardware Error Drops  %s    %s\n",
				buildBuf(buf1,ntohl(stats.statistics[FM_STAT_CARD_HARDWARE_DROPS_TX_INSIDE])),
				buildBuf(buf2,ntohl(stats.statistics[FM_STAT_CARD_HARDWARE_DROPS_TX_OUTSIDE])));

			break;
		case FMHOST:
			currTok = strtok((char *) NULL," \t\n");

			if (!currTok) {
				fprintf(theMessages,"missing host address/name\n");
			}
			else {
				hostEntry = gethostbyname(currTok);

				if (hostEntry != (struct hostent *) NULL) {
					hostAddr.s_addr = ntohl(*(unsigned long *)hostEntry->h_addr_list[0]);
				}
				else {
					hostAddr.s_addr = inet_addr(currTok);
					if (hostAddr.s_addr == -1) {
						fprintf(theMessages,"%s: Invalid IP address\n",currTok);
						return;
					}
				}

				/*
				 * Now that we have the address do the look up.
				 */
				if (IN_CLASSA(hostAddr.s_addr) || IN_CLASSD(hostAddr.s_addr)) {
					fprintf(theMessages,"class A and D addresses are not supported\n");
					return;
				}
				
				/*
				 * Send the request over to the filter.
				 */
				memset((char *) &query,0,sizeof(query));

				query.type = FM_QUERY_HOST;
				query.queryValue.addr = htonl(hostAddr.s_addr);

				/*
				 * Swap the address around for the PC (barf!). Note that we
                                 *   keep these issues of what we preswap for the PC and
                                 *   what we swap internally for Big vs. Little Endian very
                                 *   separate. It ends up we do more work but we preserve
                                 *   our sanity.
				 */
				longSwap((unsigned long *)&query.queryValue.addr,1);
				
				ack = sendMessage(FM_M_QUERY,(void *) &query,sizeof(query));

				if (ack == (unsigned char *) NULL) {
					fprintf(theMessages,"query failed\n");
					return;
				}
				fprintf(theOutput,"host %s is using class %d\n",
					currTok,
					((QueryPacket *) (ack + sizeof(FiltHeader)))->queryResult.index);
			}
			break;
		case FMCLASS:
			currTok = strtok((char *) NULL," \t\n");

			if (!currTok) {
				fprintf(theMessages,"missing class number\n");
			}
			else {
				index = atoi(currTok);
				
				if (index < 0 || index > MAX_NUM_ACCESS_LISTS - 1) {
					fprintf(theMessages,
						"index must be in range 0 - %d\n",
						MAX_NUM_ACCESS_LISTS - 1);
					return;
				}
				else {
					memset((char *) &query,0,sizeof(query));

					query.type = FM_QUERY_CLASS;
					query.queryValue.index = index;
					
					ack = sendMessage(FM_M_QUERY,(void *) &query,sizeof(query));

					if (ack == (unsigned char *) NULL) {
						fprintf(theMessages,"query failed\n");
						return;
					}
					
					/* 
					 * Make copies that are easier to reference and less expensive to
					 *   compute.
					 */    
					filterIn = ((QueryPacket *) (ack + sizeof(FiltHeader)))->queryResult.accessList.in;
					filterOut = ((QueryPacket *) (ack + sizeof(FiltHeader)))->queryResult.accessList.out;
					filterSrc = ((QueryPacket *) (ack + sizeof(FiltHeader)))->queryResult.accessList.src;
					filterUdp = ((QueryPacket *) (ack + sizeof(FiltHeader)))->queryResult.accessList.udp;

					/* Byte swap the table. Since the tables are in
					 *    consecutive order in the packet, we can swap them all at the
					 *    same time. 
					 */    
					shortSwap((unsigned short *)filterIn,
						  MAX_NUM_ACCESS_RANGES * 
						  sizeof(AccessListTableEntry) * 4);

					fprintf(theOutput,"    in\t\t    out\t\t  source\t    udp\n");
					offset = 0;
					
					while ((filterIn[offset].begin  != 0 ||
						filterOut[offset].begin != 0 ||
						filterSrc[offset].begin != 0 ||
						filterUdp[offset].begin != 0) &&
					       offset < MAX_NUM_ACCESS_RANGES) {

						if (filterIn[offset].begin != 0)
							fprintf(theOutput,"%5d-%-5d\t",
								htons(filterIn[offset].begin),
								htons(filterIn[offset].end));
						else
							fprintf(theOutput,"\t\t");

						if (filterOut[offset].begin != 0)
							fprintf(theOutput,"%5d-%-5d\t",
								htons(filterOut[offset].begin),
								htons(filterOut[offset].end));
						else
							fprintf(theOutput,"\t\t");

						if (filterSrc[offset].begin != 0)
							fprintf(theOutput,"%5d-%-5d\t",
								htons(filterSrc[offset].begin),
								htons(filterSrc[offset].end));
						else
							fprintf(theOutput,"\t\t");

						if (filterUdp[offset].begin != 0)
							fprintf(theOutput,"%5d-%-5d\t",
								htons(filterUdp[offset].begin),
								htons(filterUdp[offset].end));
						else
							fprintf(theOutput,"\t\t");

						fprintf(theOutput,"\n");
						++offset;
					}
				}
			}
			break;
	case FMALLOW:
	    memset((char *) &query,0,sizeof(query));

	    query.type = FM_QUERY_ALLOW;
	    
	    ack = sendMessage(FM_M_QUERY,(void *) &query,sizeof(query));

	    if (ack == (unsigned char *) NULL) {
		fprintf(theMessages,"query failed\n");
		return;
	    }
	    
	    memcpy(&query,ack + sizeof(FiltHeader),sizeof(QueryPacket));
	    filterAllow = query.queryResult.allow;

	    i = 0;

	    while (filterAllow[i].network.s_addr != 0L &&
		   i < MAX_NUM_ALLOW_ENTRIES) {

		longSwap((unsigned long *)(filterAllow + i),2);

		fprintf(theOutput,"global allow to address %-15s",
			inet_ntoa(filterAllow[i].network));
		fprintf(theOutput," with mask %-15s\n",
			inet_ntoa(*(struct in_addr *) &filterAllow[i].mask));

		fprintf(theOutput,"   ports\n");

		j = 0;
		shortSwap((unsigned short *)filterAllow[i].access,
			  sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES / 2);

		while (ntohs(filterAllow[i].access[j].begin) != 0 &&
		       j < MAX_NUM_ACCESS_RANGES) {
		    fprintf(theOutput,"%5d-%-5d\n",
			    ntohs(filterAllow[i].access[j].begin),
			    ntohs(filterAllow[i].access[j].end));
		    ++j;
		}

		++i;
	    }
	    
	    break;
	case FMREJECT:
	    memset((char *) &query,0,sizeof(query));
	    query.type = FM_QUERY_REJECT;
	    
	    ack = sendMessage(FM_M_QUERY,(void *) &query,sizeof(query));

	    if (ack == (unsigned char *) NULL) {
		fprintf(theMessages,"query failed\n");
		return;
	    }
	    
	    memcpy(&query,ack + sizeof(FiltHeader),sizeof(QueryPacket));
	    filterReject = query.queryResult.reject;

	    longSwap((unsigned long *) filterReject,
		     sizeof(RejectTableEntry) * MAX_NUM_REJECT_ENTRIES / sizeof(unsigned long));

	    i = 0;
	    while (filterReject[i].network.s_addr != 0L &&
		   i < MAX_NUM_REJECT_ENTRIES) {
		fprintf(theOutput,"global reject from address %-15s",
			inet_ntoa(filterReject[i].network));
		fprintf(theOutput," with mask %-15s\n",
			inet_ntoa(*(struct in_addr *) &filterReject[i].mask));
		++i;
	    }
	    break;
	case FMNETWORKS:
	    memset((char *) &query,0,sizeof(query));
	    query.type = FM_QUERY_NETWORK;

	    /*
	     * Get the list from the filter.
	     */
	    ack = sendMessage(FM_M_QUERY,(void *) &query,sizeof(query));

	    if (ack == (unsigned char *) NULL) {
		fprintf(theMessages,"query failed\n");
		return;
	    }

	    memcpy(&query,ack + sizeof(FiltHeader),sizeof(QueryPacket));
	    networks = query.queryResult.networks;

	    longSwap((unsigned long *) networks,MAX_NUM_NETWORKS);

	    i = 0;

/*
	    memcpy(&aNetwork,
			networks,
			sizeof(struct in_addr));
*/

	    while (networks[i].s_addr != 0L) {
		fprintf(theOutput,"network %15s is loaded\n",
			inet_ntoa(networks[i]));
		++i;
/*
	        memcpy(&aNetwork,
			networks + i,
			sizeof(struct in_addr));
*/
	    }

	    break;
	case EMPTY:
	    fprintf(theMessages,"missing query key word\n");
	    break;
	case UNKNOWN:
	default:
	    fprintf(theMessages,"unknown query key word\n");
	    break;
    }
}

void handleUpload(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    LoadPacket load;
    unsigned long j;
    unsigned long size;
    int result;
    int i;
    char *currTok;
    struct in_addr aHost;
        
    currTok = strtok((char *) NULL," \t\n");
    result = tokenize(currTok);

    switch (result) {
	case FMCLASSES:
	    if (accessTableLoaded == NO) {
		fprintf(theMessages,"no class table has been loaded\n");
		return;
	    }

	    /*
	     * Send the classes across one index at a time.
	     */
	    load.type = FM_LOAD_CLASS;
		
	    for (i = 0;i < MAX_NUM_ACCESS_LISTS;++i) {
		
		/*
		 * Set the flags.
		 */
		load.flags = 0;

		if (i == 0)
		    load.flags = FM_LOAD_FLAGS_BEGIN;

		if (i == MAX_NUM_ACCESS_LISTS - 1)
		    load.flags = FM_LOAD_FLAGS_END;

		/*
		 * Set the index.
		 */
		load.loadValue.index = i;

		/*
		 * Copy the access lists to the packet.
		 */
		memcpy((unsigned char *) (load.loadData.accessList.in),
		       (unsigned char *) (in + i * MAX_NUM_ACCESS_RANGES),
		       sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES);
		memcpy((unsigned char *) load.loadData.accessList.out,
		       (unsigned char *) (out + i * MAX_NUM_ACCESS_RANGES),
		       sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES);
		memcpy((unsigned char *) load.loadData.accessList.src,
		       (unsigned char *) (source + i * MAX_NUM_ACCESS_RANGES),
		       sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES);
		memcpy((unsigned char *) load.loadData.accessList.udp,
		       (unsigned char *) (udp + i * MAX_NUM_ACCESS_RANGES),
		       sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES);

		/*
		 * Byte swap 'em.
		 */
		shortSwap((unsigned short *) load.loadData.accessList.in,
			  sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES * 4);

		/*
		 * Send the packet.
		 */
		if (sendMessage(FM_M_LOAD,(unsigned char *)&load,sizeof(load)) ==
		    (unsigned char *) NULL) {
		    fprintf(theMessages,"upload of classes table failed\n");
		    break;
		}
	    }
	    uploadedAndNoWrite = YES;

	    break;
	case FMALLOW:

	    if (allowTableLoaded == NO) {
		fprintf(theMessages,"no allow table has been loaded\n");
		return;
	    }

	    /*
	     * Send the allow table across.
	     */
	    load.type = FM_LOAD_ALLOW;
		
	    memcpy((unsigned char *) load.loadData.allow,
		   (unsigned char *) allowTable,
		   sizeof(allowTable));

	    /* 
	     * Byte swap it.
	     */
	    for (i = 0;i < MAX_NUM_ALLOW_ENTRIES;++i) {
		longSwap((unsigned long *)&load.loadData.allow[i].network,2);
		shortSwap((unsigned short *)load.loadData.allow[i].access,
			  sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES / 2);
	    }

	    if (sendMessage(FM_M_LOAD,(unsigned char *)&load,sizeof(load)) ==
		(unsigned char *) NULL) {
		fprintf(theMessages,"upload of allow table failed\n");
	    }

	    uploadedAndNoWrite = YES;

	    break;
	case FMREJECT:
	    if (rejectTableLoaded == NO) {
		fprintf(theMessages,"no reject table has been loaded\n");
		return;
	    }

	    /*
	     * Send the reject table across.
	     */
	    load.type = FM_LOAD_REJECT;
		
	    memcpy((unsigned char *) load.loadData.reject,
		   (unsigned char *) rejectTable,
		   sizeof(rejectTable));

	    /*
	     * Byte swap it.
	     */
	    longSwap((unsigned long *) load.loadData.reject,
		     sizeof(rejectTable) / 4);

	    if (sendMessage(FM_M_LOAD,(unsigned char *)&load,sizeof(load)) ==
		(unsigned char *) NULL) {
		fprintf(theMessages,"upload of reject table failed\n");
	    }
	    
	    uploadedAndNoWrite = YES;
	    
	    break;
	case FMNETWORKS:
	    if (numNetworks == 0) {
		fprintf(theMessages,"no networks are loaded\n");
		return;
	    }

	    /*
	     * Send all of the currently loaded networks across.
	     */
	    load.type = FM_LOAD_NETWORK;
		
	    for (i = 0;i < numNetworks;++i) {

		if (IN_CLASSC(addrTable[i].network.s_addr))
		    size = 0x100;
		else
		    size = 0x10000;

		for (j = 0;j < size;j += 1024) {
		    
		    /*
		     * Set the flags.
		     */
		    load.flags = 0;
			
		    if (j == 0)
			load.flags = FM_LOAD_FLAGS_BEGIN;
		    
		    if ((size == 0x10000 && j == size - 1024) ||
			size == 0x100)
			load.flags |= FM_LOAD_FLAGS_END;
		    
		    /*
		     * Set the network and offset.
		     */
		    load.loadValue.networkBlock.network.s_addr = htonl(addrTable[i].network.s_addr);
		    load.loadValue.networkBlock.offset = htonl(j);
		    
		    longSwap((unsigned long *) &load.loadValue.networkBlock.network,2);

		    /*
		     * Copy the network block to the packet.
		     */
		    memcpy((unsigned char *) (load.loadData.networkBlock),
			   (unsigned char *) (addrTable[i].hostTable + j),
			   1024);

		    /*
		     * Send the packet.
		     */
		    if (sendMessage(FM_M_LOAD,(unsigned char *)&load,sizeof(load)) ==
			(unsigned char *) NULL) {
			fprintf(theMessages,"upload of networks failed\n");
			uploadedAndNoWrite = YES;
			return;
		    }
		}
		
		aHost.s_addr = ntohl(addrTable[i].network.s_addr);
		fprintf(theOutput,"uploaded network %s\n",
			inet_ntoa(aHost));
	    }
	    uploadedAndNoWrite = YES;

	    break;
	case EMPTY:
	    fprintf(theMessages,"missing upload key word\n");
	    break;
	case UNKNOWN:
	default:
	    fprintf(theMessages,"unknown upload key word\n");
	    break;
    }
}

void handleWrite(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    if (sendMessage(FM_M_WRITE,(unsigned char *) NULL,0) ==
	(unsigned char *) NULL) {
	fprintf(theMessages,"write failed\n");
    }
    else
	uploadedAndNoWrite = NO;
}

void handleClear(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
	int i;
	int result;
	char *currTok;
	StatisticsPacket stats;
	unsigned char *ack;
        
	currTok = strtok((char *) NULL," \t\n");
	result = tokenize(currTok);

	switch (result) {
		case FMTABLES:
			/* 
			* Clear memory.
			*/
			accessTableLoaded = NO;
			rejectTableLoaded = NO;
			allowTableLoaded  = NO;

			for (i = 0;i < numNetworks;++i) {
				if (addrTable[i].network.s_addr != 0)
					free(addrTable[i].hostTable);

				addrTable[i].network.s_addr = 0L;
			}

			numNetworks = 0;
			break;
		case FMSTATS:
			stats.type = FM_STATISTICS_CLEAR;
			ack = sendMessage(FM_M_STATISTICS,(void *) &stats,sizeof(StatisticsPacket));

			if (ack == (unsigned char *) NULL) {
				fprintf(theMessages,"clear of statistics table failed\n");
			}

			break;
		case EMPTY:
			fprintf(theMessages,"missing clear key word\n");
			break;
		case UNKNOWN:
		default:
			fprintf(theMessages,"unknown clear key word\n");
			break;
	}
}

void handleReset(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    int i;

    /* 
     * Reinitialize global variables.
     */
    accessTableLoaded  = NO;
    rejectTableLoaded  = NO;
    allowTableLoaded   = NO;
    uploadedAndNoWrite = NO;
    verboseMode        = NO;
    passwordSet        = NO;

    /* 
     * Clean out each loaded network.
     */
    for (i = 0;i < numNetworks;++i) {
	if (addrTable[i].network.s_addr != 0L)
	    free(addrTable[i].hostTable);

	addrTable[i].network.s_addr = 0L;
    }

    numNetworks = 0;

    /* 
     * Re-read ~/.fmrc
     */
    readRc();
}

void handleRelease(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    ReleasePacket release;
    struct in_addr network;
    int result;
    char *currTok;
        
    currTok = strtok((char *) NULL," \t\n");
    result = tokenize(currTok);

    switch (result) {
	case FMCLASSES:
	    /*
	     * Send the message to release the classes.
	     */
	    release.type = FM_RELEASE_CLASSES;

	    if (sendMessage(FM_M_RELEASE,(unsigned char *)&release,
			sizeof(release)) == (unsigned char *) NULL) {
		fprintf(theMessages,"release of classes table failed\n");
	    }

	    break;
	case FMALLOW:
	    /*
	     * Send the allow table release across.
	     */
	    release.type = FM_RELEASE_ALLOW;

	    if (sendMessage(FM_M_RELEASE,(unsigned char *)&release,
			sizeof(release)) == (unsigned char *) NULL) {
		fprintf(theMessages,"release of allow table failed\n");
	    }
	    break;
	case FMREJECT:
	    /*
	     * Send the reject table release across.
	     */
	    release.type = FM_RELEASE_REJECT;

	    if (sendMessage(FM_M_RELEASE,(unsigned char *)&release,
			sizeof(release)) == (unsigned char *) NULL) {
		fprintf(theMessages,"release of reject table failed\n");
	    }

	    break;
	case FMNETWORK:
	    currTok = strtok((char *) NULL," \t\n");

	    if (!currTok) {
		fprintf(theMessages,"release network missing network argument\n");
	    }
	    else {
		network.s_addr = inet_addr(currTok);

		if (network.s_addr == -1) {
		    fprintf(theMessages,"%s: Invalid IP network\n",currTok);
		    break;
		}
		
		/*
		 * Now that we have the address do the look up.
		 */
		if (IN_CLASSA(ntohl(network.s_addr)) || IN_CLASSD(ntohl(network.s_addr))) {
		    fprintf(theMessages,"class A and D addresses are not supported\n");
		    break;
		}
		
		/*
		 * Send the release request across.
		 */
		release.type = FM_RELEASE_NETWORK;
		release.network = network;
		longSwap((unsigned long *) &release.network,1);
		
		if (sendMessage(FM_M_RELEASE,(unsigned char *)&release,
			    sizeof(release)) == (unsigned char *) NULL) {
		    fprintf(theMessages,"release of network %s failed\n",currTok);
		}
	    }

	    break;
	case EMPTY:
	    fprintf(theMessages,"missing release key word\n");
	    break;
	case UNKNOWN:
	default:
	    fprintf(theMessages,"unknown release key word\n");
	    break;
    }
}

void handleReboot(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    if (sendMessage(FM_M_REBOOT,(void *) NULL,0) == (unsigned char *) NULL) {
	fprintf(theMessages,"reboot failed\n");
    }
}

void handleHelp(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    int result;
    char *currTok;
        
    currTok = strtok((char *) NULL," \t\n");
    result = tokenize(currTok);

    switch (result) {
	case FMSET:
	    fprintf(theOutput,"\tset (verbose|target|password|retries|timeout) <args>\n");
	    fprintf(theOutput,"\nSets the specified variable with the given value.\n");
	    fprintf(theOutput,"Examples: set verbose yes\n");
	    fprintf(theOutput,"          set target 10.1.2.3\n");
	    break;
	case FMLOAD:
	    fprintf(theOutput,"\tload (network|classes|allow|reject) <filename>\n");
	    fprintf(theOutput,"\nLoads the specified type of data from <filename>. Note\n");
	    fprintf(theOutput,"that multiple networks can be loaded but only one classes,\n");
	    fprintf(theOutput,"allow and reject table can be loaded at a time.\n");
	    fprintf(theOutput,"Examples: load network ~/fm/129.43.0.0\n");
	    fprintf(theOutput,"          load classes /networks/classes\n");
	    break;
	case FMSHOW:
	    fprintf(theOutput,"\tshow (host|class|allow|reject|target|verbose|password|retries|timeout) [<args>]\n");
	    fprintf(theOutput,"\nPerforms look ups in the tables currently loaded into fm.\n");
	    fprintf(theOutput,"Also displays variable values.\n");
	    fprintf(theOutput,"Examples: show host athena.mit.edu\n");
	    fprintf(theOutput,"          show target\n");
	    fprintf(theOutput,"          show class 18\n");
	    break;
	case FMQUERY:
	    fprintf(theOutput,"\tquery (host|class|allow|reject|stats) [<args>]\n");
	    fprintf(theOutput,"\nPerforms look ups in the tables currently loaded into the\n");
	    fprintf(theOutput,"filter (pointed to by the target variable).\n");
	    fprintf(theOutput,"Examples: query host 193.12.45.71\n");
	    fprintf(theOutput,"          query reject\n");
	    break;
	case FMUPLOAD:
	    fprintf(theOutput,"\tupload (networks|classes|allow|reject)\n");
	    fprintf(theOutput,"\nUploads the specified data to the filter. Note that this\n");
	    fprintf(theOutput,"data gets loaded into memory only and takes effect immediately\n");
	    fprintf(theOutput,"(see the write command).\n");
	    fprintf(theOutput,"Examples: upload networks\n");
	    fprintf(theOutput,"          upload classes\n");
	    break;
	case FMWRITE:
	    fprintf(theOutput,"\twrite\n");
	    fprintf(theOutput,"\nTells the filter to write all of the newly loaded tables from\n");
	    fprintf(theOutput,"memory to disk. If you do not do this after loading data and quit,\n");
	    fprintf(theOutput,"fm will warn you that the data has not been written to disk.\n");
	    break;
	case FMRELEASE:
	    fprintf(theOutput,"\trelease (classes|allow|reject|network) [<args>]\n");
	    fprintf(theOutput,"\nTells the filter to remove a specified table from the filter's\n");
	    fprintf(theOutput,"memory and disk storage. This command takes effect immediately. The\n");
	    fprintf(theOutput,"filter will resume default behavior for the specified table.\n");
	    fprintf(theOutput,"Examples: release network 128.193.0.0\n");
	    fprintf(theOutput,"          release allow\n");
	    break;
	case FMREBOOT:
	    fprintf(theOutput,"\treboot\n");
	    fprintf(theOutput,"\nTells the filter to reboot the hardware (cold boot).\n");
	    break;
	case FMCLEAR:
	    fprintf(theOutput,"\tclear (tables|statistics)\n");
	    fprintf(theOutput,"\nClears all of the tables loaded into fm or the statistics in the\n");
	    fprintf(theOutput,"filter as specified.\n");
	    break;
	case FMRESET:
	    fprintf(theOutput,"\treset\n");
	    fprintf(theOutput,"\nCompletely resets the internal state of fm. This includes rereading\n");
	    fprintf(theOutput,"the .fmrc file.\n");
	    break;
	case FMNEWPWD:
	    fprintf(theOutput,"\tnewpwd <password>\n");
	    fprintf(theOutput,"\nInstalls the specified password into the filter. If there is already a\n");
	    fprintf(theOutput,"\ncurrent password, it is used to encrypt the transfer.\n");
	    break;
	case FMQUIT:
	    fprintf(theOutput,"\tquit\n");
            fprintf(theOutput,"\nPretty self explanatory. fm will gripe if you quit and have upload'd\n");
            fprintf(theOutput,"tables into the filter without performing a write.\n");
	    break;
	case EMPTY:
	    fprintf(theOutput,"commands:\n");
	    fprintf(theOutput,"\tset (verbose|target|password|retries|timeout) <args>\n");
	    fprintf(theOutput,"\tload (network|classes|allow|reject) <filename>\n");
	    fprintf(theOutput,"\tshow (host|class|allow|reject|target|verbose|password|retries|timeout) [<args>]\n");
	    fprintf(theOutput,"\tquery (host|class|allow|reject|stats) [<args>]\n");
	    fprintf(theOutput,"\tupload (networks|classes|allow|reject)\n");
	    fprintf(theOutput,"\twrite\n");
	    fprintf(theOutput,"\trelease (classes|allow|reject|network) [<args>]\n");
	    fprintf(theOutput,"\treboot\n");
	    fprintf(theOutput,"\tclear (tables|stats)\n");
	    fprintf(theOutput,"\treset\n");
	    fprintf(theOutput,"\tnewpwd <password>\n");
	    fprintf(theOutput,"\tquit\n");
	    fprintf(theOutput,"'!' as the first character of a command passes the command to a shell\n");
	    fprintf(theOutput,"'#' is a comment\n");
	    break;
	case UNKNOWN:
	default:
	    fprintf(theMessages,"unknown help key word\n");
	    break;
    }
}

void handleInput(FILE *theInput,FILE *theOutput,FILE *theMessages)
{
    int result;
    char *currTok;
    char command[256];
        
    fprintf(theOutput,"fm> ");
    fgets(readBuffer,sizeof(readBuffer),theInput);

    while (!feof(theInput)) {

	currTok = strtok(readBuffer," \t\n");
	result = tokenize(currTok);

	switch (result) {
	    case FMSET:
		handleSet(theInput,theOutput,theMessages);
		break;
	    case FMLOAD:
		handleLoad(theInput,theOutput,theMessages);
		break;
	    case FMSHOW:
		handleShow(theInput,theOutput,theMessages);
		break;
	    case FMQUERY:
		handleQuery(theInput,theOutput,theMessages);
		break;
	    case FMUPLOAD:
		handleUpload(theInput,theOutput,theMessages);
		break;
	    case FMWRITE:
		handleWrite(theInput,theOutput,theMessages);
		break;
	    case FMQUIT:
		if (uploadedAndNoWrite)
		    fprintf(theMessages,"note: no write was performed\n");
		exit(0);
		break;
	    case FMCLEAR:
		handleClear(theInput,theOutput,theMessages);
		break;
	    case FMRESET:
		handleReset(theInput,theOutput,theMessages);
		break;
	    case FMRELEASE:
		handleRelease(theInput,theOutput,theMessages);
		break;
	    case FMREBOOT:
		handleReboot(theInput,theOutput,theMessages);
		break;
	    case FMNEWPWD:
		handleNewpwd(theInput,theOutput,theMessages);
		break;
	    case FMHELP:
		handleHelp(theInput,theOutput,theMessages);
		break;
	    case FMSHELL:
		strcpy(command,"/bin/csh -c \'");

		strcat(command,currTok + 1);
		while ((currTok = strtok((char *) NULL," "))) {
		    strcat(command," ");
		    strcat(command,currTok);
		}
		strcat(command,"'");

		if (system(command) == 127)
		    fprintf(theMessages,"shell escape failed\n");
		break;
	    case UNKNOWN:
		fprintf(theMessages,"unknown command\n");
		break;
	    case FMCOMMENT:
	    case EMPTY:
		/* Do nothing. Just read another line. */
		break;
	    default:
		fprintf(theMessages,"strange return value from lex\n");
		exit(1);
		break;
	}

        fprintf(theOutput,"fm> ");
        fgets(readBuffer,sizeof(readBuffer),theInput);
    }
}

void usage(void)
{
    fprintf(messages,"usage: fm\n");
    exit(1);
}

void readRc(void)
{
    struct passwd *pwd;
    struct stat statBuf;
    FILE *rcInput;
    FILE *nullOutput;
    char rcFilename[256];

    /* Get the users home directory.
     */
    pwd = getpwuid(getuid());

    if (pwd == (struct passwd *) NULL) {
        fprintf(messages,"Who are you?\n");
        exit(1);
    }

    /* Build a path to the user's ~/.fmrc
     */
    strcpy(rcFilename,pwd->pw_dir);
    strcat(rcFilename,"/.fmrc");

    if (stat(rcFilename,&statBuf) != -1) {
	/* Check the user's .fmrc and gripe if it is not in the correct mode.
	 */
        if ((statBuf.st_mode & 0x1FF) != 0x100) {
            fprintf(messages,"Your .fmrc must be mode 400\n");
            exit(1);
        }

	/* Process the .fmrc file.
	 */ 
        rcInput = fopen(rcFilename,"r");

        if (rcInput != (FILE *) NULL) {
            nullOutput = fopen("/dev/null","r+");
            handleInput(rcInput,nullOutput,messages);
        }
    }
}

int main(int argc,char *argv[])
{
    /*
     * Get the command line arguments.
     */
    if (argc != 1) {
        usage();
    }

    /*
     * Try to set up nit. This requires super user access.
     */
    initCommunications();

    /* Initialize the random number generator. This is used by the DES sequence number
     *   security code.
     */
    initPotp();

    /*
     * Read in the ~/.fmrc
     */
    readRc();

    /*
     * Redirect the output to /dev/null if the stdin is not a tty.
     */
    /*
    if (!isatty(0)) {
	output = fopen("/dev/null","r+");
    }
    */

    fprintf(output,"Filter Manager %s\n",VERSION);
    fprintf(output,"Copyright (C) 1993, 1995 Texas A&M University\n");

    /*
     * Handle user input. Only returns when the user has entered an EOF.
     */
    handleInput(input,output,messages);

    fprintf(output,"\n");

    if (uploadedAndNoWrite)
	fprintf(messages,"note: no write was performed\n");

    return 0;
}
