/*
     Filter - Packet filtering software for the Drawbridge package
     Copyright (C) 1993 David K. Hess, Douglas Lee Schales, David R. Safford

     Please see the file `COPYING' for the complete copyright notice.

     WD.C - Version 1.0 - 4/21/93
*/

#include <dos.h>
#include <io.h>
#include <alloc.h>
#include <process.h>
#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include <mem.h>
#include <string.h>
#include <fcntl.h>
#include <sys\stat.h>
#include <dir.h>
#include "filter.h"
#include "wd.h"

WDInterface interfaces[NUMINTERFACES];

int numInterfaces = 0;

// Structure of a packet.
//
// The beginning of a packet is 256 byte aligned and has a 4 byte header.
//
// |- byte 0 -|- byte 1 -|-- bytes 2 and 3 --|
//    status    next ptr   length of packet
//
// Note that the next ptr is a byte that references a particular 256 byte block
//   in the max 16K shared memory space. (There are 64 of these blocks.)
//
// The structure of the shared memory buffer ring follows (assuming 16K):
//
// |- block 0 -|- block 1 -|- block....-|- block 62 -|- block 63 -|
//
// There are two pointers that define the status of the ring buffer. BNRY is
//   the boundary between the currently free blocks and the packets waiting
//   to be processed. It in particular marks the last free block in the ring.
//   CURR is a pointer to the next free block that the card will use. It in
//   particular marks the first free block of the ring.
//
// Note that if BNRY + 1 == CURR then there are no packets waiting.
//
// Note also that the first 6 blocks (0 thru 5 = 1536 bytes) are used as the output
//   buffer so that the ring buffer math is NOT modulo 64.
//
// The CREG register uses the bottom 6 bits to decode where in memory the card's
//   shared memory should appear. These 6 bits correspond to bits 18-13 of the
//   memory address. (Falling on a 8K boundary).
WDInterface *WDDefine(unsigned short ioAddress,unsigned char *bufferAddress,int dataPath)
{
    int i;
    int newInterface;

    //fprintf(stdout,"Card memory is configured at %04X:%04X\n",
    //          FP_SEG(bufferAddress),
    //          FP_OFF(bufferAddress));

    newInterface = numInterfaces++;

    // Save off the info. for the interface.
    interfaces[newInterface].ioAddress     = ioAddress;
    interfaces[newInterface].bufferAddress = bufferAddress;

    // Get the ethernet address out of the rom.
    fprintf(stdout,"Ethernet address: ");
    for (i = 0;i < 6;++i) {
	interfaces[newInterface].etherAddress.bytes[i] = inportb(ioAddress + ADDROM + i);
	fprintf(stdout,"%02X",interfaces[newInterface].etherAddress.bytes[i]);
	if (i != 5)
	    fprintf(stdout,":");
    }
    fprintf(stdout,"\n");

    // Reset the card and clear shared memory decoding.
    outportb(ioAddress + CREG,MSK_RESET);
    outportb(ioAddress + CREG,0x00);

    // Set up the number of pages. Also set up 16K buffers.
    if (dataPath == 16) {
	interfaces[newInterface].numPages = 64;
	outportb(ioAddress + LAAR,LAN16ENB);
    }
    else
	interfaces[newInterface].numPages = 32;

    interfaces[newInterface].cardID        = newInterface + 1;
    interfaces[newInterface].currBuffer    = 0;
    interfaces[newInterface].transmitting  = NO;
    interfaces[newInterface].lengthBuf1    = 0;
    interfaces[newInterface].lengthBuf2    = 0;
    interfaces[newInterface].packetWaiting = NO;
    interfaces[newInterface].packetGood    = NO;
    interfaces[newInterface].packetLength  = 0;

    // Load the memory base register and enable the shared memory.
    outportb(ioAddress + CREG,FP_SEG(bufferAddress) >> 9);

    // Make sure we are on page 0.
    outportb(ioAddress + CMDR,MSK_PG0 | MSK_RD2);

    // Select FIFO threshold = 8 bytes. This enables the 16 bit datapath
    //   also if it is a 16K board.
    if (dataPath == 16)
	outportb(ioAddress + DCR,MSK_BMS | MSK_FT10 | MSK_WTS);
    else
	outportb(ioAddress + DCR,MSK_BMS | MSK_FT10);

    // Clear RBCR0 and RBCR1.
    outportb(ioAddress + RBCR0,0);
    outportb(ioAddress + RBCR1,0);

    // Turn off receiving.
    outportb(ioAddress + RCVR,MSK_MON);

    // Clear TCR.
    outportb(ioAddress + TCR,0);

    // Set up the end of the input buffer (in 256 byte pages)
    outportb(ioAddress + PSTOP,interfaces[newInterface].numPages);

    // Set up the start of the input buffer (in 256 byte pages)
    outportb(ioAddress + PSTART,STRT_PG);

    // Set up the boundary register.
    outportb(ioAddress + BNRY,STRT_PG);

    // Set up for no interrupts.
    outportb(ioAddress + ISR,0xFF);
    outportb(ioAddress + IMR,0x00);

    // Move to page one.
    outportb(ioAddress + CMDR,MSK_PG1 | MSK_RD2);

    // Set up the card's ring buffer pointer.
    outportb(ioAddress + CURR,STRT_PG + 1);

    // Set the ethernet address.
    for (i = 0;i < 6;++i)
	outportb(ioAddress + PAR0 + i,interfaces[newInterface].etherAddress.bytes[i]);

    // Set multicast address to all zeros.
    for (i = 0;i < 8;++i)
	outportb(ioAddress + MAR0 + i,0);

    // Move back to page 0.
    outportb(ioAddress + CMDR,MSK_PG0 | MSK_RD2);

    // Activate the 8390 chip.
    outportb(ioAddress + CMDR,MSK_STA | MSK_RD2);

    // Set the card into promiscuous mode.
    outportb(ioAddress + RCVR,MSK_AB | MSK_AM | MSK_PRO);

    // Return back the new interface number.
    return (WDInterface far *) (interfaces + newInterface);
}

unsigned char *WDGetReadAccess(WDInterface *interface)
{
    unsigned char bnry;
    unsigned char curr;
    unsigned char *buffer;
    unsigned char next;
    unsigned short ioAddress;

    ioAddress = interface->ioAddress;

    // Move to page zero and get the boundary register.
    outportb(ioAddress + CMDR,MSK_PG0 + MSK_RD2);
    bnry = inportb(ioAddress + BNRY);

    // Move to page 1 and get the current page register.
    outportb(ioAddress + CMDR,MSK_PG1 | MSK_RD2);
    curr = inportb(ioAddress + CURR);

//    if (bnry == curr) {
//      fprintf(stdout,"@");
//      fflush(stdout);
//    }

//    if (bnry <= curr) {
//      fprintf(stdout,"%03d-",bnry + 64 - curr);
//    }
//    else {
//      fprintf(stdout,"%03d-",bnry - curr);
//    }

    // Increment the boundary value with wrap around.
    ++bnry;
    if (bnry >= interface->numPages)
	bnry = STRT_PG;

    // Check if a packet is ready.
    if (bnry == curr) {
	return (unsigned char *) NULL;
    }
    else {
	buffer = interface->bufferAddress +
		 (((unsigned short) bnry) << 8);

	// fprintf(stdout,"%04X:%04X\n",FP_SEG(buffer),FP_OFF(buffer));

	//for (i = 0;i < 25;++i)  {
	//      fprintf(stdout,"%02X ",buffer[i]);
	//}

	//fprintf(stdout,"\n");
	//fprintf(stdout,"%02x\n",SMK_PRX | SMK_PHY);

	// Check if the packet is ok.
	if ((buffer[0] != (SMK_PRX | SMK_PHY)) &&
	    (buffer[0] != SMK_PRX)) {
	    //fprintf(stdout,"bad status\n");
	    goto badPacket;
	}
	else {

	    // Get the pointer to the next packet.
	    next = buffer[1];

	    // fprintf(stdout,"bnry = %d next = %d curr = %d\n",bnry,next,curr);

	    // Perform a sanity check on the next packet pointer.
	    if (bnry <= curr) {
		// **** BNRY **** NEXT **** CURR ****
		if (next < bnry || next > curr) {
		    //fprintf(stdout,"bad next pointer - 1\n");
		    // exit(1);
		    goto badPacket;
		}
	    }
	    else {
		if (next >= bnry) {
		    // **** CURR **** BNRY **** NEXT ****
		    if (next >= interface->numPages) {
			//fprintf(stdout,"bad next pointer - 2\n");
			goto badPacket;
		    }
		}
		else {
		    // **** NEXT **** CURR **** BNRY ****
		    if (next < STRT_PG || next > curr) {
			//fprintf(stdout,"bad next pointer - 3\n");
			goto badPacket;
		    }
		}
	    }

	    // A good packet!
	    return buffer;
	}
    }

badPacket:
//    fprintf(stdout,"bnry = %d next = %d curr = %d\n",bnry,next,curr);
//    fprintf(stdout,"buffer = %04X:%04X\n",FP_SEG(buffer),FP_OFF(buffer));

    // Set up for page 0.
    outportb(ioAddress + CMDR,MSK_PG0 | MSK_RD2);

    // Write out the new boundary value at old value + 1.
    outportb(ioAddress + BNRY,bnry);

    return (unsigned char *) NULL;
}

void WDFree(WDInterface *interface,unsigned char next)
{
    unsigned short ioAddress;

    ioAddress = interface->ioAddress;

    // Set up for page 0.
    outportb(ioAddress + CMDR,MSK_PG0 | MSK_RD2);

    // Free up the blocks up to the beginning of the next packet.
    if (--next < STRT_PG)
	next = interface->numPages - 1;

    // Write out the new boundary value.
    outportb(ioAddress + BNRY,next);
}

unsigned char WDIsBusy(WDInterface *interface)
{
    return inportb(interface->ioAddress + CMDR) & MSK_TXP;
}

void WDWrite(WDInterface *interface,unsigned short length)
{
    unsigned short ioAddress;

    // fprintf(stdout,"WRITE\n");

    ioAddress = interface->ioAddress;

    // Set up for page 0.
    outportb(ioAddress + CMDR,MSK_PG0 | MSK_RD2);

    // Set the transmission length.
    outportb(ioAddress + TBCR0,length & 0x00FF);
    outportb(ioAddress + TBCR1,(length & 0xFF00) >> 8);

    // Tell the card where the packet begins (block 0).
    outportb(ioAddress + TPSR,interface->currBuffer);

    // Send the packet.
    outportb(ioAddress + CMDR,MSK_TXP | MSK_RD2);
}

void WDMove(int length,
	    unsigned char *fromBuffer,
	    WDInterface *fromInterface,
	    unsigned char *toBuffer)
{
    unsigned char *bufferEnd;
    int bufferLength;
    int buffLeft;

    bufferEnd = fromInterface->bufferAddress + (fromInterface->numPages << 8);

    //fprintf(stdout,"dest buff = %04X:%04X packet = %04X:%04X length = %d\n",
    //          FP_SEG(toBuffer),FP_OFF(toBuffer),
    //          FP_SEG(fromBuffer),FP_OFF(fromBuffer),length);

    bufferLength = (int) (bufferEnd - fromBuffer);

    // This could be written in a more concise way but it would involve
    //   checking a conditional on each iteration of the loop. This way
    //   will be a little bit more efficient.

    // Since each block is on a word boundary we can move words for
    //   increased efficiency.
    if (bufferLength >= length) {

	// No wrap.
	length = (length + 1) >> 1;

	moveWords(length,toBuffer,fromBuffer);

	//for (i = 0;i < length;++i)
	//    ((unsigned short *)toBuffer)[i] = ((unsigned short *)fromBuffer)[i];
    }
    else {
	// Wrap.
	// Set up the first part. If the packet wraps then this first
	//   part will be word aligned.
	buffLeft = (int) (bufferEnd - fromBuffer);

	length = (length - buffLeft + 1) >> 1;

	buffLeft = buffLeft >> 1;

	// Got to wrap the packet. Do the first part.
	//for (i = 0;i < buffLeft;++i)
	//    ((unsigned short *)toBuffer)[i] = ((unsigned short *)fromBuffer)[i];
	moveWords(buffLeft,toBuffer,fromBuffer);

	// Set up the next part.
	fromBuffer = fromInterface->bufferAddress + (STRT_PG << 8);

	//toPos = i;

	// Do the second part.
	//for (i = 0;i < length;++i,++toPos)
	//    ((unsigned short *)toBuffer)[toPos] = ((unsigned short *)fromBuffer)[i];
	moveWords(length,((unsigned short *)toBuffer) + buffLeft,fromBuffer);

	//fprintf(stdout,"buffLeft = %d length = %d fromBuffer = %04X:%04X\n",
	//      buffLeft,length,FP_SEG(fromBuffer),FP_OFF(fromBuffer));
	//exit(1);
    }
}

void WDCopy(int length,
	    unsigned char *fromBuffer,
	    WDInterface *fromInterface,
	    WDInterface *toInterface,
	    int bufnum)
{
    unsigned char *toBuffer;

    toBuffer = toInterface->bufferAddress + (bufnum << 8);

    WDMove(length,fromBuffer,fromInterface,toBuffer);
}
