/*
 *	$Id: fas.c
 */

/*
 * AMIGA Emulex FAS216 scsi adaptor driver
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/device.h>
#include <sys/buf.h>
#include <sys/proc.h>
#include <scsi/scsi_all.h>
#include <scsi/scsiconf.h>
#include <vm/vm.h>
#include <vm/vm_kern.h>
#include <vm/vm_page.h>
#include <machine/pmap.h>
#include <machine/cpu.h>
#include <amiga/amiga/device.h>
#include <amiga/amiga/cc.h>
#include <amiga/amiga/custom.h>
#include <amiga/dev/fasreg.h>
#include <amiga/dev/fasvar.h>

void	fas_minphys __P((struct buf *bp));
int	fas_scsicmd __P((struct scsi_xfer *xs));

void	fasinitialize __P((struct fas_softc *));

int	fas_donextcmd(struct fas_softc *dev,
		      struct vm_link_data *vm_link_data);
void	fas_scsidone(struct fas_softc *dev, int stat);
void	fasintr(struct fas_softc *dev);

void	fasreset(struct fas_softc *dev, int how);
void	fasselect(struct fas_softc *dev, int target, int lun,
		  unsigned char *cbuf, int clen, 
		  unsigned char *buf, int len, int mode);

#ifdef DEBUG
#define QPRINTF(a) if (fas_debug > 1) printf a
int	fas_debug = 0;
#else
#define QPRINTF
#endif

/*
 * default minphys routine for sci based controllers
 */
void fas_minphys(struct buf *bp)
{	/* no max transfer at this level */
}

void fas_init_nexus(struct fas_softc *dev, struct nexus *nexus)
{
  nexus->xs = 0;
  nexus->flags = 0;
  nexus->state = FAS_NS_IDLE;
  nexus->syncper = 5;
  nexus->syncoff = 0;
  nexus->config3 = dev->sc_config3 & ~FAS_CFG3_FASTSCSI;
  nexus->max_link = 0;
  nexus->cur_link = 0;
  nexus->vm_link_data.pages = 0;
  nexus->vm_link_data.offset = 0;
}

void fasinitialize(struct fas_softc *dev)
{
  fas_regmap_p	rp;
  int		i;

  dev->sc_led_status = 0;

  TAILQ_INIT(&dev->sc_xs_pending);
  TAILQ_INIT(&dev->sc_xs_free);

  for(i=0; i<MAXPENDING; i++)
    {
      dev->sc_xs_store[i].vm_link_data.pages = 0;
      TAILQ_INSERT_TAIL(&dev->sc_xs_free, &dev->sc_xs_store[i], link);
    }

  rp = dev->sc_fas;

  if (dev->sc_clock_freq <= 10)
    dev->sc_clock_conv_fact = 2;
  else if (dev->sc_clock_freq <= 40)
    dev->sc_clock_conv_fact = 2+((dev->sc_clock_freq-10)/5);
  else
    {
      dev->sc_clock_conv_fact = 8;
      panic("fasinitialize: Clock frequence too high");
    }

  dev->sc_config1 = (dev->sc_host_id & FAS_CFG1_BUS_ID_MASK);
  dev->sc_config2 = FAS_CFG2_FEATURES_ENABLE;
  dev->sc_config3 = (dev->sc_clock_freq > 25 ? FAS_CFG3_FASTCLK : 0);

  dev->sc_timeout_val =
    1+dev->sc_timeout/(0.001/dev->sc_clock_freq*7682*dev->sc_clock_conv_fact);
  dev->sc_clock_period = 1000/dev->sc_clock_freq;

  fasreset(dev, 1 | 2);	/* Reset Chip and Bus */

  dev->sc_state = FAS_S_IDLE;
  dev->sc_units_busy = 0;
  dev->sc_units_disconnected = 0;
  dev->sc_msg_in_len = 0;
  dev->sc_msg_out_len = 0;

  dev->sc_flags = 0;

  for(i=0; i<8; i++)
    fas_init_nexus(dev, &dev->sc_nexus[i]);

  vm_map_lock(kernel_map);
  if (vm_map_findspace(kernel_map, 0, MAXPHYS+NBPG,
		       (vm_offset_t *)&dev->sc_vm_link))
    {
      vm_map_unlock(kernel_map);
      panic("FAS_SCSICMD: No VM space available.");
    }
  else
    {
      int	offset;
      
      offset = (vm_offset_t)dev->sc_vm_link - VM_MIN_KERNEL_ADDRESS;
      printf(" vm_link %x", dev->sc_vm_link, offset);
      vm_object_reference(kernel_object);
      vm_map_insert(kernel_map, kernel_object, offset,
		    (vm_offset_t)dev->sc_vm_link,
		    (vm_offset_t)dev->sc_vm_link+(MAXPHYS+NBPG));
      vm_map_unlock(kernel_map);
    }

  dev->sc_vm_link_data.pages = 0;
  dev->sc_vm_link_data.offset = 0;
}

void fas_unlink_vm_link(struct fas_softc *dev)
{
  if (dev->sc_flags & FAS_HAS_VM_LINK)
    {
      physunaccess(dev->sc_vm_link, dev->sc_vm_link_data.pages*NBPG);
      dev->sc_vm_link_data.pages = 0;
      dev->sc_vm_link_data.offset = 0;
      dev->sc_flags &= ~FAS_HAS_VM_LINK;
    }
}

void fas_link_vm_link(struct fas_softc *dev, struct vm_link_data *vm_link_data)
{
  int	i;

  if (dev->sc_flags & FAS_HAS_VM_LINK)
    fas_unlink_vm_link(dev);

  dev->sc_vm_link_data = *vm_link_data;

  if (dev->sc_vm_link_data.pages)
    {
      for(i=0; i<dev->sc_vm_link_data.pages; i++)
	physaccess(dev->sc_vm_link+i*NBPG, dev->sc_vm_link_data.pa[i],
		   NBPG, PG_CI);
      
      dev->sc_flags |= FAS_HAS_VM_LINK;
    }
}

/* used by specific scem controller */
int fas_scsicmd(struct scsi_xfer *xs)
{
  struct fas_softc	*dev;
  struct scsi_link	*slp;
  struct fas_pending	*pendp;
  int			 flags, s, target;
  struct vm_link_data	 vm_link_data;

  slp = xs->sc_link;
  dev = slp->adapter_softc;
  flags = xs->flags;
  target = slp->target;

  vm_link_data.offset = 0;
  vm_link_data.pages = 0;

  if (flags & SCSI_DATA_UIO)
    panic("fas: scsi data uio requested");

  if ((flags & SCSI_POLL) &&
      (dev->sc_xs || (dev->sc_units_busy & (1<<target))))
    panic("fas_scsicmd: busy");

  if (!dev->sc_dma_ok(dev, xs->data, xs->datalen) && !(flags & SCSI_POLL))
    {
      vm_offset_t	sva;
      int		len, n;
	    
      vm_link_data.offset = (vm_offset_t)xs->data & PGOFSET;

      sva = (vm_offset_t)xs->data & PG_FRAME;
      len = round_page(xs->data+xs->datalen-sva);
	    
      vm_link_data.pages = (len+NBPG-1)/NBPG;
      for(n=0; n<vm_link_data.pages; n++)
	{
	  vm_link_data.pa[n] = kvtop(sva);
	  sva += NBPG;
	}
    }

  s = splbio();
  pendp = dev->sc_xs_free.tqh_first;
  if (!pendp)
    {
      splx(s);
      return(TRY_AGAIN_LATER);
    }

  if (dev->sc_xs || (dev->sc_units_busy & (1<<target)))
    {
      TAILQ_REMOVE(&dev->sc_xs_free, pendp, link);
      pendp->xs = xs;
      pendp->vm_link_data = vm_link_data;
      TAILQ_INSERT_TAIL(&dev->sc_xs_pending, pendp, link);

      splx(s);
      return(SUCCESSFULLY_QUEUED);
    }

  dev->sc_xs = xs;
  splx(s);
  
  fas_donextcmd(dev, &vm_link_data);

  if (flags & SCSI_POLL)
    return(COMPLETE);
  return(SUCCESSFULLY_QUEUED);
}

fas_donextcmd(struct fas_softc *dev, struct vm_link_data *vm_link_data)
{
  struct scsi_xfer	*xs;
  struct scsi_link	*slp;
  int			 flags;
  int			 stat;

  xs = dev->sc_xs;
  slp = xs->sc_link;
  flags = xs->flags;

  if (flags & SCSI_RESET)
    fasreset(dev, 2);

  dev->sc_flags = 0;

  dev->sc_stat[0] = -5;
  if (!(flags & (SCSI_DATA_OUT | SCSI_DATA_IN)) || (flags & SCSI_POLL))
    {
      dev->sc_vm_link_data.pages = 0;
      stat = fasicmd(dev, slp->target, slp->lun,
		     xs->cmd, xs->cmdlen, xs->data, xs->datalen);
    }
  else if (!fasgo(dev, xs, vm_link_data))
    return;
  else
    stat = dev->sc_stat[0];

  fas_scsidone(dev, stat);
}

void fas_scsidone(struct fas_softc *dev, int stat)
{
  struct fas_pending	*pendp;
  struct scsi_xfer	*xs;
  struct vm_link_data	 vm_link_data;
  int			 s, donext;

  xs = dev->sc_xs;
  if (xs == NULL)
    panic("fas_scsidone: no xs!");

  xs->resid = dev->sc_len;
  xs->status = stat;

  if (stat == 0)
    xs->resid = 0;
  else
    {
      switch(stat)
	{
	case SCSI_CHECK:
	  if (stat = fasgetsense(dev, xs))
	    xs->error = XS_DRIVER_STUFFUP;
	  else
	    xs->error = XS_SENSE;
	  break;
	case SCSI_BUSY:
	  xs->error = XS_BUSY;
	  break;
	default:
	  xs->error = XS_TIMEOUT;
	  QPRINTF(("fas_scsicmd() bad %x\n", stat));
	  break;
	}
    }

  fas_unlink_vm_link(dev);

  xs->flags |= ITSDONE;

  s = splbio();
  pendp = dev->sc_xs_pending.tqh_first;
  while(pendp)
    {
      if (!(dev->sc_nexus[pendp->xs->sc_link->target].flags & FAS_UNIT_BUSY))
	break;
      pendp = pendp->link.tqe_next;
    }

  if (!pendp)
    {
      donext = 0;
      dev->sc_xs = NULL;
    }
  else
    {
      donext = 1;
      TAILQ_REMOVE(&dev->sc_xs_pending, pendp, link);
      dev->sc_xs = pendp->xs;
      vm_link_data = pendp->vm_link_data;
      pendp->xs = NULL;
      pendp->vm_link_data.pages = 0;
      TAILQ_INSERT_TAIL(&dev->sc_xs_free, pendp, link);
    }
  splx(s);
  scsi_done(xs);

  if (donext)
    fas_donextcmd(dev, &vm_link_data);
}

void fasreset(struct fas_softc *dev, int how)
{
  fas_regmap_p	rp;
  int		i;

  for(i=0; i<8; i++)
    fas_init_nexus(dev, &dev->sc_nexus[i]);

  rp = dev->sc_fas;

  if (how & 1)
    {
      rp->fas_command = FAS_CMD_RESET_CHIP;
      delay(1);
      rp->fas_command = FAS_CMD_NOP;

      rp->fas_config1 = dev->sc_config1;
      rp->fas_config2 = dev->sc_config2;
      rp->fas_config3 = dev->sc_config3;
      rp->fas_timeout = dev->sc_timeout_val;
      rp->fas_clkconv = dev->sc_clock_conv_fact & FAS_CLOCK_CONVERSION_MASK;
    }
  if (how & 2)
    {
      rp->fas_command = FAS_CMD_RESET_SCSI_BUS;
      delay(10000);

/* Skip interrupt generated by RESET_SCSI_BUS */
      while(rp->fas_status & FAS_STAT_INTERRUPT_PENDING)
	{
	  dev->sc_status = rp->fas_status;
	  dev->sc_sequence = rp->fas_seqstep;
	  dev->sc_interrupt = rp->fas_interrupt;
	  
	  delay(10000);
	}

      dev->sc_status = rp->fas_status;
      dev->sc_sequence = rp->fas_seqstep;
      dev->sc_interrupt = rp->fas_interrupt;
    }

  delay(250000);		/* RESET to SELECT DELAY */
}

void fas_save_pointers(struct fas_softc *dev)
{
  struct nexus	*nx;

  nx = dev->sc_cur_nexus;
  if (nx)
    {
      nx->cur_link	= dev->sc_cur_link;
      nx->max_link	= dev->sc_max_link;
      nx->buf		= dev->sc_buf;
      nx->len		= dev->sc_len;
      nx->dma_len	= dev->sc_dma_len;
      nx->dma_buf	= dev->sc_dma_buf;
      nx->dma_blk_len	= dev->sc_dma_blk_len;
      nx->dma_blk_ptr	= dev->sc_dma_blk_ptr;
    }
}

void fas_restore_pointers(struct fas_softc *dev)
{
  struct nexus	*nx;

  nx = dev->sc_cur_nexus;
  if (nx)
    {
      dev->sc_cur_link	  = nx->cur_link;
      dev->sc_max_link	  = nx->max_link;
      dev->sc_buf	  = nx->buf;
      dev->sc_len	  = nx->len;
      dev->sc_dma_len	  = nx->dma_len;
      dev->sc_dma_buf	  = nx->dma_buf;
      dev->sc_dma_blk_len = nx->dma_blk_len;
      dev->sc_dma_blk_ptr = nx->dma_blk_ptr;
    }
}

int fasgetsense(struct fas_softc *dev, struct scsi_xfer *xs)
{
  struct scsi_link	*slp;
  struct scsi_sense	 rqs;
  int			 stat;

  slp = xs->sc_link;
	
  rqs.opcode = REQUEST_SENSE;
  rqs.byte2 = slp->lun << 5;
#ifdef not_yet
  rqs.length = xs->req_sense_length ? xs->req_sense_length : sizeof(xs->sense);
#else
  rqs.length = sizeof(xs->sense);
#endif
  
  rqs.unused[0] = rqs.unused[1] = rqs.control = 0;
  
  stat = fasicmd(dev, slp->target, slp->lun, &rqs, sizeof(rqs),
		 &xs->sense, rqs.length, SCSI_DATA_IN);

  return(stat);
}

/* Immediate commands */
int fasiwait(struct fas_softc *dev)
{
  fas_regmap_p	rp;
  register int	wait;

  rp = dev->sc_fas;

  wait = 500000;
  while(!(rp->fas_status & FAS_STAT_INTERRUPT_PENDING) && wait)
    delay(1);

  if (wait)
    {
      dev->sc_status = rp->fas_status;
      dev->sc_sequence = rp->fas_seqstep;
      dev->sc_interrupt = rp->fas_interrupt;
    }

  return(wait == 0);
}

void fas_ixfer(struct fas_softc *dev)
{
  fas_regmap_p	 rp;
  u_char	*buf;
  int		 len, mode, phase;

  rp = dev->sc_fas;
  buf = dev->sc_buf;
  len = dev->sc_len;

  phase = dev->sc_status & FAS_STAT_PHASE_MASK;
  mode = (phase == FAS_PHASE_DATA_IN);

  while(len && ((dev->sc_status & FAS_STAT_PHASE_MASK) == phase))
    if (mode)
      {
	rp->fas_command = FAS_CMD_TRANSFER_INFO;
	
	if (fasiwait(dev)) break;
	
	*buf++ = rp->fas_fifo;
	len--;
      }
    else
      {
	len--;
	rp->fas_fifo = *buf++;
	rp->fas_command = FAS_CMD_TRANSFER_INFO;

	if (fasiwait(dev)) break;
      }

  dev->sc_buf = buf;
  dev->sc_len = len;
}

void fas_build_sdtrm(struct fas_softc *dev, int period, int offset)
{
  dev->sc_msg_out[0] = 0x01;
  dev->sc_msg_out[1] = 0x03;
  dev->sc_msg_out[2] = 0x01;
  dev->sc_msg_out[3] = period/4;
  dev->sc_msg_out[4] = offset;
  dev->sc_msg_out_len = 5;
}

void fasselect(struct fas_softc *dev, int target, int lun,
	       unsigned char *cbuf, int clen, 
	       unsigned char *buf, int len, int mode)
{
  fas_regmap_p	 rp;
  struct nexus	*nexus;
  int		 i;
  u_char	 state, ID, sync, cmd;

  if (cpu040 && len && !(mode & FAS_SELECT_I))
    dma_cachectl(buf, len);

  rp = dev->sc_fas;

  dev->sc_unit = target;
  dev->sc_lun = lun;

  if (dev->sc_vm_link_data.pages)
    dev->sc_buf = dev->sc_vm_link + dev->sc_vm_link_data.offset;
  else
    dev->sc_buf = buf;
  dev->sc_len = len;

  nexus = &dev->sc_nexus[target];
  dev->sc_cur_nexus = nexus;
  dev->sc_chain = nexus->dma;

  dev->sc_max_link = dev->sc_build_dma_chain(dev, buf, len);
  dev->sc_cur_link = 0;
  dev->sc_dma_len = 0;
  dev->sc_dma_buf = 0;
  dev->sc_dma_blk_len = 0;
  dev->sc_dma_blk_ptr = 0;

  fas_save_pointers(dev);
  nexus->xs = dev->sc_xs;

  nexus->clen = clen;
  bcopy(cbuf, nexus->cbuf, nexus->clen);

  nexus->vm_link_data = dev->sc_vm_link_data;

  nexus->flags &= FAS_SYNC_TESTED;
  nexus->state = FAS_NS_SELECTED;

  rp->fas_syncper = nexus->syncper;
  rp->fas_syncoff = nexus->syncoff;
  rp->fas_config3 = nexus->config3;
  rp->fas_config1 = dev->sc_config1;
  rp->fas_timeout = dev->sc_timeout_val;
  rp->fas_dest_id = target;

  state = ((mode & FAS_SELECT_I) ? FAS_S_IMMEDIATE : FAS_S_INTERRUPT);
  ID    = ((mode & FAS_SELECT_R) ? 0xC0 : 0x80) | lun;
  sync  = ((mode & FAS_SELECT_S) ? 1 : 0);

  if (sync && (state == FAS_S_IMMEDIATE))
    sync = 0;

  dev->sc_state = state;
  rp->fas_fifo = ID;

  if (!sync && ((nexus->flags & FAS_SYNC_TESTED) && (nexus->syncoff != 0)))
    {
      nexus->state = FAS_NS_SELECTED_SDTR;

      fas_build_sdtrm(dev, 200, 0);
      cmd = FAS_CMD_SEL_ATN_STOP;
    }
  else if (!sync || (nexus->flags & FAS_SYNC_TESTED))
    {
      for(i=0; i<clen; i++)
	rp->fas_fifo = (cbuf[i] | (i==1? (lun<<5) : 0));

      cmd = FAS_CMD_SEL_ATN;
    }
  else
    {
      nexus->state = FAS_NS_SELECTED_SDTR;

      fas_build_sdtrm(dev, ((dev->sc_clock_freq>25) ? 100 : 200), 8);
      cmd = FAS_CMD_SEL_ATN_STOP;
    }

  rp->fas_command = cmd;

  dev->sc_stat[0] = 0xff;
  dev->sc_msg_in[0] = 0xff;
  dev->sc_msg_in_len = 0;
}

int fasgo(struct fas_softc *dev, struct scsi_xfer *xs,
	  struct vm_link_data *vm_link_data)
{
  int	target, lun;

  target = xs->sc_link->target;
  lun    = xs->sc_link->lun;

  dev->sc_vm_link_data = *vm_link_data;

  fasselect(dev, target, lun, (char *)xs->cmd, xs->cmdlen, 
	    xs->data, xs->datalen, FAS_SELECT_S);

  return(0);
}

int fas_pretests(struct fas_softc *dev, fas_regmap_p rp)
{
  struct nexus	*nexus;
  int		 i;

  if (dev->sc_interrupt & FAS_INT_SCSI_RESET_DETECTED)
    {
      for(i=0; i<8; i++)
	{
	  if (dev->sc_nexus[i].xs)
	    {
	      dev->sc_xs = dev->sc_nexus[i].xs;
	      fas_scsidone(dev, -1);
	    }
	  fas_init_nexus(dev, &dev->sc_nexus[i]);
	}
      printf("fasintr: SCSI-RESET detected!");
      return(-1);
    }

  if (dev->sc_interrupt & FAS_INT_ILLEGAL_COMMAND)
    {
      printf("FIFO:");
      while(rp->fas_fifo_flags & FAS_FIFO_COUNT_MASK)
	printf(" %x", rp->fas_fifo);
      printf("\n");

      printf("CMD: %x\n", rp->fas_command);
      panic("fasintr: ILLEGAL COMMAND!");
    }

  if (dev->sc_interrupt & FAS_INT_RESELECTED)
    {
      if (dev->sc_units_disconnected)
	{
	  dev->sc_resel[0] &= ~(1<<dev->sc_host_id);

	  for(i=0; i<8; i++)
	    if (dev->sc_resel[0] & (1<<i))
	      break;

	  if (dev->sc_nexus[i].flags & FAS_DISCONNECTED)
	    {
	      dev->sc_cur_nexus = &dev->sc_nexus[i];
	      nexus = dev->sc_cur_nexus;
	      dev->sc_chain = nexus->dma;
	      dev->sc_xs = nexus->xs;
	      dev->sc_state = FAS_S_INTERRUPT;

	      nexus->flags &= ~(FAS_DISCONNECTED | FAS_GOT_DISCONN);
	      dev->sc_units_disconnected--;

	      rp->fas_dest_id = i & 7;
	      rp->fas_command = FAS_CMD_FLUSH_FIFO;
	      rp->fas_command = FAS_CMD_MESSAGE_ACCEPTED;
	      for(i=0; i<10; i++);

	      fas_restore_pointers(dev);

	      if (nexus->vm_link_data.pages)
		fas_link_vm_link(dev, &nexus->vm_link_data);

	      return(1);
	    }
	}

      panic("fasintr: Unexpected reselection!");
    }

  return(0);
}

int fas_midaction(struct fas_softc *dev, fas_regmap_p rp, struct nexus *nexus)
{
  int	i, left, len;

  if (dev->sc_interrupt & FAS_INT_DISCONNECT)
    {
      if ((nexus->flags & FAS_GOT_DISCONN) &&
	  (nexus->state != FAS_NS_DONE))
	{
	  nexus->state = FAS_NS_DISCONNECTED;

	  nexus->flags &= ~FAS_GOT_DISCONN;
	  nexus->flags |= FAS_DISCONNECTED;

	  dev->sc_units_disconnected++;

	  rp->fas_command = FAS_CMD_FLUSH_FIFO;

	  fas_unlink_vm_link(dev);

	  dev->sc_xs = 0;
	  dev->sc_state = FAS_S_IDLE;

	  rp->fas_command = FAS_CMD_ENABLE_RESEL;

	  return(1);
	}
      
      switch(nexus->state)
	{
	case FAS_NS_DONE:
	  break;
	case FAS_NS_SELECTED:
	case FAS_NS_SELECTED_SDTR:
	  if (dev->sc_timeout_val < 255)
	    dev->sc_timeout_val++;
     	  rp->fas_command = FAS_CMD_FLUSH_FIFO;
	  dev->sc_stat[0] = -2;
	  break;
	default:
	  printf("fasintr: Unexpected disconnection\n");
	  printf("fasintr: unit %d state %d phase %d\n",
		 nexus->xs->sc_link->target, nexus->state,
		 dev->sc_status & FAS_STAT_PHASE_MASK);
	  printf("fasintr: flags %x\n", nexus->flags);

	  printf("FIFO:");
	  while(rp->fas_fifo_flags & FAS_FIFO_COUNT_MASK)
	    printf(" %x", rp->fas_fifo);
	  printf("\n");

	  dev->sc_stat[0] = -3;
	  break;
	}

      if (dev->sc_units_disconnected)
	rp->fas_command = FAS_CMD_ENABLE_RESEL;

      if (dev->sc_state == FAS_S_IMMEDIATE)
	{
	  dev->sc_state = FAS_S_IDLE;
	  nexus->flags &= ~FAS_UNIT_BUSY;
	  dev->sc_led_off(dev);
	}
      else
	{
	  dev->sc_state = FAS_S_IDLE;
	  nexus->flags &= ~FAS_UNIT_BUSY;
	  dev->sc_led_off(dev);
	  fas_scsidone(dev, dev->sc_stat[0]);
	}

      return(-1);
    }

  switch(nexus->state)
    {
    case FAS_NS_SELECTED:
    case FAS_NS_SELECTED_SDTR:
      if (!(nexus->flags & FAS_UNIT_BUSY))
	dev->sc_led_on(dev);

      if (nexus->vm_link_data.pages)
	fas_link_vm_link(dev, &nexus->vm_link_data);

      nexus->flags |= FAS_UNIT_BUSY;
      break;

    case FAS_NS_DATA_IN:
    case FAS_NS_DATA_OUT:
      if (dev->sc_state == FAS_S_INTERRUPT)
	if (dev->sc_cur_link < dev->sc_max_link)
	  {
	    left = dev->sc_clear_dma(dev);
	    len  = dev->sc_dma_len;

	    if (nexus->state == FAS_NS_DATA_IN)
	      {
		if (dev->sc_dma_buf == dev->sc_bump_pa)
		  {
		    while((rp->fas_fifo_flags & FAS_FIFO_COUNT_MASK) && left)
		      dev->sc_bump_va[len-(left--)] = rp->fas_fifo;
		    
		    bcopy(dev->sc_bump_va, dev->sc_buf, len-left);
		  }
	      }
	    else
	      {
		if (dev->sc_dma_buf == dev->sc_bump_pa)
		  {
		    rp->fas_command = FAS_CMD_FLUSH_FIFO;
		    if (left)
		      panic("LEFT IN FAS_NS_DATA_OUT");
		  }
	      }
	    
	    dev->sc_len -= len-left;
	    dev->sc_buf += len-left;
	    
	    dev->sc_dma_buf += len-left;
	    dev->sc_dma_len  = left;
	    
	    dev->sc_dma_blk_ptr += len-left;
	    dev->sc_dma_blk_len -= len-left;
	    
	    if (!dev->sc_dma_blk_len)
	      dev->sc_cur_link++;
	  }
      break;

    case FAS_NS_STATUS:
      dev->sc_stat[0] = rp->fas_fifo;
      break;

    default:
      break;
    }

  return(0);
}

int fas_postaction(struct fas_softc *dev, fas_regmap_p rp, struct nexus *nexus)
{
  int		i, left, len;
  u_char	cmd;
  short		offset, period;

  cmd = 0;

  switch(dev->sc_status & FAS_STAT_PHASE_MASK)
    {
    case FAS_PHASE_DATA_OUT:
      nexus->state = FAS_NS_DATA_OUT;

      if (dev->sc_state == FAS_S_IMMEDIATE)
	fas_ixfer(dev);
      else
	{
	  dev->sc_clear_dma(dev);
	  if (dev->sc_cur_link < dev->sc_max_link)
	    {
	      if (!dev->sc_dma_blk_len)
		{
		  dev->sc_dma_blk_ptr = dev->sc_chain[dev->sc_cur_link].ptr;
		  dev->sc_dma_blk_len = dev->sc_chain[dev->sc_cur_link].len;
		  dev->sc_dma_blk_flg = dev->sc_chain[dev->sc_cur_link].flg;
		}

	      if (dev->sc_dma_blk_flg == FAS_CHAIN_BUMP)
		len = dev->sc_dma_blk_len;
	      else if (dev->sc_dma_blk_flg == FAS_CHAIN_PRG)
		len = dev->sc_dma_blk_len;
	      else
		len=dev->sc_need_bump(dev,
				      dev->sc_dma_blk_ptr,dev->sc_dma_blk_len);

	      if (len)
		{
		  dev->sc_dma_buf = dev->sc_bump_pa;
		  dev->sc_dma_len = len;
		  
		  bcopy(dev->sc_buf, dev->sc_bump_va, dev->sc_dma_len);
		}
	      else
		{
		  dev->sc_dma_buf = dev->sc_dma_blk_ptr;
		  dev->sc_dma_len = dev->sc_dma_blk_len;
		}

	      dev->sc_init_dma_out(dev, dev->sc_dma_buf, dev->sc_dma_len);
	      cmd = FAS_CMD_TRANSFER_INFO | FAS_CMD_DMA;
	    }
	  else
	    {
	      rp->fas_tc_low = 256;
	      rp->fas_tc_mid = 0;
	      rp->fas_tc_high = 0;
	      cmd = FAS_CMD_TRANSFER_PAD;
	    }
	}
      break;

    case FAS_PHASE_DATA_IN:
      nexus->state = FAS_NS_DATA_IN;

      if (dev->sc_state == FAS_S_IMMEDIATE)
	fas_ixfer(dev);
      else
	{
	  dev->sc_clear_dma(dev);
	  if (dev->sc_cur_link < dev->sc_max_link)
	    {
	      if (!dev->sc_dma_blk_len)
		{
		  dev->sc_dma_blk_ptr = dev->sc_chain[dev->sc_cur_link].ptr;
		  dev->sc_dma_blk_len = dev->sc_chain[dev->sc_cur_link].len;
		  dev->sc_dma_blk_flg = dev->sc_chain[dev->sc_cur_link].flg;
		}

	      if (dev->sc_dma_blk_flg == FAS_CHAIN_BUMP)
		len = dev->sc_dma_blk_len;
	      else if (dev->sc_dma_blk_flg == FAS_CHAIN_PRG)
		len = dev->sc_dma_blk_len;
	      else
		len=dev->sc_need_bump(dev,
				      dev->sc_dma_blk_ptr,dev->sc_dma_blk_len);

	      if (len)
		{
		  dev->sc_dma_buf = dev->sc_bump_pa;
		  dev->sc_dma_len = len;
		}
	      else
		{
		  dev->sc_dma_buf = dev->sc_dma_blk_ptr;
		  dev->sc_dma_len = dev->sc_dma_blk_len;
		}

	      dev->sc_init_dma_in(dev, dev->sc_dma_buf, dev->sc_dma_len);
	      cmd = FAS_CMD_TRANSFER_INFO | FAS_CMD_DMA;
	    }
	  else
	    {
	      rp->fas_tc_low = 256;
	      rp->fas_tc_mid = 0;
	      rp->fas_tc_high = 0;
	      cmd = FAS_CMD_TRANSFER_PAD;
	    }
	}
      break;

    case FAS_PHASE_COMMAND:
      rp->fas_command = FAS_CMD_FLUSH_FIFO;
      for(i=0; i<5; i++);
      nexus->state = FAS_NS_SVC;

      for(i=0; i<nexus->clen; i++)
	rp->fas_fifo = nexus->cbuf[i] | (i==1 ? (dev->sc_lun<<5) : 0);
      cmd = FAS_CMD_TRANSFER_INFO;
      break;

    case FAS_PHASE_STATUS:
      nexus->state = FAS_NS_STATUS;
      cmd = FAS_CMD_COMMAND_COMPLETE;
      nexus->flags |= FAS_REC_MSG;
      break;

    case FAS_PHASE_MESSAGE_OUT:
      if (dev->sc_msg_out_len)
	{
	  rp->fas_command = FAS_CMD_FLUSH_FIFO;
	  for(i=0; i<5; i++);
	  for(i=0; i<dev->sc_msg_out_len; i++)
	    rp->fas_fifo = dev->sc_msg_out[i];
	  cmd = FAS_CMD_TRANSFER_INFO;
	  dev->sc_msg_out_len = 0;
	}
      else
	{
	  rp->fas_fifo = 8;	/* NOP */
	  cmd = FAS_CMD_TRANSFER_INFO;
	}
      break;

    case FAS_PHASE_MESSAGE_IN:
      cmd = FAS_CMD_TRANSFER_INFO;
      if (!((nexus->flags ^= FAS_REC_MSG) & FAS_REC_MSG))
	{
	  dev->sc_msg_in[dev->sc_msg_in_len++] = rp->fas_fifo;
	  cmd = FAS_CMD_MESSAGE_ACCEPTED;
	  if (dev->sc_msg_in[0] >= 0x80)       ;
	  else if (dev->sc_msg_in[0] >= 0x30)  ;
	  else if (dev->sc_msg_in[0] >= 0x20)
	    {
	      if (dev->sc_msg_in_len == 2)
		nexus->flags |= FAS_HAS_MSG;
	    }
	  else if (dev->sc_msg_in[0] == 1)
	    {
	      if (dev->sc_msg_in_len >= 2)
		if ((dev->sc_msg_in[1]+2) == dev->sc_msg_in_len)
		  nexus->flags |= FAS_HAS_MSG;
	    }
	  else
	    nexus->flags |= FAS_HAS_MSG;
	}
      else
	{
	  rp->fas_command = FAS_CMD_FLUSH_FIFO;
	  cmd = FAS_CMD_TRANSFER_INFO;
	}

      if (nexus->flags & FAS_HAS_MSG)
	{
	  switch(dev->sc_msg_in[0])
	    {
	    case 0x00:	/* COMMAND COMPLETE */
	      nexus->state = FAS_NS_DONE;
	      break;
	    case 0x04:	/* DISCONNECT */
	      nexus->flags |= FAS_GOT_DISCONN;
	      break;
	    case 0x02:	/* SAVE DATA POINTER */
	      fas_save_pointers(dev);
	      break;
	    case 0x03:	/* RESTORE DATA POINTERS */
	      fas_restore_pointers(dev);
	      break;
	    case 0x07:	/* MESSAGE REJECT */
	      if (nexus->state == FAS_NS_SELECTED_SDTR)
		{
		  nexus->config3 &= ~FAS_CFG3_FASTSCSI;
		  nexus->syncper = 5;
		  nexus->syncoff = 0;

		  rp->fas_syncper = nexus->syncper;
		  rp->fas_syncoff = nexus->syncoff;
		  rp->fas_config3 = nexus->config3;
		}
	      else
		panic("fasintr: Unknown message rejected!");
	      break;
	    case 0x08:	/* MO OPERATION */
	      break;
	    case 0x01:	/* EXTENDED MESSAGE */
	      switch(dev->sc_msg_in[2])
		{
		case 0x00:	/* MODIFY DATA POINTERS */
		  break;
		case 0x01:	/* SYNCHRONOUS DATA TRANSFER REQUEST */
		  period = 4*dev->sc_msg_in[3];
		  offset = dev->sc_msg_in[4];

		  if (offset == 0)
		    period = 5*dev->sc_clock_period;

		  nexus->syncper = period/dev->sc_clock_period;
		  nexus->syncoff = offset;

		  if (period < 200)
		    nexus->config3 |= FAS_CFG3_FASTSCSI;
		  else
		    nexus->config3 &= ~FAS_CFG3_FASTSCSI;

		  nexus->flags |= FAS_SYNC_TESTED;

		  rp->fas_syncper = nexus->syncper;
		  rp->fas_syncoff = nexus->syncoff;
		  rp->fas_config3 = nexus->config3;

		  if (nexus->state == FAS_NS_SELECTED_SDTR)
		    nexus->state = FAS_NS_SELECTED;
		  break;
		case 0x02:	/* EXTENDED IDENTIFY (SCSI-1) */
		  break;
		case 0x03:	/* WIDE DATA TRANSFER REQUEST */
		  dev->sc_msg_out[0] = 0x07;
		  dev->sc_msg_out_len = 1;
		  rp->fas_command = FAS_CMD_MESSAGE_ACCEPTED;
		  cmd = FAS_CMD_SET_ATN;
		  break;
		default:
		  break;
		}
	      break;
	    default:
	      dev->sc_msg_out[0] = 0x07;
	      dev->sc_msg_out_len = 1;
	      rp->fas_command = FAS_CMD_MESSAGE_ACCEPTED;
	      cmd = FAS_CMD_SET_ATN;
	      break;
	    }
	  nexus->flags &= ~(FAS_HAS_MSG | FAS_REC_MSG);
	  dev->sc_msg_in_len = 0;
	}
      break;
    default:
      fasreset(dev, 2);
      printf("UNKNOWN PHASE! state: %d\n", dev->sc_state);
      dev->sc_led_off(dev);
      if (dev->sc_state != FAS_S_IMMEDIATE)
	fas_scsidone(dev, -4);

      return(-1);
    }

  if (cmd)
    rp->fas_command = cmd;

  return(0);
}

void fasintr(struct fas_softc *dev)
{
  fas_regmap_p	 rp;
  struct nexus	*nexus;

  rp = dev->sc_fas;

  if (fas_pretests(dev, rp))
    return;

  nexus = dev->sc_cur_nexus;
  if (dev->sc_state != FAS_S_INTERRUPT || !nexus)
    return;

  if (fas_midaction(dev, rp, nexus))
    return;

  if (fas_postaction(dev, rp, nexus))
    return;

  return;
}

int fasicmd(struct fas_softc *dev, int target, int lun,
	    u_char *cbuf, int clen, void *buf, int len, int flags)
{
  fas_regmap_p	 rp;
  struct nexus	*nexus;
  int		 state;

  dev->sc_ints_off(dev);

  rp = dev->sc_fas;

  fasselect(dev, target, lun, cbuf, clen, buf, len, FAS_SELECT_I);

  nexus = dev->sc_cur_nexus;

  while(1)
    {
      if ((nexus->state == FAS_NS_SELECTED) ||
	  (nexus->state == FAS_NS_SELECTED_SDTR))
	{
	  if (fasiwait(dev))
	    {
	      dev->sc_stat[0] = -6;
	      break;
	    }
	}
      else
	if ((nexus->state != FAS_NS_DATA_IN) &&
	    (nexus->state != FAS_NS_DATA_OUT))
	  {
	    while(fasiwait(dev));
	  }

      if (!(state = fas_pretests(dev, rp)))
	if (!(state = fas_midaction(dev, rp, nexus)))
	  state = fas_postaction(dev, rp, nexus);

      if (state == -1)
	break;
    }

  nexus->flags &= ~FAS_SYNC_TESTED;
  dev->sc_state = FAS_S_IDLE;

  dev->sc_ints_on(dev);

  return(dev->sc_stat[0]);
}
