/*
 *	$Id: flz3sc.c
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/device.h>
#include <scsi/scsi_all.h>
#include <scsi/scsiconf.h>
#include <vm/vm.h>
#include <vm/vm_kern.h>
#include <vm/vm_page.h>
#include <machine/pmap.h>
#include <amiga/amiga/custom.h>
#include <amiga/amiga/cc.h>
#include <amiga/amiga/device.h>
#include <amiga/dev/fasreg.h>
#include <amiga/dev/fasvar.h>
#include <amiga/dev/zthreebusvar.h>
#include <amiga/dev/flz3screg.h>
#include <amiga/dev/flz3scvar.h>

int flz3scprint __P((void *auxp, char *));
void flz3scattach __P((struct device *, struct device *, void *));
int flz3scmatch __P((struct device *, struct cfdata *, void *));

struct scsi_adapter flz3sc_scsiswitch = {
	fas_scsicmd,
	fas_minphys,
	0,			/* no lun support */
	0,			/* no lun support */
	fas_adinfo,
	"flz3sc",
};

struct scsi_device flz3sc_scsidev = {
	NULL,		/* use default error handler */
	NULL,		/* do not have a start functio */
	NULL,		/* have no async handler */
	NULL,		/* Use default done routine */
	"flz3sc",
	0,
};


struct cfdriver flz3sccd = {
	NULL, "flz3sc", (cfmatch_t)flz3scmatch, flz3scattach, 
	DV_DULL, sizeof(struct flz3sc_softc), NULL, 0 };

int flz3sc_init_dma_in	__P((struct fas_softc *sc, void *ptr, int len));
int flz3sc_init_dma_out	__P((struct fas_softc *sc, void *ptr, int len));
int flz3sc_clear_dma	__P((struct fas_softc *sc));
int flz3sc_dma_ok	__P((struct fas_softc *sc, void *ptr, int len));
int flz3sc_build_dma_chain __P((struct fas_softc *sc, void *p, int l));
int flz3sc_need_bump	__P((struct fas_softc *sc, void *ptr, int len));
void flz3sc_led_on	__P((struct fas_softc *sc));
void flz3sc_led_off	__P((struct fas_softc *sc));
void flz3sc_ints_on	__P((struct fas_softc *sc));
void flz3sc_ints_off	__P((struct fas_softc *sc));

/*
 * if we are an Advanced Systems & Software FastlaneZ3
 */
int flz3scmatch(struct device *pdp, struct cfdata *cdp,	void *auxp)
{
  struct zthreebus_args *zap;

  zap = auxp;
  if (zap->manid == 0x2140 && zap->prodid == 11)
    return(1);

  return(0);
}

void flz3scattach(struct device *pdp, struct device *dp, void *auxp)
{
  struct flz3sc_softc	*sc;
  struct zthreebus_args	*zap;
  flz3sc_regmap_p	 rp;
  u_int			*pte, page;

  zap = auxp;

  sc = (struct flz3sc_softc *)dp;
  sc->sc_softc.sc_fas = (void *)(((char *)zap->va) + 0x1000000);
  sc->sc_softc.sc_dma = zap->va;

  sc->sc_softc.sc_spec = &sc->sc_specific;

  sc->sc_softc.sc_init_dma_in	 = flz3sc_init_dma_in;
  sc->sc_softc.sc_init_dma_out	 = flz3sc_init_dma_out;
  sc->sc_softc.sc_clear_dma	 = flz3sc_clear_dma;
  sc->sc_softc.sc_dma_ok	 = flz3sc_dma_ok;
  sc->sc_softc.sc_build_dma_chain= flz3sc_build_dma_chain;
  sc->sc_softc.sc_need_bump	 = flz3sc_need_bump;

  sc->sc_softc.sc_led_on      = flz3sc_led_on;
  sc->sc_softc.sc_led_off     = flz3sc_led_off;

  sc->sc_softc.sc_ints_on      = flz3sc_ints_on;
  sc->sc_softc.sc_ints_off     = flz3sc_ints_off;

  sc->sc_softc.sc_clock_freq   = 40;	/* FastlaneZ3 runs at 40MHz */
  sc->sc_softc.sc_timeout      = 250;	/* Set default timeout to 250ms */
  sc->sc_softc.sc_config_flags = 0;
  sc->sc_softc.sc_host_id      = 7;

  rp = (flz3sc_regmap_p)sc->sc_softc.sc_fas;

  sc->sc_specific.portbits = 0xA0 | FLZ3SC_PB_EDI | FLZ3SC_PB_ESI;
  sc->sc_specific.hardbits = rp->hardbits;

  sc->sc_softc.sc_bump_sz = NBPG;
  sc->sc_softc.sc_bump_va = (u_char *)kmem_alloc(kernel_map,
						 sc->sc_softc.sc_bump_sz);
  sc->sc_softc.sc_bump_pa = (void *)kvtop(sc->sc_softc.sc_bump_va);

  pte = kvtopte(sc->sc_softc.sc_bump_va);
  page= (u_int)sc->sc_softc.sc_bump_pa & PG_FRAME;

  *pte = PG_V | PG_RW | PG_CI | page;
  TBIAS();

  printf(": dmabuf pa 0x%x", sc->sc_softc.sc_bump_pa);

  fasinitialize((struct fas_softc *)sc);

  sc->sc_softc.sc_link.adapter_softc = sc;
  sc->sc_softc.sc_link.adapter_targ = sc->sc_softc.sc_host_id;
  sc->sc_softc.sc_link.adapter = &flz3sc_scsiswitch;
  sc->sc_softc.sc_link.device = &flz3sc_scsidev;

  custom.intreq = INTF_PORTS;
  custom.intena = INTF_SETCLR | INTF_PORTS;

/* We don't want interrupt until we're initialized! */
  rp->hardbits = sc->sc_specific.portbits;

  printf("\n");

/* attach all scsi units on us */
  config_found(dp, &sc->sc_softc.sc_link, flz3scprint);
}

/* print diag if pnp is NULL else just extra */
int flz3scprint(void *auxp, char *pnp)
{
  if (pnp == NULL)
    return(UNCONF);

  return(QUIET);
}

int flz3sc_intr()
{
  flz3sc_regmap_p	  rp;
  struct fas_softc	 *dev;
  struct flz3sc_specific *flspec;
  int			  i, found, quickints;
  u_char		  hb;

  found = 0;
  for(i=0; i<flz3sccd.cd_ndevs; i++)
    {
      dev = flz3sccd.cd_devs[i];
      if (!dev)
	continue;

      flspec = dev->sc_spec;
      rp = (flz3sc_regmap_p)dev->sc_fas;
      hb = rp->hardbits;

      if (hb & FLZ3SC_HB_IACT)
	continue;

      found++;

      flspec->hardbits = hb;
      if ((hb & FLZ3SC_HB_CREQ) &&
	  !(hb & FLZ3SC_HB_MINT) &&
	  (rp->FAS216.fas_status & FAS_STAT_INTERRUPT_PENDING))
	{
	  quickints = 256;
	  do
	    {
	      dev->sc_status = rp->FAS216.fas_status;
	      dev->sc_sequence = rp->FAS216.fas_seqstep;
	      dev->sc_interrupt = rp->FAS216.fas_interrupt;
		  
	      if (dev->sc_interrupt & FAS_INT_RESELECTED)
		{
		  dev->sc_resel[0] = rp->FAS216.fas_fifo;
		  dev->sc_resel[1] = rp->FAS216.fas_fifo;
		}
	      fasintr(dev);
	    } while((rp->FAS216.fas_status & FAS_STAT_INTERRUPT_PENDING) &&
		    --quickints);
	}
      
/* Reset fastlane interrupt bits */
      rp->hardbits = flspec->portbits & ~FLZ3SC_PB_INT_BITS;
      rp->hardbits = flspec->portbits;
    }

  return(found);
}

/* Load transfer adress into dma register */
void flz3sc_set_dma_adr(struct fas_softc *sc, void *ptr)
{
  unsigned int			*p, d;

  d = (unsigned int)ptr;
  p = (unsigned int *)((d & 0xFFFFFF) + (int)sc->sc_dma);

  ((flz3sc_regmap_p)sc->sc_fas)->clear=0;
  *p = d;
}

/* Set DMA transfer counter */
void flz3sc_set_dma_tc(struct fas_softc *sc, unsigned int len)
{
  sc->sc_fas->fas_tc_low  = len; len >>= 8;
  sc->sc_fas->fas_tc_mid  = len; len >>= 8;
  sc->sc_fas->fas_tc_high = len;
}

/* Set DMA mode */
void flz3sc_set_dma_mode(struct fas_softc *sc, int mode)
{
  struct flz3sc_specific	*spec;

  spec = sc->sc_spec;

  spec->portbits = (spec->portbits & ~FLZ3SC_PB_DMA_BITS) | mode;
  ((flz3sc_regmap_p)sc->sc_fas)->hardbits = spec->portbits;
}

/* Initialize DMA for transfer into memory */
int flz3sc_init_dma_in(struct fas_softc *sc, void *ptr, int len)
{
  flz3sc_set_dma_adr(sc, ptr);
  flz3sc_set_dma_mode(sc, FLZ3SC_PB_ENABLE_DMA | FLZ3SC_PB_DMA_READ);
  flz3sc_set_dma_tc(sc, len);
}

/* Initialize DMA for transfer out from memory */
int flz3sc_init_dma_out(struct fas_softc *sc, void *ptr, int len)
{
  flz3sc_set_dma_adr(sc, ptr);
  flz3sc_set_dma_mode(sc, FLZ3SC_PB_ENABLE_DMA | FLZ3SC_PB_DMA_WRITE);
  flz3sc_set_dma_tc(sc, len);
}

/* Clear DMA */
int flz3sc_clear_dma(struct fas_softc *sc)
{
  flz3sc_set_dma_mode(sc, FLZ3SC_PB_DISABLE_DMA);
  flz3sc_set_dma_adr(sc, 0);

  return((((sc->sc_fas->fas_tc_high << 8) |
	    sc->sc_fas->fas_tc_mid) << 8) |
	    sc->sc_fas->fas_tc_low);
}

/* Check if address and len is ok for DMA transfer */
int flz3sc_dma_ok(struct fas_softc *sc, void *ptr, int len)
{
  int	retcode, p;

  p = (int)ptr;

  retcode = TRUE;	/* Lets asume that all is OK */

#ifdef M68040
  if (cpu040)
    {
      if (p >= 0xFFFC0000)
	retcode = FALSE;
    }
  else
#endif
    {
      if (p >= 0xFF000000)
	retcode = FALSE;
    }

  return(retcode);
}

/* Check if address and len is ok for DMA transfer */
int flz3sc_need_bump(struct fas_softc *sc, void *ptr, int len)
{
  int	p;

  p = (int)ptr;
  p &= 0x03;

  if (p)
    {
      p = 3-p;
      if (p > len)
	p = len;
    }
  else
    if (len & 3)
      p = len;

  return(p);
}

/* Interrupt driven routines */
int flz3sc_build_dma_chain(struct fas_softc *sc, void *p, int l)
{
  char	*ptr;
  int	 len,  prelen,  postlen;
  int	 n;
  void	*pa, *lastpa;
  int	 max_t;

#define set_link(n, p, l, f)\
do {\
sc->sc_chain[n].ptr   = (p);\
sc->sc_chain[n].len   = (l);\
sc->sc_chain[n++].flg = (f);\
} while(0)

  n = 0;

  if (!sc->sc_dma_ok(sc, p, l))
    {
      while(l)
	{
	  len = ((l > sc->sc_bump_sz) ? sc->sc_bump_sz : l);

	  set_link(n, p, len, FAS_CHAIN_BUMP);

	  p += len;
	  l -= len;
	}
      return(n);
    }

  if (l < 512)
    {
      set_link(n, p, l, FAS_CHAIN_BUMP);
      return(n);
    }

  ptr = p;
  len = l;

  pa = (void *)kvtop(ptr);
  prelen = ((int)p & 0x03);

  if (prelen)
    {
      prelen = 4-prelen;
      set_link(n, ptr, prelen, FAS_CHAIN_BUMP);
      ptr += prelen;
      len -= prelen;
    }

  lastpa = 0;
  while(len > 3)
    {
      pa = (void *)kvtop(ptr);
      max_t = NBPG - ((int)pa & PGOFSET);
      if (max_t > len)
	max_t = len;

      max_t &= ~3;

      if (lastpa == pa)
	sc->sc_chain[n-1].len += max_t;
      else
	set_link(n, pa, max_t, FAS_CHAIN_DMA);

      lastpa = pa+max_t;

      ptr += max_t;
      len -= max_t;
    }

  if (len)
    set_link(n, ptr, len, FAS_CHAIN_BUMP);

  return(n);
}

/* Turn on led */
void flz3sc_led_on(struct fas_softc *sc)
{
  struct flz3sc_specific	*spec;
  flz3sc_regmap_p		rp;

  spec = sc->sc_spec;
  rp = (flz3sc_regmap_p)sc->sc_fas;

  sc->sc_led_status++;

  spec->portbits |= FLZ3SC_PB_LED;
  rp->hardbits = spec->portbits;
}

/* Turn off led */
void flz3sc_led_off(struct fas_softc *sc)
{
  struct flz3sc_specific	*spec;
  flz3sc_regmap_p		rp;

  spec = sc->sc_spec;
  rp = (flz3sc_regmap_p)sc->sc_fas;

  if (sc->sc_led_status)
    sc->sc_led_status--;

  if (!sc->sc_led_status)
    {
      spec->portbits &= ~FLZ3SC_PB_LED;
      rp->hardbits = spec->portbits;
    }
}

void flz3sc_ints_on(struct fas_softc *sc)
{
  struct flz3sc_specific	*spec;
  flz3sc_regmap_p		 rp;

  spec = sc->sc_spec;
  rp = (flz3sc_regmap_p)sc->sc_fas;

  spec->portbits |= FLZ3SC_PB_EDI | FLZ3SC_PB_ESI;
  rp->hardbits = spec->portbits;
}

void flz3sc_ints_off(struct fas_softc *sc)
{
  struct flz3sc_specific	*spec;
  flz3sc_regmap_p		 rp;

  spec = sc->sc_spec;
  rp = (flz3sc_regmap_p)sc->sc_fas;

  spec->portbits &= ~(FLZ3SC_PB_EDI | FLZ3SC_PB_ESI);
  rp->hardbits = spec->portbits;
}
