/* 
 * Mach Operating System
 * Copyright (c) 1991,1990,1989 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 * 
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 * 
 * Carnegie Mellon requests users of this software to return to
 * 
 *  Software Distribution Coordinator   or   Software.Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 * 
 * any improvements or extensions that they made and grant Carnegie Mellon
 * the rights to redistribute these changes.
 **********************************************************************
 * HISTORY
 * $Log:	emul_machdep.c,v $
 * Revision 2.4  92/02/02  13:01:13  rpd
 * 	Removed MAP_UAREA code.
 * 	Fixed for gcc.
 * 	[92/01/30            rpd]
 * 
 * Revision 2.3  91/12/19  20:25:40  mrt
 * 	Updated to new copyright
 * 
 * Revision 2.2  90/11/16  11:41:28  rwd
 * 	Taken from UX21
 * 	[90/11/14  15:05:19  rwd]
 * 
 * Revision 2.8  90/08/06  15:30:57  rwd
 * 	Added pid_by_task.
 * 	[90/07/31            rwd]
 * 	include sys/types.h.
 * 	[90/07/17            rwd]
 * 
 * Revision 2.7  90/06/02  15:20:54  rpd
 * 	Updated for new IPC.
 * 	[90/05/02  15:11:13  rpd]
 * 
 * Revision 2.6  90/05/29  20:22:32  rwd
 * 	Add set_arg_addr routine (rfr)
 * 	[90/04/12            rwd]
 * 	Add set_arg_addr routine (rfr)
 * 	[90/04/12            rwd]
 * 
 * Revision 2.5  90/05/21  14:07:51  dbg
 * 	Add set_arg_addr routine
 * 	[90/04/12            rwd]
 * 
 * Revision 2.4  90/03/09  16:22:41  rwd
 * 	Check for signals after system call also, since numerous calls
 * 	no longer send messages and can't get interrupt set as a result.
 * 	Add mapped sigreturn.
 * 	[90/02/23            rwd]
 * 	Check for signals before trying the system call.
 * 	[90/01/31            rwd]
 * 
 * Revision 2.3  89/10/17  11:24:11  rwd
 * 	Added syscall init_process(-41).  Mach_Init needs this.
 * 	[89/09/29            rwd]
 * 
 * 	Removed reference to ERROR
 * 	[89/09/26            rwd]
 * 
 * Revision 2.2.1.1  89/09/21  20:35:37  dbg
 * 	Add interrupt return parameter to all calls.
 * 	[89/09/21            dbg]
 * 
 * Revision 2.2  89/08/31  16:28:19  rwd
 * 	No special case for syscall needed.  Fix up user stack whenever
 * 	ERESTART occurs.
 * 	[89/08/24            rwd]
 * 	Changed to reflect change in trampline code that pushes syscall
 * 	# on stack.
 * 	[89/08/23            rwd]
 * 
 * 	Make ERESTART code work and special case for syscall
 * 	[89/08/21            rwd]
 * 
 * Revision 2.1  89/08/04  14:04:49  rwd
 * Created.
 * 
 * 22-Jun-89  Randall Dean (rwd) at Carnegie-Mellon University
 *	Added copyright and history to file by dbg.  Fixed register
 *	format.
 *
 */

/*
 * Take/Return from signals - machine-dependent.
 */
#include <mach/mach.h>
#include <sys/types.h>
#include <sys/errno.h>
#include <sys/signal.h>
#include <sys/syscall.h>
#include <machine/psl.h>
#include <machine/vmparam.h>
#include <bsd_msg.h>	/* error code definitions */

#include "syscall_table.h"

extern mach_port_t	our_bsd_server_port;

#define	E_JUSTRETURN	255	/* return without changing registers */

/*
 * User's registers are stored on partially on the user's stack
 * and partially on the emulator's stack.
 */
struct emul_regs_2 {
	int	d0;
	int	d1;
	int	a0;
	int	a1;
	short	pad;
	int	syscode;
	short	sr;
	int	pc;
};

struct emul_regs {
	int	d2;
	int	d3;
	int	d4;
	int	d5;
	int	d6;
	int	d7;
	int	a2;
	int	a3;
	int	a4;
	int	a5;
	int	a6;
	struct emul_regs_2 *usp;
				/* pointer to rest of user registers */
};

void	take_signals();	/* forward */

int	emul_low_entry = -9;
int	emul_high_entry = 181;

extern emul_common();

void
emul_setup(task)
	task_t	task;
{
	register int i;
	register kern_return_t	rc;

	for (i = emul_low_entry;
	     i <= emul_high_entry;
	     i++) {
		rc = task_set_emulation(task,
					(vm_offset_t) emul_common,
					i);
	}
	rc = task_set_emulation(task,
			(vm_offset_t) emul_common,
			-33);
	rc = task_set_emulation(task,
			(vm_offset_t) emul_common,
			-34);
	rc = task_set_emulation(task,
			(vm_offset_t) emul_common,
			-41);
	rc = task_set_emulation(task,
			(vm_offset_t) emul_common,
			-52);
}

/*
 * System calls enter here.
 */
void
emul_syscall(regs)
	register struct emul_regs *regs;
{
	register int	syscode;
	int		opc;
	register int	error;
	register struct sysent *callp;
	int		rval[2];
	boolean_t	interrupt = FALSE;
	register struct emul_regs_2 *regs2,*regs2_mv;
	register int	*args;

	regs2 = regs->usp;
	args = (int *)(regs2+1);	/* args on stack top */
	args++;	/* point to first argument - skip return address */

	syscode = regs2->syscode;

	/*
	 * Save old PC for backup
	 */

	opc = regs2->pc - 2;

	if (syscode == 0) {
	    /*
	     * Indirect system call.
	     */
	    syscode = *args++;
	}

	/*
	 * Find system call table entry for the system call.
	 */
	if (syscode >= nsysent)
	    callp = &sysent[63];	/* nosysent */
	else if (syscode >= 0)
	    callp = &sysent[syscode];
	else {
	    /*
	     * Negative system call numbers are CMU extensions.
	     */
	    if (syscode == -33)
		callp = &sysent_task_by_pid;
	    else if (syscode == -34)
		callp = &sysent_pid_by_task;
	    else if (syscode == -41)
		callp = &sysent_init_process;
	    else if (syscode == -59)
		callp = &sysent_htg_ux_syscall;
	    else if (syscode < -ncmusysent)
		callp = &sysent[63];	/* nosysent */
	    else
		callp = &cmusysent[-syscode];
	}

	/*
	 * Set up the initial return values.
	 */
	rval[0] = 0;
	rval[1] = regs2->d1;

	/*
	 * Call the routine, passing arguments according to the table
	 * entry.
	 */
	switch (callp->nargs) {
	    case 0:
		error = (*callp->routine)(our_bsd_server_port,
				&interrupt,
				rval);
		break;
	    case 1:
		error = (*callp->routine)(our_bsd_server_port,
				&interrupt,
				args[0],
				rval);
		break;
	    case 2:
		error = (*callp->routine)(our_bsd_server_port,
				&interrupt,
				args[0], args[1],
				rval);
		break;
	    case 3:
		error = (*callp->routine)(our_bsd_server_port,
				&interrupt,
				args[0], args[1], args[2],
				rval);
		break;
	    case 4:
		error = (*callp->routine)(our_bsd_server_port,
				&interrupt,
				args[0], args[1], args[2], args[3],
				rval);
		break;
	    case 5:
		error = (*callp->routine)(our_bsd_server_port,
				&interrupt,
				args[0], args[1], args[2], args[3], args[4],
				rval);
		break;
	    case 6:
		error = (*callp->routine)(our_bsd_server_port,
				&interrupt,
				args[0], args[1], args[2],
				args[3], args[4], args[5],
				rval);
		break;

	    case -1:	/* generic */
		error = (*callp->routine)(our_bsd_server_port,
				&interrupt,
				syscode,
				args,
				rval);
		break;

	    case -2:	/* pass registers to modify */
		error = (*callp->routine)(our_bsd_server_port,
				&interrupt,
				args,
				rval,
				regs);
		regs2 = regs->usp;	/* if changed */
		break;
	}

	/*
	 * Set up return values.
	 */

	switch (error) {
	    case E_JUSTRETURN:
		/* Do not alter registers */
		break;

	    case 0:
		/* Success */
		regs2->sr &= ~SR_CC;
		regs2->d0 = rval[0];
		regs2->d1 = rval[1];
		break;

	    case ERESTART:
		/* restart call */
		regs2_mv = (struct emul_regs_2 *)((int *)regs2 - 1);
		regs2_mv->d0=regs2->d0;
		regs2_mv->d1=regs2->d1;
		regs2_mv->a0=regs2->a0;
		regs2_mv->a1=regs2->a1;
		regs2_mv->sr=regs2->sr;
		regs2_mv->pc = opc;
		regs2->pc = syscode;
		regs->usp = regs2_mv;
		break;

	    default:
		/* error */
		regs2->sr |= SR_CC;
		regs2->d0 = error;
		break;
	}

	/*
	 * Handle interrupt request
	 */
	if (error == ERESTART || error == EINTR || interrupt)
	    take_signals(regs);
}
/*
 * Exec starts here to save registers.
 */
struct execa {
    char	*fname;
    char	**argp;
    char	**envp;
};

int
e_execv(serv_port, interrupt, argp, rval, regs)
	mach_port_t		serv_port;
	boolean_t		*interrupt;
	register struct execa	*argp;
	int			*rval;
	struct emul_regs	*regs;
{
	return (e_execvx(serv_port, interrupt, argp, FALSE, regs));
}

int
e_execve(serv_port, interrupt, argp, rval, regs)
	mach_port_t		serv_port;
	boolean_t		*interrupt;
	register struct execa	*argp;
	int			*rval;
	struct emul_regs	*regs;
{
	return (e_execvx(serv_port, interrupt, argp, TRUE, regs));
}

int
e_execvx(serv_port, interrupt, argp, use_env, regs)
	mach_port_t		serv_port;
	boolean_t		*interrupt;
	register struct execa	*argp;
	struct emul_regs	*regs;
	boolean_t		use_env;
{
	register struct emul_regs_2 *regs2;
	int		entry[2];
	unsigned int	entry_count;
	vm_offset_t	arg_addr;
	register int	error;

	/*
	 * Do not have to save user registers on old stack;
	 * they will all be cleared.
	 */

	/*
	 * Call exec.  If error, return without changing registers.
	 */
	entry_count = 2;
	error = e_exec_call(serv_port,
			    interrupt,
			    argp->fname,
			    argp->argp,
			    (use_env) ? argp->envp : (char **)0,
			    &arg_addr,
			    entry,
			    &entry_count);
	if (error)
	    return (error);

	/*
	 * Put new user stack just below arguments.
	 */
	regs2 = ((struct emul_regs_2 *)arg_addr) - 1;

	/*
	 * Clear user registers except for sp, pc, sr.
	 */
	regs2->d0 = 0;
	regs2->d1 = 0;
	regs2->a0 = 0;
	regs2->a1 = 0;
	regs2->sr = PSL_USERSET;
	regs2->pc = entry[0];

	regs->d2 = 0;
	regs->d3 = 0;
	regs->d4 = 0;
	regs->d5 = 0;
	regs->d6 = 0;
	regs->d7 = 0;
	regs->a2 = 0;
	regs->a3 = 0;
	regs->a4 = 0;
	regs->a5 = 0;
	regs->a6 = 0;
	regs->usp = regs2;

	/*
	 * Clear FP state?
	 */

	/*
	 * Return to new stack.
	 */
	return (E_JUSTRETURN);
}


/*
 * Take a signal.
 */
void
take_signals(regs)
	register struct emul_regs *regs;
{
	struct emul_regs_2	save_regs2;
	register struct emul_regs_2	*regs2;

	register struct sigcontext *scp;
	register struct sigframe {
	    int		sf_signum;
	    int		sf_code;
	    struct sigcontext *sf_scp;
	} *fp;

	int	old_mask, old_onstack, sig, code, handler, new_sp;
	boolean_t	interrupt;

	/*
	 * Get anything valuable off user stack first.
	 */
	save_regs2 = *regs->usp;

	/*
	 * Get the signal to take from the server.  It also
	 * switches the signal mask and the stack, so we must
	 * be off the old user stack before calling it.
	 */
	(void) Bsd1_take_signal(our_bsd_server_port,
			&interrupt,
			&old_mask,
			&old_onstack,
			&sig,
			&code,
			&handler,
			&new_sp);

	/*
	 * If there really were no signals to take, return.
	 */
	if (sig == 0) {
	    return;
	}

	/*
	 * Put the signal context and signal frame on the signal stack.
	 */
	if (new_sp == 0) {
	    /*
	     * Build signal frame and context on user's stack.
	     */
	    new_sp = (int)regs->usp;
	}

	scp = ((struct sigcontext *)new_sp) - 1;
	fp  = ((struct sigframe *)scp) - 1;

	/*
	 * Build the argument list for the signal handler.
	 */
	fp->sf_signum = sig;
	fp->sf_code = code;
	fp->sf_scp = scp;

	/*
	 * Build the signal context to be used by sigreturn.
	 */
	scp->sc_onstack = old_onstack;
	scp->sc_mask = old_mask;
	scp->sc_sp = (int)(regs->usp+1);	/* user sp after return */
	scp->sc_pc = save_regs2.pc;
	scp->sc_ps = save_regs2.sr;

	/*
	 * Set up the new stack and handler addresses.
	 */
	regs2 = ((struct emul_regs_2 *)fp) - 1;
	*regs2 = save_regs2;
	regs->usp = regs2;
	regs2->pc = handler;
}

int
e_sigreturn(serv_port, interrupt, argp, rval, regs)
	mach_port_t		serv_port;
	boolean_t		*interrupt;
	int			*argp;
	int			*rval;
	struct emul_regs	*regs;
{
	return e_osigcleanup(serv_port, interrupt, argp, rval, regs);
}

/*
 * SUN seems to make no distinction between sigreturn and osigcleanup.
 */
int
e_osigcleanup(serv_port, interrupt, argp, rval, regs)
	mach_port_t		serv_port;
	boolean_t		*interrupt;
	int			*argp;
	int			*rval;
	struct emul_regs	*regs;
{
	register struct sigcontext	*scp;
	struct emul_regs_2		save_regs2;
	register struct emul_regs_2	*regs2;
	register int			rc;
	struct sigcontext		sc;

	/*
	 * Copy in signal context and regs2.  Pointer to sigcontext is
	 * on top of user stack (unlike all other syscalls).
	 */
	save_regs2 = *regs->usp;
	scp = *((struct sigcontext **)(regs->usp + 1));
	sc = *scp;

	/*
	 * Change signal stack and mask.  If new signals are pending,
	 * do not take them until we switch user stack.
	 */
	rc = Bsd1_sigreturn(serv_port,
			interrupt,
			sc.sc_onstack & 01,
			sc.sc_mask);

	/*
	 * Change registers.
	 */
	regs2 = ((struct emul_regs_2 *)sc.sc_sp) - 1;

	*regs2 = save_regs2;
	regs->usp = regs2;
	regs2->pc = sc.sc_pc;
	regs2->sr = (sc.sc_ps & ~PSL_USERCLR) | PSL_USERSET;

	return (E_JUSTRETURN);
}

/*
 * Wait has a weird parameter passing mechanism.
 */
int
e_wait(serv_port, interrupt, argp, rval, regs)
	mach_port_t		serv_port;
	boolean_t		*interrupt;
	int			*argp;
	int			*rval;
	struct emul_regs	*regs;
{
	register struct emul_regs_2 *regs2 = regs->usp;

	int	new_args[2];

	if ((regs2->sr & PSL_ALLCC) == PSL_ALLCC) {
	    new_args[0] = regs2->d0;	/* options */
	    new_args[1] = regs2->d1;	/* rusage_p */
	}
	else {
	    new_args[0] = 0;
	    new_args[1] = 0;
	}
	return (emul_generic(serv_port, interrupt,
			SYS_wait, &new_args[0], rval));
}

int
e_fork(serv_port, interrupt, argp, rval, regs)
	mach_port_t		serv_port;
	boolean_t		*interrupt;
	int			*argp;
	int			*rval;
	struct emul_regs	*regs;
{
	register struct emul_regs_2 *regs2 = regs->usp;
	register int error;

	struct sun_thread_state	child_regs;
	
	extern int	child_fork();

	/*
	 * Set up registers for child.  It resumes on its own stack.
	 */
	child_regs.d0  = regs2->d0;
	child_regs.d1  = regs2->d1;
	child_regs.d2  = regs->d2;
	child_regs.d3  = regs->d3;
	child_regs.d4  = regs->d4;
	child_regs.d5  = regs->d5;
	child_regs.d6  = regs->d6;
	child_regs.d7  = regs->d7;
	child_regs.a0  = regs2->a0;
	child_regs.a1  = regs2->a1;
	child_regs.a2  = regs->a2;
	child_regs.a3  = regs->a3;
	child_regs.a4  = regs->a4;
	child_regs.a5  = regs->a5;
	child_regs.a6  = regs->a6;
	child_regs.sp  = (int)regs->usp;
	child_regs.pc  = (int)child_fork;
	child_regs.sr  = regs2->sr;
	
	/* FP regs!!!! */

	/*
	 * Create the child.
	 */
	error = Bsd1_fork(serv_port, interrupt,
			 (int *)&child_regs,
			 SUN_THREAD_STATE_REGS_COUNT,
			 &rval[0]);

	if (error == 0)
	    rval[1] = 0;

	return (error);
}

vm_offset_t
set_arg_addr(arg_size)
	vm_size_t	arg_size;
{
	/*
	 * Round argument size to fullwords
	 */
	arg_size = (arg_size + NBPW - 1) & ~(NBPW - 1);

	/*
	 * Put argument list at top of stack.
	 */
	return (USRSTACK - arg_size);
}
