/************************************************************************\
*                                                                        *
*  GS: A Generalized, stochastic petri net Simulator                     *
*      V0.01      March 1989      Andreas Nowatzyk (agn@unh.cs.cmu.edu)  *
*      Carnegie-Mellon Univerity, School of Computer Science             *
*      Schenley Park, Pittsburgh, PA 15213                               *
*                                                                        *
\************************************************************************/

/* GSPN simulator, main part */

#include <stdio.h>
#include "gs.h"

main(argc, argv)
    int argc;
    char *argv[];
{
    register int i, j;

    get_opt(argc, argv);			/* process comand line */
    init();					/* initialize data structures */

    if (n_rgen < 2 || n_fire < 1000)
	err ("Bogus rgen/fire parameters\n");

    for (i = 0; i < n_rgen; i++) {
	reset_net();
	fire(n_fire);
	cleanup();
    }

    print_result();

    exit(0);					/* successful exit */
}

err(s)
    char *s;
{
    fprintf (stderr, "Error: %s\n", s);
    exit(1);
}

init ()						/* initialize data structures */
/*
 *  0. initialize random number generation
 *  1. allocate histogram space (and zero it)
 *  2. compute intergerized transition probabilities for
 *     immediate transitions. make sure that any additive combination
 *     of these un-normalized probablilities does not exceed +2^31,
 *     also insure that no probablility is == 0
 */
{
    register int i, j;
    register double s;
    register struct place *p;
    register struct hist *h;
    register struct transition *t;

    if (bim_ratio < 0 || bim_ratio > 100 || bim_alt < 0.0)
	err ("bad -b or -a option");

    rnd_init(random_seed);

    for(i = n_places, p = PLACE; i--; p++) {
	j = p->ini_mrk + 1;
	if (j < INIT_H_SIZE) j = INIT_H_SIZE;
	p->max_mrk = j;
	p->H = h = (struct hist *) malloc (j * sizeof (struct hist));
	while (j--) {
	    h->sum = 0.0;
	    h->sum_sq = 0.0;
	    h->sum_cr = 0.0;
	    h++;
	}
    }

    s = 0.0;
    for (i = n_transitions, t = TRANSITION, j = 0; i--; t++)
	if (t->type == TTY_IMM) {
	    j++;
	    s += 1.0 / t->delay;
	}
    if (j) {					/* worry about this only if there are imm transitions */
	if (s < EPSILON)
	    err ("Unreasonable transition rates for immediate transitions");
	s = 268435456.0 / s;			/* scale */
	for (i = n_transitions, t = TRANSITION; i--; t++)
	    if (t->type == TTY_IMM) {
		t->prob = 0.5 + s / t->delay;
		if (t->prob < 1)
		    t->prob = 1;
	    }
    }
}

reset_net()					/* reset net to initial state */
/*
 *  1. reset places to original # of tokens
 *  2. reset clock & queue
 *  3. reset transition enable counts
 *  4. initialize fire-queue
 *  5. reset result accumulators & state
 */
{
    register int    i, j;
    register struct place *p;
    register struct hist *h;
    register struct transition *t;
    register struct result *r;

    for (i = n_places, p = PLACE; i--; p++) {
	p->cur_mrk = p->ini_mrk;
	p->t_last = 0.0;
	for (j = p->max_mrk, h = p->H; j--; h++)
	    h->prob = 0.0;
    }

    top = 0;
    i_top = 0;
    i_prob_acc = 0;
    clock = 0.0;

    for (i = n_transitions, t = TRANSITION; i--; t++) {
	t->fire_cnt = 0;
	if (!(t->ena_cnt = t->ini_ena)) {
	    switch (t->type) {
	    case TTY_IMM:
		enqueue_i_tr(t);
		break;
	    case TTY_EXP:
		enqueue_e_tr(t);
		break;
	    case TTY_DET:
		enqueue_d_tr(t);
		break;
	    case TTY_BIM:
		enqueue_b_tr(t);
		break;
	    case TTY_NRM:
		enqueue_n_tr(t);
		break;
	    }
	}
    }

    for (i = n_results, r = RESULT; i--; r++) {
	if (r->type)
	    continue;
	r->t_last = 0.0;
	r->acc = 0.0;
	(*(r->upd_fnctn)) ();
    }
}

add_H_space (p)					/* increase the histogram size */
    register struct place *p;
{
    register int i;
    register struct hist *h;

    i = INIT_H_SIZE + p->max_mrk / 2;
    p->H = h = (struct hist *) realloc (p->H, (p->max_mrk + i) * sizeof(struct hist));
    h += p->max_mrk;
    p->max_mrk += i;
    while (i--) {
	    h->prob = 0.0;
	    h->sum = 0.0;
	    h->sum_sq = 0.0;
	    h->sum_cr = 0.0;
	    h++;
    }
}

enqueue_i_tr (t)				/* add an immediate transition to fire queue */
    register struct transition *t;
{
    t->prev = 0;
    if (t->next = i_top)
	i_top->prev = t;
    i_top = t;
    i_prob_acc += t->prob;
}

enqueue_d_tr (t)				/* add an deterministic transition to fire queue */
    register struct transition *t;
{
    register double tf;
    register struct transition *q;

    t->t_fire = tf = clock + t->delay;

    if (q = top) {
	while (q->t_fire < tf) {
	    if (!q->next) {			/* end of queue: append at end */
		q->next = t;
		t->prev = q;
		t->next = 0;
		return;
	    }
	    q = q->next;
	}

	t->next = q;				/* insert before q */
	if (!(t->prev = q->prev))		/* ... happen to be top */
	    top = t;
	else
	    q->prev->next = t;
	q->prev = t;
    } else {					/* virgin queue: insert at top */
	top = t;
	t->next = 0;
	t->prev = 0;
    }
}

enqueue_e_tr (t)				/* add an deterministic transition to fire queue */
    register struct transition *t;
{
    register double tf;
    double rnd_nedi();
    register struct transition *q;

    t->t_fire = tf = clock + rnd_nedi(t->delay);

    if (q = top) {
	while (q->t_fire < tf) {
	    if (!q->next) {			/* end of queue: append at end */
		q->next = t;
		t->prev = q;
		t->next = 0;
		return;
	    }
	    q = q->next;
	}

	t->next = q;				/* insert before q */
	if (!(t->prev = q->prev))		/* ... happen to be top */
	    top = t;
	else
	    q->prev->next = t;
	q->prev = t;
    } else {					/* virgin queue: insert at top */
	top = t;
	t->next = 0;
	t->prev = 0;
    }
}

enqueue_b_tr (t)				/* add a bimodal distr. transition to fire queue */
    register struct transition *t;
{
    register double tf;
    register struct transition *q;

    if (bim_ratio <= rnd_ri(100))
	tf = clock + t->delay;
    else
	tf = clock + t->delay * bim_alt;

    t->t_fire = tf;

    if (q = top) {
	while (q->t_fire < tf) {
	    if (!q->next) {			/* end of queue: append at end */
		q->next = t;
		t->prev = q;
		t->next = 0;
		return;
	    }
	    q = q->next;
	}

	t->next = q;				/* insert before q */
	if (!(t->prev = q->prev))		/* ... happen to be top */
	    top = t;
	else
	    q->prev->next = t;
	q->prev = t;
    } else {					/* virgin queue: insert at top */
	top = t;
	t->next = 0;
	t->prev = 0;
    }
}

enqueue_n_tr (t)				/* add a normal distr. transition to fire queue */
    register struct transition *t;
{
    register double tf;
    double rnd_01d();
    register struct transition *q;

    t->t_fire = tf = clock + t->delay * rnd_01d();

    if (q = top) {
	while (q->t_fire < tf) {
	    if (!q->next) {			/* end of queue: append at end */
		q->next = t;
		t->prev = q;
		t->next = 0;
		return;
	    }
	    q = q->next;
	}

	t->next = q;				/* insert before q */
	if (!(t->prev = q->prev))		/* ... happen to be top */
	    top = t;
	else
	    q->prev->next = t;
	q->prev = t;
    } else {					/* virgin queue: insert at top */
	top = t;
	t->next = 0;
	t->prev = 0;
    }
}

fire (n)					/* fire n-times */
    register unsigned long n;
{
    register double tf;
    register struct transition **q, *t;
    register long i;

    while (n--) {				/* fire loop */

	if (t = i_top) {			/* at least one enabled i-transition */
	    if (!(t->next)) {			/* only one! */
		(*(t->f_func)) ();
		continue;			/* fire and done! */
	    }
/*
 * Note: the i-trans selection method is moderately tricky, but correct and fast.
 */
	    i = rnd_ri(i_prob_acc);		/* roll the dice! */
	    while (0 <= (i -= t->prob))
		t = t->next;
	    (*(t->f_func)) ();			/* fire the choosen one */
	    continue;
	}

	if (!(t = top))
	    dead_net(n);

	clock = tf = t->t_fire;			/* advance clock */
	q = TBUF;

	*q = t;
	i = 1;
	for (t = t->next; t && t->t_fire == tf; i++, t = t->next)
	    *(++q) = t;				/* collect all enabled transitions */
	
	if (i > 1)				/* fire a random one if more than 1 is enabled */
	    (*(TBUF[rnd_ri(i)]->f_func))();
	else
	    (*(TBUF[0]->f_func)) ();
    }
}

dead_net(n)					/* dead-locked, print final marking */
    int n;
{
    register int i;
    register struct place *p;

    printf ("No enabled transition: time=%.2f  fire-count=%d/%d\n", clock, n_fire - n, n_fire);
    for (i = n_places,  p = PLACE; i--; p++)
	printf("Place '%s': %d token\n", p->name, p->cur_mrk);

    err("No enabled transition (deadlock)");
}

cleanup ()					/* terminate fire */
/*
 *  1. accumulate total simulation time
 *  2. update place-statistic to current time
 *  3. add up fire counts and rates
 *  4. update result statistic
 */
{
    register int i, j;
    register struct place *p;
    register struct transition *tr;
    register struct hist *h;
    register struct result *r;
    register double t, c, s;
    
    total_clock += c = clock;
    total_sq_clock += c * c;

    if (c < EPSILON)
	err("Simulation did not consume time");
    s = 1.0 / c;

    for (i = n_places,  p = PLACE; i--; p++) {
	h = p->H;
	h[p->cur_mrk].prob += clock - p->t_last;
	for (j = p->max_mrk; j--; h++) {
	    t = h->prob;
	    h->sum += t;
	    h->sum_sq += t * t;
	    h->sum_cr += t * c;
	}
    }

    for(i = n_transitions, tr = TRANSITION; i--; tr++) {
	tr->tot_fire_cnt += j = tr->fire_cnt;
	t = j * s;
	tr->sum += t;
	tr->sum_sq += t * t;
    }

    for (i = n_results,  r = RESULT; i--; r++) {
	if (r->type) {
	    (*(r->upd_fnctn))();
	} else {
	    t = r->acc + r->cur * (clock - r->t_last);
	    r->sum += t;
	    r->sum_sq += t * t;
	    r->sum_cr += t * c;
	}
    }
}

print_result ()					/* you guessed it */
{
    register int i, j;
    register struct place *p;
    register struct transition *t;
    register struct hist *h;
    register struct result *rp;
    register double x, c, q, r, s;
    double sqrt(), stud_t995();

    if (total_clock < EPSILON)
	err ("No timed transitions fired");

    q = stud_t995(n_rgen - 1);
    r = n_rgen / (total_clock * total_clock * (n_rgen - 1));
    s = 1.0 / total_clock;

    for (i = n_places, p = PLACE; i--; p++) {
	printf ("Place '%s':\n", p->name);
	for(j = 0, h = p->H; j < p->max_mrk; h++, j++) {
	    if (h->sum < EPSILON)
		continue;

						/* the next is practically verbatim from Chiola's original */
	    x = h->sum * s;
	    c = h->sum_sq + x * (x * total_sq_clock - 2 * h->sum_cr);
	    c = q * sqrt(c * r);

	    printf ("\tP(%d) = %8.6f (+/- %8.6f)\n", j, x, c);
	}
    }

    for (i = n_results, rp = RESULT; i--; rp++) {
	if (rp->type) {				/* dwell-time */
	    x = rp->sum / n_rgen;
	    c = rp->sum_sq - x * rp->sum;
	    c = q * sqrt(c / (n_rgen * (n_rgen - 1)));
	    printf ("%s = %8.6f (+/- %8.6f) [%s]\n", rp->name, x, c, (rp->type == 1) ? "time units" : "fireing / time unit");
	} else {				/* probabilities and expectations */
	    x = rp->sum * s;
	    c = rp->sum_sq + x * (x * total_sq_clock - 2 * rp->sum_cr);
	    c = q * sqrt(c * r);
	    printf ("%s = %8.6f (+/- %8.6f)\n", rp->name, x, c);
	}
    }

    for(i = n_transitions, t = TRANSITION; i--; t++) {
	x = t->sum / n_rgen;
	c = t->sum_sq - x * t->sum;
	c = q * sqrt(c / (n_rgen * (n_rgen - 1)));	
	printf("Transition %s : %8.6f (+/- %8.6f) [fireing / time-unit]\n", t->name, x, c);
    }
}

/* #define INP_FILE 1 */	/* expect an input file		 */

get_opt (argc, argv)			    /* process arguments	 */
    int argc;
    char *argv[];
{
    register int     i, j, k, aty;
    double d, *dbl;
    int l;
    struct parameter *pp;
    long    *var;
    char    opt, **str, fname[82];
    char *infn = 0;				/* input file name		 */
    char *outfn = 0;				/* output file name		 */

    aty = 0;					/* expect no sub argument	    */
    for (i = 1; i < argc; i++) {		/* scan arguments		    */
	if (argv[i][0] == '-' && !aty) {	/* got an option		    */

	    for (j = 1; argv[i][j]; j++) {	/* scan option string		    */
		opt = argv[i][j];
		for (k = 0; optv[k].c; k++)
		    if (opt == optv[k].c) {	/* found option			    */
			if (OPTV & optv[k].cf) {
			    fprintf(stderr, "Option '%c' conflicts with earlier option\n", opt);
			    goto error;
			}
			OPTV |= optv[k].bc;
			if (optv[k].ty) {
			    if (aty) {
			    fprintf(stderr, "Ambigious option/argument combination\n");
			    goto error;
				
			    }
			    aty = optv[k].ty;
			    str = optv[k].s;
			    var = optv[k].x;
			    dbl = optv[k].d;
			}
			break;
		    }
		if (!optv[k].c) {
		    fprintf(stderr, "Unknown option\n");
		    goto error;
		}
	    }
	} else {				/* got an argument		    */
	    switch (aty) {
	    case 6:				/*** double ***/
		if (1 != sscanf(argv[i], "%lf", dbl))
		    goto error;
		aty = 0;
		break;
	    case 5:				/*** parameter ***/
		for (j = n_parameters, pp = PARAMETER; j--; pp++)
		    if (!strcmp(argv[i], pp->name)) {
			if(pp->type)
			    aty = 4;		/* positive double param */
			else
			    aty = 3;		/* positive int param */
			break;
		    }
		if (j < 0) {
		    fprintf(stderr, "Undefined parameter\n");
		    goto error;
		}
		break;
	    case 4:				/*** convert a double > 0 ***/
		if (1 != sscanf(argv[i], "%lf", &d) || d < 0.0) {
		    fprintf(stderr, "Invalid value for '%s'\n", pp->name);
		    goto error;
		}
		if (d < EPSILON) d = 1e20;	/* approximate infinite delay */
		else d = 1.0 / d;
		pp->value = d;
		for (j = pp->ind, k = pp->n_used; k--; j++)
		    *d_param[j] = d;
		aty = 0;
		break;
	    case 3:				/*** integer > 0 parameter ***/
		if (1 != sscanf(argv[i], "%d", &l) || l < 0) {
		    fprintf(stderr, "Invalid value for '%s'\n", pp->name);
		    goto error;
		}
		j = pp->ind;
		if ((!*i_param[j]) != (!l)) {
		    fprintf(stderr, "Can't switch between 0 and non-0 #of tokens\n");
		    goto error;
		}
		pp->value = l;
		for (k = pp->n_used; k--; j++)
		    *i_param[j] = l;
		aty = 0;
		break;
	    case 2:				/*** string sub-argument ***/
		*str = argv[i];
		aty = 0;
		break;
	    case 1:				/*** integer sub-argument ***/
		if (1 != sscanf(argv[i], "%ld", var))
		    goto error;
		aty = 0;
		break;
	    default:				/*** unspecified arg (filename) ***/
#ifdef INP_FILE
		infn = argv[i];
		i++;
#endif
		if (i < argc) {			/* output file name present ?	 */
		    if ('-' == argv[i][0] || (i + 1) < argc)
			goto error;
		    outfn = argv[i];
		}
	    }
	}
    }

    if (infn) {
	yyin = fopen(infn, "r");
	if (!yyin) {
	    fprintf(stderr, "Could not open '%s' for read\n", infn);
	    exit(1);
	}
    } else
	yyin = stdin;

    if (outfn) {
	dstf = fopen(outfn, "w");
	if (!dstf) {
	    fprintf(stderr, "Could not open '%s' for write\n", outfn);
	    exit(1);
	}
    } else
	dstf = stdout;

    return;			/* options are ok		 */

error:				/* some trouble with the arguments */
    fprintf(stderr, "gs [-");
    for (i = 0; optv[i].c; i++)
	fprintf(stderr, "%c", optv[i].c);
    fprintf(stderr, "] [input-file] [output-file]\n");

    exit(1);
}

double stud_t995 (n)		/* approximation to Student's T-percentiles */
    int n;
{
    static double tab[30] = {
	63.657, 9.925, 5.841, 4.604, 4.032, 3.707, 3.499, 3.355, 3.250, 3.169,
	 3.106, 3.055, 3.012, 2.977, 2.947, 2.921, 2.898, 2.878, 2.861, 2.845,
	 2.831, 2.819, 2.807, 2.797, 2.787, 2.779, 2.771, 2.763, 2.756, 2.750
    };

    if (n < 1)
	err ("Student's T percentiles not defined for n < 1");

    if (n <= 30)
	return tab[n - 1];

    if (n > 200)
	return 2.576;

    n -= 30;
    if (n  < 10)
	return 2.750 - n * 0.0046;
    n -= 10;
    if (n < 20)
	return 2.704 - n * 0.0022;
    n -= 20;
    if (n < 60)
	return 2.660 - n * 0.0007;
    n -= 60;
    return 2.617 - n * 0.0005;
}

double acc_dwt(p)				/* accumulate dwell time */
    register struct place *p;
{
    register double t = 0.0;
    register struct hist *h = p->H + 1;
    register int i;

    for(i = 1; i < p->max_mrk; i++, h++)
	t += i * h->prob;

    return t;
}
