/* $Id: ps.c,v 1.21 90/05/09 22:14:48 pturner Exp Locker: pturner $
 *
 *   driver for postscript printer
 *
 * courtesy of:
 *
 * Jim Hudgens
 * hudgens@ray.met.fsu.edu
 *
 * Further modifications by,
 * Ole Holm Nielsen
 * ohnielse@ltf.dth.dk
 *
 * Further modifications by
 * Jeff Treece
 * treece@sabbagh.com
 */

#include <stdio.h>
#include "externs.h"

extern char version[];
extern double charsize;
extern double devcharsize;
extern int ptofile;
extern char printstr[];
extern char *curprint;		/* curprint = ps_prstr */

/*
 * printer control string
 */
#ifndef PS_PRSTR
char ps_prstr[128] = "/usr/ucb/lpr -h";

#else
char ps_prstr[128] = PS_PRSTR;

#endif

static char tmpbuf[64];

/* postscript page at scale = 0.25 */

#define PSXMIN 150
#define PSXMAX 2200
#define PSYMIN 150
#define PSYMAX 3200
#define DXPS 2050
#define DYPS 3050
#define CHARS 1.5

#define PSXMINP 150
#define PSXMAXP 2200
#define PSYMINP 550
#define PSYMAXP 2700
#define DXPSP 2050
#define DYPSP 2050
#define CHARSP 1.25

#define MINCOLOR 0
#define MAXCOLOR 9

#define MAXLINESTYLE 14

static int psxmin = PSXMIN;
static int psxmax = PSXMAX;
static int psymin = PSYMIN;
static int psymax = PSYMAX;
static int psdx = DXPS;
static int psdy = DYPS;

static int pscolor;
static int psdmode;
static int psfont = 0;
static double pscharsize = 1.5;
static int pslinestyle;

double xconv(), yconv();

static FILE *psout;

void putstrps(s)
    char *s;
{
    fprintf(psout, s);
}

static char *fname;

static int orientflag = 0;

int pssetmode(mode)
    int mode;
{
    char tbuf[128];
    char *mktemp();

    if (mode % 2) {
	if (!ptofile) {
	    strcpy(tmpbuf, "/usr/tmp/grtoolXXXXXX");
	    fname = mktemp(tmpbuf);
	} else {
	    fname = printstr;
	}
	if ((psout = fopen(fname, "w")) == NULL) {
	    return 0;
	}
    }
    switch (mode) {
    case 1:			/* PS landscape */
	orientflag = 1;
	pscharsize = CHARS;
	psxmin = PSXMIN;
	psxmax = PSXMAX;
	psymin = PSYMIN;
	psymax = PSYMAX;
	psdx = DXPS;
	psdy = DYPS;
	break;
    case 3:			/* PS portrait */
	orientflag = 0;
	pscharsize = CHARSP;
	psxmin = PSXMINP;
	psxmax = PSXMAXP;
	psymin = PSYMINP;
	psymax = PSYMAXP;
	psdx = DXPSP;
	psdy = DYPSP;
	break;
    case 2:
    case 4:
	putstrps("stroke\n");
	putstrps("showpage\n");
	putstrps("restore\n");
	fclose(psout);
	if (!ptofile) {
	    sprintf(tbuf, "%s %s", curprint, fname);
	    system(tbuf);
	    unlink(fname);
	}
	orientflag = 0;
	break;
    }
}

static int x1 = 99999, y1 = 99999;

#define MAXPATHLEN 500		/* 500 points in a postscript path */
static int pathlength = 0;

 /* between strokes */
/* note that this is an very conservative estimate.  datasets
   of the form: newpath x0 y0 moveto x1 y1 lineto .... xN yN lineto
   are observed to overflow around N=2000 to 3000.  Solved by using
       .... xM yM lineto stroke xM yM moveto xM1 yM1 lineto ....
   for M of 500.
 */
/* MUST make sure that the path is stroked, though... */
/* stroke when:
      finishing
      changing line styles
      changing line colors
   as well as when going pen up.
 */

void drawps(x2, y2, mode)
    int x2, y2, mode;
{
    char stmp[30];
    int xtmp, ytmp;

    if (x2 < 0 || y2 < 0)	/* Eliminate garbage on output */
	return;

    if (orientflag) {
	xtmp = y2;
	ytmp = (-x2) + psymax;

	if (mode) {
	    if (pathlength > MAXPATHLEN) {	/* stroke, moveto, lineto */
		sprintf(stmp, "k %d %d m %d %d l\n", x1, y1, xtmp, ytmp);
		pathlength = 0;
	    } else {		/* lineto */
		sprintf(stmp, "  %d %d l\n", xtmp, ytmp);
		pathlength++;
	    }
	    putstrps(stmp);
	} else {		/* a move to the same point */
	    if ((pathlength <= MAXPATHLEN) && (xtmp == x1 && ytmp == y1)) {
		return;		/* silly request */
	    }
	    sprintf(stmp, "k %d %d m\n", xtmp, ytmp);
	    putstrps(stmp);
	    pathlength = 0;
	}
	x1 = xtmp;
	y1 = ytmp;
    } else {
	if (mode) {
	    if (pathlength > MAXPATHLEN) {
		/* stroke, moveto, lineto */
		sprintf(stmp, " k %d %d m %d %d l\n", x1, y1, x2, y2);
		pathlength = 0;
	    } else {		/* lineto */
		pathlength++;
		sprintf(stmp, " %d %d l\n", x2, y2);
	    }
	    putstrps(stmp);
	} else {
	    /* a move to the same point */
	    if ((pathlength <= MAXPATHLEN) && (x2 == x1 && y2 == y1)) {
		return;		/* silly request */
	    }
	    sprintf(stmp, " k %d %d m\n", x2, y2);
	    putstrps(stmp);
	    pathlength = 0;
	}
	x1 = x2;
	y1 = y2;
    }
}

int xconvps(x)
    double x;
{
    if (orientflag) {
	return ((int) (psymin + psdy * xconv(x)));
    } else {
	return ((int) (psxmin + psdx * xconv(x)));
    }
}

int yconvps(y)
    double y;
{
    if (orientflag) {
	return ((int) (psxmin + psdx * yconv(y)));
    } else {
	return ((int) (psymin + psdy * yconv(y)));
    }
}

void pssetfont(n)
    int n;
{
    hselectfont(psfont = n);
}

int pssetcolor(c)
    int c;
{
    char stmp[50];

    if (pscolor == c) {
	return;
    }
    c = c % MAXCOLOR;
    sprintf(stmp, "k %d w\n", 2 * (c - 1) + 1);
    putstrps(stmp);
    /* force a move on the next call to drawps */
    pathlength = MAXPATHLEN + 1;
    pscolor = c;
    return c;
}

void psdrawtic(x, y, dir, updown)
    int x, y, dir, updown;
{
    switch (dir) {
    case 0:
	switch (updown) {
	case 0:
	    drawps(x, y, 0);
	    drawps(x, y + devxticl, 1);
	    break;
	case 1:
	    drawps(x, y, 0);
	    drawps(x, y - devxticl, 1);
	    break;
	}
	break;
    case 1:
	switch (updown) {
	case 0:
	    drawps(x, y, 0);
	    drawps(x + devyticl, y, 1);
	    break;
	case 1:
	    drawps(x, y, 0);
	    drawps(x - devyticl, y, 1);
	    break;
	}
	break;
    }
}

int pssetlinestyle(style)
    int style;
{
    char stmp[50];

    if (pslinestyle == style) {
	return;
    }
    switch (style) {
    case 1:			/* solid */
	strcpy(stmp, "k [] 0 d\n");
	break;
    case 2:			/* dotted */
	strcpy(stmp, "k [3 10] 0 d\n");
	break;
    case 3:			/* long dash */
	strcpy(stmp, "k [10 5] 0 d\n");
	break;
    case 4:			/* short dash */
	strcpy(stmp, "k [5 5] 0 d\n");
	break;
    case 5:			/* dot-dashed */
	strcpy(stmp, "k [10 5 3 5] 0 d\n");
	break;
    default:
	strcpy(stmp, "k [] 0 d\n");
	break;
    }
    putstrps(stmp);
    /* force a move on the next call to drawps */
    pathlength = MAXPATHLEN + 1;
    return (pslinestyle = style);
}

void dispstrps(x, y, rot, s)
    int x, y, rot;
    char *s;
{
    puthersh(x, y, pscharsize * charsize, rot, pscolor, vector, s);
}

void psleavegraphics()
{
    pssetmode(psdmode + 1);
}

/*
 * postscript initialization routine
 */
psinitgraphics(dmode)
    int dmode;
{
    char stmp[80];

    /* re-initialize each time */
    /* probably not necessary */
    x1 = 99999;
    y1 = 99999;
    pscolor = 99;
    pslinestyle = 99;
    psdmode = dmode;
    pssetmode(psdmode);
    devconvx = xconvps;
    devconvy = yconvps;
    vector = drawps;
    devwritestr = dispstrps;
    devsetcolor = pssetcolor;
    devsetfont = pssetfont;
    devsetline = pssetlinestyle;
    devdrawtic = psdrawtic;
    devleavegraphics = psleavegraphics;
    devcharsize = pscharsize;
    devxticl = 20;
    devyticl = 20;
    devarrowlength = 20;

/*    putstrps("%%! PostScript\n"); */
    putstrps("%%!PS-Adobe\n");
    sprintf(stmp, "%%%%%%%%BoundingBox: %d %d %d %d\n",
	    PSXMINP * 72 / 300, PSYMINP * 72 / 300, PSXMAXP * 72 / 300, PSYMAXP * 72 / 300);
    putstrps(stmp);
    sprintf(stmp, "%%%%%%%% Creator: %s\n", version);
    putstrps(stmp);

    putstrps("/w {setlinewidth} bind def\n");
    putstrps("/m {moveto} bind def\n");
    putstrps("/l {lineto} bind def\n");
    putstrps("/k {stroke} bind def\n");
    putstrps("/d {setdash} bind def\n");

    putstrps("save\n");
    putstrps(".25 .25 scale\n");
    putstrps("1 setlinecap\n");
    pssetcolor(1);
    setfont(2);
    setcolor(1);
    setlinestyle(0);
}
