
/*
 *           PVM 3.0:  Parallel Virtual Machine System 3.0
 *               University of Tennessee, Knoxville TN.
 *           Oak Ridge National Laboratory, Oak Ridge TN.
 *                   Emory University, Atlanta GA.
 *      Authors:  A. L. Beguelin, J. J. Dongarra, G. A. Geist,
 *          R. J. Manchek, B. K. Moore, and V. S. Sunderam
 *                   (C) 1992 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * Neither the Institutions (Emory University, Oak Ridge National
 * Laboratory, and University of Tennessee) nor the Authors make any
 * representations about the suitability of this software for any
 * purpose.  This software is provided ``as is'' without express or
 * implied warranty.
 *
 * PVM 3.0 was funded in part by the U.S. Department of Energy, the
 * National Science Foundation and the State of Tennessee.
 */

/*
 *	task.c
 *
 *	Task descriptors.
 *
$Log$
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#ifdef	SYSVSTR
#include <string.h>
#else
#include <strings.h>
#endif

#include "global.h"
#include "ddpro.h"
#include "protoglarp.h"
#include "mesg.h"
#include "pkt.h"
#include "task.h"
#include "waitc.h"
#include "myalloc.h"
#include "listmac.h"


/***************
 **  Globals  **
 **           **
 ***************/

extern void bailout();

extern struct waitc *waitlist;		/* from waitc.c */

struct task *locltasks = 0;

void task_dump();

/***************
 **  Private  **
 **           **
 ***************/

static char rcsid[] = "$Id$";
static char etext[512];		/* scratch for error log */


/*****************
 **  Interface  **
 **             **
 *****************/

/*	task_init()
*
*	Call once before any other task stuff.
*/

void
task_init()
{
	if (!locltasks) {
		locltasks = TALLOC(1, struct task, "task");
		bzero((char*)locltasks, sizeof(struct task));
		locltasks->t_link = locltasks;
		locltasks->t_rlink = locltasks;
	}
}


/*	task_new()
*
*	Make a new task descriptor and add to list of local tasks.
*/

struct task *
task_new(tid)
	int tid;
{
	struct task *tp;

	if (!(tp = TALLOC(1, struct task, "task"))) {
		log_error("task_new() can't get memory\n");
		bailout(0);
	}
	bzero((char*)tp, sizeof(struct task));
	tp->t_tid = tid;
	tp->t_txq = pk_new(0);
	tp->t_sock = -1;
	tp->t_out = -1;
	tp->t_authfd = -1;
	LISTPUTBEFORE(locltasks, tp, t_link, t_rlink);
	return tp;
}


/*	task_free()
*
*	Do low-level cleanup needed when a task exits.
*	Remove task descriptor from local tasks list and destroy it.
*	Close any fds, unlink any files, free mbufs.
*/

void
task_free(tp)
	struct task *tp;
{
	if (tp->t_link && tp->t_rlink) {
		LISTDELETE(tp, t_link, t_rlink);
	}
	if (tp->t_rxm)
		mesg_unref(tp->t_rxm);
	if (tp->t_rxp)
		pk_free(tp->t_rxp);
	pk_free(tp->t_txq);
	if (tp->t_wait)
		wait_delete(tp->t_wait);
	if (tp->t_authnam)
		PVM_FREE(tp->t_authnam);
	if (tp->t_sock != -1) {
		wrk_fds_delete(tp->t_sock, 3);
		(void)close(tp->t_sock);
	}
	if (tp->t_out != -1) {
		wrk_fds_delete(tp->t_out, 1);
		(void)close(tp->t_out);
	}
	if (tp->t_authfd != -1)
		(void)close(tp->t_authfd);
	if (tp->t_a_out)
		PVM_FREE(tp->t_a_out);
	if (tp->t_mca)
		mca_free(tp->t_mca);
	PVM_FREE(tp);
}


/*	task_find()
*
*	Find a task in local tasks list by its tid.
*/

struct task *
task_find(tid)
	int tid;
{
	struct task *tp;

	for (tp = locltasks->t_link; tp != locltasks; tp = tp->t_link)
		if (tp->t_tid == tid)
			return tp;
	return (struct task*)0;
}


/*	task_cleanup()
*
*	Do high-level cleanup needed when a task exits.
*	Wake up any entities waiting on task, free multicast context.
*	XXX should flush any partial messages, but that would be hard.  hm.
*/

void
task_cleanup(tp)
	struct task *tp;
{
	struct mesg *mp;
	struct waitc *wp, *wp2;

	/* notify anyone who asked */

	for (wp = waitlist->wa_link; wp != waitlist; wp = wp->wa_link) {
		if (wp->wa_on == tp->t_tid) {
			mp = mesg_new(0);
			mp->m_dst = wp->wa_tid;
			if (TIDISTASK(mp->m_dst))
				mp->m_cod = wp->wa_dep;
			else {
				mp->m_cod = DM_NOTIFYACK;
				mp->m_wid = wp->wa_dep;
			}
			pkint(mp, tp->t_tid);
			sendmessage(mp);
			wp2 = wp;
			wp = wp->wa_rlink;
			wait_delete(wp2);
		}
		if (wp->wa_tid == tp->t_tid)
			wp->wa_tid = 0;				/* in case tid gets recycled */
	}

	/* complete multicast */

	if (tp->t_mca) {
/* XXX should send an EOM frag to all rcpts */
		mca_free(tp->t_mca);
		tp->t_mca = 0;
	}
}


void
task_dump()
{
	struct task *tp;
	struct pkt *pp;

	log_error("task_dump()\n");
	log_error("     tid     ptid    flags   pid\n");
	for (tp = locltasks->t_link; tp != locltasks; tp = tp->t_link) {
		sprintf(etext, "%8x %8x %8x %5d\n",
				tp->t_tid, tp->t_ptid,
				tp->t_flag, tp->t_pid);
		log_error(etext);
		log_error(" txq:     pkt      src      dst     flag      len      ofs\n");
		for (pp = tp->t_txq->pk_link; pp != tp->t_txq; pp = pp->pk_link) {
			sprintf(etext, "     %08x %8x %8x %8x %8d %8d\n",
					pp,
					pp->pk_src,
					pp->pk_dst,
					pp->pk_flag,
					pp->pk_len,
					pp->pk_dat - pp->pk_buf);
			log_error(etext);
		}
	}
	log_error("\n");
}

