BeginPackage["Tessellate`", "Global`"]

(* 

This package defines some routines for calculating and visualizing
the action of Moebius transformations in the unit disk.

Requires Mathematica version 2.0 or greater.

This newest version of this package is available for anonymous ftp at
nic.funet.fi, directory "pub/math/riemann/mathematica".

Version 1.3.0. Last modified: June 14th, 1994.

Send comments and bug reports to 
  Juha Haataja, Centre for Scientific Computing (jhaataja@csc.fi).

*)

(**********************************************************************)

UnitCircle::usage = "Unit circle for drawing, for example
Show[Graphics[UnitCircle]]"

ZToReIm::usage = "Convert a complex number to a list of real and
imaginary parts: ZToReIm[z]"

ReImToZ::usage = "Convert a list of two numbers to a complex number: 
ReImToZ[{x,y}]"

AngleOfPoint::usage = "Find the angle of a complex number in relation
to another (default is origin): AngleOfPoint[z1, z2]"

RotatePoint::usage = "Rotate a complex number around another complex 
number: RotatePoint[z1 ,theta,z2]"

ZCircle::usage = "ZCircle[z,r] is a circle centered at point z in the
complex plane, having radius r."

ZLine::usage = "ZLine[{z1, z2}] is a Euclidean line from point z1 to
point z2 in the complex plane."

EuclDistance::usage = "Euclidean distance of two complex numbers: 
EuclDistance[z1, z2]"

HypDistance::usage = "Hyperbolic distance of two complex numbers:
HypDistance[z1, z2]"

DoubleRatio::usage = "Calculate the double ratio: DoubleRatio[a,b,c,d]"

MovePoint::usage = "Move a complex number a given hyperbolic distance
from origin to a given direction: MovePoint[dist, theta]"

ParallelArc::usage = "Is a geodesic through z1 and z2 an Euclidean
line: ParallelArc[z1,z2]"

TriFromSidesAndAngle::usage = "Construct a hyperbolic triangle, when
the length of two sides and the angle between are given:
TriFromSidesAndAngle[s1, s2, angle]"

TriFromAngles::usage = "Construct a triangle, when all three angles
are given: TriFromAngles[a1, a2, a3]"

UnitPolygon::usage = "Make a unit polygon of given number of sides:
UnitPolygon[n]"

RegularPolygon::usage = "Construct a regular polygon with a given
number of sides and a given angle between the sides:
RegularPolygon[n,angle]"

GeodCircleCenter::usage = "Find the center of a geodesic circle going
through two given points: GeodCircleCenter[z1, z2]"

SolveCircles::usage = "Find the common points of two circles:
SolveCircles[{{x1,y1},r1},{{x2,y2},r2}]"

CircleAroundPolygon::usage = "Find the circle drawn around a given
regular polygon: CircleAroundPolygon[{z1, z2, z3}]"

HypCenterOfPolygon::usage = "Find the hyperbolic centre of a circle
draw around a regular polygon: HypCenterOfPolygon[{z1, z2, z3}]"

HypMiddleOfArc::usage = "Find the middle point of a hyperbolic arc
segment: HypMiddleOfArc[{z1, z2}]"

HypReflect::usage = "Reflect a complex point across an euclidean line:
HypReflect[z, {z1, z2}]"

HypInvert::usage = "Invert a complex point across a geodesic:
HypInvert[z, {z1, z2}]"

HypInvertPolygon::usage = "Invert a set of points (eg. a polygon)
across a given edge (HypInvertPolygon[vertices, index]) or a set of
edges (HypInvertPolygon[vertices, {ind1, ind2, ind3}]"

PolygonInvGroup::usage = "Make an inversion group for a polygon:
PolygonInvGroup[{z1, z2, z3, z4}, level]"

MakePoint::usage = "Convert a complex number into a graphics object:
MakePoint[z]"

MakeCircle::usage = "Convert a circle into a graphics object:
MakeCircle[{z, r}]"

MakeCircleAroundPolygon::usage = "Make a circle around a regular
polygon: MakeCircleAroundPolygon[{z1, z2, z3}]"

MakeGeodOrthoCircle::usage = "Construct a circle orthogonal to a given
circle, going through a given point: MakeGeodOrthoCircle[ZCircle[center,
radius], z]"

MakeGeodArc::usage = "Make a geodesic going from a given point to a
given point: MakeGeodArc[{z1, z2}]"

MakeBisector::usage = "Make a bisector to a geodesic arc {z1,z2}:
MakeBisector[{z1,z2}]"

MakeHypPolygon::usage = "Make a hyperbolic polygon when the vertices
are given: MakeHypPolygon[{z1, z2, z3, z4}]"

MakeHypPolygonDiagonals::usage = "Make the diagonal arcs for a
hyperbolic polygon when the vertices are given:
MakeHypPolygonDiagonals[{z1, z2, z3, z4}]"

MakeColorLevel::usage = "Make the hue needed for drawing:
MakeColorLevel[i,max]"

DrawPoints::usage = "Draw a collection a complex points:
DrawPoints[{z1, z2, z3, z4}]"

DrawArcs::usage = "Draw a collection of geodesic arcs:
DrawArcs[{{z1,z2},{z3,z4}}]"

DrawCircles::usage = "Draw a list of circles: DrawCircles[{{z, r}}]"

DrawHypPolygons::usage = "Draw a list of hyperbolic polygons:
DrawHypPolygons[{pg1, pg2, pg3}, Colored -> True, Diagonals -> True]"

DrawCirclesAroundPolygons::usage = "Draw a list of circles around a
given list of regular polygons: DrawCirclesAroundPolygons[{tri1,
tri2}]"


(**********************************************************************)

(* Default options for functions *)

Options[MakeColorLevel] = {Colored -> False, GrayLevels -> False}

Options[DrawPoints] = {Colored -> False, GrayLevels -> False, 
	Prolog -> {AbsolutePointSize[4]}, 
	Trailer -> {AspectRatio -> Automatic}, Label -> ""}

Options[DrawArcs] = {Colored -> False, GrayLevels -> False, 
	Prolog -> {AbsoluteThickness[1]}, 
	Trailer -> {AspectRatio -> Automatic}, Label -> ""}

Options[DrawCircles] = {Colored -> False, GrayLevels -> False, 
	Prolog -> {AbsoluteThickness[1]}, 
	Trailer -> {AspectRatio -> Automatic}, Label -> ""}

Options[DrawHypPolygons] = {Colored -> False, GrayLevels -> False, 
	Diagonals -> False, Bisectors -> False,
	Prolog -> {AbsoluteThickness[2], UnitCircle, AbsoluteThickness[1]},
	Trailer -> {AspectRatio -> Automatic, 
		PlotRange -> {{-1.1,1.1},{-1.25,1.1}} },
	Label -> ""}

Options[DrawCirclesAroundPolygons] = {Colored -> False, GrayLevels -> False,
	Prolog -> {AbsoluteThickness[2], UnitCircle, AbsoluteThickness[1]}, 
	Trailer -> {AspectRatio -> Automatic, 
		PlotRange -> {{-1.1,1.1},{-1.25,1.1}} },
	Label -> ""}

(**********************************************************************)

(* Descriptions of options for functions *)

Colored::usage = "Pictures are drawn in color: DrawHypPolygons[pts,
Colored -> True]. Option of functions MakeColorLevel, DrawPoints,
DrawArcs, DrawCircles, DrawHypPolygons and DrawCirclesAroundPolygons"

GrayLevels::usage = "Pictures are drawn using grayscale:
DrawHypPolygons[pts, GrayLevels -> True]. Option of functions
MakeColorLevel, DrawPoints, DrawArcs, DrawCircles, DrawHypPolygons and
DrawCirclesAroundPolygons"

Prolog::usage = "List of graphics primitives prepended to actual
graphics: DrawHypPolygons[pts, Prolog ->
{AbsoluteThickness[1]}]. Option of functions DrawPoints, DrawArcs,
DrawCircles, DrawHypPolygons and DrawCirclesAroundPolygons"

Trailer::usage = "List of graphics primitives appended to actual
graphics. Option of functions DrawPoints, DrawArcs, DrawCircles,
DrawHypPolygons and DrawCirclesAroundPolygons"

Label::usage = "Title text of actual graphics: DrawHypPolygons[pts,
Label -> \"Test picture\"]. Option of functions DrawPoints, DrawArcs,
DrawCircles, DrawHypPolygons and DrawCirclesAroundPolygons"

Diagonals::usage = "Boolean-valued option telling if diagonals are
drawn: DrawHypPolygons[pts, Diagonals -> True]. Option of function
DrawHypPolygons"

Bisectors::usage = "Boolean-valued option telling if bisectors to the
polygon sides are drawn: DrawHypPolygons[pts, Bisectors ->
True]. Option of function DrawHypPolygons"

(**********************************************************************)

Begin["`Private`"]

(**********************************************************************)

(* Filtering options *)

FilterOptions[command_Symbol, opts___] :=
	Block[{keywords = First /@ Options[command]},
		Sequence @@ Select[{opts}, MemberQ[keywords, First[#]]&]]

(**********************************************************************)

(* Some automatic conversions *)

SolveCircles[ZCircle[z_/;NumberQ[z],r_/;NumberQ[r]], a_] ^:= 
	SolveCircles[Circle[ZToReIm[z],r], a]

SolveCircles[a_,ZCircle[z_/;NumberQ[z],r_/;NumberQ[r]]] ^:= 
	SolveCircles[Circle[ZToReIm[z],r], a]

SolveCircles[ZLine[{z1_/;NumberQ[z1],z2_/;NumberQ[z2]}],a_] ^:=
	SolveCircles[Line[{ZToReIm[z1], ZToReIm[z2]}], a]

SolveCircles[a_,ZLine[{z1_/;NumberQ[z1],z2_/;NumberQ[z2]}]] ^:=
	SolveCircles[a, Line[{ZToReIm[z1], ZToReIm[z2]}]]

ZToReIm[ZCircle[z_/;NumberQ[z],r_/;NumberQ[r]]] ^:= Circle[ZToReIm[z], r]

ZToReIm[ZLine[{z1_/;NumberQ[z1],z2_/;NumberQ[z2]}]] ^:= 
	Line[{ZToReIm[z1], ZToReIm[z2]}]

MakeGeodOrthoCircle[ZCircle[z_/;NumberQ[z],r_/;NumberQ[r]],p_] ^:=
	MakeGeodOrthoCircle[Circle[ZToReIm[z],r], p]

MakeGeodOrthoCircle[ZLine[{z1_/;NumberQ[z1],z2_/;NumberQ[z2]}],a_] ^:=
	MakeGeodOrthoCircle[Line[{ZToReIm[z1],ZToReIm[z2]}], a]

MakeGeodOrthoCircle[a_,ZLine[{z1_/;NumberQ[z1],z2_/;NumberQ[z2]}]] ^:=
	MakeGeodOrthoCircle[a, Line[{ZToReIm[z1],ZToReIm[z2]}]]

Unprotect[Circle]
ReImToZ[Circle[{x_/;NumberQ[x],y_/;NumberQ[y]},r_/;NumberQ[r]]] ^:= 
	ZCircle[ReImToZ[{x,y}], r]
Protect[Circle]


(**********************************************************************)

(* Some definitions *)

GeomPrecision = 16;  (* The precision of floating-point calculations *)

NGP[arg_] := N[arg, GeomPrecision]
SetAttributes[NGP, Listable]

ComparisonTolerance = NGP[10^(-10)];

NPi = NGP[Pi]
N2Pi = NGP[2*Pi]

UnitCircle = Circle[{0,0},1]

(* To check out numerical arguments *)

NumberArgQ[vars___] := And @@ Map[NumberQ[NGP[#]]&, {vars}]

(* To flatten a list of polygons *)

FlattenObjectList[l_List] :=
	FixedPoint[Replace[#1, {A___, {X___List}, B___} :> 
		{A, X, B}] &, l]

(**********************************************************************)

(* Routines for handling complex numbers *)

(* Complex number conversions *)

ZToReIm[z_] := {Re[z],Im[z]}
ReImToZ[{x_,y_}] := x + I*y

(* Angle of complex number p1 in relation to p2 *)

AngleOfPoint[p1_,p2_] := Arg[p1-p2]
AngleOfPoint[p1_] := Arg[p1]

(* Rotate a complex number around origin *)

RotatePoint[z_,theta_] := NGP[Exp[I*theta]*z]

(**********************************************************************)

(* Distance functions*)

(* Distances of two points *)

EuclDistance[{x1_,y1_}, {x2_,y2_}] := NGP[Sqrt[(x1-x2)^2 + (y1-y2)^2]]

EuclDistance[p1_,p2_:0] := 
	NGP[Sqrt[Plus @@ ({Re[p1-p2],Im[p1-p2]}^2)]]

HypDistance[p1_,p2_:0] := 
	If[SameQ[p1,p2], 0, 
	Block[{part1,part2},
		part1 = EuclDistance[1-p1*Conjugate[p2]];
		part2 = EuclDistance[p1,p2];
		If[SameQ[part1,part2], Infinity,
			Log[(part1+part2)/(part1-part2)]]]] /;
				NumberArgQ[p1,p2]

(* Move a complex number hyperbolic distance dist from origin to
direction theta *)

MovePoint[dist_,theta_:0] :=
	Block[{ndist = NGP[dist]},
	NGP[Exp[I*theta]*If[dist==Infinity, 1,
		(Exp[ndist] - 1)/(Exp[ndist] + 1)]]]

(**********************************************************************)

(* Construction of triangles and polygons *)

(* Construct a triangle, when two sides and the angle between are given *)

TriFromSidesAndAngle[dist1_, dist2_, angle_]:= 
	{0, MovePoint[dist1], MovePoint[dist2,angle]} /; 
		NumberArgQ[dist1,dist2,angle]

(* Construct a triangle, when all angles are given, and first is
   non-zero *)

TriFromAngles[a1_/;NGP[a1]!=0, a2_, a3_]:= 
	Block[{dist1, dist2},
	dist1 = If[NGP[a3]==0, Infinity,
           ArcCosh[(Cos[a1]*Cos[a3]+Cos[a2])/(Sin[a1]*Sin[a3])]];
        dist2 =If[NGP[a2]==0, Infinity,
           ArcCosh[(Cos[a1]*Cos[a2]+Cos[a3])/(Sin[a1]*Sin[a2])]];
	{0, MovePoint[dist1], MovePoint[dist2,a1]}] /;
		NumberArgQ[a1,a2,a3]

(* Construct a triangle, when all angles are given *)

TriFromAngles[a1_, a2_, a3_]:= 
	If[NGP[a1]==0 && NGP[a2]==0 
		&& NGP[a3]==0,
		{-I, Exp[I*NPi/6], Exp[I*5*NPi/6]},
		If[NGP[a1]!=0, TriFromAngles[a1, a2, a3], 
			TriFromAngles[a2, a3, a1]]] /; NumberArgQ[a1,a2,a3]


(* Make polygon of wanted size *)

UnitPolygon[sides_, rotation_, scaling_] :=
	NGP[scaling*Table[Exp[I*(rotation + i*N2Pi/sides)], 
		{i,0,sides-1}]]

UnitPolygon[sides_] := 
	Table[Exp[I*i*N2Pi/sides], {i,0,sides-1}] 

(* Construct a regular polygon *)

RegularPolygon[sides_, angle_, rotation_:0]:=
        Block[{beta = NPi/sides, dist, i, alfa},
        alfa = If[SameQ[angle,Indeterminate], NPi/4*(1-6/sides), angle/2];
        dist = MovePoint[ArcCosh[Cot[alfa]Cot[beta]]];
        NGP[UnitPolygon[sides, rotation, dist]]] 

(**********************************************************************)

(* Tool routines *)

(* Is a geodesic through z1 and z2 an Euclidean line? *)

ApproxEqual = Compile[{{z1,_Complex},{z2,_Complex}},
        Abs[z1 - z2] <= 10^(-8)]

ParallelArc = Compile[{{z1,_Complex},{z2,_Complex}},
	((Abs[z1] <= 10^(-8)) || (Abs[z2] <= 10^(-8))) ||
	((Abs[z1] > 10^(-8)) && (Abs[z2] > 10^(-8)) &&
		((Abs[NGP[Arg[z1]]-NGP[Arg[z2]]] <= 10^(-8)) ||
		(Abs[Abs[NGP[Arg[z1]]-NGP[Arg[z2]]]-NPi] <= 10^(-8)) ||
		(Abs[Abs[NGP[Arg[z1]]-NGP[Arg[z2]]]-N2Pi] <= 10^(-8))))]

(* Construct a geodesic circle going through given points *)

GeodCircleCenter = Compile[{{z1,_Complex},{z2,_Complex}},
        Block[{s1 = 1 + Abs[z1]^2, s2 = 1 + Abs[z2]^2,
                norm = 2*(Re[z2]*Im[z1] - Re[z1]*Im[z2])},
                -I*(z1*s2 - z2*s1)/norm]]

(* Find the common points of two circles or lines *)

SolveCircles[Line[{{x1_,y1_}, {x2_,y2_}}], 
		Line[{{x3_,y3_}, {x4_,y4_}}]] :=   
	If[SameQ[x1,x2], 
		If[SameQ[x3,x4], ComplexInfinity,
			Block[{k = (y4-y3)/(x4-x3)},
			ReImToZ[{x1,k*x1 + y3 - k*x3}]]],
	If[SameQ[x3,x4], 
		Block[{k = (y2-y1)/(x2-x1)},
		ReImToZ[{x3,k*x3 + y1 - k*x1}]],
	Block[{k1 = (y2-y1)/(x2-x1), k2 = (y4-y3)/(x4-x3), b1, b2},
	b1 = y1 - k1*x1; b2 = y3 - k2*x3;
	ReImToZ[{b1/(-k1 + k2) - b2/(-k1 + k2), -b2*k1/(-k1 + k2)
				+ (b1*k2)/(-k1 + k2)}]]]] /;
			NumberArgQ[x1, y1, x2, y2, x3, y3, x4, y4]
	
SolveCircles[Circle[{x1_,y1_},r1_], Circle[{x2_,y2_},r2_]] :=
        Block[{exp1, exp2, denom, points},
	exp1 = (-r1^2 + 2*r1*r2 - r2^2 + x1^2 - 2*x1*x2 + x2^2 + 
		y1^2 - 2*y1*y2 + y2^2)^(1/2);
	exp2 = (r1^2 + 2*r1*r2 + r2^2 - x1^2 + 2*x1*x2 - x2^2 - y1^2 + 
		2*y1*y2 - y2^2)^(1/2);
	denom = 2*(x1^2 - 2*x1*x2 + x2^2 + y1^2 - 2*y1*y2 + y2^2);
	points =  {{x1^3 - x1^2*x2 + x2^3 + exp1*exp2*y1 - exp1*exp2*y2 + 
		x2*(r1^2 - r2^2 + y1^2 - 2*y1*y2 + y2^2) + 
		x1*(-r1^2 + r2^2 - x2^2 + y1^2 - 2*y1*y2 + y2^2), 
		exp1*exp2*x2 + y1^3 + x1*(-(exp1*exp2) + x2*(-2*y1 - 2*y2)) + 
		(r1^2 - r2^2)*y2 - y1^2*y2 + y2^3 + x1^2*(y1 + y2) + 
		x2^2*(y1 + y2) + y1*(-r1^2 + r2^2 - y2^2)}, 
		{x1^3 - x1^2*x2 + x2^3 - exp1*exp2*y1 + exp1*exp2*y2 + 
		x2*(r1^2 - r2^2 + y1^2 - 2*y1*y2 + y2^2) + 
		x1*(-r1^2 + r2^2 - x2^2 + y1^2 - 2*y1*y2 + y2^2), 
		-(exp1*exp2*x2) + y1^3 + x1*(exp1*exp2 + x2*(-2*y1 - 2*y2)) + 
		(r1^2 - r2^2)*y2 - y1^2*y2 + y2^3 + x1^2*(y1 + y2) + 
		x2^2*(y1 + y2) + y1*(-r1^2 + r2^2 - y2^2)}}/denom;
	Map[ReImToZ,points]] /; NumberArgQ[x1, y1, r1, x2, y2, r2]

SolveCircles[Line[{{x1_,y1_}, {x2_,y2_}}], Circle[{cx_,cy_},r_]] :=
        Block[{eq, k, b, x, y, vars={x,y}, points},
	If[NGP[x1-x2] != 0,
		k = NGP[(y2-y1)/(x2-x1)]; 
		b = NGP[y1 - k*x1]; 
		y = k*x + b;
	        points = x /. NSolve[(x-cx)^2+(y-cy)^2==r^2,x];
		points = Map[ReImToZ[{#,k*#+b}]&, points],
		x = NGP[x1]; 
		points = y /. NSolve[(x-cx)^2+(y-cy)^2==r^2,y];
		points = Map[ReImToZ[{x, #}]&, points]];
	points] /; NumberArgQ[x1,y1,x2,y2,cx,cy,r]

SolveCircles[Circle[{x2_,y2_}, r2_],Line[a_List]] :=
        SolveCircles[Line[a],Circle[{x2,y2},r2]] /; NumberArgQ[x2,y2,r2]


(* Find the circle around a regular polygon *)

CircleAroundPolygon[vertices_List] :=
        Block[{eq, x, y, r, vars = {x,y,r}, solution,
                p1=ZToReIm[vertices[[1]]], p2=ZToReIm[vertices[[2]]], 
			p3=ZToReIm[vertices[[3]]]},
        eq = {(x-p1[[1]])^2+(y-p1[[2]])^2==r^2,
                (x-p2[[1]])^2+(y-p2[[2]])^2==r^2,
                (x-p3[[1]])^2+(y-p3[[2]])^2==r^2};
        solution = NSolve[eq,vars];
        First[ZCircle[ReImToZ[{x,y}], r] /. 
		Select[solution, ((r /. #) >= 0)&]]]

(* Find the hyperbolic center of a regular polygon *)

HypCenterOfPolygon[vert_List] :=
        Block[{circle, pc, v1, v2, c1, c2, points}, 
		circle = CircleAroundPolygon[vert];
		pc = First[Apply[List,circle]];
		{v1, v2} = Take[vert,2];
		c1 = If[ParallelArc[pc,v1], ZLine[{pc,v1}],
	                MakeGeodOrthoCircle[circle,v1]];
		c2 = If[ParallelArc[pc,v2], ZLine[{pc,v2}],
	                MakeGeodOrthoCircle[circle,v2]];
                points = SolveCircles[c1,c2];
		If[Length[points] == 0, center = points,
			center = If[EuclDistance[points[[1]]] <= 1,
				points[[1]],points[[2]]]]]

(* Find the middle point of a hyperbolic arc segment *)

HypMiddleOfArc[{z1_, z2_}] :=
	Block[{z0, x0, y0, r0, m, n, s, t, pararc = ParallelArc[z1,z2],
		mx, my, nx, ny, eq, k, b, x, y, cx, cy, res},
	If[pararc, crc = ZLine[{z1,z2}],
		z0 = GeodCircleCenter[z1,z2]; r0 = Abs[z0-z1];
		crc = ZCircle[z0,r0]];
	{m, n} = SolveCircles[UnitCircle, crc];
	{{mx, my},{nx, ny}} = Map[ZToReIm, {m, n}];
        s = EuclDistance[z1,n]/EuclDistance[z1,m];
	t = EuclDistance[z2,m]/EuclDistance[z2,n];
	If[pararc,
		If[Re[z1-z2]!=0, k = Im[z1-z2]/Re[z1-z2]; 
			b = Im[z1]-k*Re[z1]; y = k*x + b;
			solution = NSolve[s*((x - mx)^2 + (y - my)^2) 
				== t*((x - nx)^2 + (y - ny)^2), x];
			If[Length[solution] == 1, 
				res = {{x,y},{x,y}} /. solution,
				res = {x,y} /. solution],
			x = Re[z1]; 
			solution = NSolve[s*((x - mx)^2 + (y - my)^2)
                                == t*((x - nx)^2 + (y - ny)^2), y];
			If[Length[solution] == 1, 
				res = {{x,y},{x,y}} /. solution,
				res = {x,y} /. solution]],
		{x0, y0} = ZToReIm[z0]; 
		eq = (x - x0)^2 + (y - y0)^2 == r0^2;
		solution = NSolve[{s*((x - mx)^2 + (y - my)^2) 
			== t*((x - nx)^2 + (y - ny)^2), eq}, {x, y}];
		res = {x, y} /. solution];
        If[Abs[res[[1,1]]]^2 + Abs[res[[1,2]]]^2 <= 1.0, ReImToZ[res[[1]]],
                ReImToZ[res[[2]]]]] /; NumberArgQ[z1,z2]
	
(**********************************************************************)

(* Inversion routines *)
(* Reflect across an Euclidean line *)

HypReflect[z_,{p1_,p2_}] :=
	Block[{diff,norm,unit,normal},
	diff = p1 - p2; 
	unit = diff/Abs[diff];
	diff = z - p1;
	norm = diff - unit*(ZToReIm[diff] . ZToReIm[unit]);
	z - 2*norm] /; NumberArgQ[z,p1,p2]

(* Invert a point across a geodesic *)

HypInvert[z_,{z1_,z2_}] :=
	If[ParallelArc[z1,z2], HypReflect[z,{z1,z2}],
	Block[{center,cradius,diff,norm},
	center = GeodCircleCenter[z1,z2];
	cradius = Abs[center-z1];
	diff = z - center;
	norm = ZToReIm[diff];
	center+(cradius^2/(norm.norm))*diff]] /; NumberArgQ[z,z1,z2]

(* Invert a polygon across an given edge *)

HypInvertPolygon[vertices_List,index_Integer] :=
	Block[{vert = RotateLeft[vertices, index - 1], arc},
	arc = Take[vert,2];
	Join[arc, Map[HypInvert[#,arc]&, Drop[vert,2]]]]

(* Invert a polygon across some of its edges *)

HypInvertPolygon[vertices_List, indexes_List] :=
	Map[HypInvertPolygon[vertices,#]&, indexes]

(* Invert a polygon across all its edges *)

HypInvertPolygon[vertices_List] :=
	Map[HypInvertPolygon[vertices,#]&, Table[i,{i,1,Length[vertices]}]]

(* Make an inversion group for a polygon *)

PolygonInvGroup[vertices_List, level_Integer:1] :=
	If[level<=0, {vertices},
	Block[{index = Drop[Table[i,{i,1,Length[vertices]}],1],
		gen, all = Table[{}, {level+1}]},
	all[[1]] = {vertices};
	gen = HypInvertPolygon[vertices];
	all[[2]] = gen;
	For[i=2, i <= level, i++,
		gen = Flatten[Map[HypInvertPolygon[#,index]&, gen],1];
		all[[i+1]] = gen];
	Flatten[all,1]]]  /; Depth[vertices] <= 2

PolygonInvGroup[vertices_List, level_Integer:1] :=
	If[level<=0, {vertices},
	Block[{polys = FlattenObjectList[{vertices}]},
		Map[PolygonInvGroup[#,level]&, polys]]] /; 
					Depth[vertices] > 2


(**********************************************************************)


(* Graphics routines *)

MakePoint[z_/;NumberQ[z]] := 
	Point[{Re[z], Im[z]}]

(* Make a circle around a regular polygon *)

MakeCircleAroundPolygon[vertices_List] := 
	ZToReIm[CircleAroundPolygon[vertices]]

(* Make a geodesic arc orthogonal to a given circle or line, going
through a given point *)

MakeGeodOrthoCircle[Circle[{cx_,cy_},cr_],p_] := 
	Block[{px, py, eq, xo, yo, ro, res, pr,
		vars={xo,yo,ro}, solution},
	{px, py} = ZToReIm[p];
        eq = {ro^2+1==xo^2+yo^2,
                ro^2+cr^2==(xo-cx)^2+(yo-cy)^2,
                (px-xo)^2+(py-yo)^2==ro^2};
	solution = NSolve[eq,vars];
	If[Length[solution] >= 1, 
		res = Select[{xo,yo,ro} /. solution, 
			(Im[Last[#]] == 0 && Last[#] >= 0 &&
			Abs[(#[[1]]-px)^2 + (#[[2]]-py)^2 - #[[3]]^2] <= 
				ComparisonTolerance)& ];
		{xo, yo, ro} = First[res];
		Circle[{xo,yo},Abs[ro]],
		MakeHypOrthoLine[{{cx,cy},cr},{px,py}]]] /;
			NumberArgQ[cx,cy,cr,p]

MakeGeodOrthoCircle[Line[{{x1_,y1_},{x2_,y2_}}], p_] :=
	Block[{px, py, x, y, eqline, eq, r, k = Infinity, b, 
		solution},
	{px, py} = ZToReIm[p];
	eq = {r^2 == (px - x)^2 + (py - y)^2,
		1 + r^2 == x^2 + y^2};
	If[SameQ[x1,x2], x = x1; solution = NSolve[eq,{r,y}],
		k = (y2 - y1)/(x2 - x1);
		b = y1 - k*x1;
		y = k*x + b; solution = NSolve[eq,{r,x}]];
	If[Length[solution] >= 1,
        Circle[{x,y},Abs[r]] /. First[solution],
		If[SameQ[p,0], If[SameQ[k,Infinity], Line[{{0,0},{0,1}}], 
					Line[{{0,0},{1,-1/k}}]]
			Line[{{px,py},{0,0}}]]]] /;
                        NumberArgQ[x1,y1,x2,y2,p]

MakeHypOrthoLine[{{cx_,cy_},cr_}, {px_,py_}] :=
        Block[{eq, a, b, x, vars={a,b}, sol1, sol2, sol3},
                eq = {cy == a*cx + b, py == a*px + b};
	sol1 = First[NSolve[eq,vars]];
	sol2 = NSolve[(1+a)*x^2 + 2*a*b*x + b^2 == 1 /. sol1,{x}];
	sol3 = {x,a*x+b} /. sol2 /. sol1;
	Line[{First[sol3],Last[sol3]}]] /;
		NumberArgQ[cx,cy,cr,px,py]

(* Make a geodesic arc going from a given point z1 to a given point z2 *)

MakeGeodArc[{z1_,z2_}] := 
	If[ParallelArc[z1,z2], Line[{ZToReIm[z1],ZToReIm[z2]}],
	Block[{s1 = 1 + Abs[z1]^2, s2 = 1 + Abs[z2]^2,
                norm = 2*(Re[z2]*Im[z1] - Re[z1]*Im[z2]),
                ccenter = -I*(z1*s2 - z2*s1)/norm,
                cradius = EuclDistance[ccenter,z1],
		angle1, angle2, minang, maxang,x,y},
	angle1 = NGP[Arg[z1-ccenter]];
	angle2 = NGP[Arg[z2-ccenter]];
	minang = Min[angle1, angle2]; 
	maxang = Max[angle1, angle2]; 
	{x,y} = ZToReIm[ccenter];
	If[minang >= 0.0 && maxang >= 0.0,
		Circle[{x,y}, cradius, {minang, maxang}],
	If[minang < 0.0 && maxang < 0.0,
		Circle[{x,y}, cradius, N2Pi+{minang, maxang}],
	If[maxang - minang < NPi,
		{Circle[{x,y}, cradius, {minang+N2Pi,N2Pi}],
			Circle[{x,y},cradius,{0,maxang}]},
		Circle[{x,y},cradius,{maxang,minang+N2Pi}]]]]
	]] /; NumberArgQ[z1,z2]

(* Make a hyperbolic polygon when the vertices are given *)
	
MakeHypPolygon[vertices_List] :=
	Block[{i, len = Length[vertices]},
	Table[MakeGeodArc[vertices[[ 1+Mod[{i-1,i}, len] ]]],
		{i, 1, len}]]

(* Make the arcs from polygon edges to the centre *)

MakeHypPolygonDiagonals[vertices_List] :=
        Block[{ccenter, i, len = Length[vertices]},
	If[Mod[len,2]==0, Table[MakeGeodArc[vertices[[{i,i+len/2}]]], 
			{i, 1, len/2}],
		ccenter = HypCenterOfPolygon[vertices];
		Table[MakeGeodArc[{vertices[[i]], ccenter}], {i, 1, len}]]]

(* Make the arcs from middle points of polygon edges to the centre *)

MakeHypPolygonBisectors[vert_List] :=
        Block[{pts, centers, i, len = Length[vert]},
	pts = Transpose[{vert,RotateRight[vert]}];
	centers = Map[HypMiddleOfArc[#]&,pts];
	If[Mod[len,2]==0, 
		Table[MakeGeodArc[centers[[{i,i+len/2}]]], 
			{i, 1, len/2}],
		Table[MakeGeodArc[{centers[[i]],
			vert[[ 1+Mod[i-1+Quotient[len,2],len] ]]}],
			{i,1,len}]]]

(* Make a bisector arc to a geodesic arc {z1,z2} *)

MakeBisector[{z1_,z2_}] :=
	Block[{c1, m = HypMiddleOfArc[{z1,z2}], crc, cr, pts},
	If[ParallelArc[z1,z2], crc = ZLine[{z1,z2}],
		c1 = GeodCircleCenter[z1,z2];
		crc = Circle[ZToReIm[c1],Abs[c1-z1]]];
	cr = MakeGeodOrthoCircle[crc,m];
	pts = SolveCircles[cr,UnitCircle]] /; NumberArgQ[z1,z2]

(* Select the drawing color *)

MakeColorLevel[level_Integer,maxlevel_Integer, opts___Rule] := 
	Block[{colored, graylevels},
	colored = Colored /. {opts} /. Options[MakeColorLevel];
	graylevels = GrayLevels /. {opts} /. Options[MakeColorLevel];
	If[colored, Hue[NGP[0.9*(maxlevel-level)/maxlevel],0.6,0.65],
	  If[graylevels, GrayLevel[0.9*(maxlevel-level)/maxlevel],
	    GrayLevel[0] ] ] ]

(**********************************************************************)

(* Drawing routines *)

(* Draw a collection a complex points *)

DrawPoints[pts_List,opts___Rule] :=
        Block[{colored, graylevels, prolog, label, trailer, 
		points = Flatten[pts], max}, 
	colored = Colored /. {opts} /. Options[DrawPoints];
	graylevels = GrayLevels /. {opts} /. Options[DrawPoints];
	prolog = Prolog /. {opts} /. Options[DrawPoints];
	label = Label /. {opts} /. Options[DrawPoints];
	prolog = Join[prolog, {Text[FontForm[label, {"Times-Italic",12}], 
		{0,-1.1}]}];
	trailer = Trailer /. {opts} /. Options[DrawPoints];
	max = Length[points];
	If[colored || graylevels,
		points = Table[{MakeColorLevel[i, max, opts], 
			MakePoint[ points[[i]] ]}, {i, 1, max}],
		points = Map[MakePoint, points]];		
	Show[Graphics[Join[prolog, points]], trailer, 
		FilterOptions[Graphics, opts]]]

(* Draw a collection of geodesic arcs, DrawArcs[{{z1,z2},{z3,z4}}] *)

DrawArcs[arcs_List, opts___Rule] :=
	Block[{colored, graylevels, prolog, label, trailer, ar, max},
	colored = Colored /. {opts} /. Options[DrawArcs];
	graylevels = GrayLevels /. {opts} /. Options[DrawArcs];
        prolog = Prolog /. {opts} /. Options[DrawArcs];
        label = Label /. {opts} /. Options[DrawArcs];
        prolog = Join[prolog, {Text[FontForm[label, {"Times-Italic",12}],
                {0,-1.1}]}];
        trailer = Trailer /. {opts} /. Options[DrawArcs];
        max = Length[arcs];
	If[colored || graylevels,
	        ar = Table[{MakeColorLevel[i, max, opts],
        	        MakeGeodArc[ arcs[[i]] ]}, {i, 1, max}],
		ar = Map[MakeGeodArc, arcs]];
        Show[Graphics[Join[prolog, ar]], trailer,
		FilterOptions[Graphics, opts]]]
	
(* Draw a list of circles *)

DrawCircles[circles_List, opts___Rule] :=
        Block[{colored, graylevels, prolog, label, trailer, 
		circ = FlattenObjectList[{circles}], max},
	colored = Colored /. {opts} /. Options[DrawCircles];
	graylevels = GrayLevels /. {opts} /. Options[DrawCircles];
	prolog = Prolog /. {opts} /. Options[DrawCircles];
	label = Label /. {opts} /. Options[DrawCircles];
	prolog = Join[prolog, {Text[FontForm[label, {"Times-Italic",12}], 
		{0,-1.1}]}];
	trailer = Trailer /. {opts} /. Options[DrawCircles];
	max = Length[circ]; 
	If[colored || graylevels,
		circ = Table[{MakeColorLevel[i, max, opts], 
			ZToReIm[ circ[[i]] ]}, {i, 1, max}],
		circ = Map[ZToReIm, circ]];
	Show[Graphics[Join[prolog, circ]], trailer,
		FilterOptions[Graphics, opts]]]

(* Draw a list of hyperbolic polygons *)

DrawHypPolygons[polygons_List,opts___Rule] :=
        Block[{colored, graylevels, diagonals, bisectpts, prolog, label, 
		trailer, i, polys = FlattenObjectList[{polygons}], max, 
		lines, diags, bisect}, 
	colored = Colored /. {opts} /. Options[DrawHypPolygons];
	graylevels = GrayLevels /. {opts} /. Options[DrawHypPolygons];
	diagonals = Diagonals /. {opts} /. Options[DrawHypPolygons];
	bisectpts = Bisectors /. {opts} /. Options[DrawHypPolygons];
	prolog = Prolog /. {opts} /. Options[DrawHypPolygons];
	label = Label /. {opts} /. Options[DrawHypPolygons];
	prolog = Join[prolog, {Text[FontForm[label, {"Times-Italic",12}], 
		{0,-1.1}]}];
	trailer = Trailer /. {opts} /. Options[DrawHypPolygons];
	max  = Length[polys];
	If[colored || graylevels,
	        lines = Table[{MakeColorLevel[i, max, opts], 
			MakeHypPolygon[polys[[i]]]}, {i, 1, max}],
		lines = Map[MakeHypPolygon, polys]];
	diags = If[diagonals,
		If[colored || graylevels,
			Table[{MakeColorLevel[i,max,opts], 
			MakeHypPolygonDiagonals[polys[[i]]]}, {i,1,max}],
			Map[MakeHypPolygonDiagonals, polys]], {}];
	bisect = If[bisectpts,
		If[colored || graylevels,
			Table[{MakeColorLevel[i,max,opts], 
			MakeHypPolygonBisectors[polys[[i]]]}, {i,1,max}],
			Map[MakeHypPolygonBisectors, polys]], {}];
	lines = Join[lines, diags, bisect];
        Show[Graphics[Join[prolog, lines]], trailer,
		FilterOptions[Graphics, opts]]]

(* Draw a list of circles around a given list of regular polygons *)

DrawCirclesAroundPolygons[polygons_List, opts___Rule] := 
	Block[{colored, graylevels, diagonals, prolog, trailer, 
		i, max, polys = FlattenObjectList[{polygons}], circles},
	colored = Colored /. {opts} /. Options[DrawCirclesAroundPolygons];
	graylevels = GrayLevels /. {opts} /. Options[DrawCirclesAroundPolygons];
	diagonals = Diagonals /. {opts} /. Options[DrawCirclesAroundPolygons];
	prolog = Prolog /. {opts} /. Options[DrawCirclesAroundPolygons];
	label = Label /. {opts} /. Options[DrawCirclesAroundPolygons];
	prolog = Join[prolog, {Text[FontForm[label, {"Times-Italic",12}], 
		{0,-1.1}]}];
	trailer = Trailer /. {opts} /. Options[DrawCirclesAroundPolygons];
	max  = Length[polys];
	If[colored || graylevels,
		circles = Table[{MakeColorLevel[i,max, Colored -> colored], 
			MakeCircleAroundPolygon[polys[[i]]]}, {i, 1, max}],
		circles = Map[MakeCircleAroundPolygon, polys]];
	Show[Graphics[Join[prolog, circles]], trailer,
		FilterOptions[Graphics, opts]]]


(**********************************************************************)

End[]  (* Private context *)

(**********************************************************************)

EndPackage[]
Null

(**********************************************************************)

