BeginPackage["Moebius`", "Global`"]

(* 
Version 1.1. Last modified: May 18th, 1993.

The newest version of this package is available for anonymous ftp at
nic.funet.fi, directory "pub/sci/math/riemann/mathematica".

Send comments and bug reports to 
  Juha Haataja, Centre for Scientific Computing (jhaataja@csc.fi).

*)

(**********************************************************************)

MGenFromPoints::usage = "Given two lists of three corresponding points,
calculate the appropriate Moebius transform which transforms the first list
to the second: MGenFromPoints[{z1, z2, z3}, {w1, w2, w3}]"

MGenFromArcs::usage = "Given two hyperbolic arcs of equal length,
calculate the appropriate Moebius transform which transforms the first
to the second: MGenFromArcs[{z1, z2}, {z3, z4}]"

MGenFromPolygon::usage = "Given a polygon and a list of corresponding
edges, calculate the set of generators transforming the edges to each
other: MGenFromPolygon[{z1, z2, z3, z4}, {{1,2}, {2,3}}]"

MGenMult::usage = "Multiply together all generators from a list of
generators (MGenMult[{g1, g2, g3}]) or multiply a list of generators
by a generator (MGenMult[g, {g1, g2, g3}])"

MGenCombine::usage = "Form all combinations Dot[gen1,gen2] using
generators from two lists of generators: MGenCombine[{g1, g2}, {g3,
g4}]"

MGenNestMult::usage = "Given a list of generators {g1,g2,g3,...},
return {g1, g1.g2, g1.g2.g3, ...}. For example: MGenNestMult[{g1, g2,
g3}]"

MGenInvert::usage = "Find the inverses of a set of generators:
MGenInvert[{g1, g2, g3}]"

ApplyMGen::usage = "Move complex points with a given generator:
ApplyMGen[gen, {z1, z2, z3}]"

ApplyMGenList::usage = "Move complex points using generators from a list:
ApplyMGenList[{g1, g2, g3}, {z1, z2, z3}]"

(**********************************************************************)

Begin["`Private`"]

(**********************************************************************)

(* Given two lists of three corresponding points, calculate the
   appropriate Moebius transform which transforms the first list to the
   second. *)

MGenFromPoints[{z1_,ComplexInfinity,z3_},{w1_,w2_,w3_}] :=
	MGenFromPoints[{z1,z3,ComplexInfinity},{w1,w3,w2}]

MGenFromPoints[{ComplexInfinity,z2_,z3_},{w1_,w2_,w3_}] :=
	MGenFromPoints[{z2,z3,ComplexInfinity},{w2,w3,w1}]

MGenFromPoints[{z1_,z2_,z3_},{w1_,ComplexInfinity,w3_}] :=
	MGenFromPoints[{z1,z3,z2},{w1,w3,ComplexInfinity}]

MGenFromPoints[{z1_,z2_,z3_},{ComplexInfinity,w2_,w3_}] :=
	MGenFromPoints[{z2,z3,z1},{w2,w3,ComplexInfinity}]

MGenFromPoints[{z1_,z2_,ComplexInfinity},{w1_,w2_,ComplexInfinity}] :=
	Block[{a, b, c, d},
	a = w1 - w2; b = z1*w2 - z2*w1;
	c = 0; d = z1 - z2;
	{{a,b},{c,d}}]

MGenFromPoints[{z1_,z2_,z3_},{w1_,w2_,ComplexInfinity}] :=
	Block[{s, a, b, c, d},
	s = 1/((z3 - z1)/(z3 - z2));
	a = w1 - s*w2; b = s*z1*w2 - z2*w1;
	c = 1 - s; d = s*z1 - z2;
	{{a,b},{c,d}}]

MGenFromPoints[{z1_,z2_,ComplexInfinity},{w1_,w2_,w3_}] :=
	Block[{s, a, b, c, d},
	s = ((w3 - w1)/(w3 - w2));
	a = w1 - s*w2; b = s*z1*w2 - z2*w1;
	c = 1 - s; d = s*z1 - z2;
	{{a,b},{c,d}}]

MGenFromPoints[{z1_,z2_,z3_},{w1_,w2_,w3_}] :=
	Block[{s, a, b, c, d},
	s = ((w3 - w1)/(w3 - w2))/((z3 - z1)/(z3 - z2));
	a = w1 - s*w2; b = s*z1*w2 - z2*w1;
	c = 1 - s; d = s*z1 - z2;
	{{a,b},{c,d}}]

(**********************************************************************)

(* Given two hyperbolic arcs of equal length, calculate the appropriate
Moebius transform which transforms the first to the second *)

MGenFromArcs[{{z1_,z2_},{z3_,z4_}}] :=
	Block[{m1,m2},
        m1 = HypMiddleOfArc[{z1,z2}];
        m2 = HypMiddleOfArc[{z3,z4}];
        MGenFromPoints[{z1, m1, z2}, {z3, m2, z4}]]

(* Calculate the Moebius transformation which moves a polygon across given
pairs of edges *)

MGenFromPolygon[vertices_List, indexes_List] :=
        Block[{vert, max = Length[vertices]},
        vert = Map[Take[#,2]&,NestList[RotateLeft,vertices,max-1]];
        vert = Map[{vert[[First[#]]],Reverse[vert[[Last[#]]]]}&,indexes];
	Map[MGenFromArcs[#]&, vert]]

MGenFromPolygon[vertices_List] :=
        Block[{vert, max = Length[vertices], indexes},
	indexes = Table[1+Mod[{i,i+Quotient[max,2]},max], {i, 1, max}];
        vert = Map[Take[#,2]&,NestList[RotateLeft,vertices,max-1]];
        vert = Map[{vert[[First[#]]],Reverse[vert[[Last[#]]]]}&,indexes];
	Map[MGenFromArcs[#]&, vert]]


(**********************************************************************)

(* Combine generators *)

(* Multiply all generators from a list of generators *)

MGenMult[generators_List] :=
	Apply[Dot,generators]

(* Multiply a list of generators by a generator *)

MGenMult[gen_,generators_List] :=
	Map[Dot[gen,#]&,generators]

(* Form all combinations Dot[gen1,gen2] using generators from two lists
of generators *)

MGenCombine[generators1_,generators2_] :=
	Flatten[Map[MGenMult[#,generators2]&,generators1],1]

(* Given a list of generators {g1,g2,g3,...}, return 
	{g1, g1.g2, g1.g2.g3, ...} *)

MGenNestMult[generators_List] :=
	Map[MGenMult, Table[Take[generators,i],{i,1,Length[generators]}]]

(* Find the inverse of a Moebius transformation *)

MGenInvert[{{a_,b_},{c_,d_}}] := {{-d,b},{c,-a}}

MGenInvert[generators_List] :=
	Map[MGenInvert,generators] 

(**********************************************************************)

(* Move complex points with a given generator *)

ApplyMGen[gen_,point_] :=
	(Divide @@ (Dot[gen, {point,1}]))  /; AtomQ[point]

ApplyMGen[{{a_,b_},{c_,d_}},point_] := a/c  /; SameQ[point,ComplexInfinity]

ApplyMGen[gen_,points_List] :=
	Map[ApplyMGen[gen,#]&, points]

(* Move points using generators from a list *)

ApplyMGenList[generators_List,points_List] :=
	Map[ApplyMGen[#,points]&,generators]



(**********************************************************************)
(*                       Private routines                             *)
(**********************************************************************)

(* Some definitions *)

GeomPrecision = 16;  (* The precision of floating-point calculations *)
ComparisonTolerance = N[10^(-10), GeomPrecision];

NPi = N[Pi, GeomPrecision]
N2Pi = N[2*Pi, GeomPrecision]

(* To check out numerical arguments *)

NumberArgQ[vars___] := And @@ Map[NumberQ, {vars}]

(**********************************************************************)
(* Routines for handling complex numbers *)

ZToReIm[z_] := {Re[z],Im[z]}
ReImToZ[{x_,y_}] := x + I*y

(* Distances of two points *)

EuclDistance[{x1_,y1_}, {x2_,y2_}] := Sqrt[(x1-x2)^2 + (y1-y2)^2]

EuclDistance[p1_,p2_:0] :=
        N[Sqrt[Plus @@ ({Re[p1-p2],Im[p1-p2]}^2)], GeomPrecision]

(**********************************************************************)
(* Tool routines *)

(* Is a geodesic through z1 and z2 an Euclidean line? *)

ApproxEqual[z1_,z2_] :=
        If[N[Abs[z1 - z2]] <= ComparisonTolerance, True, False]

ParallelArc[{z1_,z2_}] :=
        If[ApproxEqual[z1,0] || ApproxEqual[z2,0], True,
        If[ApproxEqual[Abs[Arg[z1]-Arg[z2]], 0] ||
                ApproxEqual[Abs[Arg[z1]-Arg[z2]], NPi] ||
                ApproxEqual[Abs[Arg[z1]-Arg[z2]], N2Pi],
                True, False]] /; NumberArgQ[z1,z2]

(* Construct a geodesic circle going through given points *)

GeodCirclePoints[{z1_,z2_}] := 
	If[ParallelArc[{z1,z2}], {Infinity, Infinity},
	Block[{s1 = 1 + Abs[z1]^2, s2 = 1 + Abs[z2]^2,
		norm = 2*(Re[z2]*Im[z1] - Re[z1]*Im[z2]),
		circlecenter = -I*(z1*s2 - z2*s1)/norm,
		circleradius = EuclDistance[circlecenter,z1]},
	{circlecenter, circleradius}]]

(* Find the middle point of a hyperbolic arc through origo *)

HypMiddleOfArcThroughOrigo[{z1_,z2_}] :=
	Block[{k , h, p, d1, d2},
	If[Re[z1 - z2] != 0, 
		k = Im[z1 - z2]/Re[z1 - z2]; 
		p = (1 + k*I)*Sqrt[1/(1 + k^2)],
		p = I];
	d1 = EuclDistance[z1, p]; 
	d2 = EuclDistance[z2, p];
	h = Sqrt[d1*d2/((2 - d1)*(2 - d2))];
	p - 2*h/(1 + h)] 
	
(* Find the middle point of a hyperbolic arc segment *)

HypMiddleOfArc[{z1_, z2_}] :=
  If[ParallelArc[{z1, z2}], HypMiddleOfArcThroughOrigo[{z1, z2}],
   Block[{center, r, cx, cy, m1, m2, n1, n2, s, t, res, x, y},
        {center, r} = GeodCirclePoints[{z1, z2}];
        {cx, cy} = ZToReIm[center];
	cx = If[ApproxEqual[cx,0], 0, cx];
	cy = If[ApproxEqual[cy,0], 0, cy];
        {{m1, n1}, {m2, n2}} = {x, y} /.
                NSolve[{(x - cx)^2 + (y - cy)^2 == r^2, x^2 + y^2 == 1},{x,y}];
        {x1,y1} = ZToReIm[z1]; {x2,y2} = ZToReIm[z2];
        s = EuclDistance[{x1, y1}, {m2, n2}]/EuclDistance[{x1, y1}, {m1, n1}];
        t = EuclDistance[{x2, y2}, {m1, n1}]/EuclDistance[{x2, y2}, {m2, n2}];
        res = {x, y} /.
        NSolve[{s*((x - m1)^2 + (y - n1)^2) ==
                t*((x - m2)^2 + (y - n2)^2),
                (x - cx)^2 + (y - cy)^2 == r^2}, {x, y}];
        If[res[[1,1]]^2 + res[[1,2]]^2 <= 1, ReImToZ[res[[1]]],
                ReImToZ[res[[2]]]]]]

(**********************************************************************)

End[]

(**********************************************************************)

EndPackage[]
Null

(**********************************************************************)