/* matrix.c */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1993  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "mem.h"
#include "bltin.h"
#include "matrix.h"
#include "util.h"

#include <stdio.h>
#include <math.h>

/*
 * Define our own abs(). We need to do this, cause we need
 * and abs() that does double, as well as int.
 */
#define rabs(x) ((x) >= 0 ? (x) : -(x))

/* **************************************************************
 * Create a matrix.
 * ************************************************************** */
Matrix *
matrix_Create (nrow, ncol)
     int nrow, ncol;
{
  Matrix *new;

  if (nrow < 0 || ncol < 0)
    error_1 ("cannot specify a negative matrix dimension", (char *) 0);

  new = (Matrix *) MALLOC (sizeof (Matrix));

  new->type = MATRIX;
  new->name = 0;
  new->nrow = nrow;
  new->ncol = ncol;
  new->dtype = REAL;

  if (nrow * ncol != 0)
  {
    new->val.mr = (double *) MALLOC (sizeof (double) * (nrow * ncol));
  }
  else
  {
    new->nrow = 0;
    new->ncol = 0;
    new->val.mr = (double *) MALLOC (sizeof (double) * 1);
  }

  return (new);
}

/*
 * Create a COMPLEX matrix.
 */
Matrix *
matrix_CreateC (nrow, ncol)
     int nrow, ncol;
{
  Matrix *new;

  if (nrow < 0 || ncol < 0)
    error_1 ("cannot specify a negative matrix dimension", (char *) 0);

  new = (Matrix *) MALLOC (sizeof (Matrix));

  new->type = MATRIX;
  new->name = 0;
  new->nrow = nrow;
  new->ncol = ncol;
  new->dtype = COMPLEX;

  if (nrow * ncol != 0)
  {
    new->val.mc = (Complex *) MALLOC (sizeof (Complex) * (nrow * ncol));
  }
  else
  {
    new->nrow = 0;
    new->ncol = 0;
    new->val.mc = (Complex *) MALLOC (sizeof (Complex) * 1);
  }

  return (new);
}

/*
 * Create a matrix of Strings
 */
Matrix *
matrix_CreateS (nrow, ncol)
     int nrow, ncol;
{
  int i;
  Matrix *new;

  if (nrow < 0 || ncol < 0)
    error_1 ("cannot specify a negative matrix dimension", (char *) 0);

  new = (Matrix *) MALLOC (sizeof (Matrix));

  new->type = MATRIX;
  new->name = 0;
  new->nrow = nrow;
  new->ncol = ncol;
  new->dtype = STRING;

  if (nrow * ncol != 0)
  {
    new->val.ms = (char **) MALLOC (sizeof (char *) * (nrow * ncol));
  }
  else
  {
    new->nrow = 0;
    new->ncol = 0;
    new->val.ms = (char **) MALLOC (sizeof (char *) * 1);
  }

  /* temporary */
  for (i = 0; i < nrow * ncol; i++)
    new->val.ms[i] = 0;

  return (new);
}

/*
 * Create a NULL matrix.
 */
Matrix *
matrix_CreateNull ()
{
  Matrix *new;

  new = (Matrix *) MALLOC (sizeof (Matrix));

  new->type = MATRIX;
  new->name = 0;
  new->nrow = 0;
  new->ncol = 0;;
  new->dtype = 0;

  return (new);
}

/* **************************************************************
 * Free a matrix, and wipe out the structure members.
 * ************************************************************** */
void
matrix_Destroy (m)
     Matrix *m;
{
  ASSERT (m);
  {
    int i;

    if (m->dtype == REAL)
      FREE (m->val.mr);
    else if (m->dtype == COMPLEX)
      FREE (m->val.mc);
    else if (m->dtype == STRING)
    {
      for (i = 0; i < m->nrow * m->ncol; i++)
	FREE (m->val.ms[i]);
      FREE (m->val.ms);
    }

    m->type = -1;
    FREE (m->name);
    m->nrow = -1;
    m->ncol = -1;

    m->dtype = 0;
    FREE (m);
  }
}

/* **************************************************************
 * Copy a matrix. Create the new matrix, and return the new,
 * copied matrix as the result.
 * ************************************************************** */
Matrix *
matrix_Copy (m)
     Matrix *m;
{
  ASSERT (m);
  {
    int i;
    Matrix *new = 0;

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.mr[i] = m->val.mr[i];
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	new->val.mc[i].r = m->val.mc[i].r;
	new->val.mc[i].i = m->val.mc[i].i;
      }
    }
    else if (MTYPE (m) == STRING)
    {
      new = matrix_CreateS (MNR (m), MNC (m));
      for (i = 0; i < MNR (m) * MNC (m); i++)
	new->val.ms[i] = cpstr (m->val.ms[i]);
    }
    return (new);
  }
}

/* **************************************************************
 * Reshape a matrix (change nrow, ncol).
 * ************************************************************** */
Matrix *
matrix_Reshape (m, nrow, ncol)
     Matrix *m;
     int nrow, ncol;
{
  ASSERT (m);
  {
    Matrix *new;
    if (nrow * ncol != MNR (m) * MNC (m))
      error_1 (matrix_GetName (m), "incompatible dimensions");

    new = matrix_Copy (m);
    new->nrow = nrow;
    new->ncol = ncol;
    return (new);
  }
}

/* **************************************************************
 * Print out a matrix by collums
 * ************************************************************** */

static void matrix_print_real _PROTO ((Matrix * m, FILE * fn));
static void matrix_print_complex _PROTO ((Matrix * m, FILE * fn));
static void matrix_print_string _PROTO ((Matrix * m, FILE * fn));

void
matrix_Print (m, stream)
     Matrix *m;
     FILE *stream;
{
  ASSERT ((m));
  {
    if (m->name != 0 && strncmp (m->name, "-", 1))
      fprintf (stream, " %s =\n", m->name);

    if (MTYPE (m) == REAL)
      matrix_print_real (m, stream);
    else if (MTYPE (m) == COMPLEX)
      matrix_print_complex (m, stream);
    else if (MTYPE (m) == STRING)
      matrix_print_string (m, stream);
  }
}

/*
 * Print a matrix of REAL values
 */

static void
matrix_print_real (matrix, stream)
     Matrix *matrix;
     FILE *stream;
{
  char tmp[100];
  int i, j, k, nrow, ncol, npri, rem, start, width;
  int n_print, fwidth, fprec;

  fwidth = get_fwidth ();
  fprec = get_fprec ();

  width = max (fwidth, fprec + 3) + 2;
  if (width > 80)
  {
    error_1 ("format too large for REAL print", 0);
  }

  n_print = 80 / (width + 2);
  nrow = MNR (matrix);
  ncol = MNC (matrix);
  npri = MNC (matrix) / n_print;
  rem = MNC (matrix) % n_print;

  /* Special case, empty matrix */
  if (nrow == 0 && ncol == 0)
  {
    fprintf (stream, "\t[]\n");
    fflush (stream);
  }

  start = 1;
  for (i = 0; i < npri; i++)
  {
    if (npri >= 1)
      fprintf (stream, " matrix columns %d thru %d\n",
	       n_print * i + 1, (n_print) + (n_print * i));

    for (k = 1; k <= nrow; k++)	/* print all rows */
    {
      for (j = start; j <= n_print + start - 1; j++)
      {
	sprintf (tmp, "%*.*g  ", fwidth, fprec, MAT (matrix, k, j));
	fprintf (stream, "%*s", width, tmp);
      }
      fprintf (stream, "\n");
      fflush (stream);
    }
    start += n_print;		/* inc our col position */
    fprintf (stream, "\n");
    fflush (stream);
  }

  /* Now come back and write out the last few columns */
  if (!rem)
    return;
  if (npri >= 1)
    fprintf (stream, " matrix columns %d thru %d\n",
	     MNC (matrix) - rem + 1, MNC (matrix));

  for (k = 1; k <= nrow; k++)
  {
    for (i = ncol - rem + 1; i <= ncol; i++)
    {
      sprintf (tmp, "%*.*g  ", fwidth, fprec, MAT (matrix, k, i));
      fprintf (stream, "%*s", width, tmp);
      fflush (stream);
    }
    fprintf (stream, "\n");
    fflush (stream);
  }
}

/*
 * Print a matrix of COMPLEX values.
 */

static void
matrix_print_complex (matrix, stream)
     Matrix *matrix;
     FILE *stream;
{
  char tmp[100];
  int i, j, k, nrow, ncol, npri, rem, start, width;
  int n_print, fwidth, fprec;

  fwidth = get_fwidth ();
  fprec = get_fprec ();

  /* Check format */
  width = 2 * (max (fwidth, fprec + 3)) + 6;
  if (width > 80)
  {
    error_1 ("format too large for COMPLEX print", 0);
  }

  n_print = 80 / (width + 2);
  nrow = MNR (matrix);
  ncol = MNC (matrix);
  npri = MNC (matrix) / n_print;
  rem = MNC (matrix) % n_print;

  /* Special case, empty matrix */
  if (nrow == 0 && ncol == 0)
  {
    fprintf (stream, "\t[]\n");
    fflush (stream);
  }

  start = 1;
  for (i = 0; i < npri; i++)
  {
    if (npri >= 1)
      fprintf (stream, " matrix columns %d thru %d\n",
	       n_print * i + 1, (n_print) + (n_print * i));

    for (k = 1; k <= nrow; k++)	/* print all rows */
    {
      for (j = start; j <= n_print + start - 1; j++)
      {
	if (MATi (matrix, k, j) >= 0.0)
	{
	  sprintf (tmp, "%*.*g + %.*gi", fwidth, fprec, MATr (matrix, k, j),
		   fprec, MATi (matrix, k, j));
	}
	else
	{
	  sprintf (tmp, "%*.*g - %.*gi", fwidth, fprec, MATr (matrix, k, j),
		   fprec, -MATi (matrix, k, j));
	}
	fprintf (stream, "%*s", width, tmp);
      }
      fprintf (stream, "\n");
    }
    start += n_print;		/* inc our col position */
    fprintf (stream, "\n");
    fflush (stream);
  }

  /* Now come back and write out the last few colums */
  if (!rem)
    return;
  if (npri >= 1)
    fprintf (stream, " matrix columns %d thru %d\n",
	     MNC (matrix) - rem + 1, MNC (matrix));

  for (k = 1; k <= nrow; k++)
  {
    for (i = ncol - rem + 1; i <= ncol; i++)
    {
      if (MATi (matrix, k, i) >= 0.0)
      {
	sprintf (tmp, "%*.*g + %.*gi", fwidth, fprec, MATr (matrix, k, i),
		 fprec, MATi (matrix, k, i));
      }
      else
      {
	sprintf (tmp, "%*.*g - %.*gi", fwidth, fprec, MATr (matrix, k, i),
		 fprec, -MATi (matrix, k, i));
      }
      fprintf (stream, "%*s", width, tmp);
    }
    fprintf (stream, "\n");
    fflush (stream);
  }
}

/*
 * Print a matrix of STRINGS.
 */

static void
matrix_print_string (m, stream)
     Matrix *m;
     FILE *stream;
{
  int i, j, k, length, nrow, ncol, ncol_print, npri, rem;
  int start, tmp;

  nrow = MNR (m);
  ncol = MNC (m);

  /* Special case, empty matrix */
  if (nrow == 0 && ncol == 0)
  {
    fprintf (stream, "\t[]\n");
    fflush (stream);
  }

  /* Figure out the length of the longest STRING */
  length = 0;
  for (i = 0; i < nrow * ncol; i++)
  {
    if (MATsv (m, i) != 0)	/* just in case */
    {
      tmp = strlen (MATsv (m, i));
      if (tmp > length)
	length = tmp;
    }
  }

  /* Now figure out how many columns we can print on a line */
  ncol_print = TERM_WIDTH / (length + 2);

  npri = MNC (m) / ncol_print;
  rem = MNC (m) % ncol_print;

  start = 1;
  for (i = 0; i < npri; i++)
  {
    for (k = 1; k <= nrow; k++)	/* print all rows */
    {
      for (j = start; j <= ncol_print + start - 1; j++)
      {
	fprintf (stream, "%-*s", length + 2, MATs (m, k, j));
      }
      fprintf (stream, "\n");
    }
    start += ncol_print;	/* inc our col position */
    fprintf (stream, "\n");
    fflush (stream);
  }

  /* Now come back and write out the last few collums */
  if (!rem)
    return;
  for (k = 1; k <= nrow; k++)
  {
    for (i = ncol - rem + 1; i <= ncol; i++)
    {
      fprintf (stream, "%-*s", length + 2, MATs (m, k, i));
    }
    fprintf (stream, "\n");
    fflush (stream);
  }
}

/* **************************************************************
 * Write a matrix to file fn. This function is different than
 * matrix_Print because we want a leadin `#' on the annotation
 * lines so GNUPLOT will see them as comments.
 * ************************************************************** */

#define N_WRITE   5		/* # of columns to write pre line */
static void matrix_write_real _PROTO ((Matrix * m, FILE * fn));
static void matrix_write_complex _PROTO ((Matrix * m, FILE * fn));
static void matrix_write_string _PROTO ((Matrix * m, FILE * fn));

void
matrix_Write (m, fn)
     Matrix *m;
     FILE *fn;
{
  ASSERT ((m));
  {
    if (MTYPE (m) == REAL)
      matrix_write_real (m, fn);
    else if (MTYPE (m) == COMPLEX)
      matrix_write_complex (m, fn);
    else if (MTYPE (m) == STRING)
      matrix_write_string (m, fn);
  }
}

/*
 * Write out a REAL matrix
 */
static void
matrix_write_real (m, fn)
     Matrix *m;
     FILE *fn;
{
  int i, j, k, nrow, ncol, npri, rem, start;

  nrow = MNR (m);
  ncol = MNC (m);
  npri = MNC (m) / N_WRITE;
  rem = MNC (m) % N_WRITE;

  if (matrix_GetName (m) == 0)
    fprintf (fn, "# matrix : %s no_of_rows: %d no_of_cols: %d",
	     "MATRIX", MNR (m), MNC (m));
  else
    fprintf (fn, "# matrix : %s no_of_rows: %d no_of_cols: %d",
	     matrix_GetName (m), MNR (m), MNC (m));

  fprintf (fn, " REAL\n");

  start = 1;
  for (i = 0; i < npri; i++)
  {
    fprintf (fn, "# matrix columns %d thru %d\n",
	     N_WRITE * i + 1, (N_WRITE - 1) + (N_WRITE * i) + 1);

    for (k = 1; k <= nrow; k++)	/* print all rows */
    {
      for (j = start; j <= N_WRITE + start - 1; j++)
      {
	fprintf (fn, " %-24.16g", MAT (m, k, j));
      }
      fprintf (fn, "\n");
    }
    start += N_WRITE;		/* inc our col position */
  }

  /* Now come back and write out the last few columns */
  if (!rem)
    return;
  fprintf (fn, "# matrix columns %d thru %d\n",
	   MNC (m) - rem + 1, MNC (m));

  for (k = 1; k <= nrow; k++)
  {
    for (i = ncol - rem + 1; i <= ncol; i++)
    {
      fprintf (fn, " %-24.16g", MAT (m, k, i));
    }
    fprintf (fn, "\n");
  }
  fflush (fn);
  return;
}

/*
 * Write out a COMPLEX matrix
 */
static void
matrix_write_complex (m, fn)
     Matrix *m;
     FILE *fn;
{
  int i, j, k, nrow, ncol, npri, rem, start;

  nrow = MNR (m);
  ncol = MNC (m);
  npri = MNC (m) / N_WRITE;
  rem = MNC (m) % N_WRITE;

  if (matrix_GetName (m) == 0)
    matrix_SetName (m, cpstr ("M"));

  fprintf (fn, "# matrix : %s no_of_rows: %d no_of_cols: %d COMPLEX\n",
	   matrix_GetName (m), MNR (m), MNC (m));

  start = 1;
  for (i = 0; i < npri; i++)
  {
    fprintf (fn, "# matrix columns %d thru %d\n",
	     N_WRITE * i + 1, (N_WRITE - 1) + (N_WRITE * i) + 1);

    for (k = 1; k <= nrow; k++)	/* print all rows */
    {
      for (j = start; j <= N_WRITE + start - 1; j++)
      {
	fprintf (fn, " %-24.16g %-24.16g", MATr (m, k, j), MATi (m, k, j));
      }
      fprintf (fn, "\n");
    }
    start += N_WRITE;		/* inc our col position */
  }

  /* Now come back and write out the last few columns */
  if (!rem)
    return;
  fprintf (fn, "# matrix columns %d thru %d\n",
	   MNC (m) - rem + 1, MNC (m));

  for (k = 1; k <= nrow; k++)
  {
    for (i = ncol - rem + 1; i <= ncol; i++)
    {
      fprintf (fn, " %-24.16g %-24.16g", MATr (m, k, i), MATi (m, k, i));
    }
    fprintf (fn, "\n");
  }
  fflush (fn);
  return;
}

/*
 * Write out a STRING matrix.
 * Just write out a column for now.
 */
static void
matrix_write_string (m, fn)
     Matrix *m;
     FILE *fn;
{
  int i, nrow, ncol;

  nrow = MNR (m);
  ncol = MNC (m);

  if (matrix_GetName (m) == 0)
    fprintf (fn, "# matrix : %s no_of_rows: %d no_of_cols: %d",
	     "MATRIX", MNR (m), MNC (m));
  else
    fprintf (fn, "# matrix : %s no_of_rows: %d no_of_cols: %d",
	     matrix_GetName (m), MNR (m), MNC (m));

  fprintf (fn, " STRING\n");

  for (i = 0; i < nrow * ncol; i++)
  {
    fprintf (fn, "%s\n", MATsv (m, i));
  }

  fflush (fn);
  return;
}


#define NLINE 200
/* **************************************************************
 * Read a matrix from file.
 * ************************************************************** */

static void matrix_read_string _PROTO ((FILE * fn, Matrix * m));

Matrix *
matrix_Read (fn)
     FILE *fn;
{
  int i, j, k, nrow, ncol, npri, rem, start, type;
  char jnk_str[NLINE], m_name[80], dtype[20];	/* fix this someday */
  Matrix *new = 0;

  type = 0;			/* Initialize */

  fscanf (fn, "%s no_of_rows: %d no_of_cols: %d %s\n",
	  m_name, &nrow, &ncol, dtype);

  npri = ncol / N_WRITE;
  rem = ncol % N_WRITE;

  if (!strcmp (dtype, "REAL"))
  {
    new = matrix_Create (nrow, ncol);
    type = REAL;
  }
  else if (!strcmp (dtype, "COMPLEX"))
  {
    new = matrix_CreateC (nrow, ncol);
    type = COMPLEX;
  }
  else if (!strcmp (dtype, "STRING"))
  {
    new = matrix_CreateS (nrow, ncol);
    type = STRING;
    matrix_read_string (fn, new);
    new->name = cpstr (m_name);
    return (new);
  }
  new->name = cpstr (m_name);

  start = 1;
  for (i = 0; i < npri; i++)
  {
    fgets (jnk_str, NLINE, fn);
    /* fscanf(fn, "# matrix columns %*s %*s %*s\n"); */
    for (k = 1; k <= nrow; k++)	/* read all rows */
    {
      if (type == REAL)
      {
	for (j = start; j <= N_WRITE + start - 1; j++)
	  fscanf (fn, " %le", &MAT (new, k, j));
      }
      else if (type == COMPLEX)
      {
	for (j = start; j <= N_WRITE + start - 1; j++)
	  fscanf (fn, " %le %le", &MATr (new, k, j), &MATi (new, k, j));
      }
      fscanf (fn, "\n");
    }
    start += N_WRITE;		/* inc our col position */
  }

  /* Now come back and read the last few columns */
  if (!rem)
    return (new);
  fgets (jnk_str, NLINE, fn);
  /* fscanf(fn, "# matrix columns %*s %*s %*s\n"); */
  for (k = 1; k <= nrow; k++)
  {
    for (i = ncol - rem + 1; i <= ncol; i++)
      if (type == REAL)
	fscanf (fn, " %le", &MAT (new, k, i));
      else if (type == COMPLEX)
	fscanf (fn, " %le %le", &MATr (new, k, i), &MATi (new, k, i));
    fscanf (fn, "\n");
  }
  return (new);
}

static void
matrix_read_string (fn, m)
     FILE *fn;
     Matrix *m;
{
  char *ctmp;
  int i;

  ctmp = (char *) MALLOC (sizeof (char) * 1000);

  for (i = 0; i < MNR (m) * MNC (m); i++)
  {
    fscanf (fn, "%s", ctmp);
    MATsv (m, i) = cpstr (ctmp);
  }

  fscanf (fn, "\n");
  FREE (ctmp);
}

/* **************************************************************
 * Zero the elements of an existing matrix.
 * ************************************************************** */
void
matrix_Zero (m)
     Matrix *m;
{
  ASSERT (m);
  {
    int i, size;
    size = MNR (m) * MNC (m);

    if (MTYPE (m) == REAL)
    {
      for (i = 0; i < size; i++)
	m->val.mr[i] = 0.0;
    }
    else if (MTYPE (m) == COMPLEX)
    {
      for (i = 0; i < size; i++)
      {
	m->val.mc[i].r = 0.0;
	m->val.mc[i].i = 0.0;
      }
    }
    else if (MTYPE (m) == STRING)
    {
      for (i = 0; i < size; i++)
	FREE (m->val.ms[i]);
    }
  }
}

/* **************************************************************
 * Check the row and col values against the actual row and column
 * sizes of a particular matrix.
 * ************************************************************** */
void
matrix_CheckIndices (m, row, col)
     Matrix *m;
     int row, col;
{
  if ((row > MNR (m)) || (row < 1))
    error_1 (matrix_GetName (m), "Row index out-of-range");
  if ((col > MNC (m)) || (col < 1))
    error_1 (matrix_GetName (m), "Column index out-of-range");
}

/* **************************************************************
 * Append a column of zeros to the specified matrix.
 * ************************************************************** */

/*
 * Append a culmn to a REAL Matrix.
 */
int
matrix_AppendColR (m, n_col_add)
     Matrix *m;
     int n_col_add;
{
  ASSERT (m);
  ASSERT (n_col_add > 0);
  ASSERT (MTYPE (m) == REAL);
  {
    int i, n_col_new, nrow;

    n_col_new = MNC (m) + n_col_add;

    /* Special case when [m] is NULL */
    if (MNR (m) == 0)
      nrow = 1;
    else
      nrow = MNR (m);

    m->val.mr = (double *) REALLOC (m->val.mr,
				    sizeof (double) * nrow * n_col_new);

    /* now zero out newly appended columns */
    for (i = nrow * MNC (m); i < nrow * n_col_new; i++)
      m->val.mr[i] = 0.0;

    m->ncol = n_col_new;
    m->nrow = nrow;
    return (1);
  }
}

/*
 * Append a column to a COMPLEX matrix .
 */
int
matrix_AppendColC (m, n_col_add)
     Matrix *m;
     int n_col_add;
{
  ASSERT (m);
  ASSERT (n_col_add > 0);
  ASSERT (MTYPE (m) == COMPLEX);
  {
    int i, j, n_col_new, nrow;
    Complex *temp;

    n_col_new = MNC (m) + n_col_add;

    /* Special case when [m] is NULL */
    if (MNR (m) == 0)
      nrow = 1;
    else
      nrow = MNR (m);

    temp = (Complex *) MALLOC (sizeof (Complex) * nrow * n_col_new);
    /* now copy old matrix elements into new matrix */
    if (MTYPE (m) == REAL)
    {
      for (i = 0; i < MNR (m); i++)
	for (j = 0; j < MNC (m); j++)
	{
	  temp[j * MNR (m) + i].r = MAT0 (m, i, j);
	  temp[j * MNR (m) + i].i = 0.0;
	}
    }
    else if (MTYPE (m) == COMPLEX)
    {
      for (i = 0; i < MNR (m); i++)
	for (j = 0; j < MNC (m); j++)
	{
	  temp[j * MNR (m) + i].r = MATr0 (m, i, j);
	  temp[j * MNR (m) + i].i = MATi0 (m, i, j);
	}
    }
    else
      error_1 ("matrix_AppendColC: BAD INTERNAL ERROR", 0);

    /* now zero out newly appended columns */
    for (i = MNR (m) * MNC (m); i < nrow * n_col_new; i++)
    {
      temp[i].r = 0.0;
      temp[i].i = 0.0;
    }

    if (MTYPE (m) == REAL)
      FREE (m->val.mr);
    else
      FREE (m->val.mc);

    m->val.mc = temp;
    m->dtype = COMPLEX;
    m->ncol = n_col_new;
    m->nrow = nrow;
    return (1);
  }
}

/*
 * Append a column to a STRING Matrix.
 */
int
matrix_AppendColS (m, n_col_add)
     Matrix *m;
     int n_col_add;
{
  ASSERT (m);
  ASSERT (n_col_add > 0);
  ASSERT (MTYPE (m) == STRING);
  {
    int i, n_col_new, nrow;

    n_col_new = MNC (m) + n_col_add;

    /* Special case when [m] is NULL */
    if (MNR (m) == 0)
      nrow = 1;
    else
      nrow = MNR (m);

    m->val.ms = (char **) REALLOC (m->val.ms,
				   sizeof (char *) * nrow * n_col_new);

    /* now zero out newly appended columns */
    for (i = nrow * MNC (m); i < nrow * n_col_new; i++)
      m->val.ms[i] = cpstr ("");

    m->ncol = n_col_new;
    m->nrow = nrow;
    return (1);
  }
}

/* **************************************************************
 * Append a row of zeros to the specified matrix.
 * Note: This can NOT be done with a realloc(), since we store
 *       the matrix column-wise.
 * ************************************************************** */

/*
 * Append a row to a REAL Matrix.
 */

int
matrix_AppendRowR (m, n_row_add)
     Matrix *m;
     int n_row_add;
{
  ASSERT (m);
  ASSERT (n_row_add > 0);
  ASSERT (MTYPE (m) == REAL);
  {
    int i, j, n_row_new, ncol;
    double *temp;

    n_row_new = MNR (m) + n_row_add;

    /* Special case when [m] is NULL */
    if (MNC (m) == 0)
      ncol = 1;
    else
      ncol = MNC (m);

    temp = (double *) MALLOC (sizeof (double) * n_row_new * ncol);

    /* now copy old matrix elements into new matrix */
    for (i = 0; i < MNR (m); i++)
      for (j = 0; j < MNC (m); j++)
	temp[j * n_row_new + i] = MAT0 (m, i, j);

    /* now zero out remaining elements */
    for (i = MNR (m); i < n_row_new; i++)
      for (j = 0; j < ncol; j++)
	temp[j * n_row_new + i] = 0.0;

    FREE (m->val.mr);
    m->val.mr = temp;
    m->nrow = n_row_new;
    m->ncol = ncol;
    return (1);
  }
}

/*
 * Append a row to a COMPLEX Matrix.
 */
int
matrix_AppendRowC (m, n_row_add)
     Matrix *m;
     int n_row_add;
{
  ASSERT (m);
  ASSERT (n_row_add > 0);
  {
    int i, j, n_row_new, ncol;
    Complex *temp;

    n_row_new = MNR (m) + n_row_add;

    /* Special case when [m] is NULL */
    if (MNC (m) == 0)
      ncol = 1;
    else
      ncol = MNC (m);

    temp = (Complex *) MALLOC (sizeof (Complex) * n_row_new * ncol);

    /* now copy old matrix elements into new matrix */
    if (MTYPE (m) == REAL)
    {
      for (i = 0; i < MNR (m); i++)
	for (j = 0; j < MNC (m); j++)
	{
	  temp[j * n_row_new + i].r = MAT (m, (i + 1), (j + 1));
	  temp[j * n_row_new + i].i = 0.0;
	}
    }
    else
    {
      for (i = 0; i < MNR (m); i++)
	for (j = 0; j < MNC (m); j++)
	{
	  temp[j * n_row_new + i].r = MATr (m, (i + 1), (j + 1));
	  temp[j * n_row_new + i].i = MATi (m, (i + 1), (j + 1));
	}
    }
    /* now zero out remaining elements */
    for (i = MNR (m); i < n_row_new; i++)
      for (j = 0; j < ncol; j++)
      {
	temp[j * n_row_new + i].r = 0.0;
	temp[j * n_row_new + i].i = 0.0;
      }
    if (MTYPE (m) == REAL)
      FREE (m->val.mr);
    else
      FREE (m->val.mc);
    m->val.mc = temp;
    m->dtype = COMPLEX;
    m->nrow = n_row_new;
    m->ncol = ncol;
    return (1);
  }
}

/*
 * Append a row to a STRING Matrix.
 */
int
matrix_AppendRowS (m, n_row_add)
     Matrix *m;
     int n_row_add;
{
  ASSERT (m);
  ASSERT (n_row_add > 0);
  ASSERT (MTYPE (m) == STRING);
  {
    int i, j, n_row_new, ncol;
    char **temp;

    n_row_new = MNR (m) + n_row_add;

    /* Special case when [m] is NULL */
    if (MNC (m) == 0)
      ncol = 1;
    else
      ncol = MNC (m);

    temp = (char **) MALLOC (sizeof (char *) * n_row_new * ncol);

    /* now copy old matrix elements into new matrix */
    for (i = 0; i < MNR (m); i++)
      for (j = 0; j < MNC (m); j++)
	temp[j * n_row_new + i] = MATs0 (m, i, j);

    /* now zero out remaining elements */
    for (i = MNR (m); i < n_row_new; i++)
      for (j = 0; j < ncol; j++)
	temp[j * n_row_new + i] = cpstr ("");

    FREE (m->val.ms);
    m->val.ms = temp;
    m->nrow = n_row_new;
    m->ncol = ncol;
    return (1);
  }
}

/* **************************************************************
 * Convert a matrix from REAL to COMPLEX. Do the conversion in place.
 * ************************************************************** */
void
matrix_convert (m)
     Matrix *m;
{
  ASSERT (m);
  {
    int i;
    Complex *tmp;

    if (MTYPE (m) == REAL)
    {
      tmp = (Complex *) MALLOC (MNR (m) * MNC (m) * sizeof (Complex));
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	tmp[i].r = (m)->val.mr[i];
	tmp[i].i = 0.0;
      }
      m->dtype = COMPLEX;
      FREE ((m)->val.mr);
      m->val.mc = tmp;
    }
    return;
  }
}


/* **************************************************************
 * Copy a matrix, regardless of it's type, the result is complex.
 * ************************************************************** */
Matrix *
matrix_copy_complex (m)
     Matrix *m;
{
  ASSERT (m);
  {
    int i;
    Matrix *new;

    new = matrix_CreateC (MNR (m), MNC (m));
    if (MTYPE (m) == REAL)
    {
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	MATcvr (new, i) = MATrv (m, i);
	MATcvi (new, i) = 0.0;
      }
    }
    else if (MTYPE (m) == COMPLEX)
    {
      for (i = 0; i < MNR (m) * MNC (m); i++)
      {
	MATcvr (new, i) = MATcvr (m, i);
	MATcvi (new, i) = MATcvi (m, i);
      }
    }
    return (new);
  }
}

/* **************************************************************
 * Extend a matrix to include the argument indices, initialize the
 * new elements to have valaue 0.0. Return TRUE if matrix is 
 * extended, FALSE if matrix is untouched.
 * ************************************************************** */

int
matrix_Extend (m, n_row, n_col)
     Matrix *m;
     int n_row, n_col;
{
  ASSERT (m);
  ASSERT (n_row > 0);
  ASSERT (n_col > 0);
  {
    int retval = 0;

    if (n_row > MNR (m))
    {
      retval = 1;
      if (MTYPE (m) == REAL)
	matrix_AppendRowR (m, n_row - MNR (m));
      else if (MTYPE (m) == COMPLEX)
	matrix_AppendRowC (m, n_row - MNR (m));
      else if (MTYPE (m) == STRING)
	matrix_AppendRowS (m, n_row - MNR (m));
    }
    if (n_col > MNC (m))
    {
      retval = 1;
      if (MTYPE (m) == REAL)
	matrix_AppendColR (m, n_col - MNC (m));
      else if (MTYPE (m) == COMPLEX)
	matrix_AppendColC (m, n_col - MNC (m));
      else if (MTYPE (m) == STRING)
	matrix_AppendColS (m, n_col - MNC (m));
    }
    return (retval);
  }
}

/* **************************************************************
 * Assign new element values to certain elements of a matrix.
 * rv = Vector of matrix row indices.
 * cv = Vector of matrix column indices.
 * ************************************************************** */
int
matrix_Assign (m, rv, cv, rhs)
     Matrix *m, *rhs;
     Matrix *rv, *cv;
{
  ASSERT (m);
  ASSERT (rv);
  ASSERT (cv);
  ASSERT (rhs);
  {
    int i, j, csize, rsize;

    /* Check for empty indices */
    if (MNR (rv) == 0 || MNC (rv) == 0)
      return 1;
    if (MNR (cv) == 0 || MNC (cv) == 0)
      return 1;

    /* 1st check that rhs dim matches lhs dim */
    if ((MNR (rv) != 1 && MNC (rv) != 1) || (MNR (cv) != 1 && MNC (cv) != 1))
      error_1 (m->name, "indices must be VECTORS");
    if (MNR (rhs) != (MNR (rv) * MNC (rv)))
      error_1 (m->name, "RHS row dim does not match row indices dim");
    if (MNC (rhs) != (MNR (cv) * MNC (cv)))
      error_1 (m->name, "RHS col dim does not match col indices dim");

    rsize = MNR (rv) * MNC (rv);
    csize = MNR (cv) * MNC (cv);

    /* Now do assignments */
    if (MTYPE (m) == REAL && MTYPE (rhs) == REAL)
    {
      for (i = 1; i <= rsize; i++)
	for (j = 1; j <= csize; j++)
	{
	  matrix_Extend (m, (int) MATrv1 (rv, i), (int) MATrv1 (cv, j));
	  MAT (m, ((int) MATrv1 (rv, i)), ((int) MATrv1 (cv, j))) = MAT (rhs, i, j);
	}
    }
    else if (MTYPE (m) == COMPLEX || MTYPE (rhs) == COMPLEX)
    {
      if (MTYPE (m) == REAL)
	matrix_convert (m);
      if (MTYPE (rhs) == REAL)
      {
	for (i = 1; i <= rsize; i++)
	  for (j = 1; j <= csize; j++)
	  {
	    matrix_Extend (m, (int) MATrv1 (rv, i), (int) MATrv1 (cv, j));
	    MATr (m, ((int) MATrv1 (rv, i)), ((int) MATrv1 (cv, j))) = MAT (rhs, i, j);
	    MATi (m, ((int) MATrv1 (rv, i)), ((int) MATrv1 (cv, j))) = 0.0;
	  }
      }
      else
      {
	for (i = 1; i <= rsize; i++)
	  for (j = 1; j <= csize; j++)
	  {
	    matrix_Extend (m, (int) MATrv1 (rv, i), (int) MATrv1 (cv, j));
	    MATr (m, ((int) MATrv1 (rv, i)), ((int) MATrv1 (cv, j))) = MATr (rhs, i, j);
	    MATi (m, ((int) MATrv1 (rv, i)), ((int) MATrv1 (cv, j))) = MATi (rhs, i, j);
	  }
      }
    }
    else if (MTYPE (m) == STRING && MTYPE (rhs) == STRING)
    {
      for (i = 1; i <= MNC (rv); i++)
      {
	for (j = 1; j <= MNC (cv); j++)
	{
	  matrix_Extend (m, (int) MATrv1 (rv, i), (int) MATrv1 (cv, j));
	  FREE (MATs (m, ((int) MATrv1 (rv, i)), ((int) MATrv1 (cv, j))));
	  MATs (m, ((int) MATrv1 (rv, i)), ((int) MATrv1 (cv, j)))
	    = cpstr (MATs (rhs, i, j));
	}
      }
    }
    else
      error_1 (matrix_GetName (m), "invalid combination of LHS & RHS");

    return (1);
  }
}

int
matrix_Assign_el (m, ri, ci, rhs)
     Matrix *m, *rhs;
     int ri, ci;
{
  ASSERT (m);
  ASSERT (rhs);
  {
    /* 1st check that rhs dim matches lhs dim */
    if (MNR (rhs) != 1 && MNC (rhs) != 1)
      error_1 (m->name, "RHS row dim does not match row indices dim");

    /* Now do assignments */
    if (MTYPE (m) == REAL && MTYPE (rhs) == REAL)
    {
      matrix_Extend (m, ri, ci);
      MAT (m, ri, ci) = MAT (rhs, 1, 1);
    }
    else if (MTYPE (m) == COMPLEX || MTYPE (rhs) == COMPLEX)
    {
      if (MTYPE (m) == REAL)
	matrix_convert (m);
      if (MTYPE (rhs) == REAL)
      {
	matrix_Extend (m, ri, ci);
	MATr (m, ri, ci) = MAT (rhs, 1, 1);
	MATi (m, ri, ci) = 0.0;
      }
      else
      {
	matrix_Extend (m, ri, ci);
	MATr (m, ri, ci) = MATr (rhs, 1, 1);
	MATi (m, ri, ci) = MATi (rhs, 1, 1);
      }
    }
    else if (MTYPE (m) == STRING && MTYPE (rhs) == STRING)
    {
      matrix_Extend (m, ri, ci);
      FREE (MATs (m, ri, ci));
      MATs (m, ri, ci) = cpstr (MATs (rhs, 1, 1));
    }
    else
      error_1 (matrix_GetName (m), "invalid combination of LHS & RHS");

    return (1);
  }
}

int
matrix_AssignCol (m, rv, rhs)
     Matrix *m;
     Matrix *rv, *rhs;
{
  ASSERT (m);
  ASSERT (rv);
  ASSERT (rhs);
  {
    int i, j, rsize;

    /* Check for empty indices */
    if (MNR (rv) == 0 || MNC (rv) == 0)
      return 1;

    if ((MNR (rv) * MNC (rv)) != MNR (rhs))
      error_1 (m->name, "LHS and RHS row dimensions must match");
    if (MNC (m) != MNC (rhs) && MNR (m) != 0)
      error_1 (m->name, "LHS and RHS col dimensions must match");
    if (MNR (rv) != 1 && MNC (rv) != 1)
      error_1 (m->name, "row indices must be VECTOR");

    rsize = MNR (rv) * MNC (rv);

    if (MTYPE (m) == REAL && MTYPE (rhs) == REAL)
    {
      for (i = 1; i <= rsize; i++)
      {
	for (j = 1; j <= MNC (rhs); j++)
	{
	  matrix_Extend (m, (int) MATrv1 (rv, i), j);
	  MAT (m, ((int) MATrv1 (rv, i)), j) = MAT (rhs, i, j);
	}
      }
    }
    else if (MTYPE (m) == COMPLEX || MTYPE (rhs) == COMPLEX)
    {
      if (MTYPE (m) == REAL)
	matrix_convert (m);
      if (MTYPE (rhs) == REAL)
      {
	for (i = 1; i <= rsize; i++)
	{
	  for (j = 1; j <= MNC (rhs); j++)
	  {
	    matrix_Extend (m, (int) MATrv1 (rv, i), j);
	    MATr (m, ((int) MATrv1 (rv, i)), j) = MAT (rhs, i, j);
	    MATi (m, ((int) MATrv1 (rv, i)), j) = 0.0;
	  }
	}
      }
      else
      {
	for (i = 1; i <= rsize; i++)
	{
	  for (j = 1; j <= MNC (rhs); j++)
	  {
	    matrix_Extend (m, (int) MAT (rv, 1, i), j);
	    MATr (m, ((int) MATrv1 (rv, i)), j) = MATr (rhs, i, j);
	    MATi (m, ((int) MATrv1 (rv, i)), j) = MATi (rhs, i, j);
	  }
	}
      }
    }
    else if (MTYPE (m) == STRING && MTYPE (rhs) == STRING)
    {
      for (i = 1; i <= rsize; i++)
      {
	for (j = 1; j <= MNC (rhs); j++)
	{
	  matrix_Extend (m, (int) MATrv1 (rv, i), j);
	  FREE (MATs (m, ((int) MATrv1 (rv, i)), j));
	  MATs (m, ((int) MATrv1 (rv, i)), j)
	    = cpstr (MATs (rhs, i, j));
	}
      }
    }
    else
      error_1 (matrix_GetName (m), "invalid combination of LHS & RHS");
    return (1);
  }
}

int
matrix_AssignRow (m, cv, rhs)
     Matrix *m;
     Matrix *cv, *rhs;
{
  ASSERT (m);
  ASSERT (cv);
  ASSERT (rhs);
  {
    int i, j, csize;

    /* Check for empty indices */
    if (MNR (cv) == 0 || MNC (cv) == 0)
      return 1;

    if ((MNR (cv) * MNC (cv)) != MNC (rhs))
      error_1 (m->name, "LHS and RHS col dimensions must match");
    if (MNR (m) != MNR (rhs) && MNR (m) != 0)
      error_1 (m->name, "LHS and RHS row dimensions must match");
    if (MNR (cv) != 1 && MNC (cv) != 1)
      error_1 (m->name, "col indices must be VECTOR");

    csize = MNR (cv) * MNC (cv);

    if (MTYPE (m) == REAL && MTYPE (rhs) == REAL)
    {
      for (i = 1; i <= csize; i++)
      {
	for (j = 1; j <= MNR (rhs); j++)
	{
	  matrix_Extend (m, j, (int) MATrv1 (cv, i));
	  MAT (m, j, ((int) MATrv1 (cv, i))) = MAT (rhs, j, i);
	}
      }
    }
    else if (MTYPE (m) == COMPLEX || MTYPE (rhs) == COMPLEX)
    {
      if (MTYPE (m) == REAL)
	matrix_convert (m);
      if (MTYPE (rhs) == REAL)
      {
	for (i = 1; i <= csize; i++)
	{
	  for (j = 1; j < MNR (rhs); j++)
	  {
	    matrix_Extend (m, j, (int) MATrv1 (cv, i));
	    MATr (m, j, ((int) MATrv1 (cv, i))) = MAT (rhs, j, i);
	    MATi (m, j, ((int) MATrv1 (cv, i))) = 0.0;
	  }
	}
      }
      else
      {
	for (i = 1; i <= MNC (cv); i++)
	{
	  for (j = 1; j <= MNR (rhs); j++)
	  {
	    matrix_Extend (m, j, (int) MATrv1 (cv, i));
	    MATr (m, j, ((int) MATrv1 (cv, i))) = MATr (rhs, j, i);
	    MATi (m, j, ((int) MATrv1 (cv, i))) = MATi (rhs, j, i);
	  }
	}
      }
    }
    else if (MTYPE (m) == STRING && MTYPE (rhs) == STRING)
    {
      for (i = 1; i <= MNC (cv); i++)
      {
	for (j = 1; j <= MNR (rhs); j++)
	{
	  matrix_Extend (m, j, (int) MATrv1 (cv, i));
	  FREE (MATs (m, j, ((int) MATrv1 (cv, i))));
	  MATs (m, j, ((int) MATrv1 (cv, i)))
	    = cpstr (MATs (rhs, j, i));
	}
      }
    }
    else
      error_1 (matrix_GetName (m), "invalid combination of LHS & RHS");

    return (1);
  }
}

/*
 * m [ ind ] = rhs
 */

void
matrix_AssignVector (m, ind, rhs)
     Matrix *m, *ind, *rhs;
{
  ASSERT (m);
  ASSERT (ind);
  ASSERT (rhs);
  {
    int i, nr, nc, size;

    /* Check for empty indices */
    if (MNR (ind) == 0 || MNC (ind) == 0)
      return;

    /* Check for proper indices dim. */
    if (MNR (ind) != 1 && MNC (ind) != 1)
      error_1 (m->name, "MATRIX indices must be VECTOR");
    if (MNR (ind) * MNC (ind) != MNR (rhs) * MNC (rhs))
      error_1 (m->name, "LHS and RHS must have same linear dim.");

    size = MNR (ind) * MNC (ind);

    if (MTYPE (m) == REAL && MTYPE (rhs) == REAL)
    {
      /* If LHS is emptry, make it a 1x1 */
      if (MNR (m) == 0)
      {
	matrix_Extend (m, 1, 1);
      }
      for (i = 1; i <= size; i++)
      {
	if ((int) MATrv1 (ind, i) > MNR (m) * MNC (m))
	{
	  if (MNR (m) == 1)
	  {
	    nr = 1;
	    nc = (int) ceil (MATrv1 (ind, i) / ((double) MNR (m)));
	  }
	  else if (MNC (m) == 1)
	  {
	    nr = (int) ceil (MATrv1 (ind, i) / ((double) MNC (m)));
	    nc = 1;
	  }
	  else
	  {
	    nr = 1;
	    nc = (int) ceil (MATrv1 (ind, i) / ((double) MNR (m)));
	  }
	  matrix_Extend (m, nr, nc);
	}
	MATrv1 (m, ((int) MATrv1 (ind, i))) = MATrv1 (rhs, i);
      }
    }
    else if (MTYPE (m) == COMPLEX || MTYPE (rhs) == COMPLEX)
    {
      if (MTYPE (m) == REAL)
	matrix_convert (m);	/* convert to COMPLEX */
      if (MTYPE (rhs) == REAL)
      {
	/* If LHS is emptry, make it a 1x1 */
	if (MNR (m) == 0)
	{
	  matrix_Extend (m, 1, 1);
	}
	for (i = 1; i <= size; i++)
	{
	  if ((int) MATrv1 (ind, i) > MNR (m) * MNC (m))
	  {
	    if (MNR (m) == 1)
	    {
	      nr = 1;
	      nc = (int) ceil (MATrv1 (ind, i) / ((double) MNR (m)));
	    }
	    else if (MNC (m) == 1)
	    {
	      nr = (int) ceil (MATrv1 (ind, i) / ((double) MNC (m)));
	      nc = 1;
	    }
	    else
	    {
	      nr = 1;
	      nc = (int) ceil (MATrv1 (ind, i) / ((double) MNR (m)));
	    }
	    matrix_Extend (m, nr, nc);
	  }
	  MATcvr1 (m, ((int) MATrv1 (ind, i))) = MATrv1 (rhs, i);
	  MATcvi1 (m, ((int) MATrv1 (ind, i))) = 0.0;
	}
      }
      else if (MTYPE (rhs) == COMPLEX)
      {
	/* If LHS is emptry, make it a 1x1 */
	if (MNR (m) == 0)
	{
	  matrix_Extend (m, 1, 1);
	}
	for (i = 1; i <= size; i++)
	{
	  if ((int) MATrv1 (ind, i) > MNR (m) * MNC (m))
	  {
	    if (MNR (m) == 1)
	    {
	      nr = 1;
	      nc = (int) ceil (MATrv1 (ind, i) / ((double) MNR (m)));
	    }
	    else if (MNC (m) == 1)
	    {
	      nr = (int) ceil (MATrv1 (ind, i) / ((double) MNC (m)));
	      nc = 1;
	    }
	    else
	    {
	      nr = 1;
	      nc = (int) ceil (MATrv1 (ind, i) / ((double) MNR (m)));
	    }
	    matrix_Extend (m, nr, nc);
	  }
	  MATcvr1 (m, ((int) MATrv1 (ind, i))) = MATcvr1 (rhs, i);
	  MATcvi1 (m, ((int) MATrv1 (ind, i))) = MATcvi1 (rhs, i);
	}
      }
    }
    else if (MTYPE (m) == STRING && MTYPE (rhs) == STRING)
    {
      /* If LHS is emptry, make it a 1x1 */
      if (MNR (m) == 0)
      {
	matrix_Extend (m, 1, 1);
      }
      for (i = 1; i <= size; i++)
      {
	if ((int) MATrv1 (ind, i) > MNR (m) * MNC (m))
	{
	  if (MNR (m) == 1)
	  {
	    nr = 1;
	    nc = (int) ceil (MATrv1 (ind, i) / ((double) MNR (m)));
	  }
	  else if (MNC (m) == 1)
	  {
	    nr = (int) ceil (MATrv1 (ind, i) / ((double) MNC (m)));
	    nc = 1;
	  }
	  else
	  {
	    nr = 1;
	    nc = (int) ceil (MATrv1 (ind, i) / ((double) MNR (m)));
	  }
	  matrix_Extend (m, nr, nc);
	}
	FREE (MATsv1 (m, ((int) MATrv1 (ind, i))));
	MATsv1 (m, ((int) MATrv1 (ind, i)))
	  = cpstr (MATsv1 (rhs, i));
      }
    }
    else
      error_1 (matrix_GetName (m), "invalid combination of LHS and RHS");
  }
}

/* **************************************************************
 * Set the name member of a matrix.
 * ************************************************************** */
void
matrix_SetName (m, string)
     Matrix *m;
     char *string;
{
  ASSERT (m);
  FREE (m->name);
  m->name = string;
}

/* **************************************************************
 * Extract a row matrix from the input.
 * v = the vector of matrix column indices to extract.
 * ************************************************************** */
Matrix *
matrix_ExtractRowMatrix (m, v)
     Matrix *m, *v;
{
  ASSERT (m);
  ASSERT (v);
  {
    int i, j, size;
    Matrix *new = 0;

    /* Check for NULL indices */
    if (MNR (v) == 0 || MNC (v) == 0)
      return (matrix_Create (0, 0));

    size = MNR (v) * MNC (v);

    /* Now check vector values against allowable indices */
    for (i = 1; i <= size; i++)
    {
      if ((int) MATrv1 (v, i) < 1 || (int) MATrv1 (v, i) > MNC (m))
	error_1 (m->name, "Invalid column index value");
    }

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (MNR (m), size);
      for (i = 1; i <= MNR (m); i++)
      {
	for (j = 1; j <= size; j++)
	{
	  MAT (new, i, j) = MAT (m, i, ((int) MATrv1 (v, j)));
	}
      }
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (MNR (m), size);
      for (i = 1; i <= MNR (m); i++)
      {
	for (j = 1; j <= size; j++)
	{
	  MATr (new, i, j) = MATr (m, i, ((int) MATrv1 (v, j)));
	  MATi (new, i, j) = MATi (m, i, ((int) MATrv1 (v, j)));
	}
      }
    }
    else if (MTYPE (m) == STRING)
    {
      new = matrix_CreateS (MNR (m), size);
      for (i = 1; i <= MNR (m); i++)
      {
	for (j = 1; j <= size; j++)
	{
	  MATs (new, i, j) = cpstr (MATs (m, i, ((int) MATrv1 (v, j))));
	}
      }
    }
    return (new);
  }
}

/*
 * Extract a sub-matrix, grab all columns, rows spec'ed by [v]
 */
Matrix *
matrix_ExtractColMatrix (m, v)
     Matrix *m, *v;
{
  ASSERT (m);
  ASSERT (v);
  {
    int i, j, size;
    Matrix *new = 0;

    /* Check for NULL indices */
    if (MNR (v) == 0 || MNC (v) == 0)
      return (matrix_Create (0, 0));

    size = MNR (v) * MNC (v);

    /* Now check vector values against allowable indices */
    for (i = 1; i <= size; i++)
    {
      if ((int) MATrv1 (v, i) < 1 || (int) MATrv1 (v, i) > MNR (m))
	error_2 (m->name, v->name, "Invalid row index value");
    }

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (size, MNC (m));
      for (i = 1; i <= size; i++)
      {
	for (j = 1; j <= MNC (m); j++)
	{
	  MAT (new, i, j) = MAT (m, ((int) MATrv1 (v, i)), j);
	}
      }
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (size, MNC (m));
      for (i = 1; i <= size; i++)
      {
	for (j = 1; j <= MNC (m); j++)
	{
	  MATr (new, i, j) = MATr (m, ((int) MATrv1 (v, i)), j);
	  MATi (new, i, j) = MATi (m, ((int) MATrv1 (v, i)), j);
	}
      }
    }
    else if (MTYPE (m) == STRING)
    {
      new = matrix_CreateS (size, MNC (m));
      for (i = 1; i <= size; i++)
      {
	for (j = 1; j <= MNC (m); j++)
	{
	  MATs (new, i, j) = cpstr (MATs (m, ((int) MATrv1 (v, i)), j));
	}
      }
    }
    return (new);
  }
}

/*
 * Extract a single element from a matrix
 */
Matrix *
matrix_ExtractElement (m, r, c)
     Matrix *m;
     int r, c;
{
  ASSERT (m);
  {
    Matrix *new = 0;

    /* Check row value for correctness */
    if (r > MNR (m) || r < 1)
      error_1 (m->name, "Invalid row index");

    /* Check column value for correctness */
    if (c > MNC (m) || c < 1)
      error_1 (m->name, "Invalid column index");

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (1, 1);
      MAT (new, 1, 1) = MAT (m, r, c);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (1, 1);
      MATr (new, 1, 1) = MATr (m, r, c);
      MATi (new, 1, 1) = MATi (m, r, c);
    }
    else if (MTYPE (m) == STRING)
    {
      new = matrix_CreateS (1, 1);
      MATs (new, 1, 1) = cpstr (MATs (m, r, c));
    }
    return (new);
  }
}

/*
 * Extract a sub-matrix, grab a row (r), rows spec'ed by [cv]
 */
Matrix *
matrix_ExtractRow (m, r, cv)
     Matrix *m, *cv;
     int r;
{
  ASSERT (m);
  ASSERT (cv);
  {
    int i, size;
    Matrix *new = 0;

    /* Check for NULL indices */
    if (MNR (cv) == 0 || MNC (cv) == 0)
      return (matrix_Create (0, 0));

    size = MNR (cv) * MNC (cv);

    /* Check row value for correctness */
    if (r > MNR (m) || r < 1)
      error_1 (m->name, "Invalid row index");

    /* Now check vector values against allowable indices */
    for (i = 1; i <= size; i++)
    {
      if ((int) MATrv1 (cv, i) > MNC (m) || (int) MATrv1 (cv, i) < 1)
	error_2 (m->name, cv->name, "Invalid row index value");
    }

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (1, size);
      for (i = 1; i <= size; i++)
      {
	MAT (new, 1, i) = MAT (m, r, ((int) MATrv1 (cv, i)));
      }
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (1, size);
      for (i = 1; i <= size; i++)
      {
	MATr (new, 1, i) = MATr (m, r, ((int) MATrv1 (cv, i)));
	MATi (new, 1, i) = MATi (m, r, ((int) MATrv1 (cv, i)));
      }
    }
    else if (MTYPE (m) == STRING)
    {
      new = matrix_CreateS (1, size);
      for (i = 1; i <= size; i++)
      {
	MATs (new, 1, i) = cpstr (MATs (m, r, ((int) MATrv1 (cv, i))));
      }
    }
    return (new);
  }
}

/*
 * Extract a sub-matrix, grab a column (c), rows spec'ed by [v]
 */
Matrix *
matrix_ExtractColumn (m, c, rv)
     Matrix *m, *rv;
     int c;
{
  ASSERT (m);
  ASSERT (rv);
  {
    int i, size;
    Matrix *new = 0;

    /* Check for NULL indices */
    if (MNR (rv) == 0 || MNC (rv) == 0)
      return (matrix_Create (0, 0));

    size = MNR (rv) * MNC (rv);

    /* Check column value for correctness */
    if (c > MNC (m) || c < 1)
      error_1 (m->name, "Invalid column index");

    /* Now check vector values against allowable indices */
    for (i = 1; i <= size; i++)
    {
      if ((int) MATrv1 (rv, i) > MNR (m) || (int) MATrv1 (rv, i) < 1)
	error_2 (m->name, rv->name, "Invalid row index value");
    }

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (size, 1);
      for (i = 1; i <= size; i++)
      {
	MAT (new, i, 1) = MAT (m, ((int) MATrv1 (rv, i)), c);
      }
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (size, 1);
      for (i = 1; i <= size; i++)
      {
	MATr (new, i, 1) = MATr (m, ((int) MATrv1 (rv, i)), c);
	MATi (new, i, 1) = MATi (m, ((int) MATrv1 (rv, i)), c);
      }
    }
    else if (MTYPE (m) == STRING)
    {
      new = matrix_CreateS (size, 1);
      for (i = 1; i <= size; i++)
      {
	MATs (new, i, 1) = cpstr (MATs (m, ((int) MATrv1 (rv, i)), c));
      }
    }
    return (new);
  }
}

/* **************************************************************
 * Extract a sub-matrix from a given matrix, given vectors of
 * row and column indices.
 * ************************************************************** */
Matrix *
matrix_ExtractSubMatrix (m, v_row, v_col)
     Matrix *m;
     Matrix *v_row, *v_col;
{
  ASSERT (m);
  ASSERT (v_row);
  ASSERT (v_col);
  {
    int i, j, rsize, csize;
    Matrix *new = 0;

    /* Check for NULL indices */
    if (MNR (v_row) == 0 || MNC (v_row) == 0
	|| MNR (v_col) == 0 || MNC (v_col) == 0)
    {
      return (matrix_Create (0, 0));
    }

    rsize = MNR (v_row) * MNC (v_row);
    csize = MNR (v_col) * MNC (v_col);

    for (i = 1; i <= rsize; i++)
    {
      if ((int) MATrv1 (v_row, i) < 1 || (int) MATrv1 (v_row, i) > MNR (m))
	error_1 (m->name, "Invalid row index value");
    }
    for (i = 1; i <= csize; i++)
    {
      if ((int) MATrv1 (v_col, i) < 1 || (int) MATrv1 (v_col, i) > MNC (m))
	error_1 (m->name, "Invalid column index value");
    }

    if (MTYPE (m) == REAL)
    {
      new = matrix_Create (rsize, csize);
      for (i = 1; i <= rsize; i++)
	for (j = 1; j <= csize; j++)
	  MAT (new, i, j) = MAT (m, ((int) MATrv1 (v_row, i)),
				 ((int) MATrv1 (v_col, j)));
    }
    else if (MTYPE (m) == COMPLEX)
    {
      new = matrix_CreateC (rsize, csize);
      for (i = 1; i <= rsize; i++)
	for (j = 1; j <= csize; j++)
	{
	  MATr (new, i, j) = MATr (m, ((int) MATrv1 (v_row, i)),
				   ((int) MATrv1 (v_col, j)));
	  MATi (new, i, j) = MATi (m, ((int) MATrv1 (v_row, i)),
				   ((int) MATrv1 (v_col, j)));
	}
    }
    else if (MTYPE (m) == STRING)
    {
      new = matrix_CreateS (rsize, csize);
      for (i = 1; i <= rsize; i++)
	for (j = 1; j <= csize; j++)
	  MATs (new, i, j) = cpstr (MATs (m, ((int) MATrv1 (v_row, i)),
					  ((int) MATrv1 (v_col, j))));
    }
    return (new);
  }
}

/*
 * m [ mi ]
 * Extract a row or column vector from the matrix [m].
 * If [m]  ir a row, or column matrix to begin with, generate
 * a row, or column matrix.
 * If [m] has row and column dim, both larger than 1, create
 * a row matrix.
 */

Matrix *
matrix_ExtractVector (m, mi)
     Matrix *m, *mi;
{
  ASSERT (m);
  ASSERT (mi);
  {
    int i, size;
    Matrix *new = 0;

    /* Check for NULL index matrix */
    if (MNR (mi) == 0 || MNC (mi) == 0)
    {
      return (matrix_Create (0, 0));
    }

    if (MNR (mi) != 1 && MNC (mi) != 1)
      error_1 (m->name, "matrix indices must be a VECTOR");

    size = MNR (mi) * MNC (mi);

    /* Check index values */
    for (i = 0; i < MNC (mi); i++)
      if ((int) MATrv (mi, i) < 1 || (int) MATrv (mi, i) > MNR (m) * MNC (m))
	error_1 (m->name, "matrix indices out-of-bounds");

    if (MTYPE (m) == REAL)
    {
      if (MNC (m) == 1)
	new = matrix_Create (size, 1);
      else
	new = matrix_Create (1, size);
      for (i = 0; i < size; i++)
	MATrv (new, i) = MATrv (m, ((int) MATrv (mi, i)) - 1);
    }
    else if (MTYPE (m) == COMPLEX)
    {
      if (MNC (m) == 1)
	new = matrix_CreateC (size, 1);
      else
	new = matrix_CreateC (1, size);
      for (i = 0; i < size; i++)
      {
	MATcvr (new, i) = MATcvr (m, ((int) MATrv (mi, i)) - 1);
	MATcvi (new, i) = MATcvi (m, ((int) MATrv (mi, i)) - 1);
      }
    }
    else if (MTYPE (m) == STRING)
    {
      if (MNC (m) == 1)
	new = matrix_CreateS (size, 1);
      else
	new = matrix_CreateS (1, size);
      for (i = 0; i < size; i++)
	MATsv (new, i) = cpstr (MATsv (m, ((int) MATrv (mi, i)) - 1));
    }
    return (new);
  }
}

/* **************************************************************
 * Create a REAL MATRIX, and fill its contents;
 *  d1  , d2 ,   d3
 * start, end, increment
 * d1:d2 is the same as [d1, d1+1, ..., d2]
 * d1:d2 is empty if d1 > d2
 * d1:d2:d3 is the same as [d1, d1+d3, d1+2*d3, ..., d2]
 * d1:d2:d3 is empty if d3 > 0 and d1 > d2 or if d3 < 0 and d1 < d2
 * The flag argument is used to control the creation of empty
 * vectors.
 * ************************************************************** */

Matrix *
matrix_CreateFill (d1, d2, d3, flag)
     double d1, d2, d3;
     int flag;
{
  double d, dn;
  int i, n;
  Matrix *new;

  /* Check for d3 = 0, programmer messed up */
  if (d3 == 0)
    error_1 ("must use non-zero increment for vector creation", (char *) 0);

  if (flag)
  {
    /* Check for condition where we return an empty vector */
    if ((d1 > d2 && d3 > 0.0) || (d1 < d2 && d3 < 0.0))
    {
      new = matrix_Create (0, 0);
      return (new);
    }
  }

  /* Calculate n */
  dn = rabs ((d2 - d1) / d3) + 1;
  n = (int) dn;
  if ((d2 - d1) == 0.0)
    d = d3;
  else
    d = ((d2 - d1) / (rabs (d2 - d1))) * rabs (d3);
  new = matrix_Create (1, n);

  /* Fill up vector */
  for (i = 1; i <= n; i++)
    MAT (new, 1, i) = d1 + (i - 1) * d;

  return (new);
}

/* **************************************************************
 * Truncate a matrix, in place.
 * ************************************************************** */

int
matrix_Truncate (m, nr, nc)
     Matrix *m;
     int nr, nc;
{
  ASSERT (m);
  {
    int i, j;
    double *dtmp;
    Complex *ctmp;

    /* Check first */
    if (MNR (m) == nr)
      if (MNC (m) == nc)
	return (1);

    if (MTYPE (m) == REAL)
    {
      dtmp = (double *) MALLOC (sizeof (double) * (nr * nc));
      for (i = 0; i < nr; i++)
	for (j = 0; j < nc; j++)
	  dtmp[j * nr + i] = MAT0 (m, i, j);
      FREE (m->val.mr);
      m->val.mr = dtmp;
      m->nrow = nr;
      m->ncol = nc;
    }
    else if (MTYPE (m) == COMPLEX)
    {
      ctmp = (Complex *) MALLOC (sizeof (Complex) * (nr * nc));
      for (i = 0; i < nr; i++)
	for (j = 0; j < nc; j++)
	{
	  ctmp[j * nr + i].r = MATr0 (m, i, j);
	  ctmp[j * nr + i].i = MATi0 (m, i, j);
	}
      FREE (m->val.mc);
      m->val.mc = ctmp;
      m->nrow = nr;
      m->ncol = nc;
    }
    else
      error_1 ("matrix_Truncate does not support string matrices", 0);

    return (1);
  }
}
