/* math_1.c
 * Miscellaneous math functions for RLaB */

/*  This file is a part of RLaB ("Our"-LaB)
   Copyright (C) 1992, 1994  Ian R. Searle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   See the file ./COPYING
   ********************************************************************** */

#include "rlab.h"
#include "symbol.h"
#include "util.h"
#include "bltin.h"
#include "scop1.h"
#include "matop1.h"
#include "matop2.h"
#include "listnode.h"
#include "btree.h"
#include "r_string.h"
#include "fi_1.h"
#include "mathl.h"

#include <math.h>

/* **************************************************************
 * Abs function.
 * ************************************************************** */
void
Abs (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *M;

  /* Check n_args */
  if (n_args != 1)
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("abs: 1 argument allowed", 0);
  }

  M = bltin_get_numeric ("abs", d_arg, 1);
  switch (e_type (M))
  {
  case SCALAR:
    *return_ptr = (VPTR) scalar_Abs (e_data (M));
    break;

  case MATRIX:
    *return_ptr = (VPTR) matrix_Abs (e_data (M));
    break;

  default:
    remove_tmp_destroy (M);
    error_1 ("abs: invalid argument", 0);
    break;
  }

  remove_tmp_destroy (M);
  return;
}

/* **************************************************************
 * mod function.
 * ************************************************************** */
void
Mod (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *M1, *M2;
  Matrix *m;

  /* Check n_args */
  if (n_args != 2)
    error_1 ("mod: 2 arguments allowed", 0);

  /* get args from list */
  M1 = bltin_get_numeric_matrix ("mod", d_arg, 1);
  M2 = bltin_get_numeric_matrix ("mod", d_arg, 2);
  m = matrix_Mod (e_data (M1), e_data (M2));
  *return_ptr = (VPTR) m;
  remove_tmp_destroy (M1);
  remove_tmp_destroy (M2);
  return;
}

/* **************************************************************
 * log function.
 * ************************************************************** */
void
Log (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *M;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("log: 1 argument allowed", 0);

  M = bltin_get_numeric ("log", d_arg, 1);
  switch (e_type (M))
  {
  case SCALAR:
    *return_ptr = (VPTR) scalar_Log (e_data (M));
    break;

  case MATRIX:
    *return_ptr = (VPTR) matrix_Log (e_data (M));
    break;

  default:
    remove_tmp_destroy (M);
    error_1 ("log: invalid argument", 0);
    break;
  }

  remove_tmp_destroy (M);
  return;
}

/* **************************************************************
 * log10 function.
 * ************************************************************** */
void
Log10 (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *M;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("log10: 1 argument allowed", 0);

  M = bltin_get_numeric ("log10", d_arg, 1);
  switch (e_type (M))
  {
  case SCALAR:
    *return_ptr = (VPTR) scalar_Log10 (e_data (M));
    break;

  case MATRIX:
    *return_ptr = (VPTR) matrix_Log10 (e_data (M));
    break;

  default:
    remove_tmp_destroy (M);
    error_1 ("log10: invalid argument", 0);
    break;
  }

  remove_tmp_destroy (M);
  return;
}

/* **************************************************************
 * exp function.
 * ************************************************************** */
void
Exp (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *M;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("exp: 1 argument allowed", 0);

  M = bltin_get_numeric ("exp", d_arg, 1);
  switch (e_type (M))
  {
  case SCALAR:
    *return_ptr = (VPTR) scalar_Exp (e_data (M));
    break;

  case MATRIX:
    *return_ptr = (VPTR) matrix_Exp (e_data (M));
    break;

  default:
    remove_tmp_destroy (M);
    error_1 ("exp: invalid argument", 0);
    break;
  }

  remove_tmp_destroy (M);
  return;
}

/* **************************************************************
 * Square Root function.
 * ************************************************************** */
void
Sqrt (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *M;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("sqrt: 1 argument allowed", 0);

  M = bltin_get_numeric ("sqrt", d_arg, 1);
  switch (e_type (M))
  {
  case SCALAR:
    *return_ptr = (VPTR) scalar_Sqrt (e_data (M));
    break;

  case MATRIX:
    *return_ptr = (VPTR) matrix_Sqrt (e_data (M));
    break;

  default:
    remove_tmp_destroy (M);
    error_1 ("sqrt: invalid argument", 0);
    break;
  }

  remove_tmp_destroy (M);
  return;
}

/* **************************************************************
 * Integer function, cast a double to an int, back to double.
 * Kind of like an int-filter.
 * ************************************************************** */
void
Int (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *M;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("int: 1 argument allowed", 0);

  M = bltin_get_numeric ("int", d_arg, 1);
  switch (e_type (M))
  {
  case SCALAR:
    *return_ptr = (VPTR) scalar_Int (e_data (M));
    break;

  case MATRIX:
    *return_ptr = (VPTR) matrix_Int (e_data (M));
    break;

  default:
    remove_tmp_destroy (M);
    error_1 ("int: invalid argument", 0);
    break;
  }

  remove_tmp_destroy (M);
  return;
}

/* **************************************************************
 * ceil function. Smallest integer not less than x, as a double.
 * ************************************************************** */
void
Ceil (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *M;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("ceil: 1 argument allowed", 0);

  M = bltin_get_numeric ("ceil", d_arg, 1);
  switch (e_type (M))
  {
  case SCALAR:
    *return_ptr = (VPTR) scalar_Ceil (e_data (M));
    break;

  case MATRIX:
    *return_ptr = (VPTR) matrix_ElOp (e_data (M), ceil, "ceil");
    break;
    
  default:
    remove_tmp_destroy (M);
    error_1 ("ceil: invalid argument", 0);
    break;
  }

  remove_tmp_destroy (M);
  return;
}

/* **************************************************************
 * floor function. Largest integer not greater than x, as a double.
 * ************************************************************** */
void
Floor (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *M;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("floor: 1 argument allowed", 0);

  /* get args from list */

  M = bltin_get_numeric ("floor", d_arg, 1);
  switch (e_type (M))
  {
  case SCALAR:
    *return_ptr = (VPTR) scalar_Floor (e_data (M));
    break;

  case MATRIX:
    *return_ptr = (VPTR) matrix_ElOp (e_data (M), floor, "floor");
    break;
    
  default:
    remove_tmp_destroy (M);
    error_1 ("floor: invalid argument", 0);
    break;
  }
  
  remove_tmp_destroy (M);
  return;
}

/* **************************************************************
 * Round a double.
 * ************************************************************** */
void
Round (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *M;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("round: 1 argument allowed", 0);

  M = bltin_get_numeric ("round", d_arg, 1);
  switch (e_type (M))
  {
  case SCALAR:
    *return_ptr = (VPTR) scalar_Round (e_data (M));
    break;

  case MATRIX:
    *return_ptr = (VPTR) matrix_ElOp (e_data (M), rint, "rint");
    break;
    
  default:
    remove_tmp_destroy (M);
    error_1 ("round: invalid argument", 0);
    break;
  }
  
  remove_tmp_destroy (M);
  return;
}

/* **************************************************************
 * Compute an inverse.
 * ************************************************************** */
void
Inv (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  ListNode *M;

  /* Check n_args */
  if (n_args != 1)
    error_1 ("inv: 1 argument allowed", 0);

  M = bltin_get_numeric_matrix ("inv", d_arg, 1);
  *return_ptr = (VPTR) matrix_Inverse (e_data (M));
  remove_tmp_destroy (M);
  return;
}

/* **************************************************************
 * Solve a set of equations
 * ************************************************************** */

extern int matrix_is_symm _PROTO ((Matrix *m));

void
Solve (return_ptr, n_args, d_arg)
     VPTR *return_ptr;
     int n_args;
     Datum *d_arg;
{
  char *str;
  int flag;
  Matrix *sol;
  ListNode *A, *B;
  ListNode *TYPE = 0;

  /* Check n_args */
  if (!(n_args == 2 || n_args == 3))
  {
    clean_bltin_args (d_arg, n_args);
    error_1 ("solve: 2 or 3 arguments allowed", 0);
  }

  /* get args from list */
  
  A = bltin_get_numeric_matrix ("solve", d_arg, 1);
  B = bltin_get_numeric_matrix ("solve", d_arg, 2);

  flag = 0;
  if (n_args == 3)
  {
    TYPE = bltin_get_string ("solve", d_arg, 3);
    str = string_GetString (e_data (TYPE));
    if (!strncmp ("s", str, 1))
      flag = 2;
    else if (!strncmp ("S", str, 1))
      flag = 2;
    else if (!strncmp ("g", str, 1))
      flag = 1;
    else if (!strncmp ("G", str, 1))
      flag = 1;
    else
    {
      clean_bltin_args (d_arg, n_args);
      error_1 ("solve: Invalid 3rd argument", 0);
    }
  }

  if (flag == 1)
  {
    sol = matrix_solve_ge (e_data (A), e_data (B));
  }
  else if (flag == 2)
  {
    sol = matrix_solve_sy (e_data (A), e_data (B));
  }
  else
  {
    if (matrix_is_symm (e_data (A)))
      sol = matrix_solve_sy (e_data (A), e_data (B));
    else
      sol = matrix_solve_ge (e_data (A), e_data (B));
  }

  remove_tmp_destroy (A);
  remove_tmp_destroy (B);
  if (n_args == 3)
    remove_tmp_destroy (TYPE);

  *return_ptr = (VPTR) sol;
}
