 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 ;;                                                                  ;;
 ;;                                                                  ;;
 ;;         Code to add structure to a latexinfo document.           ;;
 ;;                                                                  ;;
 ;;                 Comments to Richard Caley.                       ;;
 ;;                     (rjc@cstr.ed.ac.uk)                          ;;
 ;;                                                                  ;;
 ;;                                                                  ;;
 ;; Some code to do the hard work of making the \node entries and    ;;
 ;; menus for latexinfo automagically. It understands comments       ;;
 ;; placed below sectioning commands to be the text to be inserted   ;;
 ;; into the menu. Thus:                                             ;;
 ;;                                                                  ;;
 ;;     \subsection{Tweaking Dongles}                                ;;
 ;;         \c How to tweak dongles for fun and profit.              ;;
 ;;                                                                  ;;
 ;; Would produce a menu entry like:                                 ;;
 ;;                                                                  ;;
 ;;  * Tweaking Dongles:: How to tweak dongles for fun and profit.   ;;
 ;;                                                                  ;;
 ;; This code ignores the first \node command in a file since it     ;;
 ;; does not know how you want it connected to other info files.     ;;
 ;; You will have to create that one yourself.                       ;;
 ;;                                                                  ;;
 ;; Obviously this doesn't do all of the work for you, for           ;;
 ;; instance it doesn't do anything about cross references.          ;;
 ;; However it is a good way of getting an initial latexinfo file    ;;
 ;; from which you can go on to do things properly. My strategy is   ;;
 ;; to do a rough draft almost as if writing a normal LaTeX          ;;
 ;; document, but using short section headings suitable for node     ;;
 ;; names. Then create the basic structure with latexinfo-structure  ;;
 ;; and work on from there.                                          ;;
 ;;                                                                  ;;
 ;; I think it is fairly stable, but since it has only had to eat    ;;
 ;; my LaTeX habbits there may be things it doesn't understand.      ;;
 ;;                                                                  ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; latexinfo-remove-structure added by Robert Olson, 3/30/92.
;;


(defvar latexinfo-section-commands
	'("-Heading-"
          "chapter"
	  "section"
	  "subsection"
	  "subsubsection"))

(defun latexinfo-looking-at-level ()
       "Returns the level number of the section command we are looking at, 
if any"

       (save-excursion
	(if (not (equal (char-after (point)) ?\\ ))
	    nil
	    (let ((l latexinfo-section-commands)
		  (i 1)
		  (c nil)
		  )
		 (forward-char 1)
		
		 (while (and (not c) l)
			(if (looking-at (car l))
			    (setq c i)
			    )
			(setq i (1+ i))
			(setq l (cdr l))
			)
		 c
		 )
	    )
	)
       )

(defun latexinfo-extract-name ()

       "extract the name from the section command here."

       (save-excursion
	(if (re-search-forward "{\\([^}]*\\)}" nil t)
	    (buffer-substring (match-beginning 1) (match-end 1))
	    "none"
	    )
	)
       )

(defun latexinfo-extract-commentry ()

       "Extract any comment following the current line."

       (save-excursion
	(let ((c "") s)
	     (forward-line 1)
	     (beginning-of-line)
	     (while (eolp)
		    (forward-line 1)
		    )
	
	     (while (looking-at "[ \t]*\\\\c[ \t]")
		    (re-search-forward "[ \t]*\\\\c[ \t]+" nil t)
		    (setq c (concat c " " (buffer-substring (point) 
							(save-excursion
							 (end-of-line)
							 (point)))))
		    (forward-line 1)
		    (beginning-of-line)
		    )
	     c
	     )
	)
       )
	
	       
		       
(defun latexinfo-build-section-subtree (depth)
       "Build a tree representing the structure of the
document as given by the sectioning commands."

       (let ((subtree nil) 
	     (finished nil)
	     it)

	    (while (and (not finished) (not (eobp)))
		    (if (setq it (latexinfo-looking-at-level))
			(if (<= it depth)
			    (setq finished t)
			    (setq subtree 
				  (cons
				   (list (set-marker (make-marker) (point))
					 (line-number (point))
					 (latexinfo-extract-name)
					 (latexinfo-extract-commentry)
					 (progn
					  (forward-line 1)
					  (latexinfo-build-section-subtree it)
					  )
					 )
				   subtree))
			    )
			(forward-line 1)
			)
		    )
	    (reverse subtree)
	    )
       )

(defun latexinfo-build-section-tree ()
       "Build a tree describing the section structure of BUFFER."

       (save-excursion
	(beginning-of-buffer)
	(latexinfo-build-section-subtree 0)
	)
       )

(defun latexinfo-insert-structure (nodes parent)
       "Insert the \\node commands into the buffer."

       (save-excursion
	(let (nd
	      (nds nodes)
	      (n nil)
	      (p "\t")
	      (children nil)
	      m
	      )
	(while nds
	       (setq nd (car nds))
	       (setq n (if (cdr nds)
			   (car (cdr nds))
			   nil))
	       (goto-char (car nd))
	       (open-line 2)
	       (insert "\\node "
		       (nth 2 nd) "," 
		       (if n (nth 2 n) "\t") ","
		       p ","
		       parent)
	       (setq children (cons (cons (nth 2 nd) (nth 3 nd)) children))
	       (latexinfo-insert-structure (nth 4 nd) (nth 2 nd))
	       (setq p (nth 2 nd))
	       (setq nds (cdr nds))
	       )
	(if children
	    (progn
	     (setq children (reverse children))
	     (goto-char (car (car nodes)))
	     (forward-line -2)
	     (open-line 2)
	     (insert "\\begin{menu}\n")
	     (while children
		    (insert "* " (car (car children)) 
			    "::") 
		    (insert-char ? (- 20 (length (car (car children)))))
		    (insert (cdr (car children)) "\n")
		    (setq children (cdr children))
		    )
	     (insert "\\end{menu}")
	     )
	    )
	)
	)
       )

(defun latexinfo-delete-structure (buffer)
       "Remove node and meny commands from a buffer."

       (interactive "bDelete structure from which buffer? ")

       (save-excursion
	(set-buffer buffer)
	(beginning-of-buffer)
	(re-search-forward "\\\\node\\|\\\\begin{menu}\\|\\\\end{menu}" nil t)
	(beginning-of-line)
	(if (looking-at "\\\\node")
	    (progn
	     (forward-line 1)
	    (re-search-forward "\\\\node\\|\\\\begin{menu}\\|\\\\end{menu}" nil t)
	    )
	    )
	(beginning-of-line)
	(while (re-search-forward "\\\\node\\|\\\\begin{menu}\\|\\\\end{menu}" nil t)
	       (beginning-of-line)
	       (if (looking-at "\\\\begin{menu}")
		   (while (not (looking-at "\\\\end{menu}"))
			  (kill-line 1)
			  )
		   )
	       (kill-line 1)
	       (delete-blank-lines)
	       )
	)
       )
       

		

(defun latexinfo-structure (buffer)
       "Insert a basic structure into the document based
on the sectioning commands."

       (interactive "bAdd structure to which buffer? ")

       (save-excursion
	(set-buffer buffer)
	(if (or  (re-search-forward "\\\\begin{menu}\\|\\\\end{menu}" nil t)
		 (and 
		  (search-forward "\\node" nil t)
		  (search-forward "\\node" nil t)))
	    (error "There is already structure present")
	    )
	(let ( tree )
	     (setq tree (latexinfo-build-section-tree))
	     (latexinfo-insert-structure tree "Top")
	     )
	)
       )

(defun line-number (p)
  (count-lines (point-min) p))

(defun latexinfo-remove-structure ()
  (interactive)
  (undo-boundary)
  (save-excursion
    (let (p
	  (deleting-from nil))
      (goto-char (point-min))
      (re-search-forward "\\node\\s-+Top")
      (forward-line 1)
      (while (not (eobp))
	(cond ((looking-at "^\\\\node")
	       (setq p (point))
	       (forward-line 1)
	       (kill-region p (point))
	       (if (looking-at "^\\s-*$")
		   (progn
		     (kill-line)
		     (forward-line 1))))
	      ((looking-at "^\\\\begin{menu}")
	       (setq deleting-from (point))
	       (forward-line 1))
	      ((and (looking-at "^\\\\end{menu}") deleting-from)
	       (forward-line 1)
	       (kill-region deleting-from (point))
	       (setq deleting-from nil))
	      (t
		 (forward-line 1)))))))

