/*
   This is an emulation library for picl, allowing picl programs to 
   use COMM as an interface.  However, a few things must be changed:
   
   The program initialization must use the PICall mechanism

   Also, not all routines are implemented; just the ones that we needed
 */   

#include "tools.h"
#include "comm/comm.h"
#include "system/system.h"
extern void PIdmax();

/*D
    PICL2COMM - Using Chameleon to run programs written using PICL

    Description:
    Chameleon provides a backward compatibility mode for the PICL 
    message-passing routines.  By linking with the appropriate file, many
    PICL programs can be run with few changes.

    The only change that you must make to the source code is to make the 
    program hostless and use the PICall interface.  Normally, this involves 
    only a few simple changes to main program.

    The other change is to insert the appropriate compatibilty interface
    module in the link line ahead of the Chameleon libraries.  The name of
    the interface file is ``picl2comm$(COMM).a'', where $(COMM) is the usual
    COMM variable (equal either to "", "p4", or "pvm").  For example,
    this makefile fragment links the PICL program in pi.f with Chameleon
$
$   LDIR = /usr/local/tools.core/libs/libsg/$(ARCH)
$   $(FLINKER) -g -o pi pi.f $(LDIR)/picl2comm$(COMM).a \
$                 $(LDIR)/tools$(COMM).a $(LDIR)/tools.a \
$                 $(LDIR)/system.a $(CLIB)
$
    As usual, symbol CLIB is defined by the appropriate bmake include and ARCH 
    is the architecture.

    Notes:
    Not all (or even much) of PICL is supported.  The supported routines 
    include who0, ginv0, gray0, send0, recv0, recvbegin0, recvend0, probe0, 
    recvinfo0, barrier0, sync0, bcast0, clock0, check0, gmax0, gmin0,
    gsum0, open0, and host0.

    In addition, stubs (that do nothing but allow you to link an application) 
    are provided for setarc0, close0, clocksync0, tracelevel, 
    tracenode, traceblockbegin, traceblockend, traceexit, tracefiles, and
    traceflush.

    For compatibility with codes written for the Intel ipsc/860 and Delta, 
    the intel/NX routines mynode, mypid, mclock, gdlow, and gdhigh are 
    also supported.

    One of the major features of PICL is its support of extensive tracing of 
    operations.  Chameleon provides a subset of those events.  Use the 
    command-line switches ``-event -blogfmat picl'' to produce a PICL-format
    trace file.
   
    Rs6000 limitations:
    Because of the design of the C and Fortran compilers on the IBM rs6000's, 
    it is impossible to generate a single interface file that works with both
    languages (the design of Chameleon allows this, but PICL's design does 
    not).  By default, the interface file FOR THE RS6000s ONLY is built ONLY
    for Fortran.  To build the C version INSTEAD, define BUILDPICLC when
    compiling the interface file.  That is, use

$    xlc -Drs6000 -DBUILDPICLC -D_POSIX_SOURCE -g -Dp4 \
$                           -I/usr/local/tools.core picl2comm.c

    to build the p4 version for C programs.
D*/

/* The intel versions manually remove the ``forcetype'' types and switch to 
   the forcetype code.  We should provide an option to assume ``intel source''
   that ALWAYS checks for forcetypes and removes the type information

   Another problem is that the PICL names are monocase, preventing their use
   from both C and Fortran on IBM RS6000's.  Possible solutions:

   Define a Cname and a Fortran name; then use a switch to build EITHER a
   Fortran version or a C version of this library.  The Fortran code must
   NOT call the C versions of the emulation routines (have them call a common
   version in that case).

 */

/* 
   If neither FORTRANUNDERSCORE or FORTRANCAPS is defined, 
   we can NOT distinguish between the Fortran and C versions,
   so we have to decide whether we are generating the Fortran or the C
   version.  So far, only the rs6000 requires this version.  In this case,
   we look for the ADDITIONAL define: BUILDPICLC .  In this case, 
   send0 etc will be Fortran versions; otherwise, they will be C versions.
*/
#if !defined(FORTRANUNDERSCORE) && !defined(FORTRANCAPS)
#ifndef BUILDPICLC
#define PICLFORTRANROUTINES
#else
#define PICLCROUTINES
#endif /* BUILDPICLC */
#else
#define PICLCROUTINES
#define PICLFORTRANROUTINES
#endif /* !defined(FORTRANUNDERSCORE) && !defined(FORTRANCAPS) */

#if defined(PICLCROUTINES)
#define PCsend0    send0
#define PCrecv0    recv0 
#define PCprobe0   probe0
#define PCrecvinfo0 recvinfo0
#define PCbarrier  barrier0
#define PCsync0    sync0
#define PCbcast0   bcast0
#define PCclock0   clock0
#define PCcheck0   check0
#define PCgmax0    gmax0
#define PCgmin0    gmin0
#define PCgsum0    gsum0
#define PCginv0    ginv0
#define PCgray0    gray0
#define PCsetarc0  setarc0
#define PCclose0   close0
#define PCopen0    open0
#define PChost0    host0
#define PCwho0     who0
#define PCclocksync0 clocksync0
#define PCtracelevel tracelevel
#define PCtracemarks tracemarks
#define PCtracenode  tracenode
#define PCtraceblockend traceblockend
#define PCtraceblockbegin traceblockbegin
#define PCtraceexit traceexit
#define PCtracefiles tracefiles
#define PCtraceflush traceflush
#define PCmynode     mynode
#define PCmypid      mypid
#define PCmclock     mclock
#define PCgdlow      gdlow
#define PCgdhigh     gdhigh
#define PCkillproc   killproc
#define PCrecvbegin0 recvbegin0
#define PCrecvend0   recvend0
#define PCsendbegin0 sendbegin0
#define PCsendend0   sendend0
#endif
#if defined(PICLFORTRANROUTINES)
#define PFsend0     send0_
#define PFrecv0     recv0_
#define PFprobe0    probe0_
#define PFrecvinfo0 recvinfo0_
#define PFbarrier0  barrier0_
#define PFsync0     sync0_
#define PFbcast0    bcast0_
#define PFclock0    clock0_
#define PFcheck0    check0_
#define PFgsum0     gsum0_
#define PFgmax0     gmax0_
#define PFgmin0     gmin0_
#define PFginv0     ginv0_
#define PFgray0     gray0_
#define PFsetarc0   setarc0_
#define PFclose0    close0_
#define PFopen0     open0_
#define PFhost0     host0_
#define PFwho0      who0_
#define PFclocksync0 clocksync0_
#define PFtracelevel tracelevel_
#define PFtracemarks tracemarks_
#define PFtraceblockend traceblockend_
#define PFtraceblockbegin traceblockbegin_
#define PFtraceexit traceexit_
#define PFtracenode tracenode_
#define PFtracefiles tracefiles_
#define PFtraceflush traceflush_
#define PFmynode     mynode_
#define PFmypid      mypid_
#define PFmclock     mclock_
#define PFgdlow      gdlow_
#define PFgdhigh     gdhigh_
#define PFkillproc   killproc_
#define PFrecvbegin0 recvbegin0_
#define PFrecvend0   recvend0_
#define PFsendbegin0 sendbegin0_
#define PFsendend0   sendend0_
#endif

void PCsend0(a,b,c,d)
int b, c, d;
char *a;
{
/* A destination of -1 is a send to all.  Because we don't tag the message, 
   there is NO way to implement this other than a loop over all nodes.
   There are subtle reasons why the send/recv broadcast is inferior to a 
   special-purpose broadcast that have to do with nodes terminating before they
   handle their part of the broadcast
 */
if (d < 0) {
    int dest;
    for (dest=0; dest<PInumtids; dest++) {
	if (dest != PImytid)
	    PCsend0(a,b,c,dest);
	}
    return;
    }

#ifdef intelnx
if (c > 0 && (c & 0x40000000)) {
    c = c ^ 0x40000000;
    PIbsendrr(c,a,b,d,MSG_OTHER);
    }
else {
    PIbsend(c,a,b,d,MSG_OTHER);
    }
#else
PIbsend(c,a,b,d,MSG_OTHER);
#endif
}

void PCrecv0(a,b,c)
int b, c;
char *a;
{
#ifdef intelnx
if (c > 0 && (c & 0x40000000)) {
    c = c ^ 0x40000000;
    PIbrecvrr(c,a,b,MSG_OTHER);
    }
else {
    PIbrecv(c,a,b,MSG_OTHER);
    }
#else
PIbrecv(c,a,b,MSG_OTHER);
#endif
}

int PCprobe0(a)
int a;
{
return PInprobe(a);
}

void PCrecvinfo0(a,b,c) 
int *a, *b, *c;
{
*a = RECVLEN(); *b = RECVTYPE(); *c = RECVFROM(); 
}

void PCbarrier0()
{
PIgsync(ALLPROCS);
}

void PCsync0()
{
PIgsync(ALLPROCS);
}

void PCbcast0(a,b,c,d) 
char *a;
int  b, c, d;
{
GSCATTERSRC(a,b,d,ALLPROCS,MSG_OTHER);
}

double PCclock0()
{
return SYGetElapsedTime();
}

void PCcheck0()
{}

/* For these, I really need a half reduce (only to root) 
   e is the root */
void PCgmax0(a,b,c,d,e)
void *a;
int  b, c, d, e;
{
void (*op)();
int  elmsize, datatype;
extern void PIimax(), PIfmax(), PIdmax();
char   *work;

op = 0;
switch (c) {
    case 0: datatype = MSG_OTHER; elmsize = 1; 
            break;
    case 1: datatype = MSG_INT;   elmsize = sizeof(short); 
            break;
    case 2: datatype = MSG_INT;   elmsize = sizeof(int);    op = PIimax;
            break;
    case 3: datatype = MSG_LNG;   elmsize = sizeof(long);   op = PIimax;
            break;
    case 4: datatype = MSG_FLT;   elmsize = sizeof(float);  op = PIfmax;
            break;
    case 5: datatype = MSG_DBL;   elmsize = sizeof(double); op = PIdmax;
            break;
    }
work = (char *)MALLOC( elmsize * b );   CHKPTR(work);
gsetopHalfT( a, b, work, ALLPROCS, elmsize, datatype, e, op );
FREE(work);
}

void PCgmin0(a,b,c,d,e)
void *a;
int  b, c, d, e;
{
void (*op)();
int  elmsize, datatype;
extern void PIimin(), PIfmin(), PIdmin();
char   *work;

op = 0;
switch (c) {
    case 0: datatype = MSG_OTHER; elmsize = 1; 
            break;
    case 1: datatype = MSG_INT;   elmsize = sizeof(short); 
            break;
    case 2: datatype = MSG_INT;   elmsize = sizeof(int);    op = PIimin;
            break;
    case 3: datatype = MSG_LNG;   elmsize = sizeof(long);   op = PIimin;
            break;
    case 4: datatype = MSG_FLT;   elmsize = sizeof(float);  op = PIfmin;
            break;
    case 5: datatype = MSG_DBL;   elmsize = sizeof(double); op = PIdmin;
            break;
    }
work = (char *)MALLOC( elmsize * b );   CHKPTR(work);
gsetopHalfT( a, b, work, ALLPROCS, elmsize, datatype, e, op );
FREE(work);
}

void PCgsum0(a,b,c,d,e)
void *a;
int  b, c, d, e;
{
void (*op)();
int  elmsize, datatype;
extern void PIisum(), PIfsum(), PIdsum();
char *work;

op = 0;
switch (c) {
    case 0: datatype = MSG_OTHER; elmsize = 1; 
            break;
    case 1: datatype = MSG_INT;   elmsize = sizeof(short); 
            break;
    case 2: datatype = MSG_INT;   elmsize = sizeof(int);    op = PIisum;
            break;
    case 3: datatype = MSG_LNG;   elmsize = sizeof(long);   op = PIisum;
            break;
    case 4: datatype = MSG_FLT;   elmsize = sizeof(float);  op = PIfsum;
            break;
    case 5: datatype = MSG_DBL;   elmsize = sizeof(double); op = PIdsum;
            break;
    }
work = (char *)MALLOC( elmsize * b );   CHKPTR(work);
gsetopHalfT( a, b, work, ALLPROCS, elmsize, datatype, e, op );
FREE(work);
}

static int ngray = -1;
int PCginv0( i )
int i;
{
if (ngray == -1) {
    int mask = 0x1, np = PInumtids, bits=0;
    while (mask < np) { mask <<= 1; bits++; }
    PIinitGray( bits );
    ngray  = bits;
    }
return PIinvGray( i );
}
int PCgray0( i )
int i;
{
if (ngray == -1) {
    int mask = 0x1, np = PInumtids, bits=0;
    while (mask < np) { mask <<= 1; bits++; }
    PIinitGray( bits );
    ngray  = bits;
    }
return PIGray( i );
}

/* Environment calls.  We ignore these; insert a PICall to use this package */
void PCsetarc0(i1,i2,i3,i4)
int i1,i2,i3,i4;
{}
void PCclose0(i1)
int i1;
{}
void PCopen0(i1,i2,i3)
int *i1,*i2,*i3;
{
*i1 = PInumtids;
*i2 = PImytid;
*i3 = -1;
}
int PChost0()
{
return 0;
}

void PCwho0( np, myid, host )
int *np, *myid, *host;
{
*np   = PInumtids;
*myid = PImytid;
*host = -1;        /* Never a host */
}
void PCclocksync0(i1)
int i1;
{}

/* Tracing calls */
void PCtracelevel( event, csstats, cmstats )
int event, csstats, cmstats;
{}
void PCtracemarks( markarray, size )
int *markarray, size;
{}

void PCtracenode( tsize, tflush, tsync )
int tsize, tflush, tsync;
{}

void PCtraceblockend( i1, i2, i3)
int i1, i2, i3;
{}

void PCtraceblockbegin( i1, i2, i3 )
int i1, i2, i3;
{}

void PCtraceexit()
{}

void PCtracefiles(s1,s2,i)
char *s1, *s2;
int  i;
{}

void PCtraceflush()
{}

/* 
   Fortran versions.  At this point, if the Fortran versions are being
   defined, then PFsend0 -> send0_.  See if we need to further define
   these (say by changing them to all caps or remove the trailing underscore)
 */ 
#if defined(FORTRANCAPS)
#define send0_      SEND0
#define recv0_      RECV0
#define probe0_     PROBE0
#define recvinfo0_   RECVINFO0
#define barrier0_   BARRIER0
#define sync0_      SYNC0
#define bcast0_     BCAST0
#define clock0_     CLOCK0
#define check0_     CHECK0
#define gsum0_      GSUM0
#define gmax0_      GMAX0
#define gmin0_      GMIN0
#define ginv0_      GINV0
#define gray0_      GRAY0
#define setarc0_    SETARC0
#define close0_     CLOSE0
#define open0_      OPEN0
#define host0_      HOST0
#define who0_       WHO0
#define clocksync0_ CLOCKSYNC0
#define tracelevel_ TRACELEVEL
#define tracemarks_ TRACEMARKS
#define traceblockend_ TRACEBLOCKEND
#define traceblockbegin_  TRACEBLOCKBEGIN
#define traceexit_  TRACEEXIT
#define tracenode_  TRACENODE
#define tracefiles_ TRACEFILES
#define traceflush_ TRACEFLUSH
#define recvbegin0_ RECVBEGIN0
#define recvend0_   RECVEND0
#define sendbegin0_ SENDBEGIN0
#define sendend0_   SENDEND0
#define mynode_     MYNODE
#define mypid_      MYPID
#define mclock_     MCLOCK
#define gdlow_      GDLOW
#define gdhigh_     GDHIGH
#define killproc_   KILLPROC
#elif !defined(FORTRANUNDERSCORE)
#define send0_      send0
#define recv0_      recv0
#define probe0_     probe0
#define recvinfo0_   recvinfo0
#define barrier0_   barrier0
#define sync0_      sync0
#define bcast0_     bcast0
#define clock0_     clock0
#define check0_     check0
#define gsum0_      gsum0
#define gmax0_      gmax0
#define gmin0_      gmin0
#define ginv0_      ginv0
#define gray0_      gray0
#define setarc0_    setarc0
#define close0_     close0
#define open0_      open0
#define host0_      host0
#define who0_       who0
#define clocksync0_ clocksync0
#define tracelevel_ tracelevel
#define tracemarks_ tracemarks
#define traceblockend_ traceblockend
#define traceblockbegin_  traceblockbegin
#define traceexit_  traceexit
#define tracenode_  tracenode
#define tracefiles_ tracefiles
#define traceflush_ traceflush
#define recvbegin0_ recvbegin0
#define recvend0_   recvend0
#define sendbegin0_ sendbegin0
#define sendend0_   sendend0
#define mynode_     mynode
#define mypid_      mypid
#define mclock_     mclock
#define gdlow_      gdlow
#define gdhigh_     gdhigh
#define killproc_   killproc
#endif

void PFsend0(a,b,c,d)
int *b, *c, *d;
char *a;
{
PIbsend(*c,a,*b,*d,MSG_OTHER);
}

void PFrecv0(a,b,c)
int  *b, *c;
char *a;
{
PIbrecv(*c,a,*b,MSG_OTHER);
}

int PFprobe0(a)
int *a;
{
return PInprobe(*a);
}

void PFrecvinfo0(a,b,c) 
int *a, *b, *c;
{*a = RECVLEN(); *b = RECVTYPE(); *c = RECVFROM(); 
}

void PFbarrier0()
{
PIgsync(ALLPROCS);
}

void PFsync0()
{
PIgsync(ALLPROCS);
}

void PFbcast0(a,b,c,d) 
char *a;
int  *b, *c, *d;
{
GSCATTER(a,*b,(*d)==MYPROCID,ALLPROCS,MSG_OTHER);
}

double PFclock0()
{
return SYGetElapsedTime();
}

void PFcheck0()
{}

void PFgsum0(a,b,c,d,e)
void *a;
int  *b, *c, *d, *e;
{
PCgsum0(a,*b,*c,*d,*e);
}

void PFgmax0(a,b,c,d,e)
void *a;
int  *b, *c, *d, *e;
{
PCgmax0(a,*b,*c,*d,*e);
}

void PFgmin0(a,b,c,d,e)
void *a;
int  *b, *c, *d, *e;
{
PCgmin0(a,*b,*c,*d,*e);
}

int PFginv0( i )
int *i;
{
return PCginv0( *i );
}
int PFgray0( i )
int *i;
{
return PCgray0( *i );
}
void PFsetarc0(i1,i2,i3,i4)
int *i1,*i2,*i3,*i4;
{
PCsetarc0( *i1, *i2, *i3, *i4);
}
void PFclose0(i1)
int *i1;
{
PCclose0(*i1);
}
void PFopen0(i1,i2,i3)
int *i1,*i2,*i3;
{
PCopen0( i1, i2, i3 );
}
int PFhost0()
{
return PChost0();
}
void PFwho0(i1,i2,i3)
int *i1,*i2,*i3;
{
PCwho0( i1, i2, i3 );
}
void PFclocksync0(i1)
int *i1;
{
PCclocksync0(*i1);
}

void PFtracelevel( event, csstats, cmstats )
int *event, *csstats, *cmstats;
{
PCtracelevel( *event, *csstats, *cmstats );
}
void PFtracemarks( markarray, size )
int *markarray, *size;
{
PCtracemarks( markarray, *size );
}

void PFtraceblockend( i1, i2, i3 )
int *i1, *i2, *i3;
{
PCtraceblockend( *i1, *i2, *i3 );
}
void PFtraceblockbegin( i1, i2, i3 )
int *i1, *i2, *i3;
{
PCtraceblockbegin( *i1, *i2, *i3 );
}
void PFtraceexit()
{
PCtraceexit();
}
void PFtracenode( tsize, tflush, tsync )
int *tsize, *tflush, *tsync;
{
PCtracenode( *tsize, *tflush, *tsync );
}
void PFtracefiles(s1,s2,i,d1,d2)
char *s1, *s2;
int  *i, d1, d2;
{
PCtracefiles( s1, s2, *i );
}
void PFtraceflush()
{
PCtraceflush();
}

#if !defined(intelnx) 
/* 
   The following are routines provided temporarily to assist programs
   written for the i860 and picl that, incorrectly, use i860 routines
 */
int PCmynode()
{
return PImytid;
}
int PCmypid()
{
return 0;
}
int PCmclock()
{
return (int)(SYGetCPUTime() * 1000.0);
}
void PCgdlow( val, n, buf )
double *val, *buf;
int    n;
{
GDMIN(val,n,buf,ALLPROCS);
}
void PCgdhigh( val, n, buf )    
double *val, *buf;
int    n;
{
GDMAX(val,n,buf,ALLPROCS);
}
void PCkillproc(node,pid)
int node, pid;
{
}
int PFmynode()
{
return MYPROCID;
}
int PFmypid()
{
return 0;
}
int PFmclock()
{
return (int)(SYGetCPUTime() * 1000.0);
}
void PFgdlow( val, n, buf )
double *val, *buf;
int    *n;
{
GDMIN(val,*n,buf,ALLPROCS);
}
void PFgdhigh( val, n, buf )    
double *val, *buf;
int    *n;
{
GDMAX(val,*n,buf,ALLPROCS);
}
void PFkillproc(node,pid)
int *node, *pid;
{
}
#endif

/* 
   The following are undocumented PICL routines for asynchronous IO.
   These are a problem, since they don't make it easy to manage asynchronous
   io.  There is, for example, no context.  According to the documentation in
   upgrade.1.91, these use the tag value to discriminate between messages, so
   we have to insure that the tags are different.  To make this SIMPLE and
   relatively fast, we do the following:

   We keep a list of 129 id's (this is the same that PICL claims to keep).
   Each of these contains all that is necessary to manage the recv on systems
   with nonblocking receives.
 */
typedef struct _PICLtag {
    int             tag,
                    size, 
                    to;        /* For sends */
    char            *buf;
    PIRecvId_t   aid;
    struct _PICLtag *next;
    } PICLtag;
static PICLtag PTags[129], *avail, *head = 0;
/* We might want to add allocated elements to the TAIL to get the expected
   ordering behavior */
static int     PICLtagInit=1;

/* Insert a tag into the list */
static PICLtag *InsertTag( tag, aid )
int           tag;
ASYNCRecvId_t aid;
{
int i;
PICLtag *p;

if (PICLtagInit) {
    for (i=0; i<129; i++) 
	PTags[i].next = PTags + i + 1;
    PTags[128].next = 0;
    avail       = PTags;
    head        = 0;
    PICLtagInit = 0;
    }
if (!avail) {
    SETERRC(1,"Out of PICLtags");
    return 0;
    }
p       = avail;
avail   = avail->next;
p->tag  = tag;
p->aid  = aid;
p->next = head;
head    = p;
return p;
}

/* find and remove a tag from the list */
PICLtag *FindTag( tag )
int tag;
{
PICLtag *p, *pv;

p  = head;
pv = 0;
while (p) {
    if (p->tag == tag) {
	/* Found it.  Remove it */
	if (pv) pv->next = p->next;
	else    head     = p->next;
	p->next = avail;
	avail   = p;
	return p;
	}
    pv = p;
    p  = pv->next;
    }
return 0;
}

void PCrecvbegin0( buf, size, tag )
char *buf;
int  size, tag;
{
PICLtag       *p;
PIRecvId_t aid;

PInrecv( tag, buf, size, MSG_OTHER, aid ); 
p = InsertTag( tag, aid );
p->buf  = buf;
p->size = size;
}
void PCrecvend0( tag )
int tag;
{
PICLtag       *p;
PIRecvId_t aid;

p   = FindTag( tag );
if (!p) return;
aid = p->aid;
PIwrecv( tag, p->buf, p->size, MSG_OTHER, aid );
}

void PFrecvbegin0( buf, size, tag )
char *buf;
int  *size, *tag;
{
PCrecvbegin0( buf, *size, *tag );
}
void PFrecvend0( tag )
int *tag;
{
PCrecvend0( *tag );
}

/* Nonblocking send */
void PCsendbegin0( buf, size, tag, to )
char *buf;
int  size, tag, to;
{
PICLtag       *p;
PISendId_t aid;

if (to == -1) {
    /* Simulate a TYPED broadcast */
    int i, np = PInumtids;
    for (i=0; i<np; i++) {
	if (i != PImytid) 
	    PIbsend( tag, buf, size, i, MSG_OTHER );
	}
    return;
    }
PInsend( tag, buf, size, to, MSG_OTHER, aid ); 
p = InsertTag( tag, aid );
p->buf  = buf;
p->size = size;
p->to   = to;
}
void PCsendend0( tag )
int tag;
{
PICLtag       *p;
PISendId_t aid;

p   = FindTag( tag );
if (!p) return;
aid = p->aid;
PIwsend( tag, p->buf, p->size, p->to, MSG_OTHER, aid );
}

void PFsendbegin0( buf, size, tag, to )
char *buf;
int  *size, *tag, *to;
{
PCsendbegin0( buf, *size, *tag, *to );
}
void PFsendend0( tag )
int *tag;
{
PCsendend0( *tag );
}
