/*
 * USC.C  (Source file for the Microsecond Clock package)
 *
 * Written by:  Arun Nanda    (07/17/91)
 *
 * Extensively modified by William Gropp (08/04/92).  Primarily, the value
 * returned by these routines has been converted to a 64bit "cookie"
 * (a magic value that can be converted into a time).  Macros to access the
 * clock are defined in system/time/usec.h.  Routines to convert the
 * cookie to a time are provided.
 *
 * One problem is that some timers roll-over on relatively short intervals.
 * There isn't much that can be done about this, except that the 
 * rollover time can be computed and applied when it looks like a 
 * rollover happened.
 *
 */

#include "tools.h"
#include "system/time/usecp.h"

unsigned long SYusc_MD_rollover_val = 0;

void SYusc_init()
{

#if defined(encore)

	usc_multimax_timer  = timer_init();
	SYusc_MD_rollover_val = (unsigned long) ((1<<SYusc_MD_timer_size)-1);

#elif (defined(balance) || defined(symmetry))

	unsigned long roll;

	usclk_init();

	roll = 1 << (SYusc_MD_timer_size-1);
	SYusc_MD_rollover_val = (unsigned long) (roll + roll - 1);

#elif defined(tc2000) || defined(intelnx)

	/* SY32toDBL = 1.0 +  (double)((unsigned long)0xffffffff);  */
	SY32toDBL = (1 << 16); 
	SY32toDBL = SY32toDBL * SY32toDBL;
#elif defined(cm5)
        CMMD_node_timer_clear( CM5TOOLSTIMER );
        CMMD_node_timer_start( CM5TOOLSTIMER );
#endif

}

/*ARGSUSED*/
void SYusc_MD_clock(a)
SYusc_time_t *a;
{
#if defined(USEGETTIMEOFDAY)
	struct timeval tp;

	gettimeofday(&tp,(struct timezone *)0);
	a->s1[0] = tp.tv_sec;
	a->s1[1] = tp.tv_usec;
#elif defined(cm5)
    extern double CMMD_node_timer_elapsed();
    CMMD_node_timer_stop( CM5TOOLSTIMER );
    a->s2 = CMMD_node_timer_elapsed( CM5TOOLSTIMER );
    CMMD_node_timer_start( CM5TOOLSTIMER );
#endif
}

/*ARGSUSED*/
/*@
    SYuscValue - Converts a fast usec clock value into seconds.

    Input Parameter:
.   a - usec timer structure

    Returns:
    time in seconds.
@*/
double SYuscValue( a )
SYusc_time_t *a;
{
#if defined(encore) || defined(balance) || defined(symmetry)
/* No correction for rollovers */
return (double) a->s1[0] * 1.0e-6;

#elif defined(tc2000)
return (((double)a->s1[0]) * SY32toDBL + a->s1[1]) * 1.0e-6;

#elif defined(intelnx)
return (((double)a->s1[1]) * SY32toDBL + a->s1[0]) / HWHZ;

#elif defined(USEGETTIMEOFDAY)

return (double)a->s1[0] + 1.0e-6 * a->s1[1];

#elif defined(cm5)
return a->s2;

#else

return 0.0;
#endif
}

/*ARGSUSED*/
/*@
  SYuscDiff - Returns the difference between two values.

    Input Parameters:
.   a1 - usec timer structure
.   a2 - usec timer structure

    Returns:
    Time in seconds between a2 and a1.  If a1 is null, this routine
    has the same effect as SYuscValue( a2 ).
@*/
double SYuscDiff( a1, a2 )
SYusc_time_t *a1, *a2;
{
int msecs;

if (!a1) {
    return SYuscValue( a2 );
    }
/* Must be careful here since s1 is unsigned, and the difference could
   be negative */
#if defined(encore) || defined(balance) || defined(symmetry)
if (a2->s1[0] < a1->s1[0]) {
    return (double)( a2->s1[0] - a1->s1[0] + SYusc_MD_rollover_val) * 1.0e-6;
    }
else {
    return (double)( a2->s1[0] - a1->s1[0] ) * 1.0e-6;
    }

#elif defined(tc2000)
if (a2->s1[1] < a1->s1[1]) {
    msecs = -(int)( a1->s1[1] - a2->s1[1] );
    }
else {
    msecs = (int)( a2->s1[1] - a1->s1[1] );
    }
return (double)(a2->s1[0] - a1->s1[0]) * SY32toDBL + msecs * 1.0e-6;

#elif defined(intelnx)
#ifndef FOO
/* The 64bit structure used by hwclock is struct{long slow, shigh;}. */

if (a2->s1[0] < a1->s1[0]) {
    msecs = -(int)( a1->s1[0] - a2->s1[0] );
    }
else {
    msecs = (int)( a2->s1[0] - a1->s1[0] );
    }
return (((double)(a2->s1[1] - a1->s1[1])) * SY32toDBL + msecs) / HWHZ; 
#else
esize_t diff, esub();
double fdiff;
printf( "About to do esub\n" );
printf( "(%x,%x) - (%x,%x)\n", a2->et.shigh,a2->et.slow,a1->et.shigh,a1->et.slow);
diff  = esub(a2->et,a1->et);
printf( "(%x,%x)\n", diff.shigh, diff.slow );
fdiff = ediv(diff,HWHZ);
fdiff += (double)emod(diff,HWHZ) / (double)HWHZ;
return fdiff;
#endif

#elif defined(USEGETTIMEOFDAY)

if (a2->s1[1] < a1->s1[1]) {
    msecs = -(int)( a1->s1[1] - a2->s1[1] );
    }
else {
    msecs = (int)( a2->s1[1] - a1->s1[1] );
    }
return ((double)(a2->s1[0] - a1->s1[0])) + 1.0e-6 * msecs;

#elif defined(cm5)

return a2->s2 - a1->s2;

#else

return 0.0;
#endif
}
