#ifndef lint
static char SCCSid[] = "@(#) ./sparse/mapping.c 07/23/93";
#endif

#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"

void Spimapmap( );

/*@
   SpSetMappingPtrs - Set the mapping of rows and columns for a matrix.

   Input Parameters:
.   B      - Matrix to map
.   rowmap - mapping of rows
.   colmap - mapping of columns
.   icolmap- inverse of colmap

   These should be pointers to storage that may be used by the Mapped Matrix.

   Note:
   This routine should NOT be used by the sparse routines themselves.
   The package remembers whether the user or the package set these pointers
   (to avoid some unnecessary duplication of storage).
 @*/
void SpSetMappingPtrs( B, rowmap, colmap, icolmap )
SpMat *B;
int   *rowmap, *colmap, *icolmap;
{
SPLITTOMAT(B);
if (!B->map) {
    B->map = NEW(Mapping); CHKPTR(B->map);
    }
B->map->rowmap  = rowmap;
B->map->colmap  = colmap;
B->map->icolmap = icolmap;
B->map->q_user_row  = rowmap  != 0;
B->map->q_user_col  = colmap  != 0;
B->map->q_user_icol = icolmap != 0;
}

/*+
   SpiSetMappingPtrs - Set the mapping of rows and columns for a matrix.

   Input Parameters:
.   B      - Matrix to map
.   rowmap - mapping of rows
.   colmap - mapping of columns
.   icolmap- inverse of colmap

   These should be pointers to storage that may be used by the Mapped Matrix.
+*/
void SpiSetMappingPtrs( B, rowmap, colmap, icolmap )
SpMat *B;
int   *rowmap, *colmap, *icolmap;
{
SPLITTOMAT(B);
if (!B->map) {
    B->map = NEW(Mapping); CHKPTR(B->map);
    }
B->map->rowmap  = rowmap;
B->map->colmap  = colmap;
B->map->icolmap = icolmap;
B->map->q_user_row  = 0;
B->map->q_user_col  = 0;
B->map->q_user_icol = 0;
}

/*@
  SpGetMappingPtrs - Get the mapping pointers for a sparse matrix.
  
  Input Parameter:
.  B - matrix to getting mapping pointers for

  Output Parameters:
.  rowmap, colmap, icolmap - pointers to pointers of maps.  If null,
                            that value is not returned.
 @*/
void SpGetMappingPtrs( B, rowmap, colmap, icolmap )
SpMat *B;
int   **rowmap, **colmap, **icolmap;
{
SPLITTOMAT(B);
if (!B->map) {SETERRC(ERR_NO_MAP,"No mapping in matrix"); return;}
if (rowmap)  *rowmap  = B->map->rowmap;
if (colmap)  *colmap  = B->map->colmap;
if (icolmap) *icolmap = B->map->icolmap;
}

/*@
  SpComposeMappings - Compose a mapping with an existing mapping. 

  May need to allocate map structure.
 @*/
void SpComposeMappings( mat, rowmap, columnmap )
SpMat  *mat;
int    *rowmap,*columnmap;
{
int n = mat->rows,*oldmap;

SPLITTOMAT(mat);

/* fill up the row mapping.  This isn't quite correct, as it needs to check
   for the copies and perhaps allocate new space. */
oldmap = mat->map->rowmap;
if (!oldmap) 
    {mat->map->rowmap = SpiDupMapping( n, rowmap ); CHKERR(ERR_NO_MEM);}
else
    {Spimapmap( mat->map->rowmap, rowmap, mat->map->rowmap, n ); CHKERR(1);}

/* do the column mapping */
Spimapmap( mat->map->colmap, columnmap,mat->map->colmap,mat->cols ); CHKERR(1);
}

/*
   Spimapmap - compose two integer mappings
 */
void Spimapmap( inmap, newmap, outmap, n )
int *inmap, *newmap, *outmap, n;
{
register int i;
for (i=0; i<n; i++) 
    outmap[i] = newmap[inmap[i]];
}

/*@
  SpCreateMap - Create a mapping structure. A mapping structure is
                simply a set of index sets which map the rows 
                (columns) of a sparse matrix onto a vector. This 
                is useful for two reasons. We can do reorderings
                for matrix factorizations, then the mappings are 
                simply permutations. We can also use these mappings
                to apply matrices to vectors with holes in them,
                for instance if the unknowns live on a rectangular 
                mesh from which a few nodes have been removed.
                
 @*/
Mapping *SpCreateMap( )
{
Mapping *B;

B = NEW(Mapping); CHKPTRV(B,0);
B->rowmap      = 0;
B->colmap      = 0;
B->icolmap     = 0;
B->q_user_row  = 0;
B->q_user_col  = 0;
B->q_user_icol = 0;

return B;
}

/*@
  SpDestroyMap - Destroy a mapping structure. 
 @*/
void SpDestroyMap( map )
Mapping *map;
{
if (!map->q_user_row  && map->rowmap)  SPFree( map->rowmap );
if (!map->q_user_col  && map->colmap)  SPFree( map->colmap );
if (!map->q_user_icol && map->icolmap) SPFree( map->icolmap );
FREE( map );
}

/*@
  SpCopyMappings - Copy the mapping structure.
 @*/
void SpCopyMappings( mat, rowmap, colmap, icolmap )
SpMat  *mat;
int    *rowmap,*colmap,*icolmap;
{
int m = mat->rows, n = mat->cols;
Mapping *map = mat->map;
  
if (!map) {
    map        = NEW(Mapping); CHKPTR(map);
    mat->map   = map;
    }
else {
    if (!map->q_user_row  && map->rowmap)  FREE(map->rowmap);
    if (!map->q_user_col  && map->colmap)  FREE(map->colmap);
    if (!map->q_user_icol && map->icolmap) FREE(map->icolmap);
    }

/* Initialize the mappings to null */
map->rowmap  = 0;
map->colmap  = 0;
map->icolmap = 0;
if (rowmap)
    map->rowmap  = SpiDupMapping( m, rowmap ); CHKERR(1);
if (colmap)
    map->colmap  = SpiDupMapping( n, colmap ); CHKERR(1);
if (icolmap)
    map->icolmap = SpiDupMapping( n, icolmap ); CHKERR(1);
map->q_user_row  = 0;
map->q_user_col  = 0;
map->q_user_icol = 0;
}

/*@
  SpForceColMap - Apply a column mapping directly to a matrix (leaving the
                  matrix in a non-canonical form). This may increase
                  the speed of a matrix vector multiply, since it 
                  removes one level of indirection.
                  
 @*/
void SpForceColMap( mat )
SpMat *mat;
{
int       i,j,*map;
int       *ix,M;
int       nz;

SPLITTOMAT(mat);
M = mat->rows;
if (!mat->map) return;
if (!mat->map->colmap) return;
map = mat->map->colmap;

/* fix the column indices */
for ( j=0; j<M; j++ ) {
    SpScatterFromRow( mat, j, &nz, &ix, (double **)0 );
    for ( i=0; i<nz; i++ ) {
        ix[i] = map[ix[i]];
	}
    /* BUG - if ix is not the actual storage, need to gather it back */
    }
if (!mat->map->q_user_col) FREE(mat->map->colmap);
mat->map->colmap = 0;
}

/*+
   SpiDupMapping - duplicate a mapping.  If old is null, generate the
   identity mapping 
+*/
int *SpiDupMapping( n, old )
int n, *old;
{
int *map, i;

map = (int *)MALLOC( n * sizeof(int) ); CHKPTRV(map,0); 
if (old)
    MEMCPY( map, old, n*sizeof(int) );
else 
    for (i=0; i<n; i++) map[i] = i;
return map;
}

/*@
  SpApplyColMappingFcn - Apply a column mapping using a function.

  Input parameters:
. mat - matrix to apply mapping to
. fcn - function of form fnc(int *idx, int n, void *ctx)
. ctx - context to pass to fcn
@*/
void SpApplyColMappingFcn( mat, fcn, ctx )
SpMat *mat;
void  (*fcn)(), *ctx;
{
int       j;
int       *ix,M;
int       nz;

SPLITTOMAT(mat);
M  = mat->rows;
for ( j=0; j<M; j++ ) {
    SpScatterFromRow( mat, j, &nz, &ix, (double **)0 );
    (*fcn)( ix, nz, ctx );
    }
}
