/*
     Test of the nonlinear solvers code. This example solves a nonlinear
   ODE using a "truncated Newton with line search" or "trust region 
   Newton", in both cases the Jacobian system is solved approximately,
   with truncated Newton it is solved with a relative residual decrease 
   of trunctol; with trust region Newton it is solved until
   ||step|| < delta (diameter of trust region)

     The matrix systems are treated as sparse and solved using the SV
   solvers software.
*/
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "tools.h" 
#include "nonlin/nlall.h"
#include "sparse/spmat.h"
#include "solvers/svctx.h"

typedef struct {
  int      n;                  
  SpMat    *jacobian;
  SVctx    *svctx;
  SVMETHOD method;
} SparseCntx;

void         Function();
void         InitialGuess();
void         SetUpLinearSolve();
void         LinearSolve();
void         DestroyLinearSolve();

#define ERR {printf("Error %d %s \n",__LINE__,__FILE__); exit(0);}

main(Argc,Args)
int  Argc;
char *Args[];
{
  NLCntx     *neP;
  SparseCntx *usrP;
  int         n = 10, its, i, nfunc, nsteps, nvector;
  double     *x, *f, norm,xnorm;
  NLMETHOD   nlmethod = NLNEWTONLS;
  FILE       *fp = 0;

  SYSetFPTraps();
  if (SYArgHasName( &Argc, Args, 1, "-help")) {
   fprintf( stderr, 
      "%s: [-t <trust>] [-l <linesearch>] [-t2 <trustminpack2>]\n", Args[0] );
   fprintf( stderr, "    [-log] [-n <size>]\n" );
   exit(0);
   }
  if (SYArgHasName( &Argc, Args, 1, "-t" )) nlmethod  = NLNEWTONTR;
  if (SYArgHasName( &Argc, Args, 1, "-l" )) nlmethod  = NLNEWTONLS;
  if (SYArgHasName( &Argc, Args, 1, "-t2" )) nlmethod = NLNEWTONTR2;
  SYArgGetInt( &Argc, Args, 1, "-n", &n );
  if (SYArgHasName( &Argc, Args, 1, "-log")) fp = stderr;

  usrP  = NEW(SparseCntx); if (!usrP) ERR;
  usrP->n = n;

  printf("Running Size %d \n",n);

  neP         = NLCreate(nlmethod);       if (!neP) ERR;
  NLSetVectorOps(neP,VECreate());
  DVSetDefaultFunctions(neP->vc);
  NLSetFunction(neP,Function);
  NLSetInitialGuess(neP,InitialGuess);
  NLSetStepSetUp(neP,SetUpLinearSolve);
  NLSetStepCompute(neP,LinearSolve);
  NLSetStepDestroy(neP,DestroyLinearSolve);
  neP->vec_sol = (void *) MALLOC(n*sizeof(double)); if (!neP->vec_sol) ERR;
  neP->residual_history = (double *) MALLOC(neP->max_it*sizeof(double));
  if (!neP->residual_history) ERR;
  if (fp) neP->fp = fp;

  NLSetTruncationTolerance(neP,1.e-10);

  NLSetUp(neP,usrP);

  its = NLSolve(neP,usrP); printf("Number of Newton iterations %d \n",its);

  for ( i=0; i<its+1; i++ ) printf("%d %g \n",i,neP->residual_history[i]);
 
  NLGetNumberFunctionApplicationsUsed(neP,nfunc);
  NLGetNumberStepComputationsUsed(neP,nsteps);
  NLGetNumberVectorOpsUsed(neP,nvector);
  printf("Function evaluations %d \n",nfunc);
  printf("Inverse Jacobian Applications %d \n",nsteps);
  printf("Number of vector operations %d \n",nvector);

  NLDestroy(neP,usrP);
}
/*-------------------------------------------------------------*/
void Function(nlP,usrP,u,f)
NLCntx    *nlP;
SparseCntx *usrP;
double    *u,*f;
{
  int    i,n = usrP->n;
  double uzero = 0.0, uone = 1.0, x, h, hh;

  h = 1.0/(n + 1); x = 0.0; hh = h*h;
  f[0] = (u[1] - 2.0*u[0] + uzero)/hh + u[0]*u[0]*u[0] - 3.0*x - pow(x,9);
  f[0] = -f[0];
  for ( i=1; i<n-1; i++ ) {
    x += h;
    f[i] = (u[i+1] - 2.0*u[i] + u[i-1])/hh + u[i]*u[i]*u[i] - 3.0*x - pow(x,9);
    f[i] = -f[i];
  }
  x += h;
  f[n-1] = (uone - 2.0*u[n-1] + u[n-2])/hh + u[n-1]*u[n-1]*u[n-1] - 3.0*x - pow(x,9);
  f[n-1] = -f[n-1];
}
/*-------------------------------------------------------------*/
/*
      Forms Jacobian at the point x
*/
void SetUpLinearSolve(nlP,usrP,u)
NLCntx      *nlP;
SparseCntx  *usrP;
double      *u;
{
  int     i,n = usrP->n;
  SpMat   *mat;
  double  x, h, f, g, hh;

  h = 1.0/(n + 1); hh = h*h; g = -1.0/hh;
  usrP->jacobian = mat = SpCreate(n,n,3); if (!mat) ERR;

  f =   2.0/hh  - 3.0*u[0]*u[0] ;
  SpAddValue(mat,f,0,0);
  SpAddValue(mat,g,0,1);
  
  for ( i=1; i<n-1; i++ ) {
    f =   2.0/hh  - 3.0*u[i]*u[i];
    SpAddValue(mat,f,i,i);
    SpAddValue(mat,g,i,i+1);
    SpAddValue(mat,g,i,i-1);
  }
  f =   2.0/hh - 3.0*u[n-1]*u[n-1];
  SpAddValue(mat,f,n-1,n-1);
  SpAddValue(mat,g,n-1,n-2);

  usrP->method = SVSSOR;
  usrP->svctx = SVCreate(mat,usrP->method); if (!usrP->svctx) ERR;
  SVSetAccelerator(usrP->svctx,ITCG);
  SVSetUp(usrP->svctx); CHKERR(1);
  /* SVSetMonitor(usrP->svctx,ITDefaultMonitor,(void *)0); */
}
/*-------------------------------------------------------------*/
/* This is the standard convergence test except we also make   */
/* sure that the solution norm is less then outsidedelta       */
/*-------------------------------------------------------------*/
#define  MAX(a,b)           ((a) > (b) ? (a) : (b))
double outsidedelta; /* this is a BAD HACK */
int LinearSolveConverged(itP,svctx,n,rnorm)
ITCntx      *itP;
SVctx      *svctx;
int         n;
double      rnorm;
{
  void   *v;
  double vnorm;
  if ( n == 0 ) {itP->ttol = MAX(itP->rtol*rnorm,itP->atol); return 0;}
  if ( rnorm <= itP->ttol ) return(1);
  
  v = (void *) MALLOC( svctx->size*sizeof(double));
  v = (*itP->BuildSolution)(itP,svctx,v);
  VNORM(itP->vc,svctx,v,&vnorm);
  FREE(v);
  if (vnorm > outsidedelta) return 1; 
  return(0);
}
/*-------------------------------------------------------------*/
/*             y = -J(x)\f  || y || <= delta                   */
/* This routine is supose to solve the linear system until     */
/* either the step length is greater then delta or the         */
/* relative decrease in the residual is less then rtol         */
/*-------------------------------------------------------------*/
void LinearSolve(nlP,usrP,x,f,y,fnorm,delta,rtol,gpnorm,ynorm)
NLCntx    *nlP;
SparseCntx *usrP;
double    *f,*x,*y,fnorm,*gpnorm,delta,*ynorm,rtol;
{
  double norm;
  FILE   *fp = nlP->fp;

  outsidedelta = delta;
  SVSetRelativeTol(usrP->svctx,rtol); 
  SVSetConvergenceTest(usrP->svctx, LinearSolveConverged,(void *)0); 
  SVSolve(usrP->svctx,f,y); CHKERR(1);
  VNORM(nlP->vc,usrP,y,ynorm);
  if (*ynorm > delta) {
    norm = delta/(*ynorm);
    *gpnorm = (1.0 - norm)*fnorm;
    VSCALE(nlP->vc,usrP,-norm,y);
    if (fp) fprintf(fp,"Scaling direction by %g \n",norm);
      *ynorm = delta;
  }
  else {
    *gpnorm = 0.0;
    VSCALE(nlP->vc,usrP,-1.0,y);
    if (fp) fprintf(fp,"Direction is in Trust Region \n");
  }
} 
/*-------------------------------------------------------------*/
void DestroyLinearSolve(nlP,usrP)
NLCntx     *nlP;
SparseCntx *usrP;
{
  SVDestroy(usrP->svctx);
  SpDestroy(usrP->jacobian);
}
/*-------------------------------------------------------------*/
void InitialGuess(nlP,usrP,x)
NLCntx    *nlP;
SparseCntx *usrP;
double    *x;
{
  int    n = usrP->n,i;
  for ( i=0; i<n; i++ ) x[i] = 0.0;
}
/*-------------------------------------------------------------*/
int PrintVector(n,x)
int    *n;
double *x;
{
  int i;
  for ( i=0; i<*n; i++ ) fprintf(stderr,"%d %g \n",i,x[i]);
}

