#ifndef lint
static char SCCSid[] = "@(#) ./iter/tcqmr/tcqmr.c 07/23/93";
#endif

/*
       This file contains an implementaion of Tony Chan's 
   transpose free QMR.
*/

#include <math.h>
#define IT      it

#include "tools.h"
#include "iter/itctx.h"
#include "iter/itpriv.h"
#include "iter/itfunc.h"
#include "iter/tcqmrctx.h"
#include "tcqmrp.h"

/*+
    ITTCQMRSolve - Called after a call to ITTCQMRCreate() or
    ITCreate(ITTCQMR) and the call to ITTCQMERSetUp() or ITSetUp().
    Actually solves the linear system using Tony Chan's transpose
    free QMR.
    Preferred calling sequence ITSolve(itP,usrP)

    Input Parameters:
.   itP  - the iterative context
.   usrP - the user context

    Returns:
    the number of iterations required or -1 on error.
+*/
int ITTCQMRSolve( itP, usrP )
ITCntx *itP;
void   *usrP;
{
double      rnorm0, rnorm;                      /* residual values */
double      theta, ep, cl1, sl1, cl, sl, sprod, tau_n1, f, Gamma; 
double      deltmp, rho, beta, eptmp, ta, s, c, tau_n, delta;
double      dp1, dp2, rhom1, alpha;
int         it, cerr;
ITTCQMRCntx *qmrP = (ITTCQMRCntx *)itP->MethodPrivate;

it = 0;

ITResidual(usrP,itP,x,u,v, r, v0, b );
    
NORM(r,&rnorm0);                                /*  rnorm0 = ||r|| */

SET(0.0,um1);
COPY(r,u);
rnorm = rnorm0;
SCALE(1.0/rnorm, u);
SET(0.0,vm1);
COPY(u,v);
COPY(u,v0);
SET(0.0,pvec1);
SET(0.0,pvec2);
SET(0.0,p);
theta = 0.0; 
ep    = 0.0; 
cl1   = 0.0; 
sl1   = 0.0; 
cl    = 0.0; 
sl    = 0.0;
sprod = 1.0; 
tau_n1= rnorm0;
f     = 1.0; 
Gamma = 1.0; 
rhom1 = 1.0;

/*
 CALCULATE SQUARED LANCZOS  vectors
 */
while ( !CONVERGED(rnorm,it)) {     
    if (itP->usr_monitor) {
        (*itP->usr_monitor)( itP, usrP, it, rnorm );
	}
    MATOP( u, y, vtmp );                           /* y = A*u */
    DOT( v0, y, &dp1 );
    DOT( v0, u, &dp2 );
    alpha = dp1 / dp2;                          /* alpha = v0'*y/v0'*u */
    deltmp = alpha;
    COPY(y,z);     
    DAXPY(-alpha,u,z);                         /* z = y - alpha u */
    DOT( v0, u, &rho );
    beta   = rho / (f*rhom1);
    rhom1  = rho;
    COPY(z,utmp);                               /* up1 = (A-alpha*I)*
					       (z-2*beta*p) + f*beta*
					     beta*um1 */
    DAXPY(-2.0*beta,p,utmp);
    MATOP(utmp,up1,vtmp);
    DAXPY(-alpha,utmp,up1);
    DAXPY(f*beta*beta,um1,up1);
    NORM(up1,&dp1);
    f     = 1.0 / dp1;
    SCALE(f,up1);
    DAYPX(-beta,z,p);                          /* p = f*(z-beta*p) */
    SCALE(f,p);
    COPY(u,um1);
    COPY(up1,u);
    beta  = beta/Gamma;
    eptmp = beta;
    MATOP(v,vp1,vtmp);
    DAXPY(-alpha,v,vp1);
    DAXPY(-beta,vm1,vp1);
    NORM(vp1,&Gamma);
    SCALE(1.0/Gamma,vp1);
    COPY(v,vm1);
    COPY(vp1,v);

/*
     SOLVE  Ax = b
 */
/* Apply the last two Given's (Gl-1 and Gl) rotations to (beta,alpha,Gamma) */
    if (it > 1) {
	theta =  sl1*beta;
	eptmp = -cl1*beta;
	}
    if (it > 0) {
	ep     = -cl*eptmp + sl*alpha;
	deltmp = -sl*eptmp - cl*alpha;
	}
    
    if (fabs(Gamma) > fabs(deltmp)) {
	ta = -deltmp / Gamma;
	s = 1.0 / sqrt(1.0 + ta*ta);
	c = s*ta;
	}
    else {
	ta = -Gamma/deltmp;
	c = 1.0 / sqrt(1.0 + ta*ta);
	s = c*ta;
	}

    delta  = -c*deltmp + s*Gamma;
    tau_n  = -c*tau_n1; tau_n1 = -s*tau_n1;
    COPY(vm1,pvec);
    DAXPY(-theta,pvec2,pvec);
    DAXPY(-ep,pvec1,pvec);
    SCALE(1.0/delta,pvec);
    DAXPY(tau_n,pvec,x);
    cl1 = cl; sl1 = sl; cl = c; sl = s;     

    COPY(pvec1,pvec2);
    COPY(pvec,pvec1);

    /* Compute the upper bound on the residual norm r (See QMR paper p. 13) */
    sprod = sprod*fabs(s);
    rnorm = rnorm0 * sqrt((double)it+2.0) * sprod;     
    it++; if (it > itP->max_it) {break;}
    }

/* Get floating point work */
itP->nmatop += (it * 3);
itP->nvectors += (it) * 34;

/* Need to undo preconditioning here  */
ITUnwindPre( usrP, itP, x, vtmp );

return RCONV(it);
}

/*+
    ITTCQMRSetUp - Called after a call to ITTCQMRCreate() or
    ITCreate(ITTCQMR), allocates space needed in the TCQMR solution.
    Preferred call sequence is ITSetUp(itP,usrP).

    Input Parameters:
.   itP - the iterative context
.   usrP - the user context
+*/
void ITTCQMRSetUp( itP, usrP )
ITCntx *itP;
void   *usrP;
{
int      k;
ITTCQMRCntx *qmrP = (ITTCQMRCntx *)itP->MethodPrivate;

ITCheckDef( itP );

/* Allocate array to hold pointers to user vectors.  Note that we need
   TCQMR_VECs */
qmrP->vecs = (void **) MALLOC( (unsigned)(TCQMR_VECS*sizeof(void *) ) );
qmrP->user_work = (*itP->vc->obtain_vectors)( usrP, TCQMR_VECS );
for (k=0; k<TCQMR_VECS; k++)
    qmrP->vecs[k] = qmrP->user_work[k];
}

/* -----------------------------------------------------------------*/
/*           Allows user to change work vectors                     */
/* -----------------------------------------------------------------*/
void ITTCQMRAdjustWork( itP, usrP )
ITCntx *itP;
void      *usrP;
{
ITTCQMRCntx *qmrP;

if ( itP->adjust_work_vectors ) {
   qmrP = (ITTCQMRCntx *) itP->MethodPrivate;
   if ( (*itP->adjust_work_vectors)(usrP,qmrP->user_work, TCQMR_VECS ) )
       SETERRC(27,"Could not allocate work vectors in TCQMR");
   }
}

/*+
    ITTCQMRDestroy - Destroys a iterative context variable obtained
    by a call to ITTCQMRCreate() or ITCreate(ITTCQMR). Preferred calling
    sequence ITDestroy().

    Input Parameters:
.   itP -  the iterative context
.   usrP - the user context
+*/
void ITTCQMRDestroy(itP,usrP)
ITCntx *itP;
void   *usrP;
{
ITTCQMRCntx *qmrP = (ITTCQMRCntx *) itP->MethodPrivate;

/* Free the pointer to user variables */
FREE( qmrP->vecs );

/* free work vectors */
if ( itP->vc->release_vectors ) {
    (*itP->vc->release_vectors)(usrP,qmrP->user_work, TCQMR_VECS );
    }
/* free the context variables */
FREE(qmrP); 
FREE(itP);
}

/*+
  ITTCQMRCreate - Create the iterative context for Tony Chan's 
  transpose free QMR, TCQMR. Preferred calling sequence ITCreate(ITTCQMR).
 +*/
ITCntx *ITTCQMRCreate()
{
ITCntx    *itP;
ITTCQMRCntx *qmrP;

itP = NEW(ITCntx); CHKPTRN(itP);
ITSetDefaults( itP );
qmrP = NEW(ITTCQMRCntx); CHKPTRN(qmrP);
itP->MethodPrivate = (void *) qmrP;
itP->method        = ITTCQMR;
itP->usr_monitor   = ITDefaultMonitor;
itP->converged     = ITDefaultConverged;
itP->BuildSolution = ITDefaultBuildSolution;

itP->setup         = ITTCQMRSetUp;
itP->solver        = ITTCQMRSolve;
itP->adjustwork    = ITTCQMRAdjustWork;
itP->closedown     = ITTCQMRDestroy;

return itP;
}
