#ifndef lint
static char SCCSid[] = "@(#) ./iter/iterativ.c 07/23/93";
#endif

/*
   This file contains some simple default routines.  
   These routines should be SHORT, since they will be included in every
   executable image that uses the iterative routines (note that, through
   the registry system, we provide a way to load only the truely necessary
   files) 
 */
#include "tools.h"
#include "iter/itctx.h"
#include "iter/itfunc.h"
#include "iter/itpriv.h"
#include <stdio.h>

/*@
  ITSetDefaults - Set the most basic defaults for the iterative 
  solver context. 

  Input Parameter:
. itP - iterative context
 @*/
void ITSetDefaults( itP )
ITCntx *itP;
{
  void *ITDefaultBuildSolution(), *ITDefaultBuildResidual();

  itP->max_it    = 10000;
  itP->right_inv = 1;
  itP->guess_zero= 0;
  itP->calc_eigs = 0;
  itP->calc_res  = 0;
  itP->use_pres  = 0;
  itP->rtol      = 1.0e-5;
  itP->atol      = 1.0e-50;
  itP->vec_sol   = 0;
  itP->vec_rhs   = 0;
  itP->residual_history = 0;
  itP->res_hist_size    = 0;
  itP->res_act_size     = 0;
  itP->converged = ITDefaultConverged;
  itP->usr_monitor= 0;
  itP->adjust_work_vectors = 0;
  itP->BuildSolution = ITDefaultBuildSolution;
  itP->BuildResidual = ITDefaultBuildResidual;

  itP->vc        = VECreate();
  itP->amult     = 0;
  itP->binv      = 0;
  itP->matop     = 0;
  itP->tamult    = 0;
  itP->tbinv     = 0;
  itP->tmatop    = 0;

  itP->solver    = 0;
  itP->setup     = 0;
  itP->closedown = 0;
  itP->adjustwork= 0;

  itP->MethodPrivate = 0;

  itP->nwork         = 0;
  itP->work          = 0;

  itP->nmatop        = 0;
  itP->namult        = 0;
  itP->nbinv         = 0;
  itP->nvectors      = 0;
  itP->nscalar       = 0;

  itP->matP          = 0;
  itP->monP          = 0;
  itP->cnvP          = 0;
}
/*@
   ITCheckDef - Checks the definition of the itP quantities 
   necessary for most of the solvers.

  Input Parameter:
. itP - iterative context

   Returns:
   the number of errors encountered.

   Note:
   If no binv is set, use the vector copy routine.
 @*/
int ITCheckDef( itP )
ITCntx *itP;
{
int err = 0;
  if (!itP->vec_sol) {
      SETERRC(1,"Solution vector not specified for iterative method"); err++;}
  if (!itP->vec_rhs) {
      SETERRC(2,"RHS vector not specified for iterative method"); err++;}
  if (!itP->amult)   {
      SETERRC(3,
      "Matrix-vector product routine not specified for iterative method"); 
      err++;}
  if (!itP->binv)    {
      /* SETERRC(4,"Preconditioner not specified for iterative method"); 
	 err++; */
      itP->binv = itP->vc->copy;
      if (!itP->binv) {
	  SETERRC(4,"Preconditioner not specified for iterative method"); 
	  err++;
	  }
      }
  if (!itP->vc)      {
      SETERRC(5,"Vector operations not specified for iterative method");err++;} 
return err;
}

/*ARGSUSED*/
/*@
  ITDefaultMonitor - Default code to print residual at each iteration 
  in the iterative solvers.

  Input Parameters:
. itP   - iterative context
. usrP  - user's context
. n     - iteration number
. rnorm - 2-norm residual value (may be estimated).  
 @*/
void ITDefaultMonitor(itP,usrP,n,rnorm)
ITCntx *itP;
void   *usrP;
int    n;
double rnorm;
{
  printf("%d %14.12e \n",n,rnorm);
}

/*ARGSUSED*/
/*@
  ITDefaultConverged - Default code to determine convergence in
  the iterative solvers.

  Input Parameters:
. itP   - iterative context
. usrP  - user's context
. n     - iteration number
. rnorm - 2-norm residual value (may be estimated).  

  Returns:
  1 if the iteration has converged or exceeds divergence threshold, 
  0 otherwise.
  
 @*/
int ITDefaultConverged(itP,usrP,n,rnorm)
ITCntx *itP;
void   *usrP;
int    n;
double rnorm;
{
  if ( n == 0 ) {
      itP->ttol   = MAX(itP->rtol*rnorm,itP->atol);
      itP->rnorm0 = rnorm;
      }
  if ( rnorm <= itP->ttol )      return 1;
  /* 100 is really to small; in fact, this should really be an
     adjustable parameter.  Also, we need to return failure and abort, 
     not just "done" */
  if ( rnorm >= 1.0e5*itP->rnorm0) return 1;
  return(0);
}

/*@
  ITDefaultBuildSolution - Default code to create/move the solution.

  Input Parameters:
. itP  - iterative context
. usrP - user's context
. v    - pointer to user vector  

  Returns:
  pointer to a vector containing the solution.
 @*/
void *ITDefaultBuildSolution(itP,usrP,v)
ITCntx *itP;
void   *usrP, *v;
{
if (v) {
    if (itP->right_inv) {
	/* ITUnwindPre( usrP, itP, itP->vec_sol, v ); */
	PPRE( itP->vec_sol, v );
	}
    else 
	(*itP->vc->copy)( usrP, itP->vec_sol, v );
    /* v = itP->vec_sol; */
    }
return v;
}

/*@
  ITDefaultBuildResidual - Default code to compute the residual.

  Input Parameters:
. itP  - iterative context
. usrP - user's context
. t    - pointer to temporay vector
. v    - pointer to user vector  

  Returns:
  pointer to a vector containing the residual.
 @*/
void *ITDefaultBuildResidual(itP,usrP,t,v)
ITCntx *itP;
void   *usrP, *t,*v;
{
if (v && t) {
    if (itP->right_inv) 
	ITUnwindPre( usrP, itP, itP->vec_sol, t );
    else 
	t = itP->vec_sol;
    (*itP->amult)( usrP, t, v );
    (*itP->vc->aypx)( usrP, -1.0, itP->vec_rhs, v );
    }
return v;
}

/*
  ITDefaultGetWork - Get a number of work vectors

  Input Parameters:
. itP  - iterative context
. usrP - user's context
. nw   - number of work vectors to allocate

  Note:
  Call this only if no work vectors have been allocated 
 */
void ITDefaultGetWork( itP, usrP, nw )
ITCntx *itP;
void   *usrP;
int    nw;
{
if (itP->work) ITDefaultFreeWork( itP, usrP );
itP->nwork = nw;
itP->work  = (*itP->vc->obtain_vectors)(usrP,nw); CHKPTR(itP->work);
}

/*
  ITDefaultAdjustWork - Adjust work vectors

  Input Parameters:
. itP  - iterative context
. usrP - user's context

 */
void ITDefaultAdjustWork( itP, usrP )
ITCntx *itP;
void   *usrP;
{
if ( itP->adjust_work_vectors ) {
    (*itP->adjust_work_vectors)(usrP,itP->work,itP->nwork); CHKERR(27);
    }
}

/*
  ITDefaultFreeWork - Free work vectors

  Input Parameters:
. itP  - iterative context
. usrP - user's context
 */
void ITDefaultFreeWork( itP, usrP )
ITCntx *itP;
void   *usrP;
{
if ( itP->vc->release_vectors ) {
    (*itP->vc->release_vectors)(usrP,itP->work,itP->nwork); CHKERR(27);
    }
}

/*@
ITDefaultDestroy - Destroys a iterative context variable for methods with
no separate context.  Preferred calling sequence ITDestroy().

Input Parameters: 
.   itP - the iterative context
.   usrP - the user context
@*/
void ITDefaultDestroy(itP,usrP)
ITCntx *itP;
void   *usrP;
{
if (itP->MethodPrivate) 
    SETERRC(6,"No method private area-probably corrupted iterative context");

/* free work vectors */
ITDefaultFreeWork( itP, usrP );

/* free the context variables */
FREE(itP);
}

/*@
  ITGetWorkCounts - Get the counts of the number of operations
  performed by the iterative routines

  Input Parameter:
. itP - Iterative context

  Output Parameters:
. matop - number of applications of matrix-vector product AND preconditioner
. amult - number of applications of matrix-vector product not counted in matop
. binv  - number of applications of preconditioner not counted in matop
. vecs  - number of operations on vectors
. scalars - number of operations on scalars

  Note:
  The counts provided by this routine depend on correctly counting the
  number of operations in the iterative methods; those are not currently
  very accurate.  Users are encouraged to look at the source codes and
  send fixes to gropp@mcs.anl.gov .

  This routine does NOT clear the values; use ITClearWorkCounts to do that.
@*/
void ITGetWorkCounts( itP, matop, amult, binv, vecs, scalars )
ITCntx *itP;
int    *matop, *amult, *binv, *vecs, *scalars;
{
*matop   = itP->nmatop;
*amult   = itP->namult;
*binv    = itP->nbinv;
*vecs    = itP->nvectors;
*scalars = itP->nscalar;
}
/*@
  ITClearWorkCounts - Clear the work counts that are maintained for the
  iterative solvers.

  Input Parameter:
. itP - Iterative context
@*/
void ITClearWorkCounts( itP )
ITCntx *itP;
{
itP->nmatop        = 0;
itP->namult        = 0;
itP->nbinv         = 0;
itP->nvectors      = 0;
itP->nscalar       = 0;
}
