/*
    This file contains an interface to a log file that is object-oriented
    (but does not require C++ or other O-O language).  The intent is to
    separate the functions of accessing and of formating a log file.  This
    will allow us to modify the log file format as we discover additional
    needs (such as providing for clock adjustment, definition of states
    rather than just events, binary and compressed representations, and
    self-executing formats (for example, making log files valid postscript
    programs.

    In addition, we want to separate the notion of a file from the
    log "file"; allowing the same routines to be used if the files are
    (a) distributed or (b) memory-resident.

    The operations are:

     ELopen - open a log file of a given name.
     ELclose - close a log file
     ELread  - read the next record in a log file.
     ELwrite - write a record in a log file.

     A number of formats will (eventually) be supported.  These will include
     the ALOG and PICL format at the very least.  Additional formats may
     include (a) postscript language (b) binary/compressed format
     (c) alternate clock definitions (such as time is cycle counter
     and is not yet converted)     
 */     

typedef struct {
    FILE   *fd;        /* file descriptor */
    int    rnum;       /* record number of current record */
    int    is_ascii;   /* true if ascii, otherwise binary */
    double t_to_sec;   /* Clock is assumed to be linearly scalable to
                          seconds.  This is the scaling parameter */
    int    (*read)(), (*write)();    /* Routines to read/write entries */
    
    /* There is room here for: functions for the read/write/close
       formating (binary/compressed/ascii), byte location, etc. */
    } ELData;

/* Here are some event definitions.  These include ONLY the data that is not
   considered common to all events */    
/* This is an alog event.  The clock data is a 64 bit quantity, with t1
   holding the high bits (usually 0) */
#define ALOG_MAX_STRING 13
typedef struct {
    int           task_id;
    int           i_data;
    char          c_data[ALOG_MAX_STRING];
    } ALOGentry;

/* This is a generic variable-length entry */
typedef struct {
    int  len;
    char *val;
    } PICLentry;
    
/* This is a generic event */
typedef struct {
    int kind;                /* kind of entry */
    int type;                /* event type */
    int proc_id;             /* processor id */
    unsigned long t1, t2;    /* 64-bit time (t2 is LSW, t1 is MSW) */
    union {                  /* Other stuff */
    	int       data[1];
        ALOGentry ALOG;
        PICLentry PICL;
        } X;
    } ELEntry;


/*
    Here are macros that will extract the common fields for events
 */
#define ELetype(entry)     (entry)->type
#define ELprocessor(entry) (entry)->proc_id
#define ELetimeLow(entry)  (entry)->t1
#define ELetimeHigh(entry) (entry)->t2
#define ELetime(el,entry)  (double)( (entry)->t1 ) + (double)( (entry)->t2 )
#define ELdata(entry)      (entry)->data

#define ELread(el,entry)   (*(el->read))( el, entry )
#define ELwrite(el,entry)  (*(el->write))( el, entry )

/*
    These should really be split into a read and set
 */
#define ELGetTime( el, entry, t )
#define ELSetTime( el, entry, t )

/* Another option is to allow el to hold the entry; then these interfaces
   just pass in el */    


extern ELData *ELopen();


/* These are clock synchronization events */
#define ALOG_EVENT_SYNC        -101
#define ALOG_EVENT_PAIR_A1     -102
#define ALOG_EVENT_PAIR_A2     -103
#define ALOG_EVENT_PAIR_B1     -104



