/* SCCSid = @(#) ./blkcm/bcp.h 07/23/93 */

#ifndef __BCP
#define __BCP

/* Choose general types based on specific architectures */
#include "comm/comm.h"
#ifdef DISTRIBUTED_MEMORY
#define PARALLEL_VERSION
#endif

/*
   Definitions for mesh communication routines
 */

typedef struct {
    int s1, n1, n2, n3, n4, n5;      /* strides and lengths */
    int inc1, inc2, inc3, inc4;      /* increments between elements in
                                        the two leading dimensions */
    double *p;                       /* data */
    void   *lctx;                    /* local context (used to compute p) */
    } slab;

struct BC_entry {
    int  type;
    slab src;                        /* Source or destination data */
    int  phase;                      /* Used to group into send/recv phases */
    int  id;                         /* id of destination */
    int  processor;                  /* partner processor OR offset of
                                        partner operation if local */
#ifdef DISTRIBUTED_MEMORY
    /* All of this data is set by the comm_compile routine */
    int  inplace;                    /* True if buffer is actually the
                                        source or destination (no copy needed)
                                        */
    int  rc;                         /* Communications status */
    int  mtype;                      /* Message type */
    int  act_len;                    /* buffer length */
    double *buffer;                  /* buffer for moving data */
#endif
    };
typedef struct BC_entry BCentry;

/* Advance to next program line.  on entry, a is current line, on exit,
   it is the next line.  Value is the NEXT line */
#define NEXTLINE(a) (++a)

/*
   The types are defined here.  They fall into three basic
   groups: local, send to another processor and receive from another
           processor
   The send and receive have a number of minor modes that are intended
   to provide for optimizations:
   
   SEND               - Contains info for setting a send buffer
   RECV               - Contains info for processing a receive buffer
   BUFFER             - This operation contains the actual buffer to be
                        sent or received
   SYNC_BLOCK         - (SEND ONLY).  Complete receives of all items
                        with phase less than this instruction.
   SYNC_NBR           - synchronize with neighbor (force a handshake;
                        this can be used to speedup long messages on
                        some machines.
   FORCETYPE          - use ForceType send (receive buffer is guarenteed
                        to exist on destination node)
   To make things easier on the user, the comm_compile() routine handles
   modifying the modes of SEND and RECV.

   In order to simplify the testing, the major modes occupy a full byte
   and could be accessed with a load-byte instruction (instead of using
   a mask).  This is not yet implemented.
 */

/* Major operations supported internally only */
#define BLOCK_COMM_LOCAL_SRC      0x004
#define BLOCK_COMM_LOCAL_DEST     0x008

/* Minor modes only */
#define MAJOR_MODE_MASK       0x0FF
#define BLOCK_COMM_BUFFER     0x100
#define BLOCK_COMM_SYNC_BLOCK 0x200
#define BLOCK_COMM_SYNC_NBR   0x400
#define BLOCK_COMM_FORCETYPE  0x800

/* Macros to test modes */
#define IS_SRC(a)   (((a)->type & MAJOR_MODE_MASK) == BLOCK_COMM_SRC)
#define IS_DEST(a)  (((a)->type & MAJOR_MODE_MASK) == BLOCK_COMM_DEST)
#define IS_LOCAL_SRC(a) (((a)->type & MAJOR_MODE_MASK) == BLOCK_COMM_LOCAL_SRC)
#define IS_LOCAL_DEST(a) (((a)->type & MAJOR_MODE_MASK) == BLOCK_COMM_LOCAL_DEST)
#define IS_LOCAL(a) ( (a)->type & 0x00C )
#define SET_MAJOR_MODE(a,b) (a)->type = ((a)->type & ~MAJOR_MODE_MASK) | (b)
#define GET_MAJOR_MODE(a)   ((a)->type & MAJOR_MODE_MASK)

/* Type used in validating program on distributed memory machines */
#ifdef DISTRIBUTED_MEMORY
#define VALID_TYPE       12345
#define BLK_FINDOWN_TYPE 12777
/* Number of bytes at which the nbr-pair method should be used.  This
   depends on the particular machine.  This should be handled in
   pairsync code. */
#ifndef MIN_PAIR_SIZE
#define MIN_PAIR_SIZE 1000
#endif
#endif

#ifdef intelnx
#define BASE_FORCE 0x40000000
#endif

/*
   Here are NEW data types and objects.  The intent here is to further
   simplify the process of processing the communication requests, without
   making the "program" structure do too many things.  The actual send/recv
   operations need only a type, buffer, and length; and an id for send.
   Both may need a comm_id if they are asynchronous.

   Code to move between buffers needs the local and global buffer and length.
   
   Each phase may contain 
      async recv
      prepare to send/send
      wait for recvs to complete and move
 */
typedef struct {
    int  type, to, len, rc;
    char *p;
    void *lctx;
    } BCSends;
typedef struct {
    int  type, maxlen, rc;
    char *p;
    void *lctx;
    } BCARecvs;
typedef struct {
    int        nsend, nrecv;
    BCSends    *send;
    BCARecvs   *recv;
    } BCPhaseBlk;
typedef struct {
    int        nphase;
    BCPhaseBlk *phase;
    } BCObject;
/* This is used for sorting the program */
typedef struct {
    BCentry *entry;
    int     order, idx;
    } BC_Sblock;

#endif
