#include <stdio.h>
#include <math.h>
#include <malloc.h>

#include "imath.h"
#include "sig.h"

static char sccsid[] = "@(#)trig.c	1.1 7/15/91";

/*--------------------------------------------------------------------------
 Compute the phase of a vector.
--------------------------------------------------------------------------*/
static void
phase_int16(n, ar, ai, br)
    int n;
    short *ar, *ai;
    float *br;
{
    while (n--)
	*br++ = atan2((float)*ai++, (float)*ar++);
}
static void
phase_float32(n, ar, ai, br)
    int n;
    float *ar, *ai;
    float *br;
{
    while (n--)
	*br++ = atan2(*ai++, *ar++);
}


void
sig_phase(this)
    sig_t *this;
{
    int n;
    int offset;
    float *re = (float *)malloc(this->nrec * this->reclen * sizeof(float));

    switch (this->kind) {
    case hdr_CBFP16:
	free(this->exp);
	this->exp = NULL;
	/* FALL THRU */
    case hdr_CINT16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    phase_int16(this->reclen,
		((short *)this->re) + offset,
		((short *)this->im) + offset,
		re + offset);
	break;
    case hdr_CFLOAT32:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    phase_float32(this->reclen,
		((float *)this->re) + offset,
		((float *)this->im) + offset,
		re + offset);
	break;
    default:
	badkind("sig_phase", this->kind);
    }
    this->kind = hdr_FLOAT32;
    free(this->re);
    this->re = (short *)re;
    free(this->im);
    this->im = NULL;
}

/*--------------------------------------------------------------------------
 Compute the magnitude of a vector.
--------------------------------------------------------------------------*/
static void
magnitude_int16(n, ar, ai, br)
    int n;
    short *ar, *ai;
    float *br;
{
    while (n--)
	*br++ = hypot((float)*ai++, (float)*ar++);
}

static void
magnitude_bfp16(n, ar, ai, ae, br)
    int n;
    short *ar, *ai, ae;
    float *br;
{
    float scale;

    /* scale = (float) (1 << ae); */
    scale = scalbn(1.0, ae);

    while (n--)
	*br++ = scale * hypot((float)*ai++, (float)*ar++);
}

static void
magnitude_float32(n, ar, ai, br)
    int n;
    float *ar, *ai;
    float *br;
{
    while (n--)
	*br++ = hypot(*ai++, *ar++);
}


void
sig_magnitude(this)
    sig_t *this;
{
    int n;
    int offset;
    float *re = (float *)malloc(this->nrec * this->reclen * sizeof(float));

    switch (this->kind) {
    case hdr_CINT16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    magnitude_int16(this->reclen,
		((short *)this->re) + offset,
		((short *)this->im) + offset,
		re + offset);
	break;
    case hdr_CBFP16:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    magnitude_bfp16(this->reclen,
		((short *)this->re) + offset,
		((short *)this->im) + offset,
		this->exp[n],
		re + offset);
	free(this->exp);
	this->exp = NULL;
	break;
    case hdr_CFLOAT32:
	for (n=0, offset=0; n<this->nrec; n++, offset+=this->reclen)
	    magnitude_float32(this->reclen,
		((float *)this->re) + offset,
		((float *)this->im) + offset,
		re + offset);
	break;
    default:
	badkind("sig_magnitude", this->kind);
    }
    this->kind = hdr_FLOAT32;
    free(this->re);
    this->re = (short *)re;
    free(this->im);
    this->im = NULL;
}

/*--------------------------------------------------------------------------
 Compute exp(j*(f*t + p0)).
 t is 0, 1, 2...
 f and p0 are real-valued, result is complex.
 npoints is number of samples to generate.
--------------------------------------------------------------------------*/
void
generate_tone(npoints, re, im, f, p0)
    int npoints;
    float *re, *im;
    double f, p0;
{
    int i;
    double phase;

    for (i=0, phase=p0; i<npoints; i++, phase += f) {
	phase = fmod (phase, 2.0*M_PI);
	*re++ = cos(phase);
	*im++ = sin(phase);
    }
}

/*--------------------------------------------------------------------------
 Create a signal containing a constant tone.
--------------------------------------------------------------------------*/
sig_t *
sig_tone(npoints, f, p0)
    int npoints;
    double f, p0;
{
    sig_t *sig;
    char name[1000];

    sprintf(name, "tone(%d,%g,%g)", npoints, f, p0);
    sig = sig_new(1, npoints, hdr_CFLOAT32, name);

    generate_tone(npoints, sig->re, sig->im, f, p0);

    return sig;
}
