#include <stdio.h>
#include <malloc.h>
#include <string.h>

#include "sig.h"
#include "arith.h"	/* for badkind */

static char sccsid[] = "@(#)sig.c	1.1 7/15/91";

/*--------------------------------------------------------------------------
 Create empty signal with given dimensions and name.
--------------------------------------------------------------------------*/
sig_t *
sig_new(nrec, reclen, kind, fname)
    int nrec;
    int reclen;
    int kind;
    char *fname;
{
    sig_t *sig = (sig_t *)malloc(sizeof(sig_t));

    sig->nrec = nrec;
    sig->reclen = reclen;
    sig->kind = kind;
    sig->re = (short *)malloc(nrec * reclen * sig_ELBYTES(sig));

    if (kind & hdr_KIND_COMPLEX)
	sig->im = (short *)malloc(nrec * reclen * sig_ELBYTES(sig));
    else
	sig->im = NULL;

    if (kind & hdr_KIND_BFP)
	sig->exp = (short *)calloc(nrec, sizeof(short));
    else
	sig->exp = (short *)NULL;

    strcpy(sig->fname, fname);

    return sig;
}

/*--------------------------------------------------------------------------
 Copy a signal.
--------------------------------------------------------------------------*/
sig_t *
sig_copy(this)
    sig_t *this;
{
    sig_t *sig;

    sig = sig_new(this->nrec, this->reclen, this->kind, this->fname);
    bcopy(this->re, sig->re, this->nrec * this->reclen * sig_ELBYTES(this));
    if (this->kind & hdr_KIND_COMPLEX)
	bcopy(this->im, sig->im, this->nrec * this->reclen * sig_ELBYTES(this));
    if (this->kind & hdr_KIND_BFP)
	bcopy(this->exp, sig->exp, this->nrec * sizeof(short));
    return sig;
}

/*--------------------------------------------------------------------------
 Destroy a signal.
--------------------------------------------------------------------------*/
void
sig_destroy(this)
    sig_t *this;
{
    if (this->re != NULL)
	free(this->re);
    if (this->im != NULL)
	free(this->im);
    if (this->exp != NULL)
	free(this->exp);
    free(this);
}

/*--------------------------------------------------------------------------
 Convert vector to complex by adjoining a zero imaginary componant.
 (Thank goodness floating point zero is all zero bits.)
--------------------------------------------------------------------------*/
void
sig_complex(this)
    sig_t *this;
{
    if (!(this->kind & hdr_KIND_COMPLEX)) {
	this->im = (short *)calloc(this->nrec*this->reclen, sig_ELBYTES(this));
	this->kind |= hdr_KIND_COMPLEX;
    }
}

/*--------------------------------------------------------------------------
 Convert vector to real by truncating its imaginary componant.
--------------------------------------------------------------------------*/
void
sig_real(this)
    sig_t *this;
{
    if (this->kind & hdr_KIND_COMPLEX) {
	free(this->im);
	this->im = NULL;
	this->kind &= ~hdr_KIND_COMPLEX;
    }
}

/*--------------------------------------------------------------------------
 Verify that a signal is legal.
 Checks floating point signals for NaN's.
--------------------------------------------------------------------------*/
void
sig_verify(this)
    sig_t *this;
{
    int i, j;

    switch (this->kind) {
    case hdr_CFLOAT32: {
	float *im = (float *)this->im;
	for (i=0; i<this->nrec; i++)
	    for (j=0; j<this->reclen; j++, im++)
		if (*im != *im)
		    printf("Signal %s, record %d, element %d, imag, is NaN.\n",
			this->fname, i, j);
	}
	/* FALL THRU to real */
    case hdr_FLOAT32: {
	float *re = (float *)this->re;
	for (i=0; i<this->nrec; i++)
	    for (j=0; j<this->reclen; j++, re++)
		if (*re != *re)
		    printf("Signal %s, record %d, element %d, real, is NaN.\n",
			this->fname, i, j);
	}
	break;
    default:
	/* don't complain about unknown types */
	;
    }
}

/*--------------------------------------------------------------------------
 Print a signal.
 Only print first nrec_p records, nel_p elements of each record.
--------------------------------------------------------------------------*/
void
sig_fprint(this, nrec_p, nel_p, fp)
    sig_t *this;
    int nrec_p;
    int nel_p;
    FILE *fp;
{
    int i, j;
    int offset;

    if (nel_p <= 0 || nel_p > this->reclen) nel_p = this->reclen;
    if (nrec_p <= 0 || nrec_p > this->nrec) nrec_p = this->nrec;

    fprintf(fp, "name %s:\n", this->fname);
    switch (this->kind) {
    case hdr_CINT8:
    case hdr_INT8:
	for (i=0, offset=0; i<nrec_p; i++, offset += this->reclen) {
	    fprintf(fp, "record %d re: ", i);
	    for (j=0; j<nel_p; j++) {
		if (j % 16 == 15) fprintf(fp, "\\\n  ");
		fprintf(fp, "%d ", ((char *)this->re)[j+offset]);
	    }
	    fprintf(fp, "\n");
	    if (this->kind & hdr_KIND_COMPLEX) {
		fprintf(fp, "record %d im: ", i);
		for (j=0; j<nel_p; j++) {
		    if (j % 16 == 15) fprintf(fp, "\\\n  ");
		    fprintf(fp, "%d ", ((char *)this->im)[j+offset]);
		}
		fprintf(fp, "\n");
	    }
	}
	break;
    case hdr_CINT16:
    case hdr_INT16:
	for (i=0, offset=0; i<nrec_p; i++, offset += this->reclen) {
	    fprintf(fp, "record %d re: ", i);
	    for (j=0; j<nel_p; j++) {
		if (j % 16 == 15) fprintf(fp, "\\\n  ");
		fprintf(fp, "%d ", this->re[j+offset]);
	    }
	    fprintf(fp, "\n");
	    if (this->kind & hdr_KIND_COMPLEX) {
		fprintf(fp, "record %d im: ", i);
		for (j=0; j<nel_p; j++) {
		    if (j % 16 == 15) fprintf(fp, "\\\n  ");
		    fprintf(fp, "%d ", this->im[j+offset]);
		}
		fprintf(fp, "\n");
	    }
	}
	break;
    case hdr_CBFP16:
    case hdr_BFP16:
	for (i=0, offset=0; i<nrec_p; i++, offset += this->reclen) {
	    int exp = this->exp[i];
	    fprintf(fp, "record %d exp: %d\n", i, exp);
	    fprintf(fp, "record %d re: ", i);
	    for (j=0; j<nel_p; j++) {
		if (j % 16 == 15) fprintf(fp, "\\\n  ");
		if (exp > 0)
		    fprintf(fp, "%d ", (int)this->re[j+offset] << exp);
		else
		    fprintf(fp, "%d ", (int)this->re[j+offset] >> -exp);
	    }
	    fprintf(fp, "\n");
	    if (this->kind & hdr_KIND_COMPLEX) {
		fprintf(fp, "record %d im: ", i);
		for (j=0; j<nel_p; j++) {
		    if (j % 16 == 15) fprintf(fp, "\\\n  ");
		    if (exp > 0)
			fprintf(fp, "%d ", (int)this->im[j+offset] << exp);
		    else
			fprintf(fp, "%d ", (int)this->im[j+offset] >> -exp);
		}
		fprintf(fp, "\n");
	    }
	}
	break;
    case hdr_CFLOAT32:
    case hdr_FLOAT32:
	for (i=0, offset=0; i<nrec_p; i++, offset += this->reclen) {
	    fprintf(fp, "record %d re: ", i);
	    for (j=0; j<nel_p; j++) {
		if (j % 16 == 15) fprintf(fp, "\\\n  ");
		fprintf(fp, "%g ", ((float *) this->re)[j+offset]);
	    }
	    fprintf(fp, "\n");
	    if (this->kind & hdr_KIND_COMPLEX) {
		fprintf(fp, "record %d im: ", i);
		for (j=0; j<nel_p; j++) {
		    if (j % 16 == 15) fprintf(fp, "\\\n  ");
		    fprintf(fp, "%g ", ((float *) this->im)[j+offset]);
		}
		fprintf(fp, "\n");
	    }
	}
	break;
    default:
    badkind("sig_fprint", this->kind);
    }
}
