/*--------------------------------------------------------------------------
 Macro definition/expansion module.

 Macro definition can not be nested, but macro expansion may be nested.
--------------------------------------------------------------------------*/

static char sccsid[]="@(#)macro.c	1.1 7/16/91";

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include "macro.h"

#define macro_MAX 100
#define macro_MAXLINES 1000

static macro_t macro_tab[macro_MAX];
static int macro_i = -1;

/*--------------------------------------------------------------------------
 Save a string on the heap.
--------------------------------------------------------------------------*/
static char *
strsave(s)
    char *s;
{
    char *r;
    r = malloc(strlen(s)+1);
    strcpy(r, s);
    return r;
}

/*--------------------------------------------------------------------------
 Find index of a previously defined macro.
--------------------------------------------------------------------------*/
static int
macro_find(name)
    char *name;
{
    int i;
    for (i=0; i<= macro_i; i++) {
	if (strcmp(name, macro_tab[i].name)==0)
	    return i;
    }
    return -1;
}

/*--------------------------------------------------------------------------
 Begin defining a macro.
--------------------------------------------------------------------------*/
void
macro_define(name)
    char *name;
{
    if (macro_find(name) >= 0) {
	fprintf(stderr, "macro_define: macro %s already defined.\n", name);
	exit(1);
    }
    if (++macro_i >= macro_MAX) {
	fprintf(stderr, "macro_define: too many macros.\n");
	exit(1);
    }

    macro_tab[macro_i].name = strsave(name);
    macro_tab[macro_i].lines = (char **)calloc(sizeof(char *), macro_MAXLINES);
    macro_tab[macro_i].nlines = 0;
}

/*--------------------------------------------------------------------------
 Add another line to the current macro.
--------------------------------------------------------------------------*/
void
macro_append(s)
    char *s;
{
    if (macro_tab[macro_i].nlines >= macro_MAXLINES) {
	fprintf(stderr, "macro_append: too many lines in macro %s\n",
	    macro_tab[macro_i].name);
	exit(1);
    }
    macro_tab[macro_i].lines[macro_tab[macro_i].nlines++] = strsave(s);
}

#define BLANKS " \t"
/*--------------------------------------------------------------------------
 Start playing back the named macro, with the given arguments.
 Expand $1, $2, ... in the macro to arg1, arg2, ...
 Returns NULL if no such macro.
--------------------------------------------------------------------------*/
macro_inst_t *
macro_expand_begin(s, args)
    char *s;
    char *args;
{
    macro_inst_t *inst;
    char *tok;

    inst = (macro_inst_t *)malloc(sizeof(macro_inst_t));

    if ((inst->mac = macro_find(s)) < 0)
	return NULL;

    inst->linenum = 0;
    inst->argc=0;
    for (tok=strtok(args,BLANKS); tok; tok=strtok(NULL, BLANKS))
	inst->argv[inst->argc++] = strsave(tok);

    return inst;
}

/*--------------------------------------------------------------------------
 Call this when finished with a macro expansion.
--------------------------------------------------------------------------*/
void
macro_expand_end(inst)
    macro_inst_t *inst;
{
    while (inst->argc > 0)
	free(inst->argv[--inst->argc]);
    free(inst);
}

/*--------------------------------------------------------------------------
 Call this to get the next expanded line from the macro.
 Returns 0 upon success, -1 upon end of macro.
--------------------------------------------------------------------------*/
int
macro_expand_next(inst, buf)
    macro_inst_t *inst;
    char *buf;
{
    int argnum;
    macro_t *mac;
    char *line;
    char *p, *q;

    mac = &macro_tab[inst->mac];
    if (inst->linenum >= mac->nlines)
	return -1;

    /* Grab pointer to line, copy it to buf while replacing
     * $n with arg n
     */
    line = mac->lines[inst->linenum++];

    for (p=line; *p; ) {
	char c = *p++;
	if (c != '$') {
	    /* normal char */
	    *buf++ = c;
	} else {
	    /* $number */
	    argnum = strtol(p, &p, 10);
	    if (p == NULL) {
		fprintf(stderr,
		   "macro_expand_next: expected arg number after $ in line %s of macro %s\n",
		   line, mac->name);
		exit(1);
	    }
	    if (argnum < 1 || argnum > inst->argc) {
		fprintf(stderr,
		   "macro_expand_next: macro contained $%d, must be between 1 and # of args (%d) in instance of macro %s\n",
		   argnum, inst->argc, mac->name);
		exit(1);
	    }
	    /* Copy actual parameter instead of $number */
	    q = inst->argv[argnum-1];
	    while (*q)
		*buf++ = *q++;
	}
    }
    *buf++ = 0;

    /* Success */
    return 0;
}

#ifdef TEST
lose(s)
    char *s;
{
    fprintf(stderr, "%s\n", s);
    exit(1);
}

lose1(s,a1)
    char *s;
    char *a1;
{
    fprintf(stderr, s, a1);
    fputs("\n", stderr);
    exit(1);
}

main()
{
    macro_inst_t *inst;
    char buf[128];

    macro_define("mac1");
    macro_append("m1 l1:$1");
    macro_append("m1 l2");
    macro_define("mac2");
    macro_append("m2 l1");
    macro_append("m2 l2:$2");
    macro_define("mac3");
    macro_append("m3 l1:$1,$2");
    macro_append("m3 l2");

    inst = macro_expand_begin("mac2", "arg1 arg2");
    if (macro_expand_next(inst, buf))	lose("m2 x1");
    if (strcmp(buf, "m2 l1"))		lose1("m2 l1, buf=%s", buf);
    if (macro_expand_next(inst, buf))	lose("m2 x2");
    if (strcmp(buf, "m2 l2:arg2"))	lose1("m2 l2, buf=%s", buf);
    if (!macro_expand_next(inst, buf))	lose("m2 x3");
    macro_expand_end(inst);

    inst = macro_expand_begin("mac1", "arg1 arg2");
    if (macro_expand_next(inst, buf))	lose("m1 x1");
    if (strcmp(buf, "m1 l1:arg1"))	lose1("m1 l1, buf=%s", buf);
    if (macro_expand_next(inst, buf))	lose("m1 x2");
    if (strcmp(buf, "m1 l2"))		lose1("m1 l2, buf=%s", buf);
    if (!macro_expand_next(inst, buf))	lose("m1 x3");
    macro_expand_end(inst);

    inst = macro_expand_begin("mac3", "arg1 arg2");
    if (macro_expand_next(inst, buf))	lose("m3 x1");
    if (strcmp(buf, "m3 l1:arg1,arg2"))	lose1("m3 l1, buf=%s", buf);
    if (macro_expand_next(inst, buf))	lose("m3 x2");
    if (strcmp(buf, "m3 l2"))		lose1("m3 l2, buf=%s", buf);
    if (!macro_expand_next(inst, buf))	lose("m3 x3");
    macro_expand_end(inst);

    printf("macro: Test passed\n");
    exit(0);
}
#endif
