/*--------------------------------------------------------------------------
 Signal-editing operators.
--------------------------------------------------------------------------*/
#include <stdio.h>
#include <assert.h>
#include <malloc.h>
#include <memory.h>

#include "sig.h"
#include "edit.h"

static char sccsid[] = "@(#)edit.c	1.1 7/15/91";

/*--------------------------------------------------------------------------
 Delete the first n_left elements and the last n_right elements
 of each record of the signal.
 (See also sig_edit_x().)
--------------------------------------------------------------------------*/
void
sig_delete_x(this, n_left, n_right)
    sig_t *this;
    int n_left;
    int n_right;
{
    int nrec = this->nrec;
    int oReclen = this->reclen;
    int nReclen = oReclen - n_left - n_right;
    char *thisre = (char *)this->re;
    char *thisim = (char *)this->im;
    int elbytes = sig_ELBYTES(this);
    int oRecbytes = oReclen * elbytes;
    int nRecbytes = nReclen * elbytes;

    int oOff, nOff;

    assert(n_left >= 0);
    assert(n_right >= 0);
    assert(nReclen > 0);

    /* Shift data down.  Works for all data types. */
    n_left *= elbytes;
    for (oOff=n_left,nOff=0; nrec>0; nrec--,oOff+=oRecbytes,nOff+=nRecbytes) {
	bcopy(thisre+oOff, thisre+nOff, nRecbytes);
	if (this->kind & hdr_KIND_COMPLEX)
	    bcopy(thisim+oOff, thisim+nOff, nRecbytes);
    }

    /* Resize memory block, save new size */
    this->reclen = nReclen;
    this->re = (short *)realloc(thisre, this->nrec*nRecbytes);
    if (this->kind & hdr_KIND_COMPLEX)
	this->im = (short *)realloc(thisim, this->nrec*nRecbytes);
}

/*--------------------------------------------------------------------------
 Delete the first n_skip elements and the last (reclen-n_skip-n_copy) elements
 of each record of the signal.
--------------------------------------------------------------------------*/
void
sig_edit_x(this, n_skip, n_copy)
    sig_t *this;
    int n_skip;
    int n_copy;
{
    sig_delete_x(this, n_skip, this->reclen-n_skip-n_copy);
}

/*--------------------------------------------------------------------------
 Return a signal which is the original with each record padded with n_left and 
 n_right zero elements.
--------------------------------------------------------------------------*/
sig_t *
sig_pad_x(this, n_left, n_right)
    sig_t *this;
    int n_left;
    int n_right;
{
    sig_t *sig;
    int nrec = this->nrec;
    int oReclen = this->reclen;
    int nReclen = this->reclen + n_left + n_right;
    char *thisre = (char *)this->re;
    char *thisim = (char *)this->im;
    int elbytes = sig_ELBYTES(this);
    int oRecbytes = oReclen * elbytes;
    int nRecbytes = nReclen * elbytes;
    int oOff, nOff;
    char *sigre, *sigim;

    sig = sig_new(nrec, nReclen, this->kind, this->fname);
    sigre = (char *) sig->re;
    sigim = (char *) sig->im;
    
    n_left *= elbytes;
    n_right *= elbytes;
    for (oOff=nOff=0; nrec > 0; nrec--, oOff += oRecbytes, nOff += nRecbytes) {
	/* Real */
	if (n_left > 0)
	    bzero(sigre+nOff, n_left);
	bcopy(thisre+oOff, sigre+nOff+n_left, oRecbytes);
	if (n_right > 0)
	    bzero(sigre+nOff+n_left+oRecbytes, n_right);

	/* Imaginary */
	if (sig->kind & hdr_KIND_COMPLEX) {
	    if (n_left > 0)
		bzero(sigim+nOff, n_left);
	    bcopy(thisim+oOff, sigim+nOff+n_left, oRecbytes);
	    if (n_right > 0)
		bzero(sigim+nOff+n_left+oRecbytes, n_right);
	}
    }
    if (this->kind & hdr_KIND_BFP) {
	/* Copy exponants. */
	memcpy((char *)sig->exp, (char *)this->exp, nrec * sizeof(short));
    }

    return sig;
}

/*--------------------------------------------------------------------------
 Return a signal which is the original with each record padded IN THE MIDDLE
 with n_pad zero elements starting at n_left.
--------------------------------------------------------------------------*/
sig_t *
sig_midpad_x(this, n_left, n_pad)
    sig_t *this;
    int n_left;
    int n_pad;
{
    sig_t *sig;
    int nrec = this->nrec;
    int oReclen = this->reclen;
    int nReclen = this->reclen + n_pad;
    char *thisre = (char *)this->re;
    char *thisim = (char *)this->im;
    int elbytes = sig_ELBYTES(this);
    int oRecbytes = oReclen * elbytes;
    int nRecbytes = nReclen * elbytes;
    int oOff, nOff;
    char *sigre, *sigim;

    sig = sig_new(nrec, nReclen, this->kind, this->fname);
    sigre = (char *) sig->re;
    sigim = (char *) sig->im;
    
    n_left *= elbytes;
    n_pad *= elbytes;
    for (oOff=nOff=0; nrec > 0; nrec--, oOff += oRecbytes, nOff += nRecbytes) {
	/* Real */
	bcopy(thisre+oOff, sigre+nOff, n_left);
	bzero(sigre+nOff+n_left, n_pad);
	bcopy(thisre+oOff+n_left, sigre+nOff+n_left+n_pad, oRecbytes-n_left);

	/* Imaginary */
	if (sig->kind & hdr_KIND_COMPLEX) {
	    bcopy(thisim+oOff, sigim+nOff, n_left);
	    bzero(sigim+nOff+n_left, n_pad);
	    bcopy(thisim+oOff+n_left, sigim+nOff+n_left+n_pad, oRecbytes-n_left);
	}
    }
    if (this->kind & hdr_KIND_BFP) {
	/* Copy exponants. */
	memcpy((char *)sig->exp, (char *)this->exp, nrec * sizeof(short));
    }

    return sig;
}

/*--------------------------------------------------------------------------
 Return a signal which is the original but with n_left words of left context.
 This is used for instance in performing convolution using FFTs with
 the overlap-and-discard method.
--------------------------------------------------------------------------*/
sig_t *
sig_context_x(this, n_left)
    sig_t *this;
    int n_left;
{
    sig_t *sig;
    int nrec = this->nrec;
    int oReclen = this->reclen;
    int nReclen = this->reclen + n_left;
    char *thisre = (char *)this->re;
    char *thisim = (char *)this->im;
    int elbytes = sig_ELBYTES(this);
    int oRecbytes = oReclen * elbytes;
    int nRecbytes = nReclen * elbytes;
    int oOff, nOff;
    char *sigre, *sigim;

    sig = sig_new(nrec, nReclen, this->kind, this->fname);
    sigre = (char *) sig->re;
    sigim = (char *) sig->im;

    n_left *= elbytes;
    for (oOff=nOff=0; nrec > 0; nrec--, oOff+=oRecbytes, nOff+=nRecbytes) {
	/* Real */
	if (n_left > 0) {
	    if (oOff<n_left)
		bzero(sigre+nOff, n_left);
	    else
		bcopy(thisre+oOff-n_left, sigre+nOff, n_left);
	}
	bcopy(thisre+oOff, sigre+nOff+n_left, oRecbytes);

	/* Imaginary */
	if (sig->kind & hdr_KIND_COMPLEX) {
	    if (n_left > 0) {
		if (oOff<n_left)
		    bzero(sigim+nOff, n_left);
		else
		    bcopy(thisim+oOff-n_left, sigim+nOff, n_left);
	    }
	    bcopy(thisim+oOff, sigim+nOff+n_left, oRecbytes);
	}
    }

    return sig;
}

/*--------------------------------------------------------------------------
 Redimension the signal to have given reclen.
 Input or output must have reclen=1.
 If input has reclen=1, any partial records at end are lost.

 Examples:
    nrec=1000 and reclen=1 -> redimension(100) -> nrec=10 and reclen=100,
    nrec=1000 and reclen=1 -> redimension(300) -> nrec=3 and reclen=300,
    nrec=10 and reclen=1000 -> redimension(1) -> nrec=1000 and reclen=1.

 Note: when/if records get a block-floating-point exponent...
 for output reclen=1, this operator will have to duplicate exponents;
 for input reclen=1, this operator will have to perform a relative normalization 
 (or just insist that the input must have exponent=0).
--------------------------------------------------------------------------*/
void
sig_redimension_x(this, new_reclen)
    sig_t *this;
    int new_reclen;
{
    if (this->kind & hdr_KIND_BFP) {
	badkind("sig_redimension_x", this->kind);
    }

    if (new_reclen == 1) {
	this->nrec *= this->reclen;
	this->reclen = new_reclen;
    } else {
	if (this->reclen != 1) {
	    fprintf(stderr, "sig_redimension_x: either input or output must have reclen=1\n");
	    exit(1);
	}
	if (this->nrec < new_reclen) {
	    fprintf(stderr, "sig_redimension_x: output has no records (nrec %d < new reclen %d)\n",
		this->nrec, new_reclen);
	    exit(1);
	}
	this->nrec /= new_reclen;
	this->reclen = new_reclen;
    }
}

void
sig_redimension_y(this, new_nrec)
    sig_t *this;
    int new_nrec;
{
    int nelements;
    if (this->kind & hdr_KIND_BFP) {
	badkind("sig_redimension_y", this->kind);
    }

    nelements = this->nrec * this->reclen;
    if (nelements < new_nrec) {
	fprintf(stderr, "sig_redimension_y: not enough data (nelements %d < new nrec %d)\n",
	    nelements, new_nrec);
	exit(1);
    }
    this->nrec = new_nrec;
    this->reclen = nelements / new_nrec;
}

/*--------------------------------------------------------------------------
 Rotate each record of the signal forward n places.  
 -reclen <= n <= reclen;
--------------------------------------------------------------------------*/
void
sig_rotate_x(this, n_rot)
    sig_t *this;
    int n_rot;
{
    int nrec = this->nrec;
    int reclen = this->reclen;
    char *thisre = (char *)this->re;
    char *thisim = (char *)this->im;
    int elbytes = sig_ELBYTES(this);
    int recbytes = elbytes * reclen;
    char *tmp;
    int off;
    int n;

    if (n_rot == 0) return;

    if (n_rot < -reclen || n_rot > reclen) {
	fprintf(stderr,
	    "sig_rotate_x: reclen is %d, but you asked to rotate %d places.\n",
	    reclen, n_rot);
	exit(1);
    }

    tmp = malloc(recbytes);

    n_rot *= elbytes;
    if (n_rot < 0) {
	n_rot = -n_rot;
	/* Shift Towards 0 */
	for (off=0,n=0; n<nrec; n++, off += recbytes) {
	    /* Real */
	    /* Copy n_rot bytes to save area */
	    memcpy(tmp, thisre+off, n_rot);
	    /* Shift recbytes-n_rot bytes towards start of record */
	    bcopy(thisre+off+n_rot, thisre+off, recbytes-n_rot);
	    /* Copy n_rot bytes to end of record */
	    memcpy(thisre+off+recbytes-n_rot, tmp, n_rot);

	    if (this->kind & hdr_KIND_COMPLEX) {
		/* Imag */
		/* Copy n_rot bytes to save area */
		memcpy(tmp, thisim+off, n_rot);
		/* Shift recbytes-n_rot bytes towards start of record */
		bcopy(thisim+off+n_rot, thisim+off, recbytes-n_rot);
		/* Copy n_rot bytes to end of record */
		memcpy(thisim+off+recbytes-n_rot, tmp, n_rot);
	    }
	}
    } else {
	/* Shift Towards +infinity */
	for (off=0,n=0; n<nrec; n++, off += recbytes) {
	    /* Real */
	    /* Copy n_rot bytes to save area */
	    memcpy(tmp, thisre+off+recbytes-n_rot, n_rot);
	    /* Shift recbytes-n_rot bytes towards end of record */
	    bcopy(thisre+off, thisre+off+n_rot, recbytes-n_rot);
	    /* Copy n_rot bytes to start of record */
	    memcpy(thisre+off, tmp, n_rot);

	    if (this->kind & hdr_KIND_COMPLEX) {
		/* Imag */
		/* Gee, if C had a complex data type, this
		 * code wouldn't be needed at all.  The real and
		 * imag would be mixed together.  Next time.
		 */
		/* Copy n_rot bytes to save area */
		memcpy(tmp, thisim+off+recbytes-n_rot, n_rot);
		/* Shift recbytes-n_rot bytes towards end of record */
		bcopy(thisim+off, thisim+off+n_rot, recbytes-n_rot);
		/* Copy n bytes to start of record */
		memcpy(thisim+off, tmp, n_rot);
	    }
	}
    }
    
    free(tmp);
}

/*--------------------------------------------------------------------------
 Delete the first n_begin elements and the last n_end records
 of the signal.
--------------------------------------------------------------------------*/
void
sig_delete_y(this, n_begin, n_end)
    sig_t *this;
    int n_begin;
    int n_end;
{
    int nrec = this->nrec;
    int reclen = this->reclen;
    int elbytes = sig_ELBYTES(this);
    int recbytes = reclen * elbytes;

    assert(n_begin >= 0);
    assert(n_end >= 0);
    assert(nrec - n_begin - n_end > 0);

    /* Shift data down.  Works for all data types. */
    if (n_begin > 0) {
	/* Can't use memcpy, as blocks overlap. */
	bcopy(((char *)this->re)+n_begin*recbytes, 
	    this->re, (nrec-n_begin)*recbytes);
	if (this->kind & hdr_KIND_COMPLEX)
	    bcopy(((char *)this->im)+n_begin*recbytes, 
		this->im, (nrec-n_begin)*recbytes);
	if (this->kind & hdr_KIND_BFP)
	    bcopy(this->exp+n_begin, 
		this->exp, (nrec-n_begin)*sizeof(short));
    }

    /* Resize memory block, save new size */
    this->nrec = nrec - n_begin - n_end;
    this->re = (short *)realloc(this->re, this->nrec*recbytes);
    if (this->kind & hdr_KIND_COMPLEX)
	this->im = (short *)realloc(this->im, this->nrec*recbytes);
    if (this->kind & hdr_KIND_BFP)
	this->exp = (short *)realloc(this->exp, this->nrec*sizeof(short));
}

