#include <stdio.h>
#include <assert.h>
#include <math.h>
#include <malloc.h>

#include "imath.h"
#include "sig.h"

static char sccsid[] = "@(#)ac_chirp.c	1.1 7/15/91";

/*--------------------------------------------------------------------------
 Rob Fatland's airsar (digital) chirp generator, modified to output chirp
 rather than chirp phase.  Chirp varies from -1 to 1.
 Returns buffer and number of samples computed (= tau*nu_fast).

 short **chirpbufp	pointer to output buffer will be placed here
 double tau		pulse duration microseconds
 double bw		chirp bandwidth MHz
 double f_0		chirp start frequency, equal to f_s/4 - bw/2)
 double p_0		chirp start phase
 (double f_s		Real sampling frequency MHz)
 double nu_slow	        chirp slow clock frequency
 double nu_fast	        chirp fast clock frequency

 For example,
    buflen = ac_chirp(cbuf, 10.0e-6, 20.0e6, 1.25e6, 5e6, 270e6);
--------------------------------------------------------------------------*/
int ac_chirp(chirpbufp, tau, bw, f_0, p_0, nu_slow, nu_fast)
    float **chirpbufp;
    double tau, bw, f_0, p_0;
    double nu_slow, nu_fast;
{
    double delta_t, dt, delta_nu;
    double phase, dphase, d_dphase, very_small;
    double fast_clock;
    int    index;
    int npoints;
    float *chirp;

    /* Figure out how many samples caller wants, allocate space */
    npoints = (int) (tau * nu_fast);
    assert(npoints > 10);
    assert(npoints < 50000);
    *chirpbufp = chirp = (float *)malloc(npoints * sizeof(float));

    delta_t    = 1.0 / nu_slow;   /* .2 microseconds */
    dt         = 1.0 / nu_fast;   /* .0037 microseconds */
    delta_nu   = bw / ((float)((int)(nu_slow*tau) - 1));
    phase      = p_0 / (2.0 * M_PI);
    dphase     = f_0 / nu_fast;
    d_dphase   = delta_nu / nu_fast;
    very_small = .0000000000001;

    for (index = npoints; index > 0; ){

        for (fast_clock = very_small; 
             (fast_clock < delta_t) && (index > 0); 
             fast_clock += dt, index--){

	    phase = fmod (phase, 1.0);
	    *chirp++ = cos(2.0 * M_PI * phase);
	    phase += dphase;
        }
        dphase += d_dphase;
    }
    return npoints;
}

/*--------------------------------------------------------------------------
 Create a signal containing a chirp.
--------------------------------------------------------------------------*/
sig_t *
sig_ac_chirp(tau, bw, f_0, p_0, nu_slow, nu_fast)
    double tau, bw, f_0, p_0;
    double nu_slow, nu_fast;
{
    sig_t *sig;
    char name[1000];

    sprintf(name, "ac_chirp(%g,%g,%g,%g,%g,%g)",
	tau, bw, f_0, p_0, nu_slow, nu_fast);
    sig = sig_new(1, 0, hdr_CFLOAT32, name);

    /* Create real componant */
    sig->reclen = ac_chirp(&sig->re, tau, bw, f_0, p_0, nu_slow, nu_fast);

    /* Create zeroed imaginary componant of same size */
    sig->im = (short *)calloc(sig->nrec * sig->reclen, sig_ELBYTES(sig));

    return sig;
}
