/*
 *
 * $Source: /filesv/usr/local/proj/sphinx/spx2/src/lib/cfile/RCS/tf_util.c,v $
 *
 *
 *  MODULE NAME:    tf_util.c
 *
 *
 *  AUTHORS:
 *
 *	K. Alagappan
 *
 */


/*
 * COPYRIGHT (C) 1992 DIGITAL EQUIPMENT CORPORATION
 * ALL RIGHTS RESERVED
 *
 * "Digital Equipment Corporation authorizes the reproduction,
 * distribution and modification of this software subject to the following
 * restrictions:
 * 
 * 1.  Any partial or whole copy of this software, or any modification
 * thereof, must include this copyright notice in its entirety.
 *
 * 2.  This software is supplied "as is" with no warranty of any kind,
 * expressed or implied, for any purpose, including any warranty of fitness 
 * or merchantibility.  DIGITAL assumes no responsibility for the use or
 * reliability of this software, nor promises to provide any form of 
 * support for it on any basis.
 *
 * 3.  Distribution of this software is authorized only if no profit or
 * remuneration of any kind is received in exchange for such distribution. 
 * 
 * 4.  This software and all application programs are to be used only for
 * non-commercial purposes. However, media costs associated with the
 * distribution of the software or application programs may be recovered.
 *
 */


#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/time.h>
#include "BigNum.h"
#include "BigRSA.h"
#include "cdc.h"
#include "cdc_db.h"
#include "random.h"
#include "spxapi_defs.h"

#define INT_STR  20
#define TOO_BIG -1
#define TF_LCK_RETRY ((unsigned)2)	/* seconds to sleep before
					 * retry if ticket file is
					 * locked */
extern  errno;
extern int cdc_debug;

/*
 * fd must be initialized to something that won't ever occur as a real
 * file descriptor. Since open(2) returns only non-negative numbers as
 * valid file descriptors, and spx_tf_init always stuffs the return value
 * from open in here even if it is an error flag, we must
 * 	a. Initialize fd to a negative number, to indicate that it is
 * 	   not initially valid.
 *	b. When checking for a valid fd, assume that negative values
 *	   are invalid (ie. when deciding whether spx_tf_init has been
 *	   called.)
 *	c. In spx_tf_close, be sure it gets reinitialized to a negative
 *	   number. 
 */
static  fd = -1;
static	curpos = 0;			/* Position in tfbfr */
static	lastpos = 0;			/* End of tfbfr */
static	char tfbfr[4*BUFSIZ];		/* Buffer for ticket data */

static spx_tf_gets(), spx_tf_read();

/*
 * This file contains routines for manipulating the ticket cache file.
 *
 * The principal's ticket file is in the following format:
 *
 *    localname; fullname; uidlen; uuid; before; after; s_bitlen; deleg_flag;
 *    pubRSAKey;
 *    s_privRSAKey;
 *    userticket_len; userticket;
 *    TA_1
 *    TA_2
 *    ...
 *    TA_n
 *    EOF
 *
 *      Where "TA_X" consists of the following fixed-length
 *      fields from the TA structure (see "cdc.h"):
 *
 *      caX_fullname; caX_pubRSAKey;
 *
 */

/*
 * spx_tf_init() should be called before the other ticket file routines.
 * It takes the name of the ticket file to use, "tf_name", and a
 * read/write flag "rw" as arguments. 
 *
 * It tries to open the ticket file, checks the mode, and if everything
 * is okay, locks the file.  If it's opened for reading, the lock is
 * shared.  If it's opened for writing, the lock is exclusive. 
 *
 * Returns ASUCCESS if all went well, otherwise one of the following: 
 *
 * NO_TKT_FIL   - file wasn't there
 * TKT_FIL_ACC  - file was in wrong mode, etc.
 * TKT_FIL_LCK  - couldn't lock the file, even after a retry
 */

spx_tf_init(tf_name, rw)
    char   *tf_name;
{
    int     wflag;
    short   me, getuid();
    struct stat stat_buf;

    switch (rw) {
    case R_TKT_FIL:
	wflag = 0;
	break;
    case W_TKT_FIL:
	wflag = 1;
	break;
    default:
	if (cdc_debug) fprintf(stderr, "spx_tf_init: illegal parameter\n");
	return TKT_FIL_ACC;
    }
    if (lstat(tf_name, &stat_buf) < 0)
	switch (errno) {
	case ENOENT:
	    return NO_TKT_FIL;
	default:
	    return TKT_FIL_ACC;
	}
    me = getuid();
    if ((stat_buf.st_uid != me && me != 0) ||
	((stat_buf.st_mode & S_IFMT) != S_IFREG))
	return TKT_FIL_ACC;

    /*
     * If "wflag" is set, open the ticket file in append-writeonly mode
     * and lock the ticket file in exclusive mode.  If unable to lock
     * the file, sleep and try again.  If we fail again, return with the
     * proper error message. 
     */

    curpos = sizeof(tfbfr);
    
    if (wflag) {
	fd = open(tf_name, O_RDWR, 0600);
	if (fd < 0) {
	    return TKT_FIL_ACC;
	}
	if (flock(fd, LOCK_EX | LOCK_NB) < 0) {
	    sleep(TF_LCK_RETRY);
	    if (flock(fd, LOCK_EX | LOCK_NB) < 0) {
		(void) close(fd);
		fd = -1;
		return TKT_FIL_LCK;
	    }
	}
	return ASUCCESS;
    }
    /*
     * Otherwise "wflag" is not set and the ticket file should be opened
     * for read-only operations and locked for shared access. 
     */

    fd = open(tf_name, O_RDONLY, 0600);
    if (fd < 0) {
	return TKT_FIL_ACC;
    }
    if (flock(fd, LOCK_SH | LOCK_NB) < 0) {
	sleep(TF_LCK_RETRY);
	if (flock(fd, LOCK_SH | LOCK_NB) < 0) {
	    (void) close(fd);
	    fd = -1;
	    return TKT_FIL_LCK;
	}
    }
    return ASUCCESS;
}

/*
 * spx_tf_get_pname() reads the principal's name from the ticket file. It
 * should only be called after spx_tf_init() has been called.  The
 * principal's name is filled into the "p" parameter.  If all goes well,
 * ASUCCESS is returned.  If spx_tf_init() wasn't called, TKT_FIL_INI is
 * returned.  If the name was null, or EOF was encountered, or the name
 * was longer than ANAME_SZ, TKT_FIL_FMT is returned. 
 */

spx_tf_get_pname(p)
    char   *p;
{
    if (fd < 0) {
	if (cdc_debug)
	    fprintf(stderr, "spx_tf_get_pname called before spx_tf_init.\n");
	return TKT_FIL_INI;
    }
    if (spx_tf_gets(p, ANAME_SZ) < 2)	/* can't be just a null */
	return TKT_FIL_FMT;
    return ASUCCESS;
}

/*
 * spx_tf_close() closes the ticket file and sets "fd" to -1. If "fd" is
 * not a valid file descriptor, it just returns.  It also clears the
 * buffer used to read tickets.
 *
 * The return value is not defined.
 */

spx_tf_close()
{
    if (!(fd < 0)) {
	(void) flock(fd, LOCK_UN);
	(void) close(fd);
	fd = -1;		/* see declaration of fd above */
    }
    bzero(tfbfr, sizeof(tfbfr));
}

/*
 * spx_tf_gets() is an internal routine.  It takes a string "s" and a count
 * "n", and reads from the file until either it has read "n" characters,
 * or until it reads a null byte. When finished, what has been read exists
 * in "s". If it encounters EOF or an error, it closes the ticket file. 
 *
 * Possible return values are:
 *
 * n            the number of bytes read (including null terminator)
 *              when all goes well
 *
 * 0            end of file or read error
 *
 * TOO_BIG      if "count" characters are read and no null is
 *		encountered. This is an indication that the ticket
 *		file is seriously ill.
 */

static 
spx_tf_gets(s, n)
    register char *s;
{
    register count;

    if (fd < 0) {
	if (cdc_debug)
	    fprintf(stderr, "spx_tf_gets called before spx_tf_init.\n");
	return TKT_FIL_INI;
    }
    if (curpos >= lastpos) {
      if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	spx_tf_close();
	return 0;
      }
      curpos = 0;
    }
    while ((tfbfr[curpos] == ' ') && (curpos < lastpos))  curpos++;
    for (count = n - 1; count > 0; --count) {
	if (curpos >= lastpos) {
	    if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	      spx_tf_close();
	      return 0;
	    }
	    curpos = 0;
	}
	*s = tfbfr[curpos++];
	if (*s == ';') {
	    *s = '\0';
	    return (n - count);
	}
	if (*s != '\n')  *s++;
    }
    spx_tf_close();
    return TOO_BIG;
}

static 
spx_tf_gethex(s, n)
    register char *s;
{
    register count;
    int      hex, high, low;

    if (fd < 0) {
	if (cdc_debug)
	    fprintf(stderr, "spx_tf_gets called before spx_tf_init.\n");
	return TKT_FIL_INI;
    }
    if (curpos >= lastpos) {
      if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	spx_tf_close();
	return(0);
      }
      curpos = 0;
    }
    if (tfbfr[curpos] == '\n')  curpos++;
    if (curpos >= lastpos) {
      if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	spx_tf_close();
	return(0);
      }
      curpos = 0;
    }
    if (tfbfr[curpos] == ' ')  curpos++;
    for (count = n - 1; count > 0; --count) {
	if (curpos >= lastpos) {
	    if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	      spx_tf_close();
	      return(0);
	    }
	    curpos = 0;
	}
	if (tfbfr[curpos] == ';') {
	    curpos++;
	    return (n - count);
	}
	if ((tfbfr[curpos] != '\n') && (tfbfr[curpos] != ' '))  {
	  if (curpos == lastpos-1) {
	    sscanf(&tfbfr[curpos],"%01x",&high);
	    if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	      spx_tf_close();
	      return(0);
	    }
	    curpos = 0;
	    sscanf(&tfbfr[curpos],"%01x",&low);
	    hex = high * 16 + low;
	    curpos++;
	  } else {
	    sscanf(&tfbfr[curpos],"%02x", &hex);
	    curpos += 2;
	  }
	  *s = hex;
	  *s++;
	} else curpos++;
    }
    spx_tf_close();
    return TOO_BIG;
}

static
spx_tf_getbin(s, n)
    register char *s;
{
    register count;

    if (fd < 0) return TKT_FIL_INI;

    if (curpos >= lastpos) {
      if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	spx_tf_close();
	return(0);
      }
      curpos = 0;
    }
    if ((tfbfr[curpos] == '\n') || (tfbfr[curpos] == ' '))  curpos++;

    for (count = n ; count > 0; --count) {
	if (curpos >= lastpos) {
	    if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	      spx_tf_close();
	      return(0);
	    }
	    curpos = 0;
	}
	*s++ = tfbfr[curpos];
	curpos++;
    }
    if (curpos >= lastpos) {
      if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	spx_tf_close();
	return(0);
      }
      curpos = 0;
    }
    if (tfbfr[curpos] != ';') {
      printf("spx_tf_getbin: expecting ';', got '%02x'\n",tfbfr[curpos]);
    }
    curpos++;
    return(0);
}

/*
 * spx_tf_read() is an internal routine.  It takes a string "s" and a count
 * "n", and reads from the file until "n" bytes have been read.  When
 * finished, what has been read exists in "s".  If it encounters EOF or
 * an error, it closes the ticket file.
 *
 * Possible return values are:
 *
 * n		the number of bytes read when all goes well
 *
 * 0		on end of file or read error
 */

static
spx_tf_read(s, n)
    register char *s;
    register n;
{
    register count;
    
    for (count = n; count > 0; --count) {
	if (curpos >= lastpos) {
	    if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	      spx_tf_close();
	      return 0;
	    }
	    curpos = 0;
	}
	*s++ = tfbfr[curpos++];
    }
    return n;
}
     
/*
 * spx_tf_save_cred() appends an credential info to the end of the
 * credential file.  You must call spx_tf_init() before calling
 * spx_tf_save_ta().
 *
 * Returns ASUCCESS if all goes well, TKT_FIL_INI if spx_tf_init() wasn't
 * called previously, and AFAILURE for anything else that went wrong.
 */

spx_tf_save_cred(cred)
ClaimantCred *cred;
{
    off_t   lseek();
    int     buflen, count;		/* count for write */
    char    intbuf[20], buf[PRIVATE_KEY_SIZE];

    if (fd < 0) {		/* fd is ticket file as set by spx_tf_init */
	  if (cdc_debug)
	      fprintf(stderr, "spx_tf_save_cred called before spx_tf_init.\n");
	  return TKT_FIL_INI;
    }
    /* Find the end of the ticket file */
    (void) lseek(fd, 0L, 2);

    /* Write the credential info and associated data */

    /* fullname */
    count = strlen(cred->fullname);
    if (write(fd, cred->fullname, count) != count)
      goto bad;
    write(fd, "; ", 2);

    /* uidlen */
    bzero(intbuf, 20);
    sprintf(intbuf, "%x", cred->uidlen);
    count = strlen(intbuf);
    if (write(fd, intbuf, count) != count)
	goto bad;
    write(fd, "; ", 2);

    /* uuid */
    buflen = cred->uidlen;
    if (write(fd, cred->uuid, buflen) != buflen)
	goto bad;
    write(fd, "; ", 2);

    /* life */
    bzero(intbuf, 20);
    sprintf(intbuf, "%x", cred->before);
    count = strlen(intbuf);
    if (write(fd, intbuf, count) != count)
	goto bad;
    write(fd, "; ", 2);
    bzero(intbuf, 20);
    sprintf(intbuf, "%x", cred->after);
    count = strlen(intbuf);
    if (write(fd, intbuf, count) != count)
	goto bad;
    write(fd, "; ", 2);

    /* deleg_flag */
    bzero(intbuf, 20);
    sprintf(intbuf, "%x", cred->deleg_flag);
    count = strlen(intbuf);
    if (write(fd, intbuf, count) != count)
	goto bad;
    write(fd, ";\n", 2);

    /* bitlen */
    bzero(intbuf, 20);
    sprintf(intbuf, "%x", cred->bitlen);
    count = strlen(intbuf);
    if (write(fd, intbuf, count) != count)
	goto bad;
    write(fd, ";\n", 2);

    /* s_privRSAkey */
    if (cred->deleg_flag) {
      bzero(buf, PRIVATE_KEY_SIZE);
      bcopy(cred->s_privRSAKey, buf, PRIVATE_KEY_SIZE);
      if (write(fd, buf, PRIVATE_KEY_SIZE) != PRIVATE_KEY_SIZE)
	goto bad;
      write(fd, ";\n", 2);
    }

    /* userticket_len */
    bzero(intbuf, 20);
    sprintf(intbuf, "%x", cred->userticket_len);
    count = strlen(intbuf);
    if (write(fd, intbuf, count) != count)
      goto bad;
    write(fd, "; ", 2);

    /*  userTicket  */
    bzero(buf, PRIVATE_KEY_SIZE);
    buflen = cred->userticket_len;
    bcopy(cred->userticket, buf, buflen);
    if (write(fd, buf, buflen) != buflen)
	goto bad;
    write(fd, ";\n", 2);
    /* Actually, we should check each write for success */
    return (ASUCCESS);

bad:
    printf("spx_tf_save_cred : wrote bad argument in cfile\n");
    return (AFAILURE);
}

spx_tf_markend()
{
    if (fd < 0) {		/* fd is ticket file as set by spx_tf_init */
	  if (cdc_debug)
	      fprintf(stderr, "spx_tf_save_cred called before spx_tf_init.\n");
	  return TKT_FIL_INI;
    }
    /* Find the end of the ticket file */
    (void) lseek(fd, 0L, 2);
    write(fd,";\n",2);
    return(ASUCCESS);
}

/*
 * spx_tf_save_ta() appends an incoming trusted authority to the end of the
 * credential file.  You must call spx_tf_init() before calling
 * spx_tf_save_ta().
 *
 * Returns ASUCCESS if all goes well, TKT_FIL_INI if spx_tf_init() wasn't
 * called previously, and AFAILURE for anything else that went wrong.
 */

spx_tf_save_ta(ta_cred)
TrustedCred *ta_cred;
{
    TrustedAuthority  *ta;
    off_t   lseek();
    int     count, tacnt;		/* count for write */
    char    buf[PRIVATE_KEY_SIZE], intbuf[20];
    int     buflen;


    if (fd < 0) {		/* fd is ticket file as set by spx_tf_init */
	  if (cdc_debug)
	      fprintf(stderr, "spx_tf_save_tf called before spx_tf_init.\n");
	  return TKT_FIL_INI;
    }

    /* Find the end of the ticket file */

    tacnt = ta_cred->count;
    (void) lseek(fd, 0L, 2);

    /* count */
    bzero(intbuf, 20);
    sprintf(intbuf, "%x", tacnt);
    count = strlen(intbuf);
    if (write(fd, intbuf, count) != count)
      goto bad;
    write(fd, "; ", 2);

    if (ta_cred->head == NULL) {
      if (tacnt == 0)  return(ASUCCESS);
      printf("ta_cred->head is NULL, tacnt is %d\n", tacnt);
      return(AFAILURE);
    }
    ta = ta_cred->head;

    /* Write the trusted authority and associated data */
    while (tacnt != 0) {
      /* fullname */
      count = strlen(ta->fullname);
      if (write(fd, ta->fullname, count) != count)
	goto bad;
      write(fd, "; ", 2);
      /*  uidlen  */
      bzero(intbuf, 20);
      sprintf(intbuf, "%x", ta->uidlen);
      count = strlen(intbuf);
      if (write(fd, intbuf, count) != count)
	goto bad;
      write(fd, "; ", 2);
      /* uuid */
      buflen = ta->uidlen;
      if (write(fd, ta->uuid, buflen) != buflen)
	goto bad;
      write(fd, "; ", 2);
      /* cross_certifier */
      bzero(intbuf, 20);
      sprintf(intbuf, "%x", ta->cross_certifier);
      count = strlen(intbuf);
      if (write(fd, intbuf, count) != count)
	goto bad;
      write(fd, "; ", 2);
      /* ca_pubRSAKey */
      bzero(buf, PRIVATE_KEY_SIZE);
      bcopy(ta->pubRSAKey, buf, PUBLIC_KEY_SIZE);
      if (write(fd, buf, PUBLIC_KEY_SIZE) != PUBLIC_KEY_SIZE)
	goto bad;
      write(fd, ";\n", 2);
      tacnt--;
      if (tacnt != 0) ta = ta->next;
    }
    /* Actually, we should check each write for success */
    return (ASUCCESS);
bad:
    return (AFAILURE);
}

spx_tf_get_cred(name, cred, ta_cred)
char *name;
ClaimantCred *cred;
TrustedCred  *ta_cred;
{
    char str[INT_STR];
    int  tacnt = 0;
    TrustedAuthority   *ta, *ta_new;

    if (fd < 0) {
	if (cdc_debug)
	    fprintf(stderr, "spx_tf_get_pname called before spx_tf_init.\n");
	return TKT_FIL_INI;
    }

    if (spx_tf_gets(cred->name, ANAME_SZ) < 0) {
	printf("cred->name - TKT_FIL_FMT\n");
	return(-1);
      }

    if (spx_tf_gets(cred->fullname, FULLNAME_SZ) < 0) {
	printf("cred->fullname - TKT_FIL_FMT\n");
	return(-1);
      }

    bzero(str, INT_STR);
    if (spx_tf_gets(str, INT_STR) < 0) {
	printf("uidlen - TKT_FIL_FMT\n");
	return(-1);
      }
    sscanf(str, "%x", &cred->uidlen);

    if (spx_tf_getbin(cred->uuid, cred->uidlen) < 0) {
	printf("uid - TKT_FIL_FMT\n");
	return(-1);
      }

    bzero(str, INT_STR);
    if (spx_tf_gets(str, INT_STR) < 0) {
	printf("before - TKT_FIL_FMT\n");
	return(-1);
      }
    sscanf(str, "%x", &cred->before);

    bzero(str, INT_STR);
    if (spx_tf_gets(str, INT_STR) < 0) {
	printf("after - TKT_FIL_FMT\n");
	return(-1);
      }
    sscanf(str, "%x", &cred->after);

    bzero(str, INT_STR);
    if (spx_tf_gets(str, INT_STR) < 0) {
	printf("deleg_flag - TKT_FIL_FMT\n");
	return(-1);
      }
    sscanf(str, "%x", &cred->deleg_flag);

    bzero(str, INT_STR);
    if (spx_tf_gets(str, INT_STR) < 0) {
	printf("bitlen - TKT_FIL_FMT\n");
	return(-1);
      }
    sscanf(str, "%x", &cred->bitlen);

    cred->s_privRSAKey = (RSAKeyStorage *) calloc(sizeof(RSAKeyStorage), 1);
    if (cred->deleg_flag) {
      if (spx_tf_getbin(cred->s_privRSAKey, PRIVATE_KEY_SIZE) < 0) {
	printf("s_privkey - TKT_FIL_FMT\n");
	return(-1);
      }
    } else {
      bzero(cred->s_privRSAKey, PRIVATE_KEY_SIZE);
    }

    bzero(str, INT_STR);
    if (spx_tf_gets(str, INT_STR) < 0) {
      printf("userticket_len - TKT_FIL_FMT\n");
      return(-1);
    }
    sscanf(str, "%x", &cred->userticket_len);

    if ((spx_tf_getbin(cred->userticket, cred->userticket_len)) < 0) {
	printf("userticket - TKT_FIL_FMT\n");
	return(-1);
      }

    bzero(str, INT_STR);
    if ((spx_tf_gets(str, INT_STR)) < 0) {
      printf("ta count - TKT_FIL_FMT\n");
      return(-1);
    }
    sscanf(str, "%x", &ta_cred->count);

    if (ta_cred->count == 0) {
      ta_cred->head = NULL;
      return(ASUCCESS);
    }
    ta = (TrustedAuthority *) malloc(sizeof(TrustedAuthority));
    ta_cred->head = ta;
    ta->next = NULL;
    tacnt = ta_cred->count;
    while (tacnt) {
      if (spx_tf_gets(ta->fullname, FULLNAME_SZ) < 0) {
	printf("ta fullname - TKT_FIL_FMT\n");
	return(-1);
      }
      bzero(str, INT_STR);
      if (spx_tf_gets(str, INT_STR) < 0) {
	printf("ta uidlen - TKT_FIL_FMT\n");
	return(-1);
      }
      sscanf(str, "%x", &ta->uidlen);
      if (spx_tf_getbin(ta->uuid, ta->uidlen) < 0) {
	printf("ta uid - TKT_FIL_FMT\n");
	return(-1);
      }
      bzero(str, INT_STR);
      if (spx_tf_gets(str, INT_STR) < 0) {
	printf("ta cross_certifier - TKT_FIL_FMT\n");
	return(-1);
      }
      sscanf(str, "%x", &ta->cross_certifier);
      ta->pubRSAKey = (RSAKeyStorage *) calloc(sizeof(RSAKeyStorage), 1);
      if (spx_tf_getbin(ta->pubRSAKey, PUBLIC_KEY_SIZE) < 0) {
	printf("ta pubkey - TKT_FIL_FMT\n");
	return(-1);
      }
      tacnt--;
      if (tacnt) {
	ta_new = (TrustedAuthority *) malloc(sizeof(TrustedAuthority));
	ta->next = ta_new;
	ta = ta_new;
	ta->next = NULL;
      }
    }
    return(ASUCCESS);
}
