;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald ALGEBRAIC) 


;Transforms for Simplification: Transforms rely on lower-level
;objects called expression processors; these have the basic algebraic
;simplification routines hardwired in. 

(define-operation (ALGEBRAIC-SUB-PROCESSOR soi))
(define-operation (PROCESSOR-SIMPLIFY processor expr params))
(define-operation (PROCESSOR-REDUCED-TERMS processor))
(define-predicate PROCESSOR?)
(define-operation (PROCESSOR-VALIDITY-CONDITIONS processor))
(define-operation (PARTITION-SUMMATION processor expr params))
;;;Given a summation expression a_1+...+a_n, how to break it up as
;;(b_1+...+b_n)-(c_1+...+c_n). See order.t for the default way of doing this.

(define-structure-type ALGEBRAIC-PROCESSOR
  language
  scalars-type
  exponent-processor
  coefficient-processor
  numeral-to-term-function		;a mapping from numerals to terms
  constant-recognizer-function		;which terms are numerals
  term-to-numeral-function		;a mapping from terms to numerals
  faithful-numeral-representation?	;true if map from numerals to terms is injective
  -r
  +r
  *r
  ^r
  sub-r
  /r
  reduced-terms
  handled-operators
  commutes
  expand
  cancellation-valid?
  sum-partitioner
  rewrite-rules
  (((algebraic-sub-processor soi) soi)
   ((processor-validity-conditions soi)
    (algebraic-processor-validity-conditions soi))
   ((processor-reduced-terms soi) (algebraic-processor-reduced-terms soi))
   ((partition-summation processor expr params)
    ((algebraic-processor-sum-partitioner processor) processor expr params))
   ((processor? soi) '#t)))

(define (operation-sorts op)
  (if op (make-set
	  (cons (higher-sort-range (expression-sorting op))
		(higher-sort-domains (expression-sorting op))))
      the-empty-set))

(define (ALGEBRAIC-PROCESSOR-VALIDITY-CONDITIONS processor)
  
  (if (and (not (ring-processor? processor)) (commutative? processor))
      (imps-error "ALGEBRAIC-PROCESSOR-VALIDITY-CONDITIONS: commutativity is an invalid
declaration for a non-ring algebraic processor"))
  (if (and (not (ring-processor? processor)) (or (^r processor) (/r processor)))
      (imps-error "ALGEBRAIC-PROCESSOR-VALIDITY-CONDITIONS: algebraic operation ~A is not
allowed for a non-ring algebraic processor" (or (^r processor) (/r processor))))
  (let ((sorts+  (operation-sorts (+r processor)))
	(sorts* (operation-sorts (*r processor)))
	(sorts^ (if (^r processor)
		    (make-set (list
			       (higher-sort-range (expression-sorting (^r processor)))
			       (car (higher-sort-domains (expression-sorting (^r processor))))))
		    the-empty-set))
	(sorts-sub (operation-sorts (sub-r processor)))
	(sorts-minus (operation-sorts (-r processor)))
	(sorts-/ (operation-sorts (/r processor))))
	 
    (if (and (ring-processor? processor)
	     (< 1 (cardinality (big-u (list sorts^ sorts-sub sorts-minus sorts-/ sorts+ sorts*)))))
	(imps-error "ALGEBRAIC-PROCESSOR-VALIDITY-CONDITIONS: algebraic ring operations
have improper sortings."))

    (if (< 1 (cardinality (big-u (list sorts+ sorts-sub sorts-minus))))
	(imps-error "ALGEBRAIC-PROCESSOR-VALIDITY-CONDITIONS: algebraic operations do not have identical domains and ranges")))

  (let* ((0-sort (number->scalar-constant processor 0))
	 (1-sort (number->scalar-constant processor 1))
	 
	 (0-coefficient-sort (number->scalar-constant (coefficient-processor processor) 0))
	 (1-coefficient-sort (number->scalar-constant (coefficient-processor processor) 1))

	 (0-exp-sort (number->exponent-constant processor 0))
	 (1-exp-sort (number->exponent-constant processor 1))
	 (-1-exp-sort (number->exponent-constant processor -1))
	 (formulas '())
	 (sort (car (higher-sort-domains (expression-sorting (+r processor)))))
	 (exp-sort (if (^r processor)
		       (cadr (higher-sort-domains
			      (expression-sorting (^r processor))))
		       sort))

	 (coefficient-sort 
	  (car (higher-sort-domains (expression-sorting (*r processor)))))
;;;	 (coefficient-base-sort
;;;	  (car (higher-sort-domains
;;;		(expression-sorting (+r (coefficient-processor processor))))))
;;;	 
	 (x (find-variable 'x sort))
	 (y (find-variable 'y sort))
	 (z (find-variable 'z sort))
	 (m (find-variable 'm exp-sort))
	 (n (find-variable 'n exp-sort))
	 
	 (c (find-variable 'c coefficient-sort))
	 (d (find-variable 'd coefficient-sort))

	 (+exp (lambda (a b) (apply-operator (+r (exponent-processor processor)) a b)))
	 (*exp (lambda (a b) (apply-operator (*r (exponent-processor processor)) a b)))
	 (*op (lambda (a b) (apply-operator (*r processor) a b)))

	 (*ext-op (lambda (a b) (apply-operator (*ext-r processor) a b)))
	 (+ext-op (lambda (a b) (apply-operator (+r (coefficient-processor processor)) a b)))


	 (+op (lambda (a b) (apply-operator (+r processor) a b)))
	 (/op (lambda (a b) (apply-operator (/r processor) a b)))
	 (subop (lambda (a b) (apply-operator (sub-r processor) a b)))
	 (^op (lambda (a b) (apply-operator (^r processor) a b)))
	 (-op (lambda (a) (apply-operator (-r processor) a))))

    (or 0-sort (imps-error "ALGEBRAIC-PROCESSOR-VALIDITY-CONDITIONS: processor has no zero -element."))
    
    (push formulas (equality (+op x y) (+op y x)))
    (push formulas (equality (+op x 0-sort) x))
    (push formulas (equality (+op (+op x y) z) (+op x (+op y z))))
    (if (*r processor)
	(block
	  (or 1-coefficient-sort
	      (imps-error "ALGEBRAIC-PROCESSOR-VALIDITY-CONDITIONS: processor has no multiplicative unit."))
	  
	  (if (commutative? processor)
	      (push formulas (equality (*op x y) (*op y x))))
	  (if (processor-cancellation-valid? processor)
	      (if (or (-r processor) (sub-r processor))
		  (push formulas
			(biconditional (equality (*op x y) 0-sort)
				       (disjunction (equality x 0-sort)
						    (equality y 0-sort))))
		  (push formulas
			(implication (equality (*op x y) (*op x z))
				     (equality y z)))))


	  (if (not (-r processor))
	      (push formulas (equality (*op 0-coefficient-sort x) 0-sort)))

	  (push formulas (equality (*op 1-coefficient-sort x) x))
	  (push formulas (equality (*op c (+op y z)) (+op (*op c y) (*op c z))))
	  (if (ring-processor? processor)
	      (if (not (commutative? processor))
		  (push formulas (equality (*op (+op y z) x)
					   (+op (*op y x) (*op z x))))
		  (push formulas (equality (*op (*op x y) z) (*op x (*op y z)))))
	      (if (*ext-r processor)
		  (block
		    (push formulas (equality (*op (+ext-op c d) x)
					     (+op (*op c x) (*op d x))))
		    (push formulas (equality (*op (*ext-op c d) z) (*op c (*op d z))))
		    )))))
    (if (^r processor)
	(block
	  (push formulas (implication
			  (defined-in (*op (^op x m) (^op x n)) sort)
			  (equality (^op x (+exp m n)) (*op (^op x m) (^op x n)))))
	  (push formulas (implication
			  (disjunction
			   (defined-in (*op (^op x m) (^op y m)) sort)
			   (defined-in (^op (*op x y) m) sort))
			  (equality (*op (^op x m) (^op y m)) (^op (*op x y) m))))

				       
	  (push formulas (equality (^op x 1-exp-sort) x))
	  (push formulas (implication
			  (defined-in (^op x 0-exp-sort) sort)
			  (equality (^op x 0-exp-sort) 1-sort)))
	  
	  (push formulas (implication
			  (defined-in (^op 1-sort n) sort)
			  (equality (^op 1-sort n) 1-sort)))

	  (push formulas (implication (defined-in (^op 0-sort m) sort)
				      (equality (^op 0-sort m) 0-sort)))

	  (push formulas (implication
			  (defined-in (^op (^op x m) n) sort)
			  (equality (^op (^op x m) n) (^op x (*exp m n)))))
	  (push formulas (biconditional
			  (conjunction
			   (defined-in (^op x m) sort)
			   (defined-in (^op x n) sort))
			  (defined-in (^op (^op x m) n) sort)))))
    (if (and (/r processor) (^r processor))
	(push formulas
	      (implication
	       (disjunction (is-defined (/op x y))
			    (is-defined (*op x (^op y -1-exp-sort))))
	       (equality (/op x y) (*op x (^op y -1-exp-sort))))))
    (if (sub-r processor)
	(if (-r processor);;easiest to characterize sub-r in terms of -:
	    (push formulas (equality (subop x y) (+op x (-op y))))
	    ;;otherwise do it directly:
	    (block (push formulas (equality (subop x y) (+op x (subop 0-sort y))))
		   (push formulas (equality (+op x (subop 0-sort x)) 0-sort)))))
    (if (-r processor)
	(push formulas (equality (+op x (-op x)) 0-sort)))
    (if (not (eq? processor (exponent-processor processor)))
	(set formulas
	     (append formulas (processor-validity-conditions
			       (exponent-processor processor)))))

    (if (not (eq? processor (coefficient-processor processor)))
	(set formulas
	     (append formulas (processor-validity-conditions
			       (coefficient-processor processor)))))


    ;;add other rules here.
    
    ;;;add rewrite rules:
    (union (map rewrite-rule-formula (algebraic-processor-rewrite-rules processor))
	   formulas)))

(define (ALGEBRAIC-PROCESSOR-APPLY-REWRITE-RULES processor expr params)
  (iterate loop ((rules (algebraic-processor-rewrite-rules processor))
		 (expr expr))
    (if (null? rules) expr
	(receive (new-expr reqs ())
	  ((car rules)
	   (processor-parameters-context params)
	   expr
	   (processor-parameters-persistence params))
	  (set (processor-parameters-requirements params)
	       (set-union (processor-parameters-requirements params)
			  reqs))
	  (loop (cdr rules) new-expr)))))

(define (ALGEBRAIC-PROCESSOR-INSISTENTLY-APPLY-REWRITE-RULES processor expr params)
  (let ((simp (algebraic-processor-apply-rewrite-rules processor expr params)))
    (if (eq? simp expr)
	expr
	(algebraic-processor-insistently-apply-rewrite-rules processor simp params))))


(define (ALGEBRAIC-PROCESSOR-SIMPLIFY processor expr params)
  (if (processor-reduced? processor expr params)
      expr
      (let ((expr (algebraic-processor-insistently-apply-rewrite-rules processor expr params)))
	(if (application? expr)
	    (select (operator expr)
	      (((+r processor)) (annotate-expression-as-reduced
				 processor
				 (simp+ processor expr params)
				 params))
	      (((^r processor)) (annotate-expression-as-reduced
				 processor
				 (if (and (algebraic-processor-expand processor)
					  (commutative? processor))
				     (expand^ processor expr params)
				     (simp^ processor expr params))
				 params))
	      (((*r processor)) (annotate-expression-as-reduced
				 processor
				 (if (eq? processor (coefficient-processor processor))
				     (if (algebraic-processor-expand processor)
					 (expand* processor expr params)
					 (simp* processor expr params))
				     (simp*-1 processor expr params))
				 params))
	      (((-r processor))
	       (annotate-expression-as-reduced
		processor
		(simp- processor expr params) params))
	      (((sub-r processor)) (annotate-expression-as-reduced
				    processor
				    (simp-sub processor expr params)
				    params))
	      (((/r processor))
	       (annotate-expression-as-reduced
		processor (simp/ processor expr params) params))
	      (else (simplify-by-transforms 
		     (processor-parameters-context params)
		     expr
		     (processor-parameters-persistence params))))
	  
	    (simplify-by-transforms 
	     (processor-parameters-context params)
	     expr
	     (processor-parameters-persistence params))))))

(define (ALGEBRAIC-PROCESSOR-SIMPLIFY-WITH-REQUIREMENTS processor context expr persist)
  (if (and (application? expr)
	   (memq (operator expr)
		 (algebraic-processor-handled-operators processor)))
      (let ((params (make-processor-parameters)))
	(set (processor-parameters-persistence params) persist)
	(set (processor-parameters-context params) context)
	(let ((simplified
	       (algebraic-processor-simplify processor expr params)))
	  (return simplified (processor-parameters-requirements params) '#t)))
      (return expr '() '#f)))
      

;;;(define (ALGEBRAIC-PROCESSOR-SIMPLIFY-COMPONENTS processor expr params)
;;;  (cond	((formal-symbol? expr) expr)
;;;	(else
;;;	 (apply apply-operator
;;;		(operator expr)
;;;		(map (lambda (x) (algebraic-processor-simplify processor  x params))
;;;		     (expression-components expr))))))

(define (+r processor) (algebraic-processor-+r (algebraic-sub-processor processor)))
(define (-r processor) (algebraic-processor--r (algebraic-sub-processor processor)))
(define (*r processor) (algebraic-processor-*r (algebraic-sub-processor processor)))

(define (*ext-r processor) (algebraic-processor-*r (coefficient-processor processor)))

(define (^r processor) (algebraic-processor-^r (algebraic-sub-processor processor)))
(define (sub-r processor) (algebraic-processor-sub-r (algebraic-sub-processor processor)))
(define (/r processor) (algebraic-processor-/r (algebraic-sub-processor processor)))

(define (commutative? processor)
  (algebraic-processor-commutes (algebraic-sub-processor processor)))
(define (exponent-processor processor)
  (algebraic-processor-exponent-processor (algebraic-sub-processor processor)))

(define (coefficient-processor processor)
  (algebraic-processor-coefficient-processor (algebraic-sub-processor processor)))

(define (scalars-type  processor)
  (algebraic-processor-scalars-type (algebraic-sub-processor processor)))
(define (processor-cancellation-valid? processor)
  (algebraic-processor-cancellation-valid? (algebraic-sub-processor processor)))
(define (processor-language processor)
  (algebraic-processor-language (algebraic-sub-processor processor)))

(define (processor-numeral-to-term-function processor)
  (algebraic-processor-numeral-to-term-function (algebraic-sub-processor processor)))
(define (processor-constant-recognizer-function processor)
 (algebraic-processor-constant-recognizer-function (algebraic-sub-processor processor)))
(define (processor-term-to-numeral-function processor)
  (algebraic-processor-term-to-numeral-function (algebraic-sub-processor processor)))
(define (processor-faithful-numeral-representation? processor)
  (algebraic-processor-faithful-numeral-representation? (algebraic-sub-processor processor)))

(define (ring-processor? processor)
  (and (*r processor)
       (eq? (coefficient-processor processor) processor)))

(define (BUILD-ALGEBRAIC-PROCESSOR
	 language             ;The language which contains the basic operations.
	 
	 scalars              ;A numerical data type. See numerical-objects in resources.
	 
	 operations-alist     ;operations-alist is an alist ((name-1 const-1) ... (name-n const-n))
	 
	 commutes)

 
  
  (let ((+r-op (cadr (assq '+ operations-alist)))	
	(*r-op (cadr (assq '* operations-alist)))
	(^r-op (cadr (assq '^ operations-alist)))
	(-r-op (cadr (assq '- operations-alist)))
	(sub-r-op (cadr (assq 'sub operations-alist)))
	(/r-op (cadr (assq '/ operations-alist)))
	
	(0r (cadr (assq 'zero operations-alist)))
	(1r (cadr (assq 'unit operations-alist)))
	
	(obj (make-algebraic-processor)))
    (set (algebraic-processor-language obj) language)
    
    (set (algebraic-processor-scalars-type obj) scalars)

    ;again the default
    (set (algebraic-processor-exponent-processor obj) obj)
    (set (algebraic-processor-coefficient-processor obj) obj)
    (set (algebraic-processor--r obj) -r-op)
    (set (algebraic-processor-+r obj) +r-op)
    (set (algebraic-processor-*r obj) *r-op)
    (set (algebraic-processor-^r obj) ^r-op)
    (set (algebraic-processor-sub-r obj) sub-r-op)
    (set (algebraic-processor-/r obj) /r-op)
    (set (algebraic-processor-expand obj) '#t)
    (set (algebraic-processor-commutes obj) commutes)
    (set (algebraic-processor-reduced-terms obj) (make-table))
    (set (algebraic-processor-rewrite-rules obj) '())
    
    (set (algebraic-processor-handled-operators obj)
	 (let ((accum '()))
	   (walk
	    (lambda (x) (if x (push accum x)))
	    (list -r-op +r-op *r-op ^r-op sub-r-op /r-op))
	   accum))
    (set (algebraic-processor-cancellation-valid? obj) (true? (/r obj)))
    (set (algebraic-processor-sum-partitioner obj) default-summation-partitioner)
    (use-trivial-scalar-correspondences obj 0r 1r)
    obj))


(define (USE-TRIVIAL-SCALAR-CORRESPONDENCES proc 0r 1r)
  (set (algebraic-processor-numeral-to-term-function proc)
       (lambda (n)
	 (let ((expr (numeral->repeated-sum-of-ones proc 0r 1r n)))
	   (annotate-expression-as-universally-reduced proc expr '#f)
	   expr)))
  (set (algebraic-processor-constant-recognizer-function proc)
       (lambda (expr)
	 (let ((n (repeated-sum-of-ones->numeral proc 0r 1r expr)))
	   (if n (annotate-expression-as-universally-reduced proc expr '#f))
	   n)))
  (set (algebraic-processor-term-to-numeral-function proc)
       (processor-constant-recognizer-function proc))
  (set (algebraic-processor-faithful-numeral-representation? proc) '#f)
  ;;;
  proc)

(define (PROCESSOR-SOUND-IN-THEORY? processor theory)
  (or (memq? processor (theory-valid-processors theory))
      (let ((valid?
	     (every? (lambda (x)
		       (let ((thm? (theory-theorem? theory x)))
			 (if (not thm?) (format '#t "~A fails to be a theorem.~%" x))
			 thm?))
		     (processor-validity-conditions processor))))
	(if valid? (set (theory-valid-processors theory)
			(add-set-element processor (theory-valid-processors theory))))
	valid?)))

(define (UNIVERSAL-REDUCTION-ENTRY? entry)
  (eq? entry 'universal))

(define (UNIVERSAL-REDUCTION-ENTRY) 'universal)

(define (PROCESSOR-UNIVERSALLY-REDUCED? processor expr params)
  (ignore params)
  (universal-reduction-entry? 
   (table-entry (processor-reduced-terms processor) expr)))

(define (PROCESSOR-REDUCED? processor expr params)
  (let ((entry (table-entry (processor-reduced-terms processor) expr)))
    (or (universal-reduction-entry? entry)
	(let ((val (assq (processor-parameters-context params) entry)))

	 ;;in the case of formulas, the formula is considered reduced if it is reduced in
	 ;;the same context with higher persistence.

	  (and val (<= (processor-parameters-persistence params) (cdr val)))))))


(define (ANNOTATE-EXPRESSION-AS-UNIVERSALLY-REDUCED processor expr params)
  (ignore params)
  (set  (table-entry (processor-reduced-terms processor) expr)
	(universal-reduction-entry)))

(define (ANNOTATE-EXPRESSION-AS-REDUCED processor expr params)
  (or (processor-universally-reduced? processor expr params)
      (push (table-entry (processor-reduced-terms processor) expr)
	    (cons (processor-parameters-context params)
		  (processor-parameters-persistence params))))
  expr)
;;;
;;;(define (ANNOTATE-EXPRESSION-AS-REDUCED processor expr params)
;;;  (if (not (formula? expr))
;;;      (set (table-entry (processor-reduced-terms processor) expr) expr)
;;;      (block
;;;	(push (table-entry (processor-reduced-terms processor) expr)
;;;	      (cons (processor-parameters-context params)
;;;		    (processor-parameters-persistence params)))
;;;	expr)))
;;;
;;;(define (PROCESSOR-REDUCED? processor expr params)
;;;   (if (formula? expr)
;;;       (let ((val (assq (processor-parameters-context params)
;;;			(table-entry (processor-reduced-terms processor) expr))))
;;;
;;;	 ;;in the case of formulas, the formula is considered reduced if it is reduced in
;;;	 ;;the same context with higher persistence.
;;;
;;;	 (and val (<= (processor-parameters-persistence params) (cdr val))))
;;;       (table-entry (processor-reduced-terms processor) expr)))

(define (ZZ-EXTENSION? processor)
  ;;;says when we can use LISP arbitrary precision arithmetic. Used in 
  ;;;equality algebraic processors.
  (integer-extension-type? (scalars-type processor)))


(define (BASE-SORT processor)
  (car (higher-sort-domains (expression-sorting (+r processor)))))

(define (BASE-SORTED? processor expr)
  (eq? (base-sort processor) (expression-sorting expr)))

(define (EXP-SORT processor)
  (if (^r processor)
      (cadr (higher-sort-domains
	     (expression-sorting (^r processor))))
      (base-sort processor)))


;                        MATHEMATICAL ASSUMPTIONS for ring processors
;Let B be the base sort, 
;Let E the base sort for exponents,
;Let N be the scalar semi-ring (one of the semi-rings nn, zz, qq, or zz_p.)
;(+,*) makes base-sort into a semi-ring which contains (as subalgebras) 
;the scalar semi-ring, 
;- is the inverse operation for addition, sub is subtraction. If these operations
;are present B is a ring. This is also true if N is zz, qq or zz_p.
;In particular, +,* are total functions on the base sort. 

;^:B x E -> B requires some care.

;;Here are some examples:

;;;Suppose R is a field. Then
;;; (i) x^m is defined for m a positive integer, by x*x* ...*x m-times.
;;; (ii) x^m is defined if m<0 and x<>0, by x^m=(1/x)^-m=1/(x^-m)
;;; (iii) x^0 is defined if x<>0 to be 1.
;;; (iv)  0^0 is undefined.

;;Suppose R is an integral domain. Then
;;; (i) for any integer 1<=m define x^m by x*x* ...*x m-times.
;;; (ii) If m=0 and not(x=0) define x^m=1
;;; (iii) If x=0 and not(m=0) define x^m=0.
;;; (iv)  0^0 is undefined.

;;Suppose R is a ring. Then
;;; (i) For any integer 1<=m define x^m by x*x* ...*x m-times.
;;; (ii) x^m is undefined if m<=0.

;(a)Since * is total, x^m,x^n are defined iff x^m*x^n is defined. In this case
;         x^m*x^n = x^(m+n)

;The simplifier uses this rule only in the direction lhs->rhs.
;x^m,x^n have to be kept along as convergence requirements, since x^(m+n) may converge 
;without x^m,x^n converging.

;(b)If x^a,x^b are defined iff (x^a)^b is defined. In this case
;         (x^a)^b = x^(a*b)

;The simplifier uses this rule only in the direction lhs->rhs.
;x^a,x^b have to be kept along as convergence requirements. 

;(c) x^m* y^m are defined implies (x*y)^m is defined. In this case
;          x^m * y^m = (x*y)^m
;x^m,y^m have to be retained as convergence requirements.

;However, for expansion (see expand.t), this equality is actually used both ways,
;so rhs defined -> lhs defined must also hold if expansion is used. That is to say,
;whenever expansion is used:
;(c') if (x * y)^m is defined then x^m and y^m must both be defined, or else we may be turning
;a defined expression into an undefined one.


;  (d) x^1=x is true for any value of x.

;  (e) 1^x=1 is true for any value of x. x is a convergence requirement.

;  (f) 0^n = 0 whenever this is defined.  0^n is a convergence requirement.

;  (g) x^0=1 whenever this is defined (whenever not(x=0)). In this case we require the
;      formula not(x=0).

;  (h) 0^0 is necessarily undefined. (Apply (f) and (g))


;                        
;l = (scalar-base-language processor):
;(a) (+r processor)  must an operation in l.
;(b) (^r processor), (*r processor), (-r processor), (sub-r processor), (/r processor) must either be operations in l or '()
;(c) scalars-type is a numerical-type. 
;(d) For n=0,1,2..., ((processor-numeral-to-term-function processor) (coerce-type (scalars-type processor) n)) must be non-nil.  
;(e) The set of numerical-objects which are in the range of
;(processor-term-to-numeral-function processor) is closed under numerical-x
;(x = +,*,-,^) if (xr processor) is non-nil; otherwise a run-time error may result.



(define (SCALAR-CONSTANT? processor expr)
  ((processor-constant-recognizer-function processor) expr))

(define (SCALAR-CONSTANT-=0? processor expr)
  (and (scalar-constant? processor expr)
       (numerical-=0? (scalar-constant->numerical-object processor expr))))
(define (SCALAR-CONSTANT-=1? processor expr)
  (and (scalar-constant? processor expr)
       (numerical-=1? (scalar-constant->numerical-object processor expr))))


;scalar-base-language is supposed to contain all scalar-constants.
;The following function may return () for some numbers n 
;depending on whether the there is a constant in the language 
;whose name is (coerce-type (scalars-type processor) n).

(define (NUMERICAL-OBJECT->SCALAR-CONSTANT processor num)
  ((processor-numeral-to-term-function processor) num))

(define (SCALAR-CONSTANT->NUMERICAL-OBJECT processor expr)
  ((processor-term-to-numeral-function processor) expr))

(define (NUMBER->SCALAR-CONSTANT processor n)
  (let ((try
	 ((processor-numeral-to-term-function processor)
	  (coerce-type (scalars-type processor) n))))
    (if try try (undefined (base-sort processor)))))

(define (NUMBER->EXPONENT-CONSTANT processor n)
  (number->scalar-constant (exponent-processor processor) n))


;Note that exponents may be of different type than than scalars.

(define (EXPONENT-TYPE processor)
  (scalars-type (exponent-processor processor)))

(define (INHIBIT-EXPONENTIATION? processor)
  (null? (^r processor)))

(define (INHIBIT-MULTIPLICATION? processor)
  (null? (*r processor)))

(define-operation (ALLOW-NEGATIVE-EXPONENTS? x) (true))
;this is only used for requiring convergence of subexpressions.

;if (operator expr) is associative, then associative-arguments gets all arguments in expr
;as well as arguments of subexpressions whose operator is op and therefore whose arguments
;can be combined associatively.
;IMPORTANT NOTE: associative-arguments creates new cons cells
;(e.g., it does not use any list structure from (expression-components expr))

(define (ASSOCIATIVE-ARGUMENTS expr)
  (let ((op (operator expr))
	(accumulated-args '()))
    (iterate collect-args ((expr expr))
      (if (and (application? expr)
	       (eq? (operator expr) op))
	  (walk (lambda (x) (collect-args x)) (arguments expr))
	  (push accumulated-args expr)))
    (reverse! accumulated-args)))


;;This stuff shouldn't be here:
;;;
;;;If the convergence requirements are all true and t->t' then either
;;;(a) Both t,t' are convergent and have the same value.
;;;(b) Both t,t' are divergent.
;;;On the other hand if one of the convergence requirements
;;;happens to be false, then t is undefined. Nothing is said concerning t'.
;;;Thus  the following identity is true:
;;;t = if(all-convergence-requirements,t',?sort(t))


;;;convergence handlers are objects which keep track of a context and a list of 
;;;formulas. They should be at least able to handle the operation REQUIRE-FORMULA:

(define-operation (REQUIRE-FORMULA params formula))

(define (REQUIRE-CONVERGENCE processor params expr)
  (let ((sort (base-sort processor)))
    (require-formula params (defined-in expr sort))))

(define (REQUIRE-CONVERGENCE-EVERY processor params expr-list)
  (walk (lambda (x) (require-convergence processor params x)) expr-list))

(define (REQUIRE-CONVERGENCE-EVERY-FACTOR processor params expr-list)
  (let ((expr-list-1 (reverse expr-list)))
    (require-convergence processor params (car expr-list-1))
    (require-convergence-every (coefficient-processor processor) params (cdr expr-list-1))))

;;End This stuff shouldn't be here:

(define-structure-type PROCESSOR-PARAMETERS
;;  base-sort
  context
  requirements
  persistence
  (((require-formula soi expr)
    (push (processor-parameters-requirements soi) expr))))

(set (processor-parameters-requirements (stype-master processor-parameters-stype)) '())
(set (processor-parameters-persistence (stype-master processor-parameters-stype))
     (context-simplification-persistence))
(set (processor-parameters-context (stype-master processor-parameters-stype))
     '#f)

(define (PROCESSOR-SIMPLIFY-WITH-REQUIREMENTS processor context expr persist)
  (let ((params (make-processor-parameters)))
    (set (processor-parameters-persistence params) persist)
    (set (processor-parameters-context params) context)
;;    (set (processor-parameters-base-sort params) (base-sort processor))
    (let ((simplified
	   (processor-simplify processor expr params)))
      (return simplified (processor-parameters-requirements params)))))


;some transforms for algebraic simplification:

(define (ADDITION? processor expr)
  (and (application? expr) (eq? (operator expr) (+r processor))))
(define (MULTIPLICATION? processor expr)
  (and (application? expr) (eq? (operator expr) (*r processor))))
(define (EXPONENTIATION? processor expr)
  (and (application? expr) (eq? (operator expr) (^r processor))))
(define (SIGN-NEGATION? processor expr)
  (and (application? expr) (eq? (operator expr) (-r processor))))
(define (SUBTRACTION? processor expr)
 (and (application? expr) (eq? (operator expr) (sub-r processor))))
(define (DIVISION? processor expr)
 (and (application? expr) (eq? (operator expr) (/r processor))))

;;;(define (NUMERICAL-GROUND-EXPRESSION? expr)
;;;    (cond ((numerical-constant? expr))
;;;	  ((formal-symbol? expr) nil)
;;;	  ((application? expr)
;;;	   (every (lambda (x) (numerical-ground-expression? x)) (arguments expr)))
;;;	  (else nil)))
;;;       
;tally-objects and tally-charts do the bookeeping for processing expressions
;with associative operators.

;a tally-object is to be thought of as a monomial: which has a numerical coefficient (the weight)
;a symbolic multiplicand (the label).

;;;(define-structure-type TALLY-OBJECT
;;;  label            
;;;  weight)
;;;
;;;(define (INIT-TALLY-OBJECT label weight)
;;;  (let ((to (make-tally-object)))
;;;    (set (tally-object-weight to) weight)
;;;    (set (tally-object-label to) label)
;;;    to))
;;;
;;;(define (TALLY tally-object increase proc)
;;;  (set (tally-object-weight tally-object)
;;;       (proc (tally-object-weight tally-object) increase)))
;;;

(define-integrable (init-tally-object label weight)
  (cons label weight))

(define-integrable (tally tally-object increase proc)
  (set (cdr tally-object) (proc (cdr tally-object) increase)))

(define-integrable (tally-object-label x) (car x))

(define-integrable (tally-object-weight x) (cdr x))

;a tally-chart is to be thought of as a polynomial (with possibly non-commutative operations.)
;a tally chart is essentially a kind of ASSOCIATIVE ARRAY.

(define-structure-type TALLY-CHART
 
  scalar                            ;the independent term.
 
  scalar-accumulator                ;procedure of one argument
 
  label-accumulator                 ;procedure of one argument
 
  label-equivalence                 ;two-place equivalence relation between tally-objects
 
  comparator                        ;either '() or a two-place order predicate 
                                    ;for comparing tally-objects
 
  object-list)                      ;list of tally-objects

(define (INIT-TALLY-CHART scalar-init
			  scalar-accumulator
			  label-accumulator
			  label-equivalence
			  comparator)

  (let ((atc (make-tally-chart)))
    (set (tally-chart-scalar atc) scalar-init)
    (set (tally-chart-scalar-accumulator atc) scalar-accumulator)
    (set (tally-chart-label-accumulator atc) label-accumulator)
    (set (tally-chart-label-equivalence atc) label-equivalence)
    (set (tally-chart-comparator atc) comparator)
    (set (tally-chart-object-list atc) '())
    atc))

(define (ACCUMULATE-SCALAR atc increase)
  (set (tally-chart-scalar atc)
       ((tally-chart-scalar-accumulator atc) (tally-chart-scalar atc) increase)))

(define (ACCUMULATE-LABEL atc label increase)
  (cond ((tally-chart-comparator atc)
	 (iterate loop ((rest (tally-chart-object-list atc)))
	   (cond ((null? rest)
		  (push (tally-chart-object-list atc)
			(init-tally-object label increase)))
		 (((tally-chart-label-equivalence atc)
		   label
		   (tally-object-label (car rest)))
		  (tally (car rest) increase (tally-chart-label-accumulator atc)))
		 (else (loop (cdr rest))))))
	(else
	 (cond ((and
		 (tally-chart-object-list atc)
		 ((tally-chart-label-equivalence atc)
		  label
		  (tally-object-label (car (tally-chart-object-list atc)))))
		(tally (car (tally-chart-object-list atc))
		       increase
		       (tally-chart-label-accumulator atc)))
	       (else (push (tally-chart-object-list atc) (init-tally-object label increase)))))))

(define (LABEL-TALLIES atc)
  (cond ((tally-chart-comparator atc)
	 (sort (tally-chart-object-list atc)
	       (lambda (a b) ((tally-chart-comparator atc)
			      (tally-object-label a)
			      (tally-object-label b)))))
	(else (tally-chart-object-list atc))))

(define (SIMP+ processor expr params)
  (sum-expression-list
   processor
   (map! (lambda (x)
	   (algebraic-processor-simplify processor x params))
	 (associative-arguments expr))
   params))

(define (SUM-EXPRESSION-LIST processor expr-list params)
  (let ((chart (make-weighted-sum-tally-chart processor)))
    (walk (lambda (x) (weighted-sum-accumulate-expression processor chart x)) expr-list)
    (weighted-sum-tally-chart->expression processor chart params)))

(define (MULTIPLY-EXPRESSION-LIST processor expr-list params)
  (let ((chart (make-weighted-product-tally-chart processor)))
    (walk (lambda (x) (weighted-product-accumulate-expression processor chart x))
	  expr-list)
    (weighted-product-tally-chart->expression processor chart params)))

(define (SIMP* processor expr params)
  (multiply-expression-list
   processor
   (map! (lambda (x) (if (exponentiation? processor x)
			 (simp^ processor  x params)
			 (algebraic-processor-simplify processor  x params)))
	 (associative-arguments expr))
   params))

(define (associative-scalars-and-vector expr)
  (let* ((reverse-args (reverse! (associative-arguments expr)))
	 (vector-arg (car reverse-args))
	 (scalar-args (reverse! (cdr reverse-args))))
    (return scalar-args vector-arg)))
	 
    

(define (SIMP*-1 processor expr params)
  (receive (scalars vector)
    (associative-scalars-and-vector expr)
    (let ((arg1 (algebraic-processor-simplify
		 (coefficient-processor processor)
		 (form-product-expression (coefficient-processor processor) scalars)
		 params))
	  (arg2 (algebraic-processor-simplify
		 processor
		 vector
		 params)))
      (let* ((coeffs (if (addition? (coefficient-processor processor) arg1)
			 (associative-arguments arg1)
			 (list arg1)))
	     (vectors (if (addition? processor arg2)
			  (associative-arguments arg2)
			  (list arg2)))
	     (terms (map-independently
		     (lambda (x y)
		       (receive (scalars vector)
			 (if (multiplication? processor y)
			     (associative-scalars-and-vector y)
			     (return '() y))
			 (let* ((args (cons x scalars))
				(coeff 
				 (multiply-expression-list
				  (coefficient-processor processor)
				  args
				  params)))
			   (if (or (scalar-constant-=0? processor vector)
				   (scalar-constant-=0?
				    (coefficient-processor processor) coeff))
			       (number->scalar-constant processor 0)
			       (apply-operator (*r processor) coeff vector)))))
		     coeffs vectors)))
	(sum-expression-list
	 processor
	 terms
	 params)))))

(define (STRING-OUT-NESTED-EXPONENTIATION processor expr params)
  ;;the purpose of this bit of code is to do rewrites of the kind
  ;;(x^u)^v --> x^(u*v).
  ;;This may turn a possibly non-convergent term into a convergent one (for example if u=v=-1).
  (receive (base expr-list)
    (iterate loop ((expr (1starg expr))
		   (accum (list (algebraic-processor-simplify
				 (exponent-processor processor)  (2ndarg expr) params))))
      (if (exponentiation? processor expr)
	  (loop (1starg expr)
		(cons (algebraic-processor-simplify
		       (exponent-processor processor)  (2ndarg expr) params) accum))
	  
	  ;;even the following is not completely satisfactory, because expr may contain
	  ;;hidden exponentiations which come out after simplification
	  

	  (let ((expr-1 (algebraic-processor-simplify processor expr params)))
	    (if (exponentiation? processor expr-1);;continue.
		(loop (1starg expr-1) (cons (2ndarg expr-1) accum))
		(return expr-1 accum)))))
    ;;we have to require convergence at this point.
    (if (and (^r processor) (> (length expr-list) 1))
	(walk
	 (lambda (x) (if (scalar-constant-=1? (exponent-processor processor) x)
			 '()
			 (require-convergence processor
					      params
					      (apply-operator
					       (^r processor) base x))))
	 expr-list))
    (return base expr-list)))

(define (SIMP^ processor expr params)
  ;simp^ cannot be called when expr is not an exponentiation.
  
  (receive (base expr-list)
    (string-out-nested-exponentiation processor expr params)
    (let ((expt (multiply-expression-list
		 (exponent-processor processor)
		 expr-list
		 params)))

      (^formal processor base expt params))))
				      

;REQUIREMENT: if simp- is called, (number->scalar-constant processor -1) must be non-nil.
;simp- should be called only if (-r processor) is non-nil.

(define (SIMP- processor expr params)
  (if
   (and (*r processor)
	(number->scalar-constant (coefficient-processor processor) -1))
   (algebraic-processor-simplify processor 
				 (apply-operator (*r processor)
						 (number->scalar-constant (coefficient-processor processor) -1)
						 (1starg expr)) params)
   expr))
	       


(define (SIMP-SUB processor expr params)
  (algebraic-processor-simplify
   processor 
   (apply-operator (+r processor) (1starg expr)
		   (apply-operator (*r processor)
				   (number->scalar-constant (coefficient-processor processor) -1)
				   (2ndarg expr)))
   params))

(define ORDER-BY-ALPHA-ROOTS?
  (make-simple-switch 'order-by-alpha-roots boolean? '#t))


;;;(define (QUICK-COMPARE a b)
;;;  (let ((qc
;;;	 (lambda (a b)
;;;	   (receive (a b)
;;;	     (if (order-by-alpha-roots?)
;;;		 (let ((alpha-root-a (expression-alpha-root a))
;;;		       (alpha-root-b (expression-alpha-root b)))
;;;		   (if (eq? alpha-root-a alpha-root-b)
;;;		       (return a b)
;;;		       (return alpha-root-a alpha-root-b)))
;;;		 (return a b))
;;;	     (or (< (expression-height a)
;;;		    (expression-height b))
;;;		 (and (= (expression-height a)
;;;			 (expression-height b))
;;;		      (< (imps-hash a) (imps-hash b)))
;;;		 (and (= (expression-height a)
;;;			 (expression-height b))
;;;		      (= (imps-hash a) (imps-hash b))
;;;		      (< ((*value t-implementation-env 'string-hash)
;;;			  (format nil "$~S$" a))
;;;			 ((*value t-implementation-env 'string-hash)
;;;			  (format nil "$~S$" b)))))))))
;;;    (let* ((c1 (qc a b))
;;;	   (ok? (or c1
;;;		    (eq? a b)
;;;		    (qc b a))))
;;;      (if ok?
;;;	  c1
;;;	  (block
;;;	    (imps-warning "QUICK-COMPARE: failing comparison between ~S and ~S.
;;;Please inform the implementors.  "
;;;			  a b)
;;;	    (< ((*value t-implementation-env 'descriptor-hash)
;;;		a)
;;;	       ((*value t-implementation-env 'descriptor-hash)
;;;		b)))))))
	       
	      
(define (QUICK-COMPARE a b)
  (let ((qc
	 (lambda (a b)
	   (or (< (expression-height a)
		  (expression-height b))
	       (and (= (expression-height a)
		       (expression-height b))
		    (< (imps-hash a) (imps-hash b)))
	       (and (= (expression-height a)
		       (expression-height b))
		    (= (imps-hash a) (imps-hash b))
		    (< ((*value t-implementation-env 'string-hash)
			(format nil "$~S$" a))
		       ((*value t-implementation-env 'string-hash)
			(format nil "$~S$" b))))))))
    (let* ((c1 (qc a b))
	   (ok? (or c1
		    (eq? a b)
		    (qc b a))))
      (if ok?
	  c1
	  (block
	    (imps-warning "QUICK-COMPARE: failing comparison between ~S and ~S.
Please inform the implementors.  "
			  a b)
	    (< (expression-descriptor-hash a)
	       (expression-descriptor-hash b)))))))	   
      
      

(define-operation (COMPARE-EXPRESSION-LISTS processor a b)
  (ignore processor)
  (iterate compare-firstn= ((l1 a) (l2 b))

    (cond ((null? l1) (not (null? l2)))
	  ((null? l2) '#f)
	  ((eq? (car l1) (car l2)) (compare-firstn= (cdr l1) (cdr l2)))
	  (else (quick-compare (car l1) (car l2))))))

;"rational" refers to a numerical object which names a constant in the language.

(define (+SCALAR processor rational algexps params)
  (ignore params)
;produces a sum of algexps 
;and the list algexps of "simple" algebraic-expressions.
;takes into account whether the first term of algexps is 0.
  (let ((scalar (numerical-object->scalar-constant processor rational)))
    (cond ((null? algexps) scalar)
	  ((numerical-=0? rational)
	   (form-sum-expression processor algexps))
;apply-binary-associative-operator requires algexps be non-nil. At this point, algexps must
;be non-nil 
	  (else (form-sum-expression processor (cons scalar algexps))))))

(define (*SCALAR processor rational algexps params)
;;  (ignore params)
  (let ((scalar (numerical-object->scalar-constant
		 (coefficient-processor processor)
		 rational)))
    (cond ((inhibit-multiplication? processor)
	   (*scalar-inhibiting-multiplication processor rational algexps params))
	  ((null? algexps) scalar)
	  ((scalar-constant-=0? processor (car algexps)) (car algexps))
	  ((numerical-=0? rational)
	   (require-convergence-every-factor processor params algexps)
	   (number->scalar-constant processor 0))
	  ((numerical-=1? rational)
	   (form-external-product-expression processor algexps));algexps is a non-null list
	  (else (form-external-product-expression processor (cons scalar algexps))))))

;REQUIREMENT: If an exponent can be numerical-constant=0? then (number->scalar-constant processor 1) 
;must be non-nil. This is true regardless of (inhibit-exponentiation? processor). 

(define (^FORMAL processor base exponent params)
					;produces a exponent of base to exponent. 
  (let ((^r (^r processor))
	(numerical-constant-1 (number->scalar-constant processor 1)))
    (cond ((scalar-constant-=1? processor base)
					;1^x is defined if x is of the right sort.
	   (require-formula params (defined-in exponent (exp-sort processor)))

	   ;;(require-convergence (exponent-processor processor) params exponent)
	   base)
	  ((scalar-constant-=0? processor base)
	   (require-convergence processor params (apply-operator ^r base exponent))
					;if 0^y  is defined, it must zero. 
	   base)
	  ((scalar-constant-=0? (exponent-processor processor) exponent)
					;x^0 when defined, is 1. This is always the case if not(x=0).
					;The "0" in the exponent refers to the 0
					;of the base ring for exponents. 
					;x^y is defined for x different from 0.
					;0^0 is undefined
	   (require-convergence processor params (apply-operator ^r base exponent))
;;;	 (require-formula params
;;;			  (negation (equality base
;;;					      (number->scalar-constant processor 0))))
	   numerical-constant-1)
	  ((scalar-constant-=1? (exponent-processor processor) exponent) base)
					;x^1 is always x. The "1" in the exponent refers to the 1
					;of the base ring for exponents. 
	  ((and (scalar-constant? processor base)
		(scalar-constant? (exponent-processor processor) exponent)
		(fixnum? (scalar-constant->numerical-object
			  (exponent-processor processor)
			  exponent)))
	   (let ((try
		  (numerical-object->scalar-constant
		   processor
		   (numerical-expt
		    (scalar-constant->numerical-object processor base)
		    (scalar-constant->numerical-object (exponent-processor processor) exponent)))))
	     (if (not (null? try)) try
		 (apply-operator ^r base exponent))))
	  ;;The utility of the following is questionable. To justify it,
	  ;;we need an axiom that for non-zero scalars x, x^m is always defined.
	  ((and (scalar-constant? processor base)
		(processor-faithful-numeral-representation? processor)
		
		(multiplication? (exponent-processor processor) exponent)
		(let ((args (associative-arguments exponent)))
		  (if (and (scalar-constant? (exponent-processor processor) (car args))
			   (fixnum? (scalar-constant->numerical-object
				     (exponent-processor processor)
				     (car args))))
		      args
		      '#f)))
	   =>
	   (lambda (args) 
	     (let* ((try
		     (numerical-object->scalar-constant
		      processor
		      (numerical-expt
		       (scalar-constant->numerical-object processor base)
		       (scalar-constant->numerical-object
			(exponent-processor processor) (car args))))))
	       (if (not (null? try))
		   (apply-operator
		    ^r
		    try
		    (form-product-expression (exponent-processor processor) (cdr args)))
		   (apply-operator ^r base exponent)))))
	  (else (apply-operator ^r base exponent)))))

(define (^FORMAL-INHIBITING-EXPONENTIATION processor base n params)
  (ignore params)
;this blocks use of exponents. For example, structure may not have exponent operation
;e.g., integers without exponentiation.
  (let ((numerical-constant-1 (number->scalar-constant processor 1)))
    (cond ((not (and (integer? n) (> n 0)))
	   (imps-error "^FORMAL-INHIBITING-EXPONENTIATION: cannot block use of exponents."))
	  ((numerical-=0? n)
	   (if (null? numerical-constant-1)
	       (imps-error "^FORMAL-INHIBITING-EXPONENTIATION: cannot block use of exponents.")
		       numerical-constant-1))
	  (else
	   (iterate loop ((i 0) (repeat-cons '()))
	     (if (>= i n)
		 (form-product-expression processor repeat-cons)
		 (loop (1+ i) (cons base repeat-cons))))))))

;REQUIREMENT:"rational" must be the name of some constant in the language.

(define (*SCALAR-INHIBITING-MULTIPLICATION processor rational algexps params)
;this blocks use of multiplication. For example, in a theory such as Pressburger arithmetic.
  (let ((scalar (numerical-object->scalar-constant processor rational)))
    (cond ((not (integer? rational))
	   (imps-error "*scalar-inhibiting-multiplication: cannot block use of scalars."))
	  ((null? algexps) scalar)
	  ((numerical-=0? rational) (require-convergence-every-factor processor params algexps) scalar)
	  ((numerical-=1? rational) (car algexps))
;There had better be only one element in algexps.
	  ((< rational 0)
	   (or (-r processor)
	       (imps-error "*SCALAR-INHIBITING-MULTIPLICATION: no additive invers."))
	   (apply-operator
	    (-r processor)
	    (*scalar-inhibiting-multiplication processor (numerical-minus rational) algexps params)))
	  (else (iterate loop ((i 0) (repeat-cons '()))
	     (if (>= i rational)
		 (form-sum-expression processor repeat-cons)
		 (loop (1+ i) (cons (car algexps) repeat-cons))))))))



;simp/ works by converting all denominators to exponents.
;REQUIREMENT: (number->exponent-constant processor -1) must be non-nil.

(define (SIMP/ processor expr params)
  (if (inhibit-exponentiation? processor) expr
;;;if (inhibit-exponentiation? processor) is true there is no point in doing anything.
      (let ((a1 (1starg expr))
	    (a2 (algebraic-processor-simplify processor (2ndarg expr) params))
	    (neg1 (number->exponent-constant processor -1))
	    (*r (*r processor)))
	(cond ((multiplication? processor a2)
	       (let ((s2 (map (lambda (x) (^formal processor x neg1 params))
			      (reverse (associative-arguments a2)))))
		 (algebraic-processor-simplify
		  processor  
		  (form-product-expression processor (cons a1 s2))
		  params)))
	      (else (algebraic-processor-simplify
		     processor 
		     (apply-operator *r a1 (^formal processor a2 neg1 params))
		     params))))))

(define (MAKE-WEIGHTED-SUM-TALLY-CHART processor)
  ;scalar-tally's and tally-object-weight's are numerical objects. 
  ;Tally labels are expression lists.
  (init-tally-chart (coerce-type (scalars-type processor) 0)
		    numerical-+  
		    numerical-+ 
		    ;equal?
		    (lambda (x y) (and (= (length x) (length y))
				       (every? alpha-equivalent? x y)))
		    (lambda (x y) (compare-expression-lists processor x y))))

(define (multiplicative-associative-arguments processor expr);;*ext-r
  (if (multiplication? processor expr)
      (let ((accumulated-args '()))
	(iterate collect-args ((expr expr) (side 'right))
	  (if (if (eq? side 'right)
		  (multiplication? processor expr)
		  (multiplication? (coefficient-processor processor) expr))
	      (block (collect-args (1starg expr) 'left)
		     (collect-args (2ndarg expr) 'right))
	      (push accumulated-args expr)))
	(reverse! accumulated-args))
      '()))

(define (WEIGHTED-SUM-ACCUMULATE-EXPRESSION processor ptc x)
  (let ((1num (coerce-type (scalars-type (coefficient-processor processor)) 1)))
    (cond ((scalar-constant? processor x)
	   (accumulate-scalar ptc (scalar-constant->numerical-object processor x)))
	  ((formal-symbol? x) (accumulate-label ptc (list x) 1num))
	  ((addition? processor x)
	   (walk (lambda (z) (weighted-sum-accumulate-expression processor ptc z)) (arguments x)))
	  ((multiplication? processor x)
	   (let ((arguments (multiplicative-associative-arguments processor x)))
	     (cond ((scalar-constant? (coefficient-processor processor) (car arguments))
		    (accumulate-label
		     ptc
		     (cdr arguments)
		     (scalar-constant->numerical-object
		      (coefficient-processor processor)
		      (car arguments))))
		   (else (accumulate-label ptc arguments 1num)))))
	  (else (accumulate-label ptc (list x) 1num)))))

;REQUIREMENT: (tally-chart-scalar ptc) must be the name of some constant in the language.

(define (WEIGHTED-SUM-TALLY-CHART->EXPRESSION processor ptc params)
  (iterate loop ((accum nil) (fc-tally-list (label-tallies ptc)))
    (cond ((null? fc-tally-list) (+scalar processor
					  (tally-chart-scalar ptc)
					  (reverse accum)
					  params))
	  ((numerical-=0? (tally-object-weight (car fc-tally-list)))
	   (require-convergence-every-factor processor params (tally-object-label (car fc-tally-list)))
	   (loop accum (cdr fc-tally-list)))
	  (else
	   (let ((weighted-product (*scalar processor
					    (tally-object-weight (car fc-tally-list))
					    (tally-object-label (car fc-tally-list))
					    params)))
	     (loop (cons weighted-product accum) (cdr fc-tally-list)))))))

  

(define (MAKE-WEIGHTED-PRODUCT-TALLY-CHART processor)
  (init-tally-chart (coerce-type (scalars-type processor) 1)
		    numerical-*
		    append!
		    ;eq?
		    alpha-equivalent?
		    (if (commutative? processor) quick-compare '())))

(define (WEIGHTED-PRODUCT-ACCUMULATE-EXPRESSION processor mtc x)
  (let ((1exp (if (inhibit-exponentiation? processor)
		  1
		  (number->exponent-constant processor 1))))
    (cond ((scalar-constant? processor x)
	   (accumulate-scalar mtc (scalar-constant->numerical-object processor x)))
					;
	  ((formal-symbol? x) (accumulate-label mtc x (list 1exp)))
	  ((multiplication? processor x)
	   (walk (lambda (z)
		   (weighted-product-accumulate-expression processor mtc z))
		 (arguments x)))
	  ((exponentiation? processor x)
	   (accumulate-label mtc (1starg x) (list (2ndarg x))))
	  (else (accumulate-label mtc x (list 1exp))))))

(define (WEIGHTED-PRODUCT-TALLY-CHART->EXPRESSION processor mtc params)
  (let ((sub (exponent-processor processor)))
    (iterate loop ((accum nil) (be-tally-list (label-tallies mtc)))
      (cond ((null? be-tally-list) (*scalar processor
					    (tally-chart-scalar mtc)
					    accum
					    params))
	    (else
	     (let ((exponent-list (tally-object-weight (car be-tally-list)))
		   (base (tally-object-label (car be-tally-list))))
					;if there is any possibility of cancellation between exponents,
					;require convergence.
	       (if (and (^r processor) (> (length exponent-list) 1))
		   (walk
		    (lambda (x) (require-convergence processor params (apply-operator (^r processor) base x)))
		    exponent-list))
	       (let ((factor (if (inhibit-exponentiation? processor)
				 (^formal-inhibiting-exponentiation
				  processor
				  base
				  (apply + exponent-list)
				  params)
				 (^formal processor
					  base
					  (sum-expression-list
					   sub
					   exponent-list
					   params)
					  params))))
		 (cond ((scalar-constant-=1? processor factor)
			(loop accum (cdr be-tally-list)))
		       ((scalar-constant-=0? processor factor)
			(require-convergence-every processor params accum)
			(map (lambda (x)
			       (require-convergence processor params (tally-object-label x))
			       (require-convergence-every processor params
							  (tally-object-weight x)))
			     (cdr be-tally-list))
			factor)
		       ((scalar-constant? processor factor)
			(accumulate-scalar mtc
					   (scalar-constant->numerical-object
					    processor factor))
			(loop accum (cdr be-tally-list)))
		       (else (loop (cons factor accum) (cdr be-tally-list)))))))))))



(define (FORM-SUM-EXPRESSION processor exprs)
  (let ((+r (+r processor)))
    (if (null? exprs) (number->scalar-constant processor 0)
	(iterate loop ((exprs (cdr exprs)) (sum (car exprs)))
	  (if (null? exprs) sum
	      (loop (cdr exprs) (apply-operator +r sum (car exprs))))))))

(define (FORM-EXTERNAL-PRODUCT-EXPRESSION processor exprs)
  (let ((*r (*r processor))
	(*ext-r (*ext-r processor)))
    (if (null? exprs)
	(number->scalar-constant processor 1)
	(if (null? (cdr exprs))
	    (car exprs)
	    (iterate loop ((exprs (cdr exprs)) (prod (car exprs)))
	      (if (null? (cdr exprs))
		  (apply-operator *r prod (car exprs))
		  (loop (cdr exprs) (apply-operator *ext-r prod (car exprs)))))))))

(define FORM-PRODUCT-EXPRESSION form-external-product-expression)
  
;;;  (let ((*r (*r processor)))
;;;    (if (null? exprs) (number->scalar-constant processor 1)
;;;	(iterate loop ((exprs (cdr exprs)) (prod (car exprs)))
;;;	  (if (null? exprs) prod
;;;	      (loop (cdr exprs) (apply-operator *r prod (car exprs)))))))

;;for rings

(define (REPEATED-SUM-OF-ONES->NUMERAL processor zero unit expr)
  (iterate loop ((top '#t) (expr expr))
  (cond ((eq? expr zero) (coerce-type (scalars-type processor) 0))
	((eq? expr unit) (coerce-type (scalars-type processor) 1))
	((and top (sign-negation? processor expr))
	 (let ((n (loop '#f (1starg expr))))
	   (if n (numerical-minus n) '#f)))
	((and (addition? processor expr) (eq? (2ndarg expr) unit))
	 (let ((n (loop '#f (1starg expr))))
	   (if n (numerical-+ 1 n) '#f)))
	(else '#f))))
	
(define (NUMERAL->REPEATED-SUM-OF-ONES processor zero unit n)
  (or ((numerical-type-recognizer (scalars-type processor)) n)
      (imps-error "~A is not of numerical type ~A."  n (scalars-type processor)))
  (cond ((numerical-=0? n) zero)
	((numerical-=1? n) unit)
	((numerical-< n (coerce-type (scalars-type processor) 0))
	 (if (-r processor)
	     (apply-operator (-r processor)
			 (numeral->repeated-sum-of-ones processor zero unit (numerical-minus n)))
	     '#f))
	(else (apply-operator
	       (+r processor)
	       (numeral->repeated-sum-of-ones
		processor
		zero
		unit
		(numerical-+ n (numerical-minus (coerce-type (scalars-type processor) 1))))
		unit))))

(define (USE-NUMERALS-FOR-GROUND-TERMS processor)
  (let ((language (processor-language processor)))
    ;;;(set (processor-cancellation-valid? processor) '#t)
    (set (algebraic-processor-faithful-numeral-representation? processor) '#t)
    (set (algebraic-processor-numeral-to-term-function processor)
	 (lambda (x) (find-constant language x)))
    (set (algebraic-processor-term-to-numeral-function processor)
	 (lambda (x) (name x)))
    (set (algebraic-processor-constant-recognizer-function processor)
	 (lambda (expr)
	   ((numerical-type-recognizer (scalars-type processor)) (name expr))))
    processor))
