;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald scripts)

(define (build-script arg-count forms protocol recognizer the-name)
  (let ((command
	 (build-universal-command
	  (object
	      (lambda (sqn . args)
		(or (= arg-count (length args))
		    (if the-name
			(imps-error "Wrong number of arguments to script named ~S" the-name)
			(imps-error "Wrong number of arguments to (anonymous) script given
                                     by ~A" forms)))
		(execute-command-sequence-for-scripts
		 sqn
		 (cons '() args)	; Create null name environment for locals 
		 forms))
	    ((print self port)
	     (format
	      port "#{Script: ~%~A}"
	      (with-output-to-string
	       la
	       (pretty-print forms la)))))
	  the-name (always '#t) protocol)))
    (cond ((not the-name))
	  ((eq? recognizer '#t)
	   (push *universally-applicable-command-names*
		 (string-downcase (format nil "~A" the-name))))
	  (recognizer
	   (push *command-applicability-table*
		 (cons (eval recognizer imps-impl-env) (list command))))
	  (else '#t))
    command))



(define (sqn-coerce-to-expression sqn term-or-term-string)
  (cond ((expression? term-or-term-string) term-or-term-string)
	((string? term-or-term-string)
	 (sequent-read (sequent-node-sequent sqn) term-or-term-string))
	(else (imps-error "~A is neither an expression nor a string."
			  term-or-term-string ))))

(define (coerce-to-theorem thm-or-thm-name)
  (cond ((theorem? thm-or-thm-name)  thm-or-thm-name)
	((symbol? thm-or-thm-name) (name->theorem thm-or-thm-name))
	(else (name->theorem (name thm-or-thm-name)))))

(define (coerce-to-translation trans-or-trans-name)
  (cond ((translation? trans-or-trans-name)  trans-or-trans-name)
	((symbol? trans-or-trans-name) (name->translation trans-or-trans-name))
	(else (name->translation (name trans-or-trans-name)))))

(define (coerce-to-macete macete-or-macete-name)
  (cond ((macete? macete-or-macete-name)  macete-or-macete-name)
	((symbol? macete-or-macete-name) (name->macete macete-or-macete-name))
	(else (name->macete (name macete-or-macete-name)))))

(define (dg-coerce-to-sequent-node dg x)
  (cond ((sequent-node? x) x)
	((number? x) (sequent-unhash-in-graph x dg))
	((and (list? x)
	      (= (length x) 2)
	      (string? (car x))
	      (string? (cadr x))
	      (read-sequent-and-post (car x) (cadr x) dg)))
	(else "DG-COERCE-TO-SEQUENT-NODE: ~A is neither a sequent-node,
 an index of one or a list of a context string and an assertio string." x)))

;;;(define (sqn-coerce-to-assumption sqn assum-str-or-num . dont-raise-error)
;;;  (cond ((formula? assum-str-or-num)
;;;	 (if (mem? alpha-equivalent? assum-str-or-num (sequent-node-assumptions sqn))
;;;	     assum-str-or-num
;;;	     (or (any (lambda (ass) (if (quick-match? ass assum-str-or-num)
;;;					ass
;;;					'#f))
;;;		      (sequent-node-assumptions sqn))
;;;		 (if dont-raise-error
;;;		     '#f
;;;		     (imps-error "SQN-COERCE-TO-ASSUMPTION: ~A 
;;;is neither an assumption, a formula which matches an assumption, a string
;;;representing one, or an index."
;;;				 assum-str-or-num)))))
;;;
;;;	((string? assum-str-or-num)
;;;	 (apply sqn-coerce-to-assumption
;;;		sqn
;;;		(sequent-read
;;;		 (sequent-node-sequent sqn)
;;;		 assum-str-or-num)
;;;		dont-raise-error))
;;;	((number? assum-str-or-num)
;;;	 (if (>= assum-str-or-num (length (sequent-node-assumptions sqn)))
;;;	     (if dont-raise-error
;;;		 '#f
;;;		 (imps-error
;;;		  "SQN-COERCE-TO-ASSUMPTION: Not enough assumptions for index ~D."
;;;		  assum-str-or-num))
;;;	     (nth (sequent-node-assumptions sqn) assum-str-or-num)))
;;;	(else
;;;	 (if dont-raise-error
;;;	     '#f
;;;	     (imps-error "SQN-COERCE-TO-ASSUMPTION: ~A 
;;;is neither an assumption, a formula which matches an assumption, a string
;;;representing one, or an index."
;;;			 assum-str-or-num)))))


(define (sqn-coerce-to-assumption sqn assum-str-or-num . dont-raise-error)
  (cond ((formula? assum-str-or-num)
	 (if (mem? alpha-equivalent? assum-str-or-num (sequent-node-assumptions sqn))
	     assum-str-or-num
	     (or (any (lambda (ass) (if (quick-match? ass assum-str-or-num)
					ass
					'#f))
		      (sequent-node-assumptions sqn))
		 (if dont-raise-error
		     '#f
		     (imps-error "SQN-COERCE-TO-ASSUMPTION: ~A 
is neither an assumption, a formula which matches an assumption, a string
representing one, or an index."
				 assum-str-or-num)))))

	((string? assum-str-or-num)
	 (apply sqn-coerce-to-assumption
		sqn
		(sequent-read
		 (sequent-node-sequent sqn)
		 assum-str-or-num)
		dont-raise-error))
	((number? assum-str-or-num)
	 (if (>= assum-str-or-num (length (sequent-node-assumptions sqn)))
	     (if dont-raise-error
		 '#f
		 (imps-error
		  "SQN-COERCE-TO-ASSUMPTION: Not enough assumptions for index ~D."
		  assum-str-or-num))
	     (nth (sequent-node-assumptions sqn) assum-str-or-num)))

	;; A new case which allows us to select patterns in more subtle ways

	((list? assum-str-or-num)
	 (let ((exprs 
		(map
		 
		 (lambda (x)
		   (cond ((string? x)
			  (sequent-read
			   (sequent-node-sequent sqn)
			   x))
			 ((expression? x) x)
			 (else 
			  (imps-error "SQN-COERCE-TO-ASSUMPTION: bad list pattern."))))
		 
		 assum-str-or-num)))
	   (or (any (lambda (ass)
		      (if (quick-match-list? ass exprs)
			  ass
			  '#f))
		      (sequent-node-assumptions sqn))
	       (if dont-raise-error
		   '#f
		   (imps-error "SQN-COERCE-TO-ASSUMPTION: ~A 
is neither an assumption, a formula which matches an assumption, a string
representing one, or an index."
			       assum-str-or-num)))))
	   
	   

	(else
	 (if dont-raise-error
	     '#f
	     (imps-error "SQN-COERCE-TO-ASSUMPTION: ~A 
is neither an assumption, a formula which matches an assumption, a string
representing one, or an index."
			 assum-str-or-num)))))



(define (quick-match-list? expr pattern-list)
  (let ((paths-to-first-pattern
	 (sort-paths-1!
	  (paths-to-satisfying-virtual-occurrences
	   expr
	   (lambda (subexpr) (quick-match? subexpr (car pattern-list)))
	   -1))))
    (if (null? paths-to-first-pattern)
	'#f
      (iterate loop ((pattern-list (cdr pattern-list))

		     ;;take the rightmost occurrence of the first pattern.
		     ;;if any occurrence is going to match, this must be it.

		     (last-path 
		      (car paths-to-first-pattern)))
	       (if (null? pattern-list)
		   '#t
		   ;;no more patterns and we're still here ... Matches!
		 (let* ((paths-to-next-pattern
			 (sort-paths-1!
			  (paths-to-satisfying-virtual-occurrences
			   expr
			   (lambda (subexpr)
			     (quick-match? subexpr (car pattern-list)))
			   -1)))
			(path-to-next


			 (any
			  (lambda (x)
			    (and (not (path-extends? x last-path))
				 (not (path-extends? last-path x))
				 (list-ordering-1 last-path x)
				 x))
				 
			  ;; found cannot be the null path because 
			  ;; the null path is extended by any path.

			  paths-to-next-pattern)))
		   (if path-to-next
		       (loop (cdr pattern-list) path-to-next)
		     '#f)))))))

(define (coerce-to-command command-or-name)
  (let ((thing
	 (cond ((dg-command? command-or-name) command-or-name)
	       ((symbol? command-or-name) (name->command command-or-name))
	       (else (name->command (name command-or-name))))))
    (or (dg-command? thing)
	(imps-error "~A cannot be coerced into a command" command-or-name))
    thing))

(define interactive-command-applier
  (lambda (sqn command-or-name args)
    (deduction-graph-apply-command-interface
     (sequent-node-graph sqn)
     command-or-name
     (list sqn)
     args
     '())))

(define script-command-applier
  (lambda (sqn command-or-name args)
    (let ((command (coerce-to-command command-or-name)))
      (apply command sqn args))))

(define (interpret-command-sequence sqn command-applier display command-script)
        
  ;;Each entry in the command script is
  
  ;; (a) The name of a single proof command.
  ;; (b) A list (proof-command-name args)
  ;; (c) A list (control-command-name args)
  (bind (((imps-raise-error?) '#f))
    (let ((goal sqn)
	   ;;
	   ;; (dg (sequent-node-graph sqn))
	  )
      (iterate loop ((sqn goal) (command-script command-script))
	(cond ((immediately-grounded? goal)		; Grounded so stop!
	       (return (immediately-grounded? goal) sqn))
	      ((or (null? command-script)		; No more commands left
		   (not (sequent-node? sqn)))		; Sqn isn't a sequent node 
	       (return (immediately-grounded? goal) sqn))
	      ((symbol? (car command-script))

	       ;;If the expression is a symbol evaluate it

	       (let ((evaled 
		      (interpret-command-argument sqn display (car command-script))))
		 (if (list? evaled)
		     (command-applier sqn (car evaled) (cdr evaled))
		     (command-applier sqn evaled '())))
	       (loop
		(sequent-node-first-unsupported-relative sqn)
		(cdr command-script)))
	      ((command-sequence-keyword? (caar command-script))
	       (loop (interpret-keyword
		      command-applier
		      sqn
		      (caar command-script)
		      display
		      (cdar command-script))
		     (cdr command-script)))
	      (else 
	       (command-applier
		sqn
		(interpret-command-argument sqn display (caar command-script))
		(map
		 (lambda (arg)
		   (interpret-command-argument sqn display arg))
		 (cdar command-script)))
	       (loop
		(sequent-node-first-unsupported-relative sqn)
		(cdr command-script))))))))
 
(define (execute-command-sequence-for-scripts sqn display command-script)
  (interpret-command-sequence sqn script-command-applier display command-script))

(define (execute-command-sequence dg-or-sqn command-script)
  (or (deduction-graph? dg-or-sqn)
      (sequent-node? dg-or-sqn)
      (imps-error
       "EXECUTE-COMMAND-SEQUENCE: ~A is not a deduction graph or sequent node."
       dg-or-sqn))
  
  (let ((sqn (if (deduction-graph? dg-or-sqn)
		 (deduction-graph-goal dg-or-sqn)
		 dg-or-sqn)))
    (if (sequent-node-grounded? sqn)
	(imps-warning "execute-command-sequence:  Starting node ~s is grounded."
		      sqn))
    (receive (() sqn)
      (interpret-command-sequence sqn interactive-command-applier '(()) command-script)
      sqn)))

(define (script-add-binding-to-display display varname val)
  (set (car display)
       (cons
	(cons varname val)
	(car display))))

(let ((keyword-proc-alist '()))
  (define (command-sequence-keyword? word)
    (assq? word keyword-proc-alist))

  (define (add-command-keyword word proc)
    (push keyword-proc-alist (cons word proc)))

  (define (interpret-keyword command-applier sqn word display arg)
    (apply (cdr (assq word keyword-proc-alist)) command-applier sqn display arg)))

(add-command-keyword
 'move-to-ancestor 
 (lambda (ca sqn display arg)
   (ignore ca display)
   (nth-ancestor sqn arg)))

(add-command-keyword
 'move-to-descendent 
 (lambda (ca sqn display arg)
   (ignore ca display)
   (deduction-graph-follow-path sqn arg)))

(add-command-keyword
 'move-to-sibling 
 (lambda (ca sqn display arg)
   (ignore ca display)
   (deduction-graph-find-sibling sqn arg)))

(define (interpret-block ca sqn display . commands)
  (ca sqn 'annotate '(begin-block))
  (receive (() sqn)
    (interpret-command-sequence sqn ca display commands)
    (and
     (sequent-node? sqn)
     (>0? (deduction-graph-unended-block-count (sequent-node-graph sqn)))
     (ca sqn 'annotate '(end-block)))
    sqn))

(define (interpret-skip ca sqn display . args)
  (ignore ca args display)
  sqn)

(define (interpret-let-script ca sqn display varname arg-count script-form)
  (let ((obj (build-script arg-count script-form '#f '#f '#f))
	(dollar-varname (concatenate-symbol '$ varname)))
    (ca sqn 'annotate (list 'let-script varname arg-count script-form))
    (script-add-binding-to-display
     display varname
     (join
       (object obj ((dg-command-name self) dollar-varname))
       obj))
    sqn))

(define (interpret-let-macete ca sqn display varname macete-form)
  (let ((interpreted-macete-form
	 (interpret-command-argument sqn display macete-form))
	(dollar-varname (concatenate-symbol '$ varname)))
    (ca sqn 'annotate (list 'let-macete varname interpreted-macete-form))
    (script-add-binding-to-display
     display varname
     (join
       (object '() ((name self) dollar-varname))
       (build-macete-from-sexp interpreted-macete-form '#f))))
  sqn)

(define (interpret-let-val ca sqn display varname val-form)
  (let ((obj (interpret-command-argument sqn display val-form)))
    (ca sqn 'annotate (list 'let-val varname val-form))
    (script-add-binding-to-display display varname obj)
    sqn))

(add-command-keyword 'block interpret-block)  
(add-command-keyword 'skip interpret-skip)
(add-command-keyword 'let-macete interpret-let-macete)
(add-command-keyword 'let-script interpret-let-script)
(add-command-keyword 'let-val interpret-let-val)

(define (interpret-label-node ca sqn display . args)
  (ignore display ca)
  (let ((sym (car args)))
    (dg-register-node sym sqn)
    sqn))

(define (interpret-jump-to-node ca sqn display . args)
  (ignore display ca)
  (let ((new-sqn (dg-get-registered-node (sequent-node-graph sqn) (car args))))
    (if (sequent-node? new-sqn)
	new-sqn
	sqn)))

(add-command-keyword 'label-node   interpret-label-node)
(add-command-keyword 'jump-to-node interpret-jump-to-node)

(define (interpret-if ca sqn display . args)
  (destructure (((test conseq altern) args))
    (receive (succeeds? new-sqn)
      (interpret-script-condition ca sqn display test)
      (receive (() newer-sqn)
	(interpret-command-sequence
	 new-sqn ca display
	 (list (if succeeds? conseq altern)))
	newer-sqn))))

(define (interpret-while ca sqn display . args)
  (destructure (((test . body) args))
    (receive (succeeds? new-sqn)
      (interpret-script-condition ca sqn display test)
      (if succeeds?
	  (receive (() newer-sqn)
	    (interpret-command-sequence new-sqn ca display body)
	    (apply interpret-while ca newer-sqn display args))
	  new-sqn))))

(define (interpret-for-nodes ca sqn display . args)
  (destructure (((list-spec . body) args))
    (let ((nodes (interpret-for-nodes-list-spec sqn display list-spec)))
      (iterate iter ((nodes nodes)
		     (new-sqn '()))
	(if (null? nodes)
	    (or new-sqn sqn)
	    (receive (grounded? newer-sqn)
	      (interpret-command-sequence (car nodes) ca display body)
	      (iter (cdr nodes) newer-sqn)))))))

(add-command-keyword 'if interpret-if)
(add-command-keyword 'while interpret-while)
(add-command-keyword 'for-nodes interpret-for-nodes)


;; Returns two values:
;; 1.  The true or false result of evaluating the condition
;; 2.  The newly current node after evaluation (which may be different from the
;;     originally current node, because some scripts cause inferences to
;;     execute).
;; 

(define (interpret-script-condition ca sqn display sexp)
  (or (pair? sexp)
      (imps-error
       "interpret-script-condition: Bad condition ~S~%should be pair."
       sexp))
  (let ((proc (retrieve-script-condition-proc (car sexp))))
    (if (procedure? proc)
	(proc ca sqn display (cdr sexp))
	(imps-error
	 "interpret-script-condition: Bad condition, unrecognized keyword~ ~s"
	 (car sexp)))))

;; condition-keyword-proc-alist is maintained as an alist of codition keywords
;; and procedures.  Each procedure must take the current sqn and the cdr of the
;; condition sexp.  It must return two values, namely the true or false result
;; and the node current after executing any embedded commands.  

(let ((condition-keyword-proc-alist '()))
  
  (define (add-condition-keyword word proc)
    (push condition-keyword-proc-alist (cons word proc)))

  (define (retrieve-script-condition-proc word)
    (cond ((assq word condition-keyword-proc-alist) => cdr)
	  (else '#f))))

(add-condition-keyword
 'matches?
 (lambda (ca sqn display rest)
   (ignore ca)
   (return (sqn-matches? 
	    sqn		
	    (interpret-command-argument sqn display (car rest))
	    (interpret-command-argument sqn display (cdr rest)))
	   sqn)))

(add-condition-keyword
 'minor?
 (lambda (ca sqn display rest)
   (ignore ca display rest)
   ;; Temporary version till minor premise better defined. 
   (return (any?
	    (lambda (infn) (not (eq? sqn (car (inference-node-hypotheses infn)))))
	    (sequent-node-out-arrows sqn))
	   sqn)))

(add-condition-keyword
 'major?
 (lambda (ca sqn display rest)
   (ignore ca display rest)
   ;; Temporary version till major premise better defined. 
   (return (any?
	    (lambda (infn) (eq? sqn (car (inference-node-hypotheses infn))))
	    (sequent-node-out-arrows sqn))
	   sqn)))
  
(add-condition-keyword
 'generated-by-rule?
 (lambda (ca sqn display rest)
   (ignore ca display)
   (return (any?
	    (lambda (rule-name)
	      (any?
	       (lambda (infn) (eq? rule-name (inference-node->symbol infn)))
	       (sequent-node-out-arrows sqn)))
	    rest)
	   sqn)))    

(add-condition-keyword
 'succeeds?
 (lambda (ca sqn display rest)
   (interpret-command-sequence sqn ca display rest)))    

(add-condition-keyword
 'progresses?
 (lambda (ca sqn display rest)
   (let ((old (deduction-graph-last-index (sequent-node-graph sqn))))
       (receive (() new-sqn)
	 (interpret-command-sequence sqn ca display rest)
	 (return (< old (deduction-graph-last-index (sequent-node-graph sqn)))
		 new-sqn)))))

(add-condition-keyword
 'not
 (lambda (ca sqn display rest)
   (receive (satisfied? new-sqn)
     (interpret-script-condition ca sqn display (car rest))
     (return (not satisfied?) new-sqn))))


(define (interpret-conjunctive-condition ca sqn display rest)
  (if (null? rest)
      (return '#t sqn)
      (receive (satisfied? new-sqn)
	(interpret-script-condition ca sqn display (car rest))
	(if satisfied?
	    (interpret-conjunctive-condition ca new-sqn display (cdr rest))
	    (return satisfied? new-sqn)))))

(define (interpret-disjunctive-condition ca sqn display rest)
  (if (null? rest)
      (return '#f sqn)
      (receive (satisfied? new-sqn)
	(interpret-script-condition ca sqn display (car rest))
	(if satisfied?
	    (return satisfied? new-sqn)
	    (interpret-disjunctive-condition ca new-sqn display (cdr rest))))))

(add-condition-keyword 'and interpret-conjunctive-condition)
(add-condition-keyword 'or  interpret-disjunctive-condition)

;; Javier: I reversed the order of arguments here.

(define (sqn-matches? sqn assert assums)
  (and
   (every? 
    (lambda (x)
      (sqn-coerce-to-assumption sqn x '#t))
    assums)
   (if (list? assert)
       (quick-match-list?
	(sequent-node-assertion sqn)
	(map (lambda (x)
	       (sqn-coerce-to-expression sqn x))
	     assert))
       (quick-match?
	(sequent-node-assertion sqn)
	(sqn-coerce-to-expression sqn assert)))))    


(let ((node-list-keyword-proc-alist '()))
  
  (define (add-node-list-keyword word proc)
    (push node-list-keyword-proc-alist (cons word proc)))

  (define (retrieve-script-node-list-proc word)
    (cond ((assq word node-list-keyword-proc-alist) => cdr)
	  (else '#f))))

(define (interpret-for-nodes-list-spec sqn display sexp)
  (or (pair? sexp)
      (imps-error
       "interpret-for-nodes-list-spec: Bad condition ~S~%should be pair."
       sexp))
  (let ((proc (retrieve-script-node-list-proc (car sexp))))
    (if (procedure? proc)
	(proc sqn display (cdr sexp))
	(imps-error
	 "interpret-for-nodes-list-spec: Bad condition, unrecognized keyword~ ~s"
	 (car sexp)))))  

(add-node-list-keyword
 'node-and-siblings
 (lambda (sqn display arg)
   (ignore arg display)
   (let ((infn (car (sequent-node-out-arrows sqn))))
     (if (inference-node? infn)
	 (inference-node-hypotheses infn)
	 '()))))

(add-node-list-keyword
 'minor-premises
 (lambda (sqn display arg)
   (ignore arg display)
   (let ((infn (car (sequent-node-out-arrows sqn))))
     (if (inference-node? infn)
	 ;; Temporary till improved treatment of minor premises
	 ;; 
	 (cdr (inference-node-hypotheses infn))
	 '()))))

(add-node-list-keyword
 'unsupported-descendents 
 (lambda (sqn display arg)
   (ignore arg display)
   (unsupported-descendents sqn)))

(define (script-comment ca sqn display comment-string)
  (ignore ca display)
  (let* ((dg (sequent-node-graph sqn))
	 (entry (car (deduction-graph-history dg))))
    (set (dg-history-entry-comments entry)
	 (cons comment-string (dg-history-entry-comments entry)))
  sqn))

(let ((command-argument-keyword-proc-alist '()))
  
  (define (add-command-argument-keyword word proc)
    (push command-argument-keyword-proc-alist (cons word proc)))

  (define (interpret-command-argument sqn display arg)
    (cond ((and (list? arg)
		(symbol? (car arg))
		(assq (car arg) command-argument-keyword-proc-alist))
	   =>
	   (lambda (found)
	     ((cdr found)
	      sqn display 
	      (map (lambda (x) 
		     (interpret-command-argument sqn display x))
		   (cdr arg)))))
	  ((and (symbol? arg)
		(let ((ch (char (symbol->string arg))))
		  (and (char= ch #\$)
		       (assq ch command-argument-keyword-proc-alist))))
	   =>
	   (lambda (found) ((cdr found) sqn display arg)))
	  ((list? arg) (map (lambda (x) (interpret-command-argument sqn display x))
			    arg))
	  (else arg))))

;;;(define (retrieve-command-argument-proc arg)
;;;    (cond ((and (list? arg)
;;;		(symbol? (car arg))
;;;		(assq (car arg) command-argument-keyword-proc-alist))
;;;	   => cdr)
;;;	  ((and (symbol? arg)
;;;		(let ((ch (char (symbol->string arg))))
;;;		  (and (char= ch #\$)
;;;		       (assq ch command-argument-keyword-proc-alist))))
;;;	   => cdr)
;;;	  (else '#f)))

;;;(define (interpret-command-argument sqn display arg)
;;;  (cond ((retrieve-command-argument-proc arg)
;;;	 =>
;;;	 (lambda (proc)
;;;	   (proc sqn display arg)))
;;;	((pair? arg)
;;;	 (map
;;;	  (lambda (sub-arg)
;;;	    (interpret-command-argument sqn display sub-arg))
;;;	  arg))
;;;	(else arg)))


(add-command-argument-keyword
 #\$
 (lambda (sqn display arg)
   (ignore sqn)
   (let ((str (symbol->string arg)))
     (let* ((objs (read-objects-from-string (nthchdr str 1)))
	    (first (and objs (car objs))))
       (cond ((not (= (length objs) 1))
	      (imps-error "command-argument-keyword: Bad $ arg ~S" arg))
	     ((and (integer? first)
		   (>0? first)
		   (<= first (length display)))
	      (nth display first))
	     ((and (= (length objs) 1)
		   (symbol? first)
		   (char= (nthchar str 1) #\$))
	      first)
	     ((and (symbol? first)
		   (assq first (nth display 0)))
	      => cdr)
	     (else
	      (imps-error "command-argument-keyword: Bad $ arg ~S" arg)))))))
 
;;; No need to take cdr anymore.
 
(add-command-argument-keyword
 '%
 (lambda (sqn display bongos-socks+tamales)
   (ignore sqn display)
   (destructure (((format-string . args) bongos-socks+tamales))
     (apply format nil format-string args))))

(add-command-argument-keyword
 '%sym
 (lambda (sqn display format-string+args)
   (ignore sqn display)
   (destructure (((format-string . args) format-string+args))
     (string->symbol 
      (string-upcase 
       (apply format nil format-string args))))))

(add-command-argument-keyword
 '~*
 (lambda (sqn display expr-strs)
   (ignore display)
   (let ((accum '())
	 (exprs
	  (map
	   (lambda (x) (sequent-read (sequent-node-sequent sqn) x))
	   expr-strs)))
     (walk
      (lambda (assum)

	(if (every?
	     (lambda (expr)     
	       (not (quick-match? assum expr)))
	     exprs)
	    (push accum assum)))
      (sequent-node-assumptions sqn))
     accum)))

(add-command-argument-keyword
 '*
 (lambda (sqn display expr-strs)
   (ignore display)
   (let ((accum '())
	 (exprs
	  (map
	   (lambda (x) (sequent-read (sequent-node-sequent sqn) x))
	   expr-strs)))
     (walk
      (lambda (assum)

	(if (any?
	     (lambda (expr)     
	       (quick-match? assum expr))
	     exprs)
	    (push accum assum)))
      (sequent-node-assumptions sqn))
     accum)))      

;;;(add-command-argument-keyword
;;; '%
;;; (lambda (sqn display format-string+args)
;;;   (ignore sqn)
;;;   (destructure (((format-string . args) format-string+args))
;;;     (apply format nil format-string args))))

(add-command-keyword 'script-comment script-comment)


(define (build-readable-comment-form comments)
  `(script-comment ,(apply string-append (reverse comments))))

(define (history-entry-jump history-entry)
  (dg-history-entry-jump-from-previous-expectation history-entry))
  

(define (build-readable-form obj)
  (cond ((null? obj) '())
	((string? obj) obj)
	((number? obj) obj)
	((symbol? obj) obj)
	((proper-list? obj)
	 (map build-readable-form obj))
	((macete? obj) (name obj))
	((sequent-node? obj) (sequent-node-number obj))
	((expression? obj) (qp obj))
	((dg-command? obj) (name obj))
	((inductor? obj) (name obj))
	(else (imps-error "BUILD-READABLE-FORM: weird object ~A" obj))))

(comment
 (let ((previous (dg-history-entry-previous-entry history-entry))
	(sqn (dg-history-entry-sequent-node history-entry)))
    (cond ((null? previous) '())
	  ((eq? (dg-history-entry-first-unsupported-relative previous)
		sqn)
	   '())
	  ((sequent-node? (dg-history-entry-first-unsupported-relative previous))
	   (receive (index path)
	     (relative-position-in-dg
	      (dg-history-entry-first-unsupported-relative previous)
	      sqn)
	     (cons index path)))

	  ;; if (dg-history-entry-first-unsupported-relative previous) is
	  ;; nil, we better signal that fact without raising an error or
          ;; otherwise interrupting the computation.

	  (else (imps-warning "HISTORY-ENTRY-JUMP: jump undefined.")
		'()))))
	   

(comment
 (define (deduction-graph-readable-history-list dg)
   ;; The deduction-graph-history is built backwards, that is the latest
   ;; command is always consed on to the front of the last.
   (receive (command-forms ())
     (iterate iter ((command-forms '())
		    (history (deduction-graph-history dg)))
       (if (null? history)
	   (return command-forms '())
	   (destructure*
	       (((history-entry . history-rest) history)
		(command (dg-history-entry-command history-entry))
		(args (dg-history-entry-arguments history-entry))
		(comments (dg-history-entry-comments history-entry)))
	     (cond 
	      ((and (eq? command (name->command 'annotate))
		    (memq? 'begin-block args))
	       (receive (subforms rest)
		 (iter '() history-rest)
		 (iter `((block ,@subforms) . ,command-forms) rest)))
	      ((and (eq? command (name->command 'annotate))
		    (memq? 'end-block args))
	       (return command-forms history-rest))
	      ((and (eq? command (name->command 'annotate))
		    (memq? (car args) '(let-macete let-script)))
	       (iter (cons args command-forms) history-rest))
	      (else 
	       (let* ((jump (history-entry-jump history-entry))
		      (command-forms-1 
		       (if comments
			   (cons (build-readable-comment-form comments)
				 command-forms)
			   command-forms))
		      (command-forms-2
		       (cons
			(if args
			    (cons (dg-command-name command) (build-readable-form args))
			    (dg-command-name command))
			command-forms-1))
		      (command-forms-3
		       (if jump
			   (append! (build-readable-jump-form jump) command-forms-2)
			   command-forms-2)))
		 (iter command-forms-3 history-rest)))))))
     command-forms)))

(define (rectify-deduction-graph-history-list dg)
  ;; The deduction-graph-history is built backwards, that is the latest
  ;; command is always consed on to the front of the last.
  (receive (command-forms ())
    (iterate iter ((command-forms '())
		   (history (deduction-graph-history dg)))
      (if (null? history)
	  (return command-forms '())
	  (destructure*
	      (((history-entry . history-rest) history)
	       (sqn (dg-history-entry-sequent-node history-entry))
	       (command (dg-history-entry-command history-entry))
	       (args (dg-history-entry-arguments history-entry))
	       (comments (dg-history-entry-comments history-entry))
	       (jump (history-entry-jump history-entry)))


	    (cond 
	     ((eq? command (name->command 'annotate))
	      (let* ((command-forms-1 
		      (if comments
			  (cons (build-readable-comment-form comments)
				command-forms)
			  command-forms))
		     (command-forms-2
		      (cons args command-forms-1))
		     (command-forms-3
		      (if jump
			  (append! (build-readable-jump-form jump) command-forms-2)
			  command-forms-2)))
		
		(iter command-forms-3 history-rest)))


	     (else 
	      (let* ((command-forms-1 
		      (if comments
			  (cons (build-readable-comment-form comments)
				command-forms)
			  command-forms))
		     (command-forms-2
		      (cons
		       (if args
			   (cons (dg-command-name command)
				 (build-readable-form
				  (deinterpret-arguments-for-command
				   sqn 
				   command args)))
			   (dg-command-name command))
		       command-forms-1))
		     (command-forms-3
		      (if jump
			  (append! (build-readable-jump-form jump) command-forms-2)
			  command-forms-2)))
		(iter command-forms-3 history-rest)))))))
    command-forms))

;;;  This "special case" is wrong.  The jump has to be included here also.
;;;	      (iter (cons args (if comments 
;;;				   (cons (build-readable-comment-form comments)
;;;					 command-forms)
;;;				   command-forms))
;;;		    history-rest)


(comment
 (define (rectify-deduction-graph-history-list dg)
  ;; The deduction-graph-history is built backwards, that is the latest
  ;; command is always consed on to the front of the last.
  (receive (command-forms ())
    (iterate iter ((command-forms '())
		   (history (deduction-graph-history dg)))
      (if (null? history)
	  (return command-forms '())
	  (destructure*
	      (((history-entry . history-rest) history)
	       (command (dg-history-entry-command history-entry))
	       (args (dg-history-entry-arguments history-entry))
	       (comments (dg-history-entry-comments history-entry)))
	    (cond 
	     ((eq? command (name->command 'annotate))
	      (iter (cons args command-forms) history-rest))
	     (else 
	      (let* ((jump (history-entry-jump history-entry))
		     (command-forms-1 
		      (if comments
			  (cons (build-readable-comment-form comments)
				command-forms)
			  command-forms))
		     (command-forms-2
		      (cons
		       (if args
			   (cons (dg-command-name command) (build-readable-form args))
			   (dg-command-name command))
		       command-forms-1))
		     (command-forms-3
		      (if jump
			  (append! (build-readable-jump-form jump) command-forms-2)
			  command-forms-2)))
		(iter command-forms-3 history-rest)))))))
    command-forms)))

(define (parenthesize-history-list history-list)
  (receive (forms ())
    (iterate loop ((history-list  history-list)
		   (forms nil))
      (cond ((null? history-list)
	     (return (reverse! forms) '()))
	    ((not (pair? (car history-list)))
	     (loop (cdr history-list)
		   (cons (car history-list) forms)))
	    ((eq? (caar history-list) 'end-block)
	     (return (reverse! forms) (cdr history-list)))
	    ((eq? (caar history-list) 'begin-block)
	     (receive (new-forms rest)
	       (loop (cdr history-list) '())
	       (loop rest `((block ,@new-forms) . ,forms))))
	    (else
	     (loop (cdr history-list)
		   (cons (car history-list) forms)))))
    forms))

(define (deduction-graph-readable-history-list dg)
  (parenthesize-history-list
   (rectify-deduction-graph-history-list dg)))

(define (build-readable-jump-form jump)
  (destructure (((up . down) jump))
    (cond ((and (= up 1)
		(= (length down) 1)
		(not (pair? (car down))))
	   `((move-to-sibling ,(car down))))
	  ((null? down) `((move-to-ancestor ,up)))
	  (else `((move-to-ancestor ,up) (move-to-descendent ,down))))))


(define (shallowest-distinguishing-pattern expr exprs)
  (let ((height (expression-height expr)))
    (iterate loop ((n 0))
      (let ((pattern (display-at-depth expr n)))
	(cond ((<= height n) pattern)
	      ((any?
	       (lambda (x) (quick-match? x pattern))
	       (delq expr exprs))
	      (loop (1+ n)))
	      (else pattern))))))

(define (display-at-depth expr n)
  (cond ((formal-symbol? expr) expr)
	((= n 0) 
	 (cook-up-var-for-sort (expression-sorting expr)))
	(else (apply (expression-quasi-constructor-if-enabled-or-constructor expr)
		     (map 
		      (lambda (x) (display-at-depth x (- n 1))) 
		      (expression-quasi-components-if-enabled-or-components expr))))))
		     
(define (cook-up-var-for-sort sort)
  (if (higher-sort? sort)
      (new-variable sort 'f '())
      (new-variable sort (concatenate-symbol (char (symbol->string (name sort)))) '())))

(let ((command-deinterpreter-alist '()))
  (define (deinterpret-arguments-for-command sqn command arguments)
    (let ((found (assq (dg-command-name command) command-deinterpreter-alist)))
      (if found 
	  ((cdr found) sqn arguments)
	  arguments)))

  (define (add-command-argument-deinterpreter command-name proc)
    (push command-deinterpreter-alist (cons command-name proc))))


(add-command-argument-deinterpreter 
 'instantiate-universal-antecedent
 (lambda (sqn arguments)
   (destructure (((assumption instances) arguments))
     (list
      (shallowest-distinguishing-pattern
       (sqn-coerce-to-assumption sqn assumption '#t)
       (sequent-node-assumptions sqn))
      instances))))

(let ((single-arg-deinterpreter
       (lambda (sqn arguments)
	 (list (shallowest-distinguishing-pattern
		(sqn-coerce-to-assumption sqn (car arguments) '#t)
		(sequent-node-assumptions sqn))))))

  (add-command-argument-deinterpreter 
   'backchain
   single-arg-deinterpreter)

  (add-command-argument-deinterpreter 
   'backchain-backwards
   single-arg-deinterpreter)
  
  (add-command-argument-deinterpreter 
   'contrapose
   single-arg-deinterpreter)

  (add-command-argument-deinterpreter 
   'simplify-antecedent
   single-arg-deinterpreter)

  (add-command-argument-deinterpreter 
   'incorporate-antecedent
   single-arg-deinterpreter)

  (add-command-argument-deinterpreter 
   'antecedent-inference
   single-arg-deinterpreter))
