;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald SORTINGS)



; name is an operation applying to:
;  sorts
;  formal symbols
;  constructors
;  languages
;  schematic formulas
;  inferences
; it returns a symbol (or '#f) in all cases.  

(define-operation (NAME x)
  (imps-error "NAME:  Operation not handled by ~S." x))

; Return the set of all base sorts occurring in something.  

(define-operation (BASE-SORTS sorting-or-expression-or-language)
  (imps-error "BASE-SORTS: Operation not handled by ~S."
	      sorting-or-expression-or-language))

; A base sorting is an entity having the following information:
;  1.  A name (a symbol).
;  2.  A category, either IND or PROP.
;  3.  A type.
;  4.  An enclosing sort.  This is either '#t or else it is
;      another sort of the same type.  
;
; A base sorting *is* a type if it is identical with its own type.  It is a
; floating sort if its type is neither IND nor PROP.  The set of base sorts of
; a base sorting is its singleton.   

;; An enclosing sort is a sort guaranteed to include the given one, and
;; distinct from the given one if possible (i.e., unless the given sort is a
;; type).


; A higher sorting is an entity having the following information:
;  1.  A name, either a symbol or '#f.
;  2.  A non-null list of domains.
;  3.  A range.
;  4.  An enclosing sort.  This is either Domains -> Range or else a sort that
;      has those same domains and range.  
;
; Various relevant facts about a higher sort may be computed:
;  1.  Its category, which is (recursively) the category of its range.
;  2.  A set of base sorts, namely the union of the (recursive) base sorts of
;      its domains and range.
;  3.  A type, which is that higher sort whose domains and range are 
;      (recursively) the types of the domains and range of this sort.  
;  A sorting is either a base sorting or a higher sorting.
;  NB.  A higher sort includes all functions in domains->range if it has no name.
;  	Justification: names for higher sorts are introduced to present
;  	subsorts of the functions in domains->range.  


(define-structure-type base-sort
  name
  category
  type
  enclosing-sort
  numerical?
  alpha-hash						;for hashing compound exprs

  (((print self port)
    (print (base-sort-name self) port))
   ((base-sorts self)
    (list self))
   ((name self) (base-sort-name self))
   ((two-d-table-hash self) (base-sort-alpha-hash self))))

(set (base-sort-numerical? (stype-master base-sort-stype)) '#f)

(lset base-sort-table (make-two-d-table 'base-sort-table))

(define (retrieve-base-sort name category is-type? enclosing-sort)
  (let ((entry-list (two-d-table-entry base-sort-table name category)))
    (if entry-list
	(any
	 (lambda (s)
	   (and (if is-type?
		    (eq? s (base-sort-type s))
		    (and (not (eq? s (base-sort-type s)))
			 (eq? enclosing-sort (base-sort-enclosing-sort s))))
		s))
	 entry-list)
	'#f)))

(define (build-base-sort name category is-type? enclosing-sort)
   (cond ((or (eq? name 'ind)
	     (eq? name 'prop))
	 (imps-error "build-base-sort: Reserved name ~S" name))
	((and (not (symbol? name))
	      (not (anonymous-name? name)))
	 (imps-error "build-base-sort: name must be a symbol or anonymous-name, not ~S" name))
	((not (boolean? is-type?))
	 (imps-error "build-base-sort: is-type? must be a boolean, not ~S" is-type?))
	((and (not (eq? category IND))
	      (not (eq? category PROP)))
	 (imps-error "build-base-sort: category must be IND or PROP, not ~S"
		     category))
	((and (not is-type?)
	      (not (base-sort? enclosing-sort)))
	 (imps-error "build-base-sort: enclosing-sort must be a sort, not ~S"
		     enclosing-sort)))

  (or (retrieve-base-sort name category is-type? enclosing-sort)
      (let* ((s (make-base-sort))
	     (type (if is-type? s (retrieve-base-sort
				   (base-sort-name (base-sort-type enclosing-sort))
				   category '#t '#t))))
	(or type
	    (imps-error "build-base-sort: can't find type ~S" (base-sort-type enclosing-sort)))
	
	(set (base-sort-name s) name)
	(set (base-sort-category s) category)
	(set (base-sort-type s) type)
	(set (base-sort-enclosing-sort s) enclosing-sort)
	(set (base-sort-alpha-hash s) (descriptor-hash s))
	(let ((entry-list (two-d-table-entry base-sort-table name category)))
	  (if entry-list
	      (set (two-d-table-entry base-sort-table name category)
		   (cons s entry-list))
	      (set (two-d-table-entry base-sort-table name category) (list s))))
	s)))

(define (build-base-type name category)
  (build-base-sort name category '#t '#t))

(define IND
  (let ((s (make-base-sort)))
    (set (base-sort-name s) 'ind)
    (set (base-sort-category s) s)
    (set (base-sort-type s) s)
    (set (base-sort-enclosing-sort s) '#t)
    (set (base-sort-alpha-hash s) (descriptor-hash s))
    (set (two-d-table-entry base-sort-table 'ind s) (list s))
    s))

(define PROP
  (let ((s (make-base-sort)))
    (set (base-sort-name s) 'prop)
    (set (base-sort-category s) s)
    (set (base-sort-type s) s)
    (set (base-sort-enclosing-sort s) '#t)
    (set (base-sort-alpha-hash s) (descriptor-hash s))
    (set (two-d-table-entry base-sort-table 'prop s) (list s))
    s))

(define (base-sort-is-type? s)
  (eq? s (base-sort-type s)))

(define (base-sort-floats? s)
  (and (not (eq? (base-sort-type s) ind))
       (not (eq? (base-sort-type s) prop))))

(define (rename-base-sort s renamer)
  (let ((name (renamer (base-sort-name s)))
	(category (base-sort-category s))
	(is-type? (base-sort-is-type? s))
	(enclosing-sort (base-sort-enclosing-sort s)))	
    (build-base-sort
     name
     category
     is-type?
     (or is-type? (rename-base-sort enclosing-sort renamer)))))
  


;; A higher-sort with a name is a subsort of the higher-sort that results by
;; replacing the name with '#F.  A theory may assert that it is an improper
;; subsort.  

(define-structure-type HIGHER-SORT
  name							;non-'#f means SUBSORT
  range
  domains
  category
  type
  enclosing-sort
  level							;1 greater than max of range, domains
  bases
  numerical?
  alpha-hash						;for hashing compound exprs

  (((print self port)
    (if (print-sorts-verbosely?)
	(format port "#{IMPS-higher-sort ~A (~S -> ~S) ~S}"
		(or (higher-sort-name self)
		    "")
		(higher-sort-domains self)
		(higher-sort-range self)
		(object-hash self))
	(print (or (higher-sort-name self)
		   (reverse (cons (higher-sort-range self)
				  (reverse (higher-sort-domains self)))))
	       port)))
   ((base-sorts self) (higher-sort-bases self))
   ((name self) (higher-sort-name self))
   ((two-d-table-hash self) (higher-sort-alpha-hash self))))

(set (higher-sort-numerical? (stype-master higher-sort-stype)) '#f)

(define (sort-level s)
  (if (base-sort? s)
      0
      (higher-sort-level s)))

(define (higher-sort-is-type? s)
  (eq? s (higher-sort-type s)))

(define (sort? s)
  (or (base-sort? s)
      (higher-sort? s)))

(define (sort-category s)
  (if (base-sort? s)
      (base-sort-category s)
      (higher-sort-category s)))

(define (sort-is-type? s)
  (if (base-sort? s)
      (base-sort-is-type? s)
      (higher-sort-is-type? s)))

(define (type-of-sort s)
  (if (base-sort? s)
      (base-sort-type s)
      (higher-sort-type s)))

(define (numerical? s)
  (if (base-sort? s)
      (base-sort-numerical? s)
      (higher-sort-numerical? s)))

(define (sort-alpha-hash s)
  (if (base-sort? s)
      (base-sort-alpha-hash s)
      (higher-sort-alpha-hash s)))

(define (floating-sort? s)
  (if (base-sort? s)
      (base-sort-floats? s)
      (any? base-sort-floats? (base-sorts s))))

(define (PROP-SORTING? s)
  (eq? (sort-category s) prop))

(define (IND-SORTING? s)
  (eq? (sort-category s) ind))

(define (rename-sort s renamer)
  (if (base-sort? s)
      (rename-base-sort s renamer)
      (rename-higher-sort s renamer)))

(define (rename-higher-sort s renamer)
  (let ((domains         (map (lambda (s) (rename-sort s renamer))
			      (higher-sort-domains s)))
	(range		 (rename-sort (higher-sort-range s) renamer))
	(name		 (renamer (higher-sort-name s)))
	(enclosing-sort	 (rename-sort (higher-sort-enclosing-sort s) renamer)))
    (build-higher-sort domains range name enclosing-sort)))

(lset higher-sort-table (make-two-d-table 'higher-sort-table))

;;; NAME may be either a symbol, anonynomous name, or '#F.  

(define (retrieve-higher-sort domains range name)
  (let ((entry-list (table-entry higher-sort-table range)))
    (if entry-list
	(any
	 (lambda (s)
	   (and (eq? name (higher-sort-name s))
		(equal? domains (higher-sort-domains s))
		s))
	 entry-list)
	'#f)))

(define (non-nullary-and-domains-are-sorts? domains)
  (and (not (null? domains))
       (every? sort? domains)))

(define (build-maximal-higher-sort domains range)
  (imps-enforce sort? range)
  (imps-enforce non-nullary-and-domains-are-sorts? domains)
  
  (or (retrieve-higher-sort domains range '#f)
      (let ((s (make-higher-sort)))
	(set (higher-sort-name s) '#f)
	(set (higher-sort-range s) range)
	(set (higher-sort-domains s) domains)
	(set (higher-sort-bases s) (set-union (base-sorts range)
					      (collect-set base-sorts domains)))
	(set (higher-sort-category s) (sort-category range))
	(set (higher-sort-type s)
	     (if (and (sort-is-type? range)
		      (every? sort-is-type? domains))
		 s
		 (build-maximal-higher-sort (map type-of-sort domains)
					    (type-of-sort range))))
	(set (higher-sort-enclosing-sort s) s)
	(set (higher-sort-level s)
	     (1+ (apply max (sort-level range) (map sort-level domains))))
	(set (higher-sort-alpha-hash s) (descriptor-hash s))
	(push (table-entry higher-sort-table range) s)
	s)))

(define (build-higher-sort domains range name enclosing-sort)
  (imps-enforce sort? range)
  (imps-enforce non-nullary-and-domains-are-sorts? domains)
  (imps-enforce (lambda (n) (or (not n) (symbol? n) (anonymous-name? n))) name)
  (imps-enforce higher-sort? enclosing-sort)
  (or (and (eq? range (higher-sort-range enclosing-sort))
	   (equal? domains (higher-sort-domains enclosing-sort)))
      (imps-error "build-higher-sort: Mis-matched enclosing sort ~S;~%Domains ~S;~%Range~S~%"
		  enclosing-sort
		  domains
		  range))
  
  (or (retrieve-higher-sort domains range name)
      (let ((s (make-higher-sort)))
	(set (higher-sort-name s) name)
	(set (higher-sort-range s) range)
	(set (higher-sort-domains s) domains)
	(set (higher-sort-bases s) (set-union (base-sorts range)
					      (collect-set base-sorts domains)))
	(set (higher-sort-category s) (sort-category range))
	(set (higher-sort-type s)
	     (cond (name 
		    (build-maximal-higher-sort (map type-of-sort domains)
					       (type-of-sort range)))
		   ((and (sort-is-type? range)
			 (every? sort-is-type? domains))
		    s)
		   (else (build-maximal-higher-sort (map type-of-sort domains)
						    (type-of-sort range)))))
	(set (higher-sort-enclosing-sort s) enclosing-sort)
	(set (higher-sort-level s)
	     (1+ (apply max (sort-level range) (map sort-level domains))))
	(set (higher-sort-alpha-hash s) (descriptor-hash s))
	(push (table-entry higher-sort-table range) s)
	s)))

(define (higher-sort-exhausts-domains->range? s)
  (false? (higher-sort-name s)))


(define PRINT-SORTS-VERBOSELY?
  (make-simple-switch 'print-sorts-verbosely? boolean? '#f))

(define SORTING? sort?)

(lset *SORTING-LUBS* (make-two-d-table '*SORTING-LUBS*))

(define (sorting-lub sort1 sort2)
  (if (not (eq? (type-of-sort sort1)
		(type-of-sort sort2)))
      (imps-error "sorting-lub  Attempt to take LUB of incomparable sorts ~S~&~S."
		  sort1 sort2))
  (let ((value (or (two-d-table-entry *sorting-lubs* sort1 sort2)
		   (two-d-table-entry *sorting-lubs* sort2 sort1))))
    (or value
	(let ((value (compute-sorting-lub sort1 sort2)))
	  (set (two-d-table-entry *sorting-lubs* sort1 sort2) value)
	  value))))

(define (sorting-leq sort1 sort2)
  (and (eq? (type-of-sort sort1)
	    (type-of-sort sort2))
       (eq? sort2 (sorting-lub sort1 sort2))))

(define (compute-sorting-lub sort1 sort2)
  ;;
  ;; Precondition:  sort1 and sort2 have the same type.
  ;; 
  (cond ((eq? sort1 sort2) sort1)
	((sort-is-type? sort1)
	 sort1)
	((sort-is-type? sort2)
	 sort2)
	((and (name sort1)
	      (name sort2))
	 (let ((e1 (immediately-enclosing-sorts sort1))
	       (e2 (immediately-enclosing-sorts sort2)))
	   (iterate iter ((e1 e1))
	     (cond ((memq? (car e1) e2) (car e1))
		   ((null? (cdr e1)) (sorting-lub (car e1) (last e2)))
		   (else (iter (cdr e1)))))))
	((name sort1)
	 (sorting-lub (enclosing-sort sort1) sort2))
	((name sort2)
	 (sorting-lub sort1 (enclosing-sort sort2)))
	(else						;must be higher maximal non-type
	 (let ((domains (map sorting-lub
			     (higher-sort-domains sort1)
			     (higher-sort-domains sort2)))
	       (range (sorting-lub (higher-sort-range sort1)
				   (higher-sort-range sort2))))
	   (build-maximal-higher-sort domains range)))))       

(define (immediately-enclosing-sorts sort1)
  (if (and (not (sort-is-type? sort1))
	   (name sort1))				;if it's a subsort
      (cons sort1					;cons it onto the others
	    (immediately-enclosing-sorts (enclosing-sort sort1)))
      (list sort1)))					;else we're done...

(define SORT-NECESSARILY-INCLUDED? sorting-leq)

;; (SORTS-MAY-OVERLAP? sort1 sort2) means that e.g. equations between terms of
;; these types are permitted.  

(define (SORTS-MAY-OVERLAP? sort1 sort2)
  (eq? (type-of-sort sort1)
       (type-of-sort sort2)))

;; (SORTS-NECESSARILY-DISJOINT? sort1 sort2) means that there is no overlap in
;; any interpretation of the theory.  

(define (SORTS-NECESSARILY-DISJOINT? sort1 sort2)
  (not (sorts-may-overlap? sort1 sort2)))

;;(or (not (eq? (sort-category sort1)
;;		(sort-category sort2)))
;;      (and (not (floating-sort? sort1))
;;	   (not (floating-sort? sort2))
;;	   (not (eq? (type-of-sort sort1)
;;		     (type-of-sort sort2))))
;;      (and (floating-sort? sort1)			;floaters only float up
;;	   (not (floating-sort? sort2))
;;	   (< (sort-level sort2)
;;	      (sort-level sort1)))
;;      (and (floating-sort? sort2)			;floaters only float up
;;	   (not (floating-sort? sort1))
;;	   (< (sort-level sort1)
;;	      (sort-level sort2))))
  
(define-integrable (EQUAL-SORTINGS? s1 s2) (eq? s1 s2))
(define SORTINGS-EQUAL? EQUAL-SORTINGS?)

(define (collect-named-sorts sort)
  (iterate iter ((sorts (list sort))
		 (named-sorts '()))
    (cond ((null? sorts) (reverse! named-sorts))
	  ((not (null? (name (car sorts))))
	   (iter (cdr sorts)
		 (add-set-element (car sorts) named-sorts)))
	  (else
	   (iter (append (higher-sort-domains (car sorts))
			 (cons (higher-sort-range (car sorts))
			       (cdr sorts)))
		 named-sorts)))))

(define (SORT->LIST h)
  (or (name h)
      (append (map sort->list (higher-sort-domains h))
	      (list (sort->list (higher-sort-range h))))))

(define (HEREDITARY-SORT-DOMAINS sorting)
  (if (base-sort? sorting)
      nil
      (let ((rest (hereditary-sort-domains (higher-sort-range sorting))))
	(if (null? rest)
	    (higher-sort-domains sorting)
	    (append (higher-sort-domains sorting) rest)))))

(define (HEREDITARY-TYPE-DOMAINS sorting)
  (map type-of-sort (hereditary-sort-domains sorting)))

(define (SORTING-PERMISSIBLE?  sorting the-home-language)
  (let ((bases (and the-home-language (base-sorts the-home-language))))
    (let ((check-language?
	   (if the-home-language			;if home language is passed,
	       (lambda (s)				;sort symbols must be in it
		 (memq? s bases))
	       (lambda (()) '#t))))
      (if (base-sort? sorting)
	  (check-language? sorting)
	  (and
	   (higher-sort? sorting)
	   (iterate make-checks? ((sorts (cons (higher-sort-range sorting)
					       (higher-sort-domains sorting))))
	     (cond ((null? sorts) '#t)
		   ((base-sort? (car sorts))
		    (and (check-language? (car sorts))
			 (make-checks? (cdr sorts))))
		   ((higher-sort? (car sorts))
		    (make-checks?
		     (cons (higher-sort-range (car sorts))
			   (append 
			    (higher-sort-domains (car sorts))
			    (cdr sorts)))))
		   (else '#f))))))))
	

							; SAME-SORTED? applies to two expressions and returns true if they
							; have the same sorting. 

(define (SAME-SORTED? exp1 exp2)
  (equal-sortings? (expression-sorting exp1)
		   (expression-sorting exp2)))

							; SINGLE-SORTED-LIST? takes a single argument, a list of expressions, and
							; returns true if all its elements have the same sorting.

(define (SINGLE-SORTED-LIST? exps)
  (or (null? exps)
      (let ((sorting (expression-sorting (car exps))))
	(every?
	 (lambda (expr) (equal-sortings? sorting (expression-sorting expr))) 
	 (cdr exps)))))

(define-constant SORTING-SKELETONS-MATCH? sorts-may-overlap?)

(define-constant SORTINGS-MATCH-EXACTLY? eq?)
	   
(define-predicate sort-resolver?)

							; NAME->SORT "reads" a sort name (i.e. a symbol) into the sort structure
							; associated with that name in the language or sort-resolver.  

(define-operation  (name->sort sort-resolver symbol)
  (imps-error "name->sort: operation not handled ~S ~S" sort-resolver symbol))

(define-operation (sort-names-resolved resolver)
  (imps-error "sort-names-resolved: operation not handled ~S" resolver))

(define-operation (sorts-resolved resolver)
  (imps-error "sorts-resolved: operation not handled ~S" resolver))

(define-operation (name-sort-alist resolver)
  (imps-error "name-sort-alist: operation not handled ~S" resolver))

(define (make-type-resolver ind-floater-names prop-floater-names)
  (let ((names-resolved
	 (set-union (list 'ind 'prop)
		    (append prop-floater-names ind-floater-names))))
    (imps-enforce (object (lambda (l) (and (every? (lambda (n) (or (symbol? n)
								   (anonymous-name? n)))
						   l)
					   (is-set? l)))
		    ((print self port)
		     (format
		      port
		      "#{Procedure: tests argument is a set of symbols and anonymous names}"))) 
		  names-resolved)
    (let* ((name-type-alist
	    (cons*
	     (cons 'ind ind)
	     (cons 'prop prop)
	     (append
	      (map
	       (lambda (n)
		 (cons n (build-base-sort n prop '#t '#t)))
	       prop-floater-names)
	      (map
	       (lambda (n)
		 (cons n (build-base-sort n ind '#t '#t)))
	       ind-floater-names))))
	   (sort-set
	    (set-union (list ind prop)
		       (map cdr name-type-alist))))
      (object '()
	((name->sort self symbol)
	 (cond ((assq symbol name-type-alist)
		=> cdr)
	       (else '#f)))
	((sort-resolver? self) '#t)
	((name-sort-alist self) name-type-alist)
	((sort-names-resolved self) names-resolved)
	((base-sorts self) sort-set)
	((sorts-resolved self) sort-set)))))

(define NULL-SORT-RESOLVER
  (make-type-resolver '() '()))

(define (extend-sort-resolver resolver new-name sorting)
  (let ((names-resolved
	 (add-set-element new-name (sort-names-resolved resolver)))
	(bases (if (base-sort? sorting)
		   (add-set-element sorting (base-sorts resolver))
		   (base-sorts resolver)))
	(sort-set (add-set-element sorting (sorts-resolved resolver)))
	(alist (cons (cons new-name sorting)
		     (name-sort-alist resolver))))
    (imps-enforce is-set? names-resolved)
    (imps-enforce (lambda (n) (or (symbol? n)
				  (anonymous-name? n)))
		  new-name)
    (object '()
      ((sort-resolver? self) '#t)
      ((sort-names-resolved self) names-resolved)
      ((name-sort-alist self) alist)
      ((base-sorts self) bases)
      ((sorts-resolved self) sort-set)
      ((name->sort self symbol)
       (cond ((assq symbol alist)
	      => cdr)
	     (else '#f))))))

(define (rename-sort-resolver resolver renamer)
  (or (eq? 'prop (renamer 'prop))
      (imps-error "rename-sort-resolver:  Bad renamer maps 'PROP to ~S"
		  (renamer 'prop)))
  (let ((names-resolved
	 (map renamer (names-resolved resolver)))
	(alist (map (lambda (p)
		      (cons (renamer (car p))
			    (rename-sort (cdr p) renamer)))
		    (name-sort-alist resolver)))
	(bases (map (lambda (s) (rename-sort s renamer))
		    (base-sorts resolver)))
	(sort-set (map cdr alist)))
    (imps-enforce is-set? names-resolved)
    (object '()
      ((sort-resolver? self) '#t)
      ((sort-names-resolved self) names-resolved)
      ((name-sort-alist self) alist)
      ((base-sorts self) bases)
      ((sorts-resolved self) sort-set)
      ((name->sort self symbol)
       (cond ((assq symbol alist)
	      => cdr)
	     (else '#f))))))

(define (join-sort-resolvers resolvers)
  (imps-enforce (lambda (l)
		  (every? sort-resolver? l))
		resolvers)
  (cond ((null? resolvers)
	 null-sort-resolver)
	((null? (cdr resolvers))
	 (car resolvers))
	(else
	 (let* ((names-resolved
		 (collect-set sort-names-resolved resolvers)) 
		(sort-set (collect-set sorts-resolved resolvers))
		(bases (collect-set base-sorts resolvers))
		(alist (map
			(lambda (n)
			  (cons n (any
				   (lambda (r) (name->sort r n))
				   resolvers)))
			names-resolved)))
	   ;; do extra check here
	   (object '()
	     ((sort-resolver? self) '#t)
	     ((base-sorts self) bases)
	     ((sorts-resolved self) sort-set)
	     ((name-sort-alist self) alist)
	     ((name->sort self symbol)
	      (cond ((assq symbol alist)
		     => cdr)
		    (else '#f)))
	     ((sort-names-resolved self) names-resolved))))))
     
(define (subtract-sort-resolvers res1 res2)
  (let ((names-resolved
	 (set-difference (sort-names-resolved res1)
			 (sort-names-resolved res2)))
	(sort-set (set-difference (sorts-resolved res1)
				  (sorts-resolved res2)))
	(bases (set-difference (base-sorts res1)
			       (base-sorts res2)))
	(alist (let ((a1 (name-sort-alist res1))
		     (names2 (sort-names-resolved res2)))
		 (iterate iter ((alist '())
				(a1 a1))
		   (cond ((null? a1) (reverse! alist))
			 ((memq? (caar a1)		;first name
				 names2)
			  (iter alist (cdr a1)))
			 (else
			  (iter (cons (car a1) alist)
				(cdr a1))))))))
		     
    (object '()
      ((sort-resolver? self) '#t)
      ((base-sorts self) bases)
      ((sorts-resolved self) sort-set)
      ((name-sort-alist self) alist)
      ((name->sort self symbol)
       (cond ((assq symbol alist)
	      => cdr)
	     (else '#f)))
      ((sort-names-resolved self) names-resolved))))

(define (MAKE-SORT-RESOLVER-FROM-NAMED-SORTS sorts)
  (let ((resolver null-sort-resolver))
    (iterate loop ((sorts sorts))
      (if (null? sorts)
	  resolver
	  (let ((sort (car sorts)))
	    (extend-sort-resolver resolver (name sort) sort)
	    (loop (cdr sorts)))))))

(define (list->sort resolver lst)
  (if (or (symbol? lst)
	  (anonymous-name? lst))
      (name->sort resolver lst)
      (let ((component-sorts
	     (map (lambda (lst2)
		    (list->sort resolver lst2))
		  lst)))
	(receive (range domains)
 	  (last-&-all-but-last component-sorts) 
 	  (if (and (sort? range)
		   (every? sort? domains))
	      (build-maximal-higher-sort domains range)
	      '#f)))))

(define (BUILD-NEW-SORT enclosing-sort sort-name)
  (if (base-sort? enclosing-sort)
      (build-base-sort
       sort-name
       (sort-category enclosing-sort)
       '#f
       enclosing-sort)
      (build-higher-sort
       (higher-sort-domains enclosing-sort)
       (higher-sort-range enclosing-sort)
       sort-name
       enclosing-sort)))

(define (ENCLOSING-SORT sort)
  (cond ((sort-is-type? sort) sort)
	((base-sort? sort) (base-sort-enclosing-sort sort))
	(else 
	 (higher-sort-enclosing-sort sort))))

(define (COMMON-ENCLOSING-SORT sort1 sort2)
  (sorting-lub sort1 sort2))

(define (COMMON-ENCLOSING-SORT-LIST sort-list1 sort-list2)
  (iterate iter ((s1 sort-list1) (s2 sort-list2) (new-sort-list '()))
    (if (null? s1)
	(reverse new-sort-list)
	(iter (cdr s1) 
	      (cdr s2) 
	      (cons (common-enclosing-sort (car s1) (car s2))
		    new-sort-list)))))


(define NOMINAL-UPPER-BOUND common-enclosing-sort)

(define (ENCLOSING-MAXIMAL-HIGHER-SORT sort)
  (or (higher-sort? sort)
      (imps-error "ENCLOSING-MAXIMAL-HIGHER-SORT: ~S ~A."
		  sort "is not a higher sort"))
  (if (name sort)
      (build-maximal-higher-sort (higher-sort-domains sort) (higher-sort-range sort))
      sort))

(define (CONTAINS-SORT? resolver sort)
  (subset? (collect-named-sorts sort) (sorts-resolved resolver)))

(define (BIG-SORTING-LUB sorts)
  (or (not (null? sorts))
      (imps-error "BIG-SORTING-LUB: ~S is null." sorts))
  (iterate loop ((sorts (cdr sorts)) (lub (car sorts)))
    (if (null? sorts)
	lub
	(loop (cdr sorts) (sorting-lub lub (car sorts))))))

(define (SORTING-LIST-LEQ sort-list1 sort-list2)
  (or (= (length sort-list1) (length sort-list2))
      (imps-error "SORTING-LIST-LEQ: ~S and ~S have different lengths."
		  sort-list1 sort-list2))
  (iterate loop ((sort-list1 sort-list1) (sort-list2 sort-list2))
    (or (null? sort-list1)
	(and (sorting-leq (car sort-list1) (car sort-list2))
	     (loop (cdr sort-list1) (cdr sort-list2))))))
	
(define (make-sorting-hash-table . maybe-id)
  (apply make-hash-table sort? sort-alpha-hash eq? '#f maybe-id))
	 
