;; Copyright (c) 1990-1994 The MITRE Corporation
;; 
;; Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;;   
;; The MITRE Corporation (MITRE) provides this software to you without
;; charge to use, copy, modify or enhance for any legitimate purpose
;; provided you reproduce MITRE's copyright notice in any copy or
;; derivative work of this software.
;; 
;; This software is the copyright work of MITRE.  No ownership or other
;; proprietary interest in this software is granted you other than what
;; is granted in this license.
;; 
;; Any modification or enhancement of this software must identify the
;; part of this software that was modified, by whom and when, and must
;; inherit this license including its warranty disclaimers.
;; 
;; MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;; OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;; OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;; FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;; SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;; SUCH DAMAGES.
;; 
;; You, at your expense, hereby indemnify and hold harmless MITRE, its
;; Board of Trustees, officers, agents and employees, from any and all
;; liability or damages to third parties, including attorneys' fees,
;; court costs, and other related costs and expenses, arising out of your
;; use of this software irrespective of the cause of said liability.
;; 
;; The export from the United States or the subsequent reexport of this
;; software is subject to compliance with United States export control
;; and munitions control restrictions.  You agree that in the event you
;; seek to export this software or any derivative work thereof, you
;; assume full responsibility for obtaining all necessary export licenses
;; and approvals and for assuring compliance with applicable reexport
;; restrictions.
;; 
;; 
;; COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(provide 'theory-information)
(defun make-obarray ()
  (make-vector 509 0))

(defconst imps-obarray (make-obarray))

(defconst imps-intern-TAGS-syntax-table
  (let ((table (copy-syntax-table scheme-mode-syntax-table)))
    (modify-syntax-entry ?, "." table)
    (modify-syntax-entry ? "." table)
    table))


(defun imps-intern-properties (sym)
  "Gross dynamic binding hack.  kind, dir, file, line must be global in environment.  "
  (if (eq kind 'theory-ensemble)
      'ignore 
    (push-prop sym 'kind kind)
    (push-prop sym 'directory dir)
    (push-prop sym 'file file)
    (push-prop sym 'line line)))

(defun push-prop (sym prop value)
  (put sym prop
       (cons value
	     (get sym prop))))

(defun get-prop-nth (sym prop n)
  (nth
   n
   (get sym prop)))

(defvar imps-find-definition-offset 0)
(defvar imps-last-definition '())

(defun imps-find-definition (sym &optional next)
  (interactive
   (if current-prefix-arg
       (list imps-last-definition t)
     (list 
      (intern-soft
       (completing-read  "Name: " imps-obarray 'always t nil)
       imps-obarray)
      nil)))
  (setq imps-find-definition-offset
	(if next      
	    (1+ imps-find-definition-offset)
	  0))
  (setq imps-last-definition sym)
  (let ((remaining (- (length (get sym 'directory))
		      (1+ imps-find-definition-offset))))
    (if (>= remaining 0)
	(let ((directory (get-prop-nth sym 'directory imps-find-definition-offset))
	      (file (get-prop-nth sym 'file imps-find-definition-offset))
	      (line (get-prop-nth sym 'line imps-find-definition-offset)))
	  (if (not file)
	      (error "%s definition not found."
		       sym)
	    (condition-case bad-name 
		(progn
		  (find-file-other-window
		   (expand-file-name
		    (substitute-in-file-name
		     (format "$THEORIES/%s%s"
			      directory
			      file))))
		  (goto-line line)
		  (beginning-of-defun 1)
		  (if (> remaining 0)
		      (message "%d additional entries under %s" remaining sym)))
	      (error (message "Loaded interactively, source file unknown")))))
      (setq imps-find-definition-offset 0)
      (error "No more entries for %s" sym))))

(defun imps-loaded-p (thm-name)
  (let ((sym (intern-soft thm-name imps-obarray)))
    (not
     (not
      (get-literal-from-tea
       (format "(and (file-loaded? \"%s\" imps-implementation-env) 1)"
	       (expand-file-name
		(substitute-in-file-name
		 (format "$THEORIES/%s%s"
			 (get-prop-nth sym 'directory 0)
			 (get-prop-nth sym 'file 0))))))))))

(fset 'imps-theorem-loaded-p 'imps-loaded-p)

(defun imps-require-theorem (thm-name)
  (interactive
   (list
    (completing-read "Theorem name: " imps-obarray 'always t nil)))
  (let ((sym (intern-soft thm-name imps-obarray)))
    (if (y-or-n-p
	 (format "Really require file $THEORIES/%s%s? "
		 (get-prop-nth sym 'directory 0)
		 (get-prop-nth sym 'file 0)))
	(tea-eval-expression
	 (format "(*require nil '(theories %s%s) imps-implementation-env)"
		 (get-prop-nth sym 'directory 0)
		 (get-prop-nth sym 'file 0)))
      (error "Not requiring file $THEORIES/%s%s."
	     (get-prop-nth sym 'directory 0)
	     (get-prop-nth sym 'file 0)))))

(defun imps-xview-theorem (theorem)
  "Run xview on the theorem named THEOREM."
  (interactive
   (list (completing-read "Theorem name: " imps-obarray 'kind-is-theorem-p nil nil)))
  (tea-eval-expression
   (format "(cond ((name->theorem-1 '%s) => xview)(else (imps-warning \"theorem %s not loaded\")))" theorem theorem)))

(defun imps-xview-macete (macete)
  "Run xview on the macete named MACETE."
  (interactive
   (list (read-macete)))
  (tea-eval-expression
   (format "(cond ((name->macete '%s) => xview)
		  (else (imps-warning \"theorem %s not loaded\")))"
	    macete macete)))

(defun imps-xview-sqns (sqn-nos)
  (interactive
   (list (read-from-minibuffer "Sqn nos: " nil nil nil)))
  (tea-eval-expression
   (format
    "(xview
      (map
       (lambda (sqn-no)
	 (sequent-unhash-in-graph-by-number sqn-no %d))
       '(%s)))"
    dg-number sqn-nos)))

(defun imps-get-theorem-var-sorts (thm-name)
  (let ((sym (intern thm-name imps-obarray)))
    ;;
    ;;  (or (get sym 'var-sorts)
    ;;  
    (let ((var-sorts
	   (get-literal-from-tea
	    (format
	     "(theorem-name->var-sort-list '%s)"
	     thm-name))))
      (if (listp var-sorts)
	  (put (intern thm-name imps-obarray)
	       'var-sorts
	       var-sorts)
	(error "Theorem %s not loaded.  Consider M-x imps-require-theorem."
	       thm-name)))
    ;;
    ;; )
    ;;
    ))

(defun kind-is-translation-p (sym) (memq 'translation (get sym 'kind)))
(defun kind-is-theory-p (sym) (memq 'theory (get sym 'kind)))
(defun kind-is-theorem-p (sym) (memq 'theorem (get sym 'kind)))
(defun kind-is-inductor-p (sym) (memq 'inductor (get sym 'kind)))
(defun kind-is-macete-p (sym)
  (let ((kind (get sym 'kind)))
    (or (memq 'theorem (get sym 'kind))
	(memq 'compound-macete kind)
	(memq 'schematic-macete kind))))
(defun kind-is-bnf-p (sym) (memq 'bnf (get sym 'kind)))


 
(defun complete-macete-name (str)
  (if (and (<= 3 (length str))
	   (string= "tr%" (substring str 0 3)))
      (let ((completion (try-completion (substring str 3) imps-obarray)))
	(if (stringp completion)
	    (concat "tr%" completion)
	  completion))
    (try-completion str imps-obarray 'kind-is-macete-p)))
	  
(defun all-macete-completions (str)
  (append
   (mapcar '(lambda (str2)
	      (concat "tr%" str2))
	   (and (<= 3 (length str))
		(string= "tr%" (substring str 0 3))
		(all-completions (substring str 3) imps-obarray)))
   (all-completions str imps-obarray 'kind-is-macete-p)))

(defun insert-complete-macete-name ()
  (interactive)
  (let ((str (buffer-string)))
    (let ((completion (complete-macete-name str)))
      (cond ((or (eq t completion)
		 (string= completion str))
	     (with-output-to-temp-buffer " *Completions*"
	       (display-completion-list (all-macete-completions str))))
	    ((stringp completion)
	     (erase-buffer)
	     (insert completion))
	    ((null completion)
	     (ding)
	     (message "No match."))))))
	   
(defconst macete-minibuffer-map (copy-keymap minibuffer-local-map))
(define-key macete-minibuffer-map " " 'insert-complete-macete-name)
(define-key macete-minibuffer-map "C-m" 'insert-complete-macete-name)
  
(defun read-macete ()
  (interactive)
  (let ((minibuffer-local-map macete-minibuffer-map))
    (read-minibuffer "Macete name: ")))


(defvar imps-commands nil
  "*The IMPS commands known currently known to emacs, 
presented as a list suitable for completing-read.")
(defconst imps-commands-file
  (expand-file-name
	   (substitute-in-file-name "$IMPS/../etc/imps-commands")))
(defun imps-read-commands-from-file ()
  (let ((tmp-buff (get-buffer-create " *imps-commands-tmp*")))
    (set-buffer tmp-buff)
    (erase-buffer)
    (if (file-readable-p imps-commands-file)
	(insert-file imps-commands-file)
      (error "IMPS command file unreadable.  Try executing (t-e-write-commands) in IMPS."))
    (goto-char (point-min))
    (setq imps-commands
	  (read (current-buffer)))))

(defun add-imps-command (command-string retrieval-protocol-symbol)
  (setq imps-commands
	(cons
	 (list command-string retrieval-protocol-symbol 
	       'default-argument-transmission-protocol)
	 imps-commands)))

(defun imps-save-obarray ()
  (interactive)
  (or (file-directory-p
       (expand-file-name
	(substitute-in-file-name
	 "~/imps/theories")))
      (error "No directory ~/imps/theories.  Please create and re-execute"))
  (if (y-or-n-p
       "Really create personal IMPS obarray? ")
      (write-imps-obarray (expand-file-name
			   (substitute-in-file-name
			    "~/imps/theories/obarray")))
    (error "Not written.")))

(defun write-imps-obarray (filename)
  (let ((buff (find-file-noselect filename)))
    (set-buffer buff)
    (erase-buffer)
    (mapatoms 'write-imps-atom imps-obarray)
    (save-buffer)))
     

(defun write-imps-atom (sym)
  (insert
   "("
   (symbol-name sym)
   " (kind "
   (format "%s" (get sym 'kind))
   ") (directory "
   (format "%s" (get sym 'directory))
   ") (file "
   (format "%s" (get sym 'file))
   ") (line "
   (format "%s" (get sym 'line))
   "))\n"))


(defun read-imps-obarray (filename)
  (let ((buff (get-buffer-create " imps obarray"))
	(next-end (point-min)))
    (set-buffer buff)
    (insert-file-contents filename t)
    (set-syntax-table scheme-mode-syntax-table)
    (goto-char next-end)
    (while (setq next-end (scan-sexps (point) 1))
      (narrow-to-region (point) next-end)
      (down-list 1)
      (let* ((bounds (next-symbol-boundaries (point)))
	     (sym (intern (buffer-substring (car bounds) (cdr bounds)) imps-obarray)))
	(goto-char (cdr bounds))
	(condition-case done
	    (while t
	      (let ((next-sexp (read buff)))
		(put sym (car next-sexp) (car (cdr next-sexp)))))
	  (invalid-read-syntax (widen)))))))

(defun read-new-obarray-entry (buffer)
  (let* ((name (read buffer))
	 (kind (read buffer))
	 (dir (read buffer))
	 (file (read buffer))
	 (line (read buffer))
	 (def-form-name (read buffer))
	 (section-name
	  (condition-case la 
	      (read buffer)
	    (error nil))))
    (let ((sym (intern  (prin1-to-string name) imps-obarray)))
      (if (memq section-name (get sym 'section))
	  nil
	(push-prop sym 'kind kind)
	(push-prop sym 'directory dir)
	(push-prop sym 'file file)
	(push-prop sym 'line line)
	(push-prop sym 'def-form-name def-form-name)
	(push-prop sym 'section section-name)))))

(defun augment-imps-obarray-from-file (filename)
  (interactive "fSection auxiliary filename: ")
  (let ((buff (get-buffer-create " imps obarray"))
	(next-end (point-min)))
    (set-buffer buff)
    (erase-buffer)
    (if (not (file-readable-p filename))
	nil
      (insert-file-contents filename t)
      (set-buffer-modified-p nil)
      (set-syntax-table scheme-mode-syntax-table)
      (goto-char next-end)
      (while (search-forward "#{" nil 0)
	(forward-sexp 1)
	(read-new-obarray-entry buff)))
    (kill-buffer buff)))
	    
(defun imps-rebuild-obarray () 
  (interactive)
  (or (and (y-or-n-p "Really rebuild imps obarray? ")
	   (or (string= (user-login-name) "guttman")
	       (y-or-n-p "You're not Guttman: are you sure? ")))
      (error "Not rebuilt."))
  (setq imps-obarray (make-obarray))
  (imps-intern-from-tags-1 imps-tags-file)
  (write-imps-obarray (expand-file-name
			(substitute-in-file-name
			 "$IMPS/../etc/obarray"))))


(define-key scheme-mode-map "\C-c." 'imps-find-definition)
(define-key scheme-mode-map "\C-cd" 'imps-process-and-send-def-form)
(define-key inferior-tea-mode-map "\C-c." 'imps-find-definition)

(modify-syntax-entry ? " " scheme-mode-syntax-table)

;; I'm changing this to t so that we may rely on the call to
;; maybe-install-emacs-obarray in finish-load as defined in $IMPS/imps.t 

(defvar imps-symbols-loaded t)

(or
 imps-symbols-loaded
 (progn
   (message "Reading IMPS symbols...")
   (let ((user-obarray
	  (and (file-readable-p
		(expand-file-name
		 (substitute-in-file-name
		  "~/imps/theories/obarray"))))))
     (if (and user-obarray
	      (y-or-n-p "Load personal IMPS obarray? "))
	 (progn (message "Loading personal IMPS obarray.")
		(read-imps-obarray (expand-file-name
				    (substitute-in-file-name
				     "~/imps/theories/obarray"))))
       (message "Loading IMPS system symbols.")
       (read-imps-obarray (expand-file-name
			   (substitute-in-file-name
			    "$IMPS/../etc/obarray")))))
   (setq imps-symbols-loaded t)))

(imps-read-commands-from-file)

;; 
;; (message "Interning names from IMPS tags tables...")
;; (imps-intern-from-tags)
;; (message "Interning names from IMPS tags tables... Done.")
